/*
 * java.lang.Thread: part of the Java Class Libraries project.
 * Copyright (C) 1998, 2001 Free Software Foundation
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */

package java.lang;

/**
 ** Thread represents a single thread of execution in the VM.
 ** When an application VM starts up, it creates a new Thread
 ** which calls the main() method of a particular class.  There
 ** may be other Threads running, such as the garbage collection
 ** thread.<P>
 **
 ** Threads have names to identify them.  These names are not
 ** necessarily unique.<P>
 **
 ** Every Thread has a priority, as well, which tells the VM
 ** which Threads get more running time.<P>
 **
 ** There are two methods of creating a Thread: you may
 ** subclass Thread and implement the <CODE>run()</CODE> method, at which
 ** point you may start the Thread by calling its <CODE>start()</CODE>
 ** method, or you may implement <CODE>Runnable</CODE> in the class you
 ** want to use and then call new <CODE>Thread(your_obj).start()</CODE>.
 **
 ** @specnote it is unclear at what point a Thread should be added to a
 **           ThreadGroup, and at what point it should be removed.
 **           Should it be inserted when it starts, or when it is
 **           created?  Should it be removed when it is suspended or
 **           interrupted?  The only thing that is clear is that the
 **           Thread should be removed when it is stopped.
 ** @author John Keiser
 ** @version 1.1.0, Aug 6 1998
 ** @since JDK1.0
 **/

public class Thread implements Runnable {


    /*From kissme */

   private ThreadGroup   _group;
   private Runnable      toRun;
   private String 	    _name;

	/** The maximum priority for a Thread.
	 ** @XXX find out the value for this.
	 **/
	public static final int MAX_PRIORITY  = 100;

	/** The priority a Thread gets by default.
	 ** @XXX find out the value for this.
	 **/
	public static final int NORM_PRIORITY = 50;

	/** The minimum priority for a Thread.
	 ** @XXX find out the value for this.
	 **/
	public static final int MIN_PRIORITY = 0;

	static int numAnonymousThreadsCreated = 0;

   static int numThreadsCreated = 0;

  /* Static init and native methods */
  
  //  private native void _createJNIData(); 
  

  /* The currently executing thread. */
  private static Thread _currentThread;  //HmmM!

  /* C initialisation routine */
  private static native void _initC();

  /* Static initialiser to set the C variables */
  static
  {
    _initC();
    ThreadGroup.root.setMaxPriority(MAX_PRIORITY);
  }


   /* constructors *********************************************************/
  
  /* This constructor code is peculiar in that it has to be executed before 
     there is an active thread - thus, the very first time a thread is 
     created, currentThread etc. are invalid. That's no problem, however, 
     since we know what we are doing. 
     The interpreter must know not to try to switch threads in this case. 
     When classes are loading and being initialised it runs in single-thread 
     mode. When an application is running, it runs in multi-threaded mode. 
     */
  
  public Thread()
  {
    this(null, null, null);
  }


  public Thread(Runnable target)
  {
    this(null, target);
  }


  public Thread(ThreadGroup group,  Runnable target)
  {
    this(group, target, "Thread-" + (++numThreadsCreated));
  }
  
  public Thread(String name)
  {
	this(null, null, name);
  }
  
  public Thread(ThreadGroup group, String name)
  {
	this( group, null, name);
  }
  
  public Thread(Runnable target, String name)
  {
    this( null, target, name);
  }
  
  public Thread(ThreadGroup group,
		Runnable target,
		String name)
  {
    
    toRun = target;

    if (name == null)
      {
	_name = new String("Thread");
      }
    else
      {
	_name = name;
      }

    if (group == null)
      {
	_group = ThreadGroup.root;
      }
    else
      {
	_group = group;
      }
    
	_group.addThread(this);

    }


    /* public methods *******************************************************/ 

    public static int activeCount() 
    { 
	if (_currentThread._group == null) 
	    { 
		return 1; 
	    } 
	else 
	    { 
		return _currentThread._group.activeCount(); 
	    } 
    } 
 
    public void checkAccess() 
    { 
	if (currentThread()._group != _group) 
	    { 
		throw( new SecurityException()); 
	    } 
    } 
 
 
    //I want to change this for native threads, we do a native call and get the real thread object
    //Or we make it _currentThread non-static -- hmmmm

    public static Thread currentThread() 
    {
	return nativeCurrentThread();
    }
    //ok here we go
    public static native Thread nativeCurrentThread();
 
    public native void destroy(); 
 
    public static void dumpStack() {} 
 
    public static int enumerate(Thread tarray[]) 
    { 
	if (_currentThread._group == null) 
	    { 
		tarray[0] = _currentThread; 
		return 1; 
	    } 
	else 
	    { 
		return _currentThread._group.enumerate(tarray); 
	    } 
    } 
 
    public final String getName() 
    { 
	return _name; 
    } 
 
    public final ThreadGroup getThreadGroup() 
    { 
	return _group; 
    } 
 
    public final int getPriority() 
    { 
	return MAX_PRIORITY; 
    } 
 
    public synchronized void interrupt() 
    { 

    } 
 
    public final native boolean isAlive(); 
 
  
    public final boolean isDaemon() 
    { 
      return nativeIsDaemon(); 
    } 

    public final void setDaemon(boolean on) 
    { 
      nativeSetDaemon(on);
    }

    public native boolean nativeIsDaemon();
    public native void nativeSetDaemon(boolean on);
 
    public boolean isInterrupted() 
    { 
      return false;
    } 
 
    public final void join() throws InterruptedException 
    { 
	join(0,0); 
    } 
 
    public final synchronized void join(long timeout, int nanos) throws InterruptedException 
    { 
	if (nanos >= 500000 || (nanos != 0 && timeout == 0)) 
	    { 
		timeout++; 
	    } 
	join(timeout); 
    } 
 
    public final native void join(long millis) throws InterruptedException; 
 
    public final native void resume(); 
 
    /** The method of Thread that will be run if there is no Runnable
     ** object associated with the Thread.<P>
     **
     ** Thread's implementation does nothing at all.
     **/

    public void run() {
        if (toRun != null)
            toRun.run();
    }
        


    public final void setName(String name)
    {
	_name = name; 
    }
    public final void setPriority(int newPriority) 
    {
	if ((newPriority < MIN_PRIORITY) || (newPriority > MAX_PRIORITY) )
	    {
		throw (new IllegalArgumentException());
	    }
	else
	    {
		
	    }
    } 
 
    public static void sleep(long millis) throws InterruptedException 
    { 
	    sleep(millis, 0); 
    } 
 
    public static native void sleep(long millis, int nanos) throws InterruptedException; 
    
    public synchronized native void start(); 
    
    public final synchronized void stop() 
    { 
	stop(new ThreadDeath()); 
    } 
 
    public final synchronized void stop(Object o) 
    { 

    } 
 
    public final native void suspend(); 
    
    public String toString()
    {
	if(_name == null)
	    return "kamikaze thread";
	if(_group != null)
	    return "Thread[" + _name + "," + MAX_PRIORITY + "," + _group.getName() + "]";
	else 
	    return "Thread[" + _name + "," + MAX_PRIORITY + ", nogroup]";
    }
 
    public static native void yield(); 

    protected void finalize()
    {
	_group.removeThread(this);
    }

	public ClassLoader getContextClassLoader() {
		return(this.getClass().getClassLoader()); // For now
	}

 


}
