#include <string.h>
#include <stdlib.h>
#include <stdio.h>

#include <glade/glade.h>
#include "kipina-i18n.h"
#include "kpsettings.h"
#include "kputil.h"

/**
 * kp_date_valid:
 * @date: A #KPDate
 *
 * Checks if this DMY is valid julian day.
 *
 * Returns: TRUE if date is valid and FALSE otherwise.
 */
gboolean
kp_date_valid (KPDate *date)
{
  return g_date_valid_dmy (date->d, date->m, date->y);
}


/**
 * kp_date_set_time:
 * @date: #KPDate
 * @time_: time value, just like time_t
 *
 * Used to create 
 */
void
kp_date_set_time (KPDate *date, GTime time_)
{
  GDate *gdate = g_date_new ();
  g_date_set_time (gdate, time_);
  /*TODO: wtf?-) */

  date->d = g_date_get_day (gdate);
  date->m = g_date_get_month (gdate);
  date->y = g_date_get_year (gdate);
}


/**
 * kp_date_new_dmy:
 * @d: Day
 * @m: Month (1-12)
 * @y: Year
 *
 * Just create a new #KPDate.
 * 
 * Returns: New #KPDate
 */
KPDate *
kp_date_new_dmy (guint d, guint m, guint y)
{
  KPDate *date;

  g_return_val_if_fail (g_date_valid_dmy (d, m, y), NULL);
  
  date = g_new0 (KPDate, 1);

  date->d = d;
  date->m = m;
  date->y = y;
  
  return date;
}


/**
 * kp_date_free:
 * @date: A #KPDate
 *
 * Free the memory used by @date.
 */
void
kp_date_free (KPDate *date)
{
  g_return_if_fail (date != NULL);
  g_free (date);
}


gchar *
kp_date_seconds_to_string (gdouble sec)
{
  GString *string;
  gboolean started = FALSE;
  gchar *ret;
  guint h;
  guint m;
  guint s;
  gdouble ms = 0;

  string = g_string_new (NULL);
  
  /* hours */
  h = (guint)(sec / 3600);
  if (h > 0) {
    started = TRUE;
    g_string_append_printf (string, "<b>%u</b> h", h);
  }

  m  = (guint)((sec - h * 3600) / 60);
  if (m > 0) {
    if (started)
      g_string_append_c (string, ' ');
    g_string_append_printf (string, "<b>%u</b> min", m);
    started = TRUE;
  }

  s = sec - h * 3600 - m * 60;

  if (s > 0) {
    if (started)
      g_string_append_c (string, ' ');
    g_string_append_printf (string, "<b>%u</b> s", s);
    started = TRUE;
  }
  
  if (h * 3600 + m * 60 + s < sec)
    ms = sec - (guint) sec;
 
  if (ms > 0) {
    if (started)
      g_string_append_c (string, ' ');

    g_string_append_printf (string, "<b>%u</b> ms", (guint)(ms * 1000));
  }
 
  ret = string->str;
  g_string_free (string, FALSE);
  
  return ret;
}

gchar *
kp_date_mseconds_to_std_string (guint32 msec)
{
  GString *string;
  gchar *ret;
  guint sec;
  guint h;
  guint m;
  guint s;
  guint ms;

  sec = msec / 1000;

  string = g_string_new (NULL);
 
  /* hours */
  h = (sec / 3600);
  if (h)
    g_string_append_printf (string, "%.2u:", h);

  m  = ((sec - h * 3600) / 60);
  g_string_append_printf (string, "%.2u:", m);

  s = sec - h * 3600 - m * 60;
  g_string_append_printf (string, "%.2u", s);
 
  /* (h * 3600 * 1000 + m * 60 * 1000 + s * 1000 < msec)*/

  ms = (msec % 1000) ? msec % 1000 : 0;
 
  if (ms > 0)
    g_string_append_printf (string, ".%.3u", ms);

  ret = string->str;
  g_string_free (string, FALSE);
  
  return ret;
}


/**
 * kp_is_valid_number:
 * @possible_number: String that may contain a valid numeric value
 * 
 * Find out if the string contains some numeric data.
 * 
 * Returns: TRUE if string contains numeric data
 * only and FALSE otherwise.
 */
gboolean
kp_is_valid_number (const gchar *possible_number)
{
  return ((gint) g_ascii_strtod (possible_number, NULL)) != 0;
}

/**
 * kp_number:
 * @possible_number: String that may contain a valid double value
 * 
 * Try to parse a string to a double.
 *
 * Returns: The double value if @possible_number is
 * valid and 0.0 otherwise.
 */
gdouble
kp_number (const gchar *possible_number)
{
  guint i=0;
  if (possible_number == NULL)
    return -1.0;
  
  while (possible_number[i] != '\0') {
    if (g_ascii_isdigit (possible_number[i]) != TRUE
     && possible_number[i] != ','
     && possible_number[i] != '.')
    {
      return -1.0;
    }

    i++;
  }
    
  return g_ascii_strtod (possible_number, NULL);
}

/*
 * Test if time string is in one of the following formats:
 *
 * If len is 2, it's assumed to be: MM
 * Otherwise, if there are no colons: HHMMSS
 * If there are colons, HH:MM:SS
 * If there are colons and dot: HH:MM:SS.ddd
 * If there are no colons, but dot: HHMMSS.ddd
 *
 * This way we can be RunLog 5.0-compatible. Unlike RunLog,
 * we support also milliseconds. (That should be enough for
 * most people ;)
 *
 * Returns: the number of seconds (Can be decimal number as milliseconds
 * exist too).
 */
gint32
kp_duration_str_to_ms (const gchar *time_str)
{
  gdouble val = 0.0;
  guint coeff[6] = { 1, 10, 60, 600, 3600, 36000 };
  guint len;
  guint n;
  guint i;
  gdouble num;
  gchar **str;
  gchar *tmp;

  g_return_val_if_fail (time_str != NULL, -1.0);
  
  len = strlen (time_str);

  /* MM */
  if (len == 1 || len == 2)
    return kp_number (time_str);

  tmp = g_strdup (time_str);

  /* .ddd */
  if (len > 4 && time_str[len - 4] == '.') {
    tmp[len - 4] = '\0';
    val = kp_number (&tmp[len - 3]);
    len = strlen (tmp);
  }

  /* HMM, HHMM, HMMSS, HHMMSS */
  if ((num = kp_number (tmp)) >= 1 && len <= 6) {
    for (i = len-1;;) {
      g_assert (i <= len-1);
      val += coeff[len-1-i] * g_ascii_digit_value (tmp[i]) * 1000;

      if (i==0)
        break;
      i--;
    }
    return val;
  }

  /* HH:MM, HH:MM:SS */
  /* Split the string */
  str = g_strsplit (tmp, ":", 4);
  for (n=1; str[n-1] != NULL; n++)
    if (n == 4) {
      val = -1;
      goto error;
    }
  
  /* Count the sum */
  for (i=0; i <= (n - 2) && str[i] != NULL;) {
    if ((num = kp_number (str[i])) >= 0) {
      if (num > 59) {
        val = -1;
        goto error;
      }
      val += coeff[n - i*2 + (n == 3)] * num * 1000;
    }
    if (i == (n - 2))
      break;
    i++;
  }

error:
  g_strfreev (str);
  g_free (tmp);
  return val;
}



/*
 * Test if time string is in one of the following formats:
 *
 * If len is 2, it's assumed to be: MM
 * Otherwise, if there are no colons: HHMMSS
 * If there are colons, HH:MM:SS
 * If there are colons and dot: HH:MM:SS.ddd
 * If there are no colons, but dot: HHMMSS.ddd
 *
 * This way we can be RunLog 5.0-compatible. Unlike RunLog,
 * we support also milliseconds. (That should be enough for
 * most people ;)
 *
 * Returns: the number of seconds (Can be decimal number as milliseconds
 * exist too).
 */
gdouble
kp_is_valid_duration_entry_str (const gchar *time_str)
{
  gdouble val = 0.0;
  guint coeff[6] = { 1, 10, 60, 600, 3600, 36000 };
  guint len;
  guint n;
  guint i;
  gdouble num;
  gchar **str;
  gchar *tmp;

  g_return_val_if_fail (time_str != NULL, -1.0);
  
  len = strlen (time_str);

  /* MM */
  if (len == 1 || len == 2)
    return kp_number (time_str);

  tmp = g_strdup (time_str);

  /* .ddd */
  if (len > 4 && time_str[len - 4] == '.') {
    tmp[len - 4] = '\0';
    val = kp_number (&tmp[len - 3]) / (gdouble) 1000;
    len = strlen (tmp);
  }

  /* HMM, HHMM, HMMSS, HHMMSS */
  if ((num = kp_number (tmp)) >= 1 && len <= 6) {
    for (i = len-1;;) {
      g_assert (i <= len-1);
      val += coeff[len-1-i] * g_ascii_digit_value (tmp[i]);

      if (i==0)
        break;
      i--;
    }
    return val;
  }

  /* HH:MM, HH:MM:SS */
  /* Split the string */
  str = g_strsplit (tmp, ":", 4);
  for (n=1; str[n-1] != NULL; n++)
    if (n == 4) {
      val = -1.0;
      goto error;
    }
  
  /* Count the sum */
  for (i=0; i <= (n - 2) && str[i] != NULL;) {
    if ((num = kp_number (str[i])) >= 0) {
      if (num > 59) {
        val = -1.0;
        goto error;
      }
      val += coeff[n - i*2 + (n == 3)] * num;
    }
    if (i == (n - 2))
      break;
    i++;
  }

error:
  g_strfreev (str);
  g_free (tmp);
  return val;
}

#if 0
int main (void) {
  
  printf("%8s => %.2f\n", "22", is_valid_duration_entry_str ("22"));
  printf("%8s => %.2f\n", "22:15", is_valid_duration_entry_str ("22:15"));
  printf("%8s => %.2f\n", "220011", is_valid_duration_entry_str ("220011"));
  printf("%8s => %.2f\n", "22229", is_valid_duration_entry_str ("22229"));
  printf("%8s => %.2f\n", "2222", is_valid_duration_entry_str ("2222"));
  printf("%8s => %.2f\n", "222", is_valid_duration_entry_str ("222"));
  printf("%8s => %.2f\n", "22:22:11", is_valid_duration_entry_str ("22:22:11"));
  printf("%8s => %.2f\n", "22:22:11:", is_valid_duration_entry_str ("22:22:11:"));
  printf("%8s => %.3f\n", "22:22:11.301", is_valid_duration_entry_str ("22:22:11.301"));
  printf("%8s => %.3f\n", "222211.211", is_valid_duration_entry_str ("222211.211"));
  printf("%8s => %.2f\n", "22:22:11", is_valid_duration_entry_str ("22:22:11"));
 
  printf("\n");
  return 0;
}
#endif

gboolean
kp_is_valid_time_str (const gchar * time_str)
{
  gchar **str;
  guint i;

  str = g_strsplit (time_str, ":", 4);

  if (str[0] == NULL)
    return FALSE;

  if (!g_unichar_isdigit (str[0][0]))
    goto error;

  for (i = 0; str[i] != NULL && i < 4; i++) {
    if (g_utf8_strlen (str[i], -1) > 2)
      goto error;
    if (!g_unichar_isdigit (str[i][0]))
      goto error;
    if (str[i][1] != '\0' && !g_unichar_isdigit (str[i][1]))
      goto error;
  }
  if (i != 3)
    goto error;
  return TRUE;

error:
  g_strfreev (str);
  return FALSE;
}

gboolean
kp_is_valid_date_str (const gchar * date_str)
{
  gchar **str;
  guint i;
  gint d;
  gint m;
  gint y;

  str = g_strsplit (date_str, ".", 4);

  if (str[0] == NULL)
    return FALSE;

  for (i = 0; str[i] != NULL && i < 4; i++);
  /* nothing */

  if (i != 3) {
    g_strfreev (str);
    return FALSE;
  }

  d = (gint) g_strtod (str[0], NULL);
  m = (gint) g_strtod (str[1], NULL);
  y = (gint) g_strtod (str[2], NULL);

  if (y < 1960 || y > 2100) {
    g_strfreev (str);
    return FALSE;
  }

  if (g_date_valid_dmy (d, m, y)) {
    g_strfreev (str);
    return TRUE;
  }

  if (str)
    g_strfreev (str);

  return FALSE;
}

GValue *
kp_g_value_from_string (const gchar * string)
{
  GValue *value;
  gchar *err;
  gdouble d;
  guint i;

  d = strtod (string, &err);
  value = g_new0 (GValue, 1);

  if (string != err) {
    if (d != ((guint) d)) {
      g_value_init (value, G_TYPE_DOUBLE);
      g_value_set_double (value, d);
      return value;
    }
    else if (d > 0.0) {
      i = (guint) d;
      g_value_init (value, G_TYPE_UINT);
      g_value_set_uint (value, i);
      return value;
    }
  }
  if (strcmp (string, "true") == 0) {
    g_value_init (value, G_TYPE_BOOLEAN);
    g_value_set_boolean (value, TRUE);
    return value;
  }
  if (strcmp (string, "false") == 0) {
    g_value_init (value, G_TYPE_BOOLEAN);
    g_value_set_boolean (value, FALSE);
    return value;
  }

  /* hmmh, it should be just a string then.. */
  g_value_init (value, G_TYPE_STRING);
  g_value_set_string (value, string);

  return value;
}

gchar *
kp_g_value_to_string (const GValue * value)
{
  GType type = G_VALUE_TYPE (value);

  switch (type) {
  case G_TYPE_STRING:
    return g_strdup (g_value_get_string (value));

  case G_TYPE_UINT:
    return g_strdup_printf ("%u", g_value_get_uint (value));

  case G_TYPE_DOUBLE:
    return g_strdup_printf ("%.3f", g_value_get_double (value));

  case G_TYPE_BOOLEAN:
    if (g_value_get_boolean (value))
      return g_strdup ("true");
    else
      return g_strdup ("false");

  default:
    return g_strdup ("");
  }
}

/*
 * The Following code is taken from GtkCalendar's source.
 * It's modified a little bit to fit this program.
 */

/***************************************************************************
 * The following date routines are taken from the lib_date package.  Keep
 * them seperate in case we want to update them if a newer lib_date comes
 * out with fixes. */

const guint month_length[2][13] = {
  {0, 31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31},
  {0, 31, 29, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31}
};

const guint days_in_months[2][14] = {
  {0, 0, 31, 59, 90, 120, 151, 181, 212, 243, 273, 304, 334, 365},
  {0, 0, 31, 60, 91, 121, 152, 182, 213, 244, 274, 305, 335, 366}
};

static gchar *
month_names[12] = {
  N_("January"),
  N_("February"),
  N_("March"),
  N_("April"),
  N_("May"),
  N_("June"),
  N_("July"),
  N_("August"),
  N_("September"),
  N_("October"),
  N_("November"),
  N_("December"),
};

/**
 * kp_month_num_by_year_day_number:
 * @day_num: Number of day between 1-366
 * @is_leap: TRUE if the year is a leap year and FALSE otherwise
 *
 * Return the month that some day number is in.
 * 
 * Returns: Month number between 1 and 12 or 0 if params are
 * invalid.
 */
guint
kp_month_num_by_year_day_number (guint day_num, gboolean is_leap)
{
  guint m;
  g_return_val_if_fail (day_num != 0, 0);
  g_return_val_if_fail (day_num <= (guint) (365 + is_leap), 0);

  for (m=12; m > 0; m--)
    if (day_num > days_in_months[is_leap][m])
      return m;
  g_return_val_if_reached (0);
}


/**
 * kp_get_month_len
 * @m: Number of the month (0-11)
 * @is_leap: TRUE if the year which the month is in, is a leap year
 * 
 * Get number of days in the month.
 *
 * Returns: Number of the days or 0 if month number is invalid.
 */
guint 
kp_get_month_len (guint m, gboolean is_leap)
{
  g_return_val_if_fail (m <= 12, 0);
  return month_length[is_leap][m];
}


gboolean
kp_leap (guint year)
{
  return ((((year % 4) == 0) && ((year % 100) != 0)) || ((year % 400) == 0));
}

guint
kp_day_of_week (guint year, guint mm, guint dd)
{
  glong days;

  days = kp_calc_days (year, mm, dd);
  if (days > 0L) {
    days--;
    days %= 7L;
    days++;
  }
  return ((guint) days);
}

guint
kp_weeks_in_year (guint year)
{
  return (52 +
          ((kp_day_of_week (year, 1, 1) == 4) || (kp_day_of_week (year, 12, 31) ==
                                               4)));
}

gboolean
kp_check_date (guint year, guint mm, guint dd)
{
  if (year < 1)
    return FALSE;
  if ((mm < 1) || (mm > 12))
    return FALSE;
  if ((dd < 1) || (dd > month_length[kp_leap (year)][mm]))
    return FALSE;
  
  return TRUE;
}

guint
kp_week_number (guint year, guint mm, guint dd)
{
  guint first;

  first = kp_day_of_week (year, 1, 1) - 1;
  return ((guint) ((kp_dates_difference (year, 1, 1, year, mm, dd) + first) / 7L) +
          (first < 4));
}

glong
kp_year_to_days (guint year)
{
  return (year * 365L + (year / 4) - (year / 100) + (year / 400));
}

glong
kp_calc_days (guint year, guint mm, guint dd)
{
  gboolean lp;

  if (year < 1)
    return (0L);
  if ((mm < 1) || (mm > 12))
    return (0L);
  if ((dd < 1) || (dd > month_length[(lp = kp_leap (year))][mm]))
    return (0L);
  return (kp_year_to_days (--year) + days_in_months[lp][mm] + dd);
}

gboolean
kp_week_of_year (guint * week, guint * year, guint mm, guint dd)
{
  if (kp_check_date (*year, mm, dd)) {
    *week = kp_week_number (*year, mm, dd);
    if (*week == 0)
      *week = kp_weeks_in_year (--(*year));
    else if (*week > kp_weeks_in_year (*year)) {
      *week = 1;
      (*year)++;
    }
    return TRUE;
  }
  return FALSE;
}

glong
kp_dates_difference (guint year1, guint mm1, guint dd1, guint year2, guint mm2,
                  guint dd2)
{
  return (kp_calc_days (year2, mm2, dd2) - kp_calc_days (year1, mm1, dd1));
}

/*** END OF lib_date routines ********************************************/


void
kp_print_debug_string (const gchar *file, const gchar *function, guint line,
    const gchar *message, ...)
{
  va_list args;
  gchar *format;

  if (!kp_settings_get_bool ("debug"))
    return;

  format = g_strdup_printf ("%s: %d, in %s(): %s\n", file, line, function,
                            message);
  
  va_start (args, message);
  vprintf (format, args);
  va_end (args);

  g_free (format);
}

/**
 * kp_get_month_name:
 * @m: month number between 0 and 11
 * 
 * Get localized month name.
 *
 * Returns: Newly-allocated string that must be freed or
 * NULL if @m is invalid.
 */
gchar *
kp_get_month_name (guint m)
{
  g_return_val_if_fail (m < 12, NULL);
  
  return g_strdup (month_names[m]);
}


/**
 * kp_get_month_num:
 * @m: localized month name.
 *
 * Get number of the month.
 *
 * Returns: The number of the month between 0 and 11 or -1 if the month
 * is invalid.
 */
gint
kp_get_month_num (const gchar *m)
{
  guint i;

  for (i=0; i < 12; i++)
    if (strcmp (m, month_names[i]) == 0)
      return i;

  g_warning ("Invalid localized month name passed to %s: \"%s\"\n",
             __PRETTY_FUNCTION__, m);

  return -1;
}
    
