#include <string.h>
#include <stdio.h>
#include <time.h>

#include <glib-object.h>
#include <gdk/gdkkeysyms.h>
#include <gdk/gdk.h>
#include <glib.h>

#include "../kipina-i18n.h"
#include "../kparray2d.h"
#include "../kputil.h"
#include "kpcalendarentryinfodialog.h"
#include "kpnewsplitworkoutdialog.h"
#include "kpnewworkoutdialog.h"
#include "kpnewcommentdialog.h"
#include "kpcalendarview.h"
#include "kpviewmodel.h"
#include "kpguiutils.h"

/* KPViewModel interface stuff */

static void       kp_calendar_view_model_init     (KPViewModelIface *iface);
static void       kp_calendar_view_set_dmy        (KPViewModel *view,
                                                   guint d,
                                                   guint m,
                                                   guint y);
static void       kp_calendar_view_get_dmy        (KPViewModel *view,
                                                   guint *d,
                                                   guint *m,
                                                   guint *y);
static void       kp_calendar_view_set_view_type  (KPViewModel *view,
                                                   KPViewModelType type);
static
KPViewModelType   kp_calendar_view_get_view_type  (KPViewModel *view);
static void       kp_calendar_view_set_log        (KPViewModel *view,
                                                   KPTrainingLog *log);
static void       kp_calendar_view_unset_log      (KPViewModel *view);
/* Some function typedefs */
typedef void    (*DayNumberFunc)                  (KPCalendarView *cv);
typedef void    (*MarkTextFunc)                   (KPCalendarView *cv,
                                                   GString *buffer,
                                                   guint col, guint row);

/* Functions to calculate correct day numbers */
static void       day_num                         (KPCalendarView *cv);
static void       week_num                        (KPCalendarView *cv);
static void       month_num                       (KPCalendarView *cv);
static void       year_num                        (KPCalendarView *cv);
static void       all_time_num                    (KPCalendarView *cv);

/* Functions to get marks */
static void       basic_mark                      (KPCalendarView *cv,
                                                   GString *buffer,
                                                   guint col, guint row);
static void       year_mark                       (KPCalendarView *cv,
                                                   GString *buffer,
                                                   guint col, guint row);
/* GUI-related functions */
static void       update                          (KPCalendarView *cv);
static void       update_marks                    (KPCalendarView *cv);
static gboolean   show_popup_menu                 (KPCalendarView *cv,
                                                   GdkEventButton *event);

static guint      get_hash_key_dmy                (guint d,
                                                   guint m,
                                                   guint y);
static GdkColor  *get_color                       (const gchar *color,
                                                   GtkWidget *widget);
static void       kp_calendar_view_set_title_date (KPCalendarView *cv);

static GtkWidget *create_calendar_view_popup_menu (KPCalendarView *cv);

/* Helper functions for keyboard navigation */
static void       find_index                      (KPCalendarView *cv,
                                                   guint *x, guint *y);
static void       find_first                      (KPCalendarView *cv,
                                                   guint *x, guint *y);
static void       find_last                       (KPCalendarView *cv,
                                                   guint *x, guint *y);

/* Helper functions for mouse selections */
static gint       get_col_from_x                  (KPCalendarView *cv,
                                                   gdouble x);
static gint       get_row_from_y                  (KPCalendarView *cv,
                                                   gdouble y);
static gdouble    row_height                      (KPCalendarView *cv);
static gdouble    col_width                       (KPCalendarView *cv);

/* Callbacks (GtkWidget & GObject stuff) */
static void       cv_popup_add_split_workout      (GtkMenuItem *item,
                                                   KPCalendarView *cv);
static void       cv_popup_add_workout            (GtkMenuItem *item,
                                                   KPCalendarView *cv);
static void       cv_popup_add_comment            (GtkMenuItem *item,
                                                   KPCalendarView *cv);
static void       cv_popup_properties             (GtkMenuItem *item,
                                                   KPCalendarView *cv);
static void       cv_popup_delete                 (GtkMenuItem *item,
                                                   KPCalendarView *cv);
static void       kp_calendar_view_destroy        (GtkObject *object);
static gboolean   kp_calendar_view_expose         (GtkWidget *widget,
                                                   GdkEventExpose *event);

static void       kp_calendar_view_size_request   (GtkWidget *widget,
                                                   GtkRequisition *requisition);

static void       kp_calendar_view_size_allocate  (GtkWidget *widget,
                                                   GtkAllocation *allocation);

static void       kp_calendar_view_init           (KPCalendarView *view);
static void       kp_calendar_view_realize        (GtkWidget *widget);
static void       kp_calendar_view_unrealize      (GtkWidget *widget);

static gint       kp_calendar_view_button_press   (GtkWidget *widget,
                                                   GdkEventButton *event);

static gint       kp_calendar_view_key_press      (GtkWidget *widget,
                                                   GdkEventKey *event);
static void       kp_calendar_view_class_init     (KPCalendarViewClass *class);
static void       kp_calendar_view_get_property   (GObject *object,
                                                   guint prop_id,
                                                   GValue *value,
                                                   GParamSpec *pspec);
static void       kp_calendar_view_set_property   (GObject *object,
                                                   guint prop_id,
                                                   const GValue *value,
                                                   GParamSpec *pspec);
static void       log_connect_signals             (KPTrainingLog *log,
                                                   KPCalendarView *cv);
static void       log_disconnect_signals          (KPTrainingLog *log,
                                                   KPCalendarView *cv);
static void       log_entry_removed               (KPTrainingLog *log,
                                                   guint d, guint m, guint y,
                                                   const gchar * mark_str,
                                                   KPCalendarView *cv);
static void       log_entry_added                 (KPTrainingLog *log,
                                                   guint d, guint m, guint y,
                                                   const gchar *mark,
                                                   KPCalendarView *cv);




static struct KPCalendarViewTypeData_
{
  KPViewModelType         type;

  guint                   rows;
  guint                   cols;

  guint                   margin_top;
  guint                   margin_right;
  guint                   margin_bottom;
  guint                   margin_left;

  guint                   padding_x;
  guint                   padding_y;

  guint                   skip_padding_n_x;
  guint                   skip_padding_n_y;

  DayNumberFunc           number_func;
  MarkTextFunc            mark_func;
  guint                 **days;
  guint                 **months;
  guint                 **years;
  gchar                ***marks;

  const gchar            *str;

  gboolean                show_all_months_as_active;
  gdouble                 default_font_size;
}
cvt_data[] = {
  {
  KP_VIEW_MODEL_TYPE_DAY, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0,
      day_num,
      basic_mark,
      NULL,
      NULL,
      NULL,
      NULL,
      N_("day"),
      TRUE, PANGO_SCALE_LARGE,},
  {
  /* Week calendar */
  KP_VIEW_MODEL_TYPE_WEEK, 7, 1, 0, 0, 0, 0, 0, 0, 0, 0,
      week_num,
      basic_mark,
      NULL,
      NULL,
      NULL,
      NULL,
      N_("week"),
      TRUE, PANGO_SCALE_LARGE,},
  {
  /* Month calendar */
  KP_VIEW_MODEL_TYPE_MONTH, 6, 7, 0, 0, 0, 0, 0, 0, 0, 0,
      month_num,
      basic_mark,
      NULL,
      NULL,
      NULL,
      NULL,
      N_("month"),
      FALSE, PANGO_SCALE_MEDIUM,},
  {
  /* Year calendar */
  KP_VIEW_MODEL_TYPE_YEAR, 24, 21, 0, 0, 0, 0, 6, 6, 7, 6,
      year_num,
      year_mark,
      NULL,
      NULL,
      NULL,
      NULL,
      N_("year"),
      TRUE, PANGO_SCALE_XX_SMALL,
  },
  {
  /* All time calendar, not suppored */
  KP_VIEW_MODEL_TYPE_ALL_TIME, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0,
      all_time_num,
      NULL,
      NULL,
      NULL,
      NULL,
      NULL,
      N_("all time"),
      TRUE, PANGO_SCALE_XX_SMALL,
  },

};

typedef struct KPCalendarViewPrivateData_ KPCalendarViewPrivateData;
typedef struct KPCalendarViewTypeData_ KPCalendarViewTypeData;

struct KPCalendarViewPrivateData_
{
  KPViewModelType     type;
  KPCalendarViewTypeData *cvt;

  GdkWindow          *header_win;
  GdkWindow          *main_win;

  gchar               title_str[255];

  GtkWidget          *title_label;
  GtkWidget          *navi_buttons[4];

  guint               year;
  guint               week;
  guint               month;
  guint               day;

  gboolean            week_start_monday;
  gboolean            marks_need_update;

  gint                active_x;
  gint                active_y;

  KPTrainingLog      *log;

  guint               line_spacing;
 
  GtkWidget          *popup_mi_properties;
  GtkWidget          *popup_mi_delete;
  GtkWidget          *popup_menu_title;
  GtkWidget          *popup_menu;
  GtkWidget          *area;
};

static GHashTable  *marks_table = NULL;
static GHashTable  *allocated_colors = NULL;


#define SELECTED_BG_COLOR(widget) (&widget->style->base[ \
    GTK_WIDGET_HAS_FOCUS (widget) ? GTK_STATE_SELECTED : GTK_STATE_ACTIVE])

#define KP_CALENDAR_VIEW_PRIVATE_DATA(widget) (((KPCalendarViewPrivateData*) \
      (KP_CALENDAR_VIEW (widget)->private_data)))

#define CVT(object) ((object)->cvt[(object)->type])

enum {
  PROP_0,
  PROP_LINE_SPACING,
};

enum {
  DAY_SELECTED_SIGNAL,
  LAST_SIGNAL
};

static guint kp_calendar_view_signals[LAST_SIGNAL] = { 0 };

GType
kp_calendar_view_get_type (void)
{
  static GType kp_calendar_view_type = 0;

  if (!kp_calendar_view_type) {
    static const GTypeInfo kp_calendar_view_info = {
      sizeof (KPCalendarViewClass),
      NULL,
      NULL,
      (GClassInitFunc) kp_calendar_view_class_init,
      NULL,
      NULL,
      sizeof (KPCalendarView),
      0,
      (GInstanceInitFunc) kp_calendar_view_init,
      NULL
    };
    static const GInterfaceInfo view_model_info = {
      (GInterfaceInitFunc) kp_calendar_view_model_init,
      NULL,
      NULL
    };
    kp_calendar_view_type = g_type_register_static (GTK_TYPE_WIDGET,
                                                   "KPCalendarView",
                                                   &kp_calendar_view_info,
                                                    0);
    g_type_add_interface_static (kp_calendar_view_type,
                                 KP_TYPE_VIEW_MODEL,
                                &view_model_info);
  }
  return kp_calendar_view_type;
}


static void
kp_calendar_view_model_init (KPViewModelIface *iface)
{
  iface->set_dmy = kp_calendar_view_set_dmy;
  iface->get_dmy = kp_calendar_view_get_dmy;
  iface->set_log = kp_calendar_view_set_log;
  iface->unset_log = kp_calendar_view_unset_log;
  iface->set_view_type = kp_calendar_view_set_view_type;
  iface->get_view_type = kp_calendar_view_get_view_type;
}

static void
kp_calendar_view_init (KPCalendarView *cv)
{
  KPCalendarViewPrivateData *p_data;
  GtkWidget *popup_menu;

  cv->private_data = g_malloc (sizeof *p_data);
  p_data = KP_CALENDAR_VIEW_PRIVATE_DATA (cv);
  p_data->popup_menu = NULL;
  p_data->line_spacing = 0;
  
  if (marks_table == NULL)
    marks_table = g_hash_table_new_full (g_direct_hash, NULL, NULL, NULL);
  
  p_data->marks_need_update = TRUE;
  p_data->cvt = &cvt_data[0];
  p_data->active_x = -1;
  p_data->active_y = -1;

  GTK_WIDGET_SET_FLAGS (GTK_WIDGET (cv), GTK_CAN_FOCUS);

  g_assert (KP_IS_CALENDAR_VIEW (cv));
  popup_menu = create_calendar_view_popup_menu (cv);
  kp_calendar_view_attach_popup_menu (cv, GTK_MENU (popup_menu));
}

static void
kp_calendar_view_set_property (GObject *object, guint prop_id,
                               const GValue *value, GParamSpec *pspec)
{
  KPCalendarViewPrivateData *p_data;

  p_data = KP_CALENDAR_VIEW_PRIVATE_DATA (object);
  
  switch (prop_id)
  {
    case PROP_LINE_SPACING:
      p_data->line_spacing = g_value_get_uint (value);
      break;

    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }
}

static void
kp_calendar_view_get_property (GObject *object, guint prop_id,
                               GValue *value, GParamSpec *pspec)
{
  KPCalendarViewPrivateData *p_data;

  p_data = KP_CALENDAR_VIEW_PRIVATE_DATA (object);
  
  switch (prop_id)
  {
    case PROP_LINE_SPACING:
      g_value_set_uint (value, p_data->line_spacing);
      break;

    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }
}

static void
kp_calendar_view_class_init (KPCalendarViewClass *class)
{
  GtkWidgetClass *widget_class = GTK_WIDGET_CLASS (class);
  GtkObjectClass *gtk_object_class = GTK_OBJECT_CLASS (class);
  GObjectClass *object_class = G_OBJECT_CLASS (class);

  gtk_object_class->destroy = kp_calendar_view_destroy;
  widget_class->realize = kp_calendar_view_realize;
  widget_class->unrealize = kp_calendar_view_unrealize;
  widget_class->expose_event = kp_calendar_view_expose;
  widget_class->size_request = kp_calendar_view_size_request;
  widget_class->size_allocate = kp_calendar_view_size_allocate;
  widget_class->button_press_event = kp_calendar_view_button_press;
  widget_class->key_press_event = kp_calendar_view_key_press;
  object_class->set_property = kp_calendar_view_set_property;
  object_class->get_property = kp_calendar_view_get_property;

  g_object_class_install_property (object_class,
                                   PROP_LINE_SPACING,
                                   g_param_spec_uint ("line-spacing",
                                  "Line spacing",
                                  "Space in pixels between the lines of text "
                                  "in the calendar.",
                                   0,
                                   32,
                                   3,
                                   G_PARAM_READABLE|G_PARAM_WRITABLE));
  
  kp_calendar_view_signals[DAY_SELECTED_SIGNAL]
    = g_signal_new ("day_selected",
                    G_OBJECT_CLASS_TYPE (object_class),
                    G_SIGNAL_RUN_FIRST | G_SIGNAL_ACTION,
                    G_STRUCT_OFFSET (KPCalendarViewClass, day_selected),
                    NULL,
                    NULL,
                    g_cclosure_marshal_VOID__POINTER,
                    G_TYPE_NONE,
                    1,
                    G_TYPE_POINTER);
}

static void
kp_calendar_view_size_request (GtkWidget * widget, GtkRequisition * requisition)
{
  requisition->width = 300;
  requisition->height = 200;
}

static void
kp_calendar_view_size_allocate (GtkWidget * widget, GtkAllocation * allocation)
{
  KPCalendarViewPrivateData *p_data;
  KPCalendarView       *cal = NULL;

  g_return_if_fail (widget != NULL);
  g_return_if_fail (KP_IS_CALENDAR_VIEW (widget));
  g_return_if_fail (allocation != NULL);

  widget->allocation = *allocation;


  if (GTK_WIDGET_REALIZED (widget)) {
    cal = KP_CALENDAR_VIEW (widget);
    p_data = KP_CALENDAR_VIEW_PRIVATE_DATA (cal);

    gdk_window_move_resize (widget->window,
                            allocation->x,
                            allocation->y,
                            allocation->width,
                            allocation->height);
  }
}


static void
kp_calendar_view_realize (GtkWidget * widget)
{
  KPCalendarView       *cal;
  GdkWindowAttr       attributes;
  gint                attributes_mask;

  g_return_if_fail (KP_IS_CALENDAR_VIEW (widget));

  GTK_WIDGET_SET_FLAGS (widget, GTK_REALIZED);
  cal = KP_CALENDAR_VIEW (widget);

  attributes.x = widget->allocation.x;
  attributes.y = widget->allocation.y;
  attributes.width = widget->allocation.width;
  attributes.height = widget->allocation.height;
  attributes.wclass = GDK_INPUT_OUTPUT;
  attributes.window_type = GDK_WINDOW_CHILD;
  attributes.event_mask = gtk_widget_get_events (widget)
    | GDK_BUTTON_PRESS_MASK| GDK_EXPOSURE_MASK | GDK_KEY_PRESS_MASK;
  attributes.visual = gtk_widget_get_visual (widget);
  attributes.colormap = gtk_widget_get_colormap (widget);

  attributes_mask = GDK_WA_X | GDK_WA_Y | GDK_WA_VISUAL | GDK_WA_COLORMAP;
  widget->window =
    gdk_window_new (widget->parent->window, &attributes, attributes_mask);
  widget->style = gtk_style_attach (widget->style, widget->window);

  gdk_window_set_user_data (widget->window, widget);
  gtk_style_set_background (widget->style, widget->window, GTK_STATE_ACTIVE);

  kp_debug ("KPCalendarView realized.\n");

  update (KP_CALENDAR_VIEW (widget));
}

static void
kp_calendar_view_unrealize (GtkWidget *widget)
{
  KPCalendarViewPrivateData *p_data;
  GtkWidgetClass *parent_class;
  
  p_data = KP_CALENDAR_VIEW_PRIVATE_DATA (widget);

  kp_debug ("KPCalendarView unrealized\n");
  
  parent_class = g_type_class_peek_parent (
      KP_CALENDAR_VIEW_GET_CLASS (widget));
        
  if (GTK_WIDGET_CLASS (parent_class)->unrealize)
    (* GTK_WIDGET_CLASS (parent_class)->unrealize) (widget);
}

GtkWidget *
kp_calendar_view_new (KPViewModelType type, guint d, guint m, guint y)
{
  KPCalendarViewPrivateData *p_data;
  GtkWidget *widget;
  GDate *date;
  g_return_val_if_fail (type < KP_VIEW_MODEL_TYPE_N, NULL);

  widget = g_object_new (kp_calendar_view_get_type (), NULL);

  p_data = KP_CALENDAR_VIEW_PRIVATE_DATA (widget);
  p_data->type = type;

  if (d == 0 || m == 0 || y == 0) {
    date = g_date_new ();
    g_date_set_time (date, time (NULL));
  }
  else
    date = g_date_new_dmy (d, m, y);

  p_data->year = g_date_get_year (date);
  p_data->month = g_date_get_month (date);
  p_data->day = g_date_get_day (date);

  if (cvt_data[type].number_func)
    cvt_data[type].number_func (KP_CALENDAR_VIEW (widget));

  update (KP_CALENDAR_VIEW (widget));
  g_date_free (date);
  
  find_index (KP_CALENDAR_VIEW (widget), &p_data->active_x, &p_data->active_y);
 
  return widget;
}

static void
kp_calendar_view_get_dmy (KPViewModel *view, guint *d, guint *m, guint *y)
{
  kp_calendar_view_get_date (KP_CALENDAR_VIEW (view), d, m, y);
}

static void
kp_calendar_view_set_dmy (KPViewModel *view, guint d, guint m, guint y)
{
  kp_calendar_view_set_date (KP_CALENDAR_VIEW (view), d, m, y);
  gtk_widget_queue_draw (GTK_WIDGET (view));
}

static void
add_mark (KPCalendarEntry *entry, KPCalendarView *cv)
{
  GDate *date;
  guint d, m, y;
  gchar *mark;

  date = kp_calendar_time_get_date (kp_calendar_entry_get_date (entry));
  d = g_date_get_day (date);
  m = g_date_get_month (date);
  y = g_date_get_year (date);
  
  mark = kp_calendar_entry_to_string (entry);
  
  kp_calendar_view_add_mark (cv, d, m, y, 0, 0, mark);

  g_free (mark);
}

static void
kp_calendar_view_set_log (KPViewModel *view, KPTrainingLog *log)
{
  KPCalendarViewPrivateData *p_data;
  g_return_if_fail (KP_IS_CALENDAR_VIEW (view));
  g_return_if_fail (KP_IS_TRAINING_LOG (log));

  p_data = KP_CALENDAR_VIEW_PRIVATE_DATA (view);
  p_data->log = log;

  kp_debug ("Connecting signals.");
  log_connect_signals (log, KP_CALENDAR_VIEW (view));

  kp_training_log_foreach (log, (GFunc) add_mark, view);
}
  
static void
kp_calendar_view_unset_log (KPViewModel *view)
{
  KPCalendarViewPrivateData *p_data;
  g_return_if_fail (KP_IS_CALENDAR_VIEW (view));

  p_data = KP_CALENDAR_VIEW_PRIVATE_DATA (view);
  
  log_disconnect_signals (p_data->log, KP_CALENDAR_VIEW (view));
  kp_calendar_view_clear_marks (KP_CALENDAR_VIEW (view));
  p_data->log = NULL;
}


static void
log_connect_signals (KPTrainingLog *log, KPCalendarView *cv)
{
  g_signal_connect (G_OBJECT (log), "entry-removed",
                    G_CALLBACK (log_entry_removed), cv);
  g_signal_connect (G_OBJECT (log), "entry-added",
                    G_CALLBACK (log_entry_added), cv);
}


static void
log_disconnect_signals (KPTrainingLog *log, KPCalendarView *cv)
{
  g_signal_handlers_disconnect_by_func (log, log_entry_removed, cv);
  g_signal_handlers_disconnect_by_func (log, log_entry_added, cv);
}

  
static void
log_entry_removed (KPTrainingLog *log, guint d, guint m, guint y,
                   const gchar *mark_str, KPCalendarView *cv)
{
  g_return_if_fail (KP_IS_TRAINING_LOG (log));
  g_return_if_fail (KP_IS_CALENDAR_VIEW (cv));

  kp_debug ("Removing mark: %s\n", mark_str);
  
  kp_calendar_view_remove_mark (cv, d, m, y, mark_str);
}


static void
log_entry_added (KPTrainingLog *log, guint d, guint m, guint y,
                 const gchar *mark, KPCalendarView *cv)
{
  g_return_if_fail (KP_IS_TRAINING_LOG (log));
  g_return_if_fail (KP_IS_CALENDAR_VIEW (cv));
  
  kp_calendar_view_add_mark (cv, d, m, y, 0, 0, mark);
}


static void
kp_calendar_view_set_view_type (KPViewModel *view, KPViewModelType type)
{
  kp_calendar_view_set (KP_CALENDAR_VIEW (view), type);
  gtk_widget_queue_draw (GTK_WIDGET (view));
}


static KPViewModelType
kp_calendar_view_get_view_type (KPViewModel *view)
{
  KPCalendarViewPrivateData *p_data;

  g_return_val_if_fail (KP_IS_CALENDAR_VIEW (view), KP_VIEW_MODEL_TYPE_N);
  
  p_data = KP_CALENDAR_VIEW_PRIVATE_DATA (KP_CALENDAR_VIEW (view));
  g_return_val_if_fail (p_data != NULL, KP_VIEW_MODEL_TYPE_N);
  
  return p_data->type;
}

/**
 * kp_calendar_view_get_end_date:
 * @cv: A #KPCalendarView
 *
 * Find out what is the latest date in the current view.
 * 
 * Returns: a #GDate that represents last day in the current view.
 */
GDate *
kp_calendar_view_get_end_date (KPCalendarView *cv)
{
  KPCalendarViewPrivateData *p_data;
  GDate *date;
  guint current;
  guint max;
  guint i,j;

  date = g_date_new ();
  
  g_return_val_if_fail (KP_IS_CALENDAR_VIEW (cv), NULL);
  p_data = KP_CALENDAR_VIEW_PRIVATE_DATA (cv);

  max = 0;
  
  for (i=0; i < CVT (p_data).rows; i++)
    for (j=0; j < CVT (p_data).cols; j++) {
      /* In yearview there can be 0s too */
      if (CVT (p_data).days[i][j] > 0) {
        g_date_set_day (date, CVT (p_data).days[i][j]);
        g_date_set_month (date, CVT (p_data).months[i][j]);
        g_date_set_year (date, CVT (p_data).years[i][j]);

        current = g_date_get_julian (date);
        if (current > max)
          max = g_date_get_julian (date);
      }
    }
  g_date_set_julian (date, max);
  g_return_val_if_fail (g_date_valid (date), NULL);

  return date;
}

/**
 * kp_calendar_view_get_start_date:
 * @cv: A #KPCalendarView
 *
 * Find out what is the earliest date in the current view.
 * 
 * Returns: a #GDate that represents first day in the current view.
 */
GDate *
kp_calendar_view_get_start_date (KPCalendarView *cv)
{
  KPCalendarViewPrivateData *p_data;
  GDate *date;
  guint current;
  guint min;
  guint i,j;

  g_return_val_if_fail (KP_IS_CALENDAR_VIEW (cv), NULL);
  p_data = KP_CALENDAR_VIEW_PRIVATE_DATA (cv);

  min = G_MAXUINT;
  date = g_date_new ();
  
  for (i=0; i < CVT (p_data).rows; i++)
    for (j=0; j < CVT (p_data).cols; j++) {
      /* In yearview there can be 0s too */
      if (CVT (p_data).days[i][j] > 0) {
        g_date_set_day (date, CVT (p_data).days[i][j]);
        g_date_set_month (date, CVT (p_data).months[i][j]);
        g_date_set_year (date, CVT (p_data).years[i][j]);

        current = g_date_get_julian (date);
        if (current < min)
          min = g_date_get_julian (date);
      }
    }
  g_date_set_julian (date, min);
  g_return_val_if_fail (g_date_valid (date), NULL);
  
  return date;
}


/**
 * kp_calendar_view_get_date:
 * @cv: A #KPCalendarView
 * @d: A location to store day number or NULL
 * @m: A location to store month number or NULL
 * @y: A location to store year number or NULL
 *
 * Finds out the date of the currently selected day. 
 */
void
kp_calendar_view_get_date (KPCalendarView * cv, guint * d, guint * m, guint * y)
{
  KPCalendarViewPrivateData *p_data;

  g_return_if_fail (KP_IS_CALENDAR_VIEW (cv));
  p_data = KP_CALENDAR_VIEW_PRIVATE_DATA (cv);

  if (d)
    *d = p_data->day;
  if (m)
    *m = p_data->month;
  if (y)
    *y = p_data->year;
}


/**
 * kp_calendar_view_set_date:
 * @cv: A #KPCalendarView
 * @d: Day number between 1 and 31
 * @m: Month number between 1 and 12
 * @y: Year number in format yyyy
 *
 * Sets the active day in the view to the day represented by 
 * @d, @m and @y.
 */
void
kp_calendar_view_set_date (KPCalendarView *cv, guint d, guint m, guint y)
{
  KPCalendarViewPrivateData *p_data;
  GDate *date;

  g_return_if_fail (KP_IS_CALENDAR_VIEW (cv));
  p_data = KP_CALENDAR_VIEW_PRIVATE_DATA (cv);

  date = g_date_new_dmy (d, m, y);
  g_return_if_fail (g_date_valid (date));

  p_data->day = d;
  p_data->month = m;
  p_data->year = y;
  p_data->marks_need_update = TRUE;
  update (cv);

  g_date_free (date);
}

void
kp_calendar_view_remove_day_from_hash (gpointer key, GSList *list,
                                       gpointer data)
{
  GSList *tmp = list;
  
  while (tmp) {
    kp_debug ("Freeing: %s\n", tmp->data);
    g_free (tmp->data);
    tmp = tmp->next;
  }
  g_slist_free (list);
}

void
kp_calendar_view_clear_marks (KPCalendarView *cv)
{
  KPCalendarViewPrivateData *p_data;

  p_data = KP_CALENDAR_VIEW_PRIVATE_DATA (cv);

  kp_debug ("Removing marks from KPCalendarView.\n");

  g_hash_table_foreach (marks_table, (GHFunc)kp_calendar_view_remove_day_from_hash,
                        NULL);
  g_hash_table_destroy (marks_table);
  marks_table = g_hash_table_new_full (g_direct_hash, NULL, NULL, NULL);
    
  p_data->marks_need_update = TRUE;
  update (cv);
}


/**
 * kp_calendar_view_set:
 * @cv: A #KPCalendarView
 * @type: A #KPViewModelType
 *
 * Set the view type of this KPCalendarView. There are currently 
 * some views available: year, month, week and day view.
 */
void
kp_calendar_view_set (KPCalendarView *cv, KPViewModelType type)
{
  KPCalendarViewPrivateData *p_data;

  g_return_if_fail (KP_IS_CALENDAR_VIEW (cv));
  p_data = KP_CALENDAR_VIEW_PRIVATE_DATA (cv);
  
  if (type != p_data->type) {
    p_data->marks_need_update = TRUE;
    p_data->type = type;
    update (cv);
  }
}

/**
 * kp_calendar_view_attatch_popup_menu:
 * @cv: A #KPCalendarView
 * @menu: A #GtkMenu
 *
 * Attach a GtkMenu to KPCalendarView so when KPCalendarView
 * receives "popup-menu" event, this menu will be shown.
 * This function could be useful if #KPCalendarView would be used
 * in some other application in the future..
 */
void
kp_calendar_view_attach_popup_menu (KPCalendarView *cv, GtkMenu *menu)
{
  KPCalendarViewPrivateData *p_data;

  g_return_if_fail (KP_IS_CALENDAR_VIEW (cv));
  
  p_data = KP_CALENDAR_VIEW_PRIVATE_DATA (cv);

  p_data->popup_menu = GTK_WIDGET (menu);
}

  
static GtkWidget *
create_calendar_view_popup_menu (KPCalendarView *cv)
{
  KPCalendarViewPrivateData *p_data;
  GtkWidget *mi_split_workout;
  GtkWidget *mi_workout;
  GtkWidget *mi_comment;
  GtkWidget *menu;

  p_data = KP_CALENDAR_VIEW_PRIVATE_DATA (cv);

  GladeXML *xml = kp_gui_load ("popups", "cv_popup_menu");
  p_data->popup_menu_title = KP_W (xml, "title");
  p_data->popup_mi_properties = KP_W (xml, "properties");
  p_data->popup_mi_delete = KP_W (xml, "delete");
    
  mi_split_workout = KP_W (xml, "split_workout");
  mi_workout = KP_W (xml, "workout");
  mi_comment = KP_W (xml, "comment");
  menu = KP_W (xml, "cv_popup_menu");
  
  g_signal_connect (G_OBJECT (mi_split_workout), "activate",
                    G_CALLBACK (cv_popup_add_split_workout), cv);
  g_signal_connect (G_OBJECT (mi_workout), "activate",
                    G_CALLBACK (cv_popup_add_workout), cv);
  g_signal_connect (G_OBJECT (mi_comment), "activate",
                    G_CALLBACK (cv_popup_add_comment), cv);
  g_signal_connect (G_OBJECT (p_data->popup_mi_delete), "activate",
                    G_CALLBACK (cv_popup_delete), cv);
  g_signal_connect (G_OBJECT (p_data->popup_mi_properties), "activate",
                    G_CALLBACK (cv_popup_properties), cv);
  
  g_object_unref (xml);

  return menu;
}

void
kp_calendar_view_open_to_window (KPViewModelType type, guint d, guint m,
                                 guint y)
{
  GtkWidget *window;
  GtkWidget *calendar;

  window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
  gtk_window_set_position (GTK_WINDOW (window), GTK_WIN_POS_MOUSE);

  calendar = kp_calendar_view_new (type, d, m, y);
  gtk_container_add (GTK_CONTAINER (window), calendar);
  gtk_widget_show_all (window);
}

gboolean
kp_calendar_view_get_active (KPCalendarView *cv)
{
  KPCalendarViewPrivateData *p_data;
  p_data = KP_CALENDAR_VIEW_PRIVATE_DATA (cv);
  
  return (p_data->active_x >= 0 && p_data->active_y >= 0);
}

void
kp_calendar_view_set_active (KPCalendarView *cv, gboolean activate)
{
  KPCalendarViewPrivateData *p_data;
  p_data = KP_CALENDAR_VIEW_PRIVATE_DATA (cv);
 
  if (activate)
    find_index (cv, &p_data->active_x, &p_data->active_y);
  else {
    p_data->active_x = -1;
    p_data->active_y = -1;
    update (cv);
  }
}

static void
kp_calendar_view_set_title_date (KPCalendarView * cv)
{
  KPCalendarViewPrivateData *p_data;
  struct tm tm;
  GDate *date;
  gchar buf[255];
  gchar *format;

  g_return_if_fail (KP_IS_CALENDAR_VIEW (cv));
  p_data = KP_CALENDAR_VIEW_PRIVATE_DATA (cv);
  date = g_date_new_dmy (p_data->day, p_data->month, p_data->year);

  switch (p_data->type) {
  case KP_VIEW_MODEL_TYPE_WEEK:
    format = _("Week %V %Y");
    break;
  case KP_VIEW_MODEL_TYPE_MONTH:
    format = "%B %Y";
    break;
  case KP_VIEW_MODEL_TYPE_YEAR:
    format = "%Y";
    break;
  case KP_VIEW_MODEL_TYPE_ALL_TIME:
  case KP_VIEW_MODEL_TYPE_DAY:
    format = "%x";
    break;
    /* This would be a bug */
  default:
    format = "";                /* stupid compiler.. */
    g_assert_not_reached ();
  }
  g_date_to_struct_tm (date, &tm);
  strftime (buf, sizeof (buf) - 1, format, &tm);
  g_snprintf(p_data->title_str,
             sizeof (p_data->title_str) - 1,
            "<span size=\"x-large\" weight=\"bold\">%s</span>",
             buf);
  g_date_free (date);
}

/**********************************************************************
* Marks handling ..
*********************************************************************/

/**
 * Updates the state of the current view's.
 */
static void
update_marks (KPCalendarView *cv)
{
  KPCalendarViewPrivateData *p_data;
  GSList             *list;
  guint               key;
  guint               i, j;

  g_return_if_fail (KP_IS_CALENDAR_VIEW (cv));
  p_data = KP_CALENDAR_VIEW_PRIVATE_DATA (cv);

  if (!p_data->marks_need_update)
    return;

  for (i = 0; i < CVT (p_data).rows; i++)
    for (j = 0; j < CVT (p_data).cols; j++) {
      key = get_hash_key_dmy (CVT (p_data).days[i][j],
                              CVT (p_data).months[i][j], p_data->year);

      list = g_hash_table_lookup (marks_table, GUINT_TO_POINTER (key));
      CVT (p_data).marks[i][j] = (list) ? list->data : NULL;
    }
  p_data->marks_need_update = FALSE;

  kp_debug ("Marks updated: %u marks found.",
            g_hash_table_size (marks_table));
}


void
kp_calendar_view_remove_mark (KPCalendarView *cv, guint d, guint m, guint y,
                              const gchar *mark)
{
  KPCalendarViewPrivateData *p_data;
  GSList *list = NULL;
  GSList *days_list;
  guint key;
  guint len;

  g_return_if_fail (KP_IS_CALENDAR_VIEW (cv));
  p_data = KP_CALENDAR_VIEW_PRIVATE_DATA (cv);

  key = get_hash_key_dmy (d, m, y);
  
  days_list = g_hash_table_lookup (marks_table, GUINT_TO_POINTER (key));
  list = days_list;

  g_return_if_fail (days_list != NULL);

  while (days_list) {
    if (strcmp (days_list->data, mark) == 0)
      break;
    
    days_list = days_list->next;
  }
  if (!days_list)
    return;
  
  len = g_slist_length (list);

  kp_debug ("The list of this day contains %u entries.\n", len);
  
  if (len == 1) {
    g_slist_free (list);
    g_hash_table_remove (marks_table, GUINT_TO_POINTER (key));
  } else {
    list = g_slist_remove_link (list, days_list);

    kp_debug ("The new list contains %u marks.", g_slist_length (list));
    
    g_hash_table_insert (marks_table, GUINT_TO_POINTER (key), list);
    days_list->data = NULL;
    g_slist_free_1 (days_list);
  }
  p_data->marks_need_update = TRUE;
  update (cv);
}


/** 
 * Add mark to a given date. If h or/and m are set, they are used
 *  to decide the order of marks in the same day.
 */
void
kp_calendar_view_add_mark (KPCalendarView *cv, guint d, guint m, guint y,
                           guint h, guint min, const gchar *str)
{
  KPCalendarViewPrivateData *p_data;
  gboolean            already_in_table;
  GSList             *list = NULL;
  guint               key;

  g_return_if_fail (KP_IS_CALENDAR_VIEW (cv));
  p_data = KP_CALENDAR_VIEW_PRIVATE_DATA (cv);

  key = get_hash_key_dmy (d, m, y);
  list = g_hash_table_lookup (marks_table, GUINT_TO_POINTER (key));

  already_in_table = (list != NULL);
  list = g_slist_append (list, g_strdup (str));

  kp_debug ("KPCalendarView has %u marks for this day.\n",
             g_slist_length (list));
  
  if (!already_in_table)
    g_hash_table_insert (marks_table, GUINT_TO_POINTER (key), list);

  p_data->marks_need_update = TRUE;
  update (cv);
}

G_CONST_RETURN gchar*
kp_calendar_view_get_current_mark (KPCalendarView *cv) 
{
  KPCalendarViewPrivateData *p_data;
  
  p_data = KP_CALENDAR_VIEW_PRIVATE_DATA (cv);

  g_return_val_if_fail (p_data != NULL, NULL);
  g_return_val_if_fail ((guint)p_data->active_x <= CVT (p_data).cols, NULL);
  g_return_val_if_fail ((guint)p_data->active_y <= CVT (p_data).rows, NULL);

  if (!kp_calendar_view_get_active (cv))
    return NULL;

  return CVT (p_data).marks[p_data->active_y][p_data->active_x];
}


void
kp_calendar_view_remove_current_mark (KPCalendarView *cv)
{
  KPCalendarViewPrivateData *p_data;
  const gchar *mark;
  GSList *days_list;
  GSList *list;
  guint key;
  guint len;
  
  p_data = KP_CALENDAR_VIEW_PRIVATE_DATA (cv);
  
  g_return_if_fail (kp_calendar_view_get_active (cv));
  
  key = get_hash_key_dmy (p_data->day, p_data->month, p_data->year);
  
  mark = CVT (p_data).marks[p_data->active_y][p_data->active_x];
  
  g_return_if_fail (mark != NULL);
  
  days_list = g_hash_table_lookup (marks_table, GUINT_TO_POINTER (key));

  g_return_if_fail (days_list != NULL);

  list = g_slist_nth (days_list, g_slist_index (days_list, mark));
  len = g_slist_length (days_list);

  if (len == 1) {
    g_slist_free (days_list);
    days_list = NULL;
    g_hash_table_remove (marks_table, GUINT_TO_POINTER (key));
  } else {
    days_list = g_slist_remove_link (days_list, list);
    g_hash_table_replace (marks_table, GUINT_TO_POINTER (key), days_list);
    g_slist_free_1 (list);
  }
  
  p_data->marks_need_update = TRUE;
  update (cv);
}


/*
20030810

2003 * 10000 = 20030000
+ 08 * 100 = 800
+ 10 * 1 = 10
= 20030810
*/
static              guint
get_hash_key_dmy (guint d, guint m, guint y)
{
  return (y * 10000 + m * 100 + d);
}

/**********************************************************************
* WEEKLY VIEW
*********************************************************************/

static void
all_time_num (KPCalendarView *cv)
{
  KPCalendarViewPrivateData *p_data;
  guint cols, rows;

  p_data = KP_CALENDAR_VIEW_PRIVATE_DATA (cv);
  cols = CVT (p_data).cols;
  rows = CVT (p_data).rows;

  if (CVT (p_data).days == NULL) {
    CVT (p_data).years = (guint **) kp_array_2d_alloc (1, 1, sizeof (guint));
    CVT (p_data).months = (guint **) kp_array_2d_alloc (1, 1, sizeof (guint));
    CVT (p_data).days = (guint **) kp_array_2d_alloc (1, 1, sizeof (guint));
    CVT (p_data).marks = (gchar ***) kp_array_2d_alloc (1, 1, sizeof (gchar *));
  }
  CVT (p_data).years[0][0] = p_data->year;
  CVT (p_data).months[0][0] = p_data->month;
  CVT (p_data).days[0][0] = p_data->day;
}


static void
day_num (KPCalendarView * cv)
{
  KPCalendarViewPrivateData *p_data;
  guint               cols, rows;

  p_data = KP_CALENDAR_VIEW_PRIVATE_DATA (cv);
  cols = CVT (p_data).cols;
  rows = CVT (p_data).rows;

  if (CVT (p_data).days == NULL) {
    CVT (p_data).years = (guint **) kp_array_2d_alloc (1, 1, sizeof (guint));
    CVT (p_data).months = (guint **) kp_array_2d_alloc (1, 1, sizeof (guint));
    CVT (p_data).days = (guint **) kp_array_2d_alloc (1, 1, sizeof (guint));
    CVT (p_data).marks = (gchar ***) kp_array_2d_alloc (1, 1, sizeof (gchar *));
  }
  CVT (p_data).years[0][0] = p_data->year;
  CVT (p_data).months[0][0] = p_data->month;
  CVT (p_data).days[0][0] = p_data->day;
}

static void
year_num (KPCalendarView * cv)
{
  KPCalendarViewPrivateData *p_data;
  guint               type = KP_VIEW_MODEL_TYPE_YEAR;
  guint             **months = cvt_data[type].months;
  guint             **days = cvt_data[type].days;
  guint               cols = cvt_data[type].cols;
  guint               rows = cvt_data[type].rows;
  guint               ndays_in_prev_month;
  guint               ndays_in_month;
  guint               first_day, day;
  guint               row, col, i;
  guint               month;
  guint               year;
  guint               s_row = 0;
  guint               s_col = 0;

  g_return_if_fail (KP_IS_CALENDAR_VIEW (cv));
  p_data = KP_CALENDAR_VIEW_PRIVATE_DATA (cv);
  month = p_data->month;
  year = p_data->year;

  if (days == NULL) {
    g_return_if_fail (months == NULL);

    CVT (p_data).marks = (gchar ***) kp_array_2d_alloc (rows, cols, sizeof (gchar *));
    CVT (p_data).years = (guint **) kp_array_2d_alloc (rows, cols, sizeof (guint));
    CVT (p_data).months = (guint **) kp_array_2d_alloc (rows, cols, sizeof (guint));
    CVT (p_data).days = (guint **) kp_array_2d_alloc (rows, cols, sizeof (guint));
  }

  for (i = 1; i <= 12; i++) {

    if (i <= 3)
      s_row = 0;
    if (i > 3 && i <= 6)
      s_row = 1;
    if (i > 6 && i <= 9)
      s_row = 2;
    if (i > 9 && i <= 12)
      s_row = 3;

    s_col = i - s_row * 3 - 1;

    /*  x x x (1,0) (2,0) (3,0)
     *  x x x (1,1) (2,1) (3,1)
     *  x x x
     *  x x x
     */

    ndays_in_month = kp_get_month_len (i, kp_leap (year));
    first_day = kp_day_of_week (year, i, 1);

    /* Week starts on monday, see below */
    first_day--;

    /* FIXME: this should be handled.. */
    /*  if (calendar->display_flags & CALENDAR_WEEK_START_MONDAY)
       first_day--;
       else
       first_day %= 7;
     */

    /* Compute days of previous month */
    if (i > 1)
      ndays_in_prev_month = kp_get_month_len (i - 1, kp_leap (year));
    else
      ndays_in_prev_month = kp_get_month_len (12, kp_leap (year));

    day = ndays_in_prev_month - first_day + 1;
    row = 0;
    if (first_day > 0) {
      for (col = 0; col < first_day; col++) {
        CVT (p_data).days[s_row * 6 + row][s_col * 7 + col] = 0;
        CVT (p_data).months[s_row * 6 + row][s_col * 7 + col] = (i == 1) ?
          12 : i - 1;
        CVT (p_data).years[s_row * 6 + row][s_col * 7 + col] = (i == 1) ? 
          p_data->year - 1 : p_data->year;
        day++;
      }
    }
    /* Compute days of current month */
    col = first_day;
    for (day = 1; day <= ndays_in_month; day++) {
      CVT (p_data).days[s_row * 6 + row][s_col * 7 + col] = day;        /*day */
      CVT (p_data).months[s_row * 6 + row][s_col * 7 + col] = i;
      CVT (p_data).years[s_row * 6 + row][s_col * 7 + col] = p_data->year;

      col++;
      if (col == 7) {
        row++;
        col = 0;
      }
    }
    /* Compute days of next month */
    day = 1;
    for (; row <= 5; row++) {
      for (; col <= 6; col++) {
        CVT (p_data).days[s_row * 6 + row][s_col * 7 + col] = 0;
        CVT (p_data).months[s_row * 6 + row][s_col * 7 + col] =
          (i == 12) ? 1 : i + 1;
        CVT (p_data).years[s_row * 6 + row][s_col * 7 + col] = 
          (i == 12) ? p_data->year + 1 : p_data->year;
        day++;
      }
      col = 0;
    }
  }
}

static void
week_num (KPCalendarView * cv)
{
  KPCalendarViewPrivateData *p_data;
  guint               year;
  guint               cols, rows;
  guint               i;
  GDate              *date;

  p_data = KP_CALENDAR_VIEW_PRIVATE_DATA (cv);
  cols = CVT (p_data).cols;
  rows = CVT (p_data).rows;

  year = p_data->year;
  date = g_date_new_dmy (p_data->day, p_data->month, p_data->year);
  g_date_subtract_days (date, g_date_get_weekday (date) - 1);

  g_return_if_fail (year > 1900);
  g_return_if_fail (year < 2200);

  if (CVT (p_data).days == NULL) {
    CVT (p_data).marks = (gchar ***) kp_array_2d_alloc (rows, cols, sizeof (gchar *));
    CVT (p_data).years = (guint **) kp_array_2d_alloc (rows, cols, sizeof (guint));
    CVT (p_data).months = (guint **) kp_array_2d_alloc (rows, cols, sizeof (guint));
    CVT (p_data).days = (guint **) kp_array_2d_alloc (rows, cols, sizeof (guint));
  }
  for (i = 0; i < rows; i++) {
    CVT (p_data).years[i][0] = g_date_get_year (date);
    CVT (p_data).months[i][0] = g_date_get_month (date);
    CVT (p_data).days[i][0] = g_date_get_day (date);
    g_date_add_days (date, 1);
  }
  g_date_free (date);
}

/**********************************************************************
* MONTHLY VIEW
*********************************************************************/

static void
month_num (KPCalendarView * cv)
{
  KPCalendarViewPrivateData *p_data;
  guint               type = KP_VIEW_MODEL_TYPE_MONTH;
  guint             **months = cvt_data[type].months;
  guint             **days = cvt_data[type].days;
  guint               cols = cvt_data[type].cols;
  guint               rows = cvt_data[type].rows;
  guint               ndays_in_prev_month;
  guint               ndays_in_month;
  guint               first_day, day;
  guint               row, col;
  guint               month;
  guint               year;

  g_return_if_fail (KP_IS_CALENDAR_VIEW (cv));
  p_data = KP_CALENDAR_VIEW_PRIVATE_DATA (cv);
  month = p_data->month;
  year = p_data->year;

  if (days == NULL) {
    g_return_if_fail (months == NULL);
  
    CVT (p_data).marks = (gchar ***) kp_array_2d_alloc (rows, cols, sizeof (gchar *));
    CVT (p_data).years = (guint **) kp_array_2d_alloc (rows, cols, sizeof (guint));
    CVT (p_data).months = (guint **) kp_array_2d_alloc (rows, cols, sizeof (guint));
    CVT (p_data).days = (guint **) kp_array_2d_alloc (rows, cols, sizeof (guint));
  }
  ndays_in_month = kp_get_month_len (month, kp_leap (year));
  first_day = kp_day_of_week (year, month, 1);

  /* Week starts on monday, see below */
  first_day--;

  /* FIXME: this should be handled.. */
  /*  if (calendar->display_flags & CALENDAR_WEEK_START_MONDAY)
     first_day--;
     else
     first_day %= 7;
   */

  /* Compute days of previous month */
  if (month > 1)
    ndays_in_prev_month = kp_get_month_len (month - 1, kp_leap (year));
  else
    ndays_in_prev_month = kp_get_month_len (12, kp_leap (year));

  /* This algorithm to count day numbers is taken from
   * GtkCalendar */

  day = ndays_in_prev_month - first_day + 1;
  row = 0;
  if (first_day > 0) {
    for (col = 0; col < first_day; col++) {
      CVT (p_data).days[row][col] = 0;
      CVT (p_data).months[row][col] = 0;
      CVT (p_data).years[row][col] = 0;
      day++;
    }
  }
  /* Compute days of current month */
  col = first_day;
  for (day = 1; day <= ndays_in_month; day++) {
    CVT (p_data).days[row][col] = day;
    CVT (p_data).months[row][col] = month;
    CVT (p_data).years[row][col] = p_data->year;

    col++;
    if (col == 7) {
      row++;
      col = 0;
    }
  }
  /* Compute days of next month */
  day = 1;
  for (; row <= 5; row++) {
    for (; col <= 6; col++) {
      CVT (p_data).days[row][col] = 0;
      CVT (p_data).months[row][col] = 0;
      CVT (p_data).years[row][col] = 0;
      day++;
    }
    col = 0;
  }
}

static void
basic_mark (KPCalendarView *cv, GString *buffer, guint col, guint row)
{
  KPCalendarViewPrivateData *p_data;
  GSList *list;
  guint key;
  guint n;
  
  p_data = KP_CALENDAR_VIEW_PRIVATE_DATA (cv);

  if (p_data->type == KP_VIEW_MODEL_TYPE_ALL_TIME)
    CVT (p_data).marks[0][0] = "Not supported";
  
  if (CVT (p_data).marks[row][col] != NULL) {
        
    key = get_hash_key_dmy (CVT (p_data).days[row][col],
                            CVT (p_data).months[row][col],
                            CVT (p_data).years[row][col]);
    
    list = g_hash_table_lookup (marks_table, GUINT_TO_POINTER (key));

    n = (list) ? g_slist_length (list) : 0;
       
    if (n > 1)
      g_string_printf (buffer, "%u.%u [%u/%u]\n%s",
                       CVT (p_data).days[row][col],
                       CVT (p_data).months[row][col],
                       g_slist_index (list, CVT (p_data).marks[row][col]) + 1, n,
                       CVT (p_data).marks[row][col]);
  else
    g_string_printf (buffer, "%u.%u\n%s",
                     CVT (p_data).days[row][col],
                     CVT (p_data).months[row][col],
                     CVT (p_data).marks[row][col]);
      
  } else if (CVT (p_data).days[row][col] != 0)
    g_string_printf (buffer, "%u.%u",
                     CVT (p_data).days[row][col],
                     CVT (p_data).months[row][col]);
  else
    g_string_printf (buffer, "%s", "");
}      

static void
year_mark (KPCalendarView *cv, GString *buffer, guint col, guint row)
{
  KPCalendarViewPrivateData *p_data;
  GSList *list = NULL;
  guint key;
  guint n;
  
  p_data = KP_CALENDAR_VIEW_PRIVATE_DATA (cv);

  if (CVT (p_data).marks[row][col] != NULL) {
    key = get_hash_key_dmy (CVT (p_data).days[row][col],
                            CVT (p_data).months[row][col],
                            CVT (p_data).years[row][col]);
    
    list = g_hash_table_lookup (marks_table, GUINT_TO_POINTER (key));
  }

  n = (list) ? g_slist_length (list) : 0;

/*  if (n > 0)
    g_string_printf (buffer, "<span size=\"x-small\">%u</span>", n);
  else if (CVT (p_data).days[row][col] > 0)
    g_string_printf (buffer, "<span size=\"x-small\">-</span>");
  else
    g_string_printf (buffer, "<span size=\"x-small\">%s</span>", "");*/

  if (n > 0)
    g_string_printf (buffer, "%u", n);
  else if (CVT (p_data).days[row][col] > 0)
    g_string_printf (buffer, "-");
  else
    g_string_printf (buffer, "%s", "");
}

/*
* Update GUI.
*/
static void
update (KPCalendarView *cv)
{
  KPCalendarViewPrivateData *p_data;

  g_return_if_fail (KP_IS_CALENDAR_VIEW (cv));
  p_data = KP_CALENDAR_VIEW_PRIVATE_DATA (cv);
  
  if (CVT (p_data).number_func)
    CVT (p_data).number_func (cv);
  update_marks (cv);

  kp_calendar_view_set_title_date (cv);
 
  gtk_widget_queue_draw (GTK_WIDGET (cv));
  
  if (kp_calendar_view_get_active (cv))
    find_index (cv, &p_data->active_x, &p_data->active_y);
}


/**********************************************************************
 * CALLBACKs
 **********************************************************************/

static void
kp_calendar_view_destroy (GtkObject *object)
{
  KPCalendarViewPrivateData *p_data;
  guint i;

  g_return_if_fail (KP_IS_CALENDAR_VIEW (object));
  p_data = KP_CALENDAR_VIEW_PRIVATE_DATA (KP_CALENDAR_VIEW (object));
  if (KP_CALENDAR_VIEW (object)->private_data == NULL)
    return;

  for (i = 0; i < KP_VIEW_MODEL_TYPE_N; i++) {
    
    if (p_data->cvt[i].days != NULL) {
      kp_array_2d_free ((void **)p_data->cvt[i].days,
                        p_data->cvt[i].rows,
                        p_data->cvt[i].cols);
      p_data->cvt[i].days = NULL;
    }
    if (p_data->cvt[i].months) {
      kp_array_2d_free ((void **) p_data->cvt[i].months,
                        p_data->cvt[i].rows,
                        p_data->cvt[i].cols);
      p_data->cvt[i].months = NULL;
    }
    if (p_data->cvt[i].marks) {
      kp_array_2d_free ((void **) p_data->cvt[i].marks,
                        p_data->cvt[i].rows,
                        p_data->cvt[i].cols);
      p_data->cvt[i].marks = NULL;
    }
  }

  /* g_hash_table_destroy (marks_table); */

  if (p_data)
    g_free (p_data);
  
  KP_CALENDAR_VIEW (object)->private_data = NULL;
}



static GdkColor
*get_color (const gchar *color, GtkWidget *widget)
{
  GdkColor *c;
  
  if (!allocated_colors)
    allocated_colors = g_hash_table_new_full (g_direct_hash, g_str_equal,
                                             (GDestroyNotify) gdk_color_free,
                                              g_free);    
  
  c = g_hash_table_lookup (allocated_colors, color);
  if (c)
    return c;
  
  c = g_malloc (sizeof *c);
  gdk_color_parse (color, c);
  
  if (gdk_colormap_alloc_color (gtk_widget_get_colormap (widget), c, FALSE, TRUE))  
    g_hash_table_insert (allocated_colors, (gpointer)color, c);
  
  return c;
}


static void
cv_popup_add_split_workout (GtkMenuItem *item, KPCalendarView *cv)
{
  KPCalendarViewPrivateData *p_data;
  GtkWidget *dialog;
  KPDate date;

  p_data = KP_CALENDAR_VIEW_PRIVATE_DATA (cv);
  g_return_if_fail (p_data != NULL);
 
  kp_calendar_view_get_date (cv, &date.d, &date.m, &date.y);
  
  dialog = kp_new_split_workout_dialog_new (&date, p_data->log);
  gtk_dialog_run (GTK_DIALOG (dialog));
  gtk_widget_destroy (dialog);
}

static void
cv_popup_add_workout (GtkMenuItem *item, KPCalendarView *cv)
{
  KPCalendarViewPrivateData *p_data;
  GtkWidget *dialog;
  KPDate date;

  p_data = KP_CALENDAR_VIEW_PRIVATE_DATA (cv);
  g_return_if_fail (p_data != NULL);

  kp_calendar_view_get_date (cv, &date.d, &date.m, &date.y);
 
  dialog = kp_new_workout_dialog_new (&date, p_data->log);
  gtk_dialog_run (GTK_DIALOG (dialog));
  gtk_widget_destroy (dialog);
}

static void
cv_popup_add_comment (GtkMenuItem *item, KPCalendarView *cv)
{
  KPCalendarViewPrivateData *p_data;
  KPComment *comment;
  GtkWidget *dialog;
  guint d, m, y;
  GDate *date;

  p_data = KP_CALENDAR_VIEW_PRIVATE_DATA (cv);
  g_return_if_fail (p_data != NULL);

  kp_calendar_view_get_date (cv, &d, &m, &y);
  date = g_date_new_dmy (d, m, y);
  comment = kp_comment_new (NULL, NULL);
  
  dialog = kp_new_comment_dialog_new (date, comment);
  gtk_dialog_run (GTK_DIALOG (dialog));
  gtk_widget_destroy (dialog);
  g_date_free (date);

  if (strlen (comment->title->str) > 0) 
    kp_training_log_add (p_data->log, KP_CALENDAR_ENTRY (comment));
  else 
    g_object_unref (G_OBJECT (comment));
}


static void
cv_popup_properties (GtkMenuItem *item, KPCalendarView *cv)
{
  KPCalendarViewPrivateData *p_data;
  KPCalendarEntry *entry;
  GtkWidget *dialog;
  const gchar *mark;
  guint d, m, y;
  
  p_data = KP_CALENDAR_VIEW_PRIVATE_DATA (cv);
  
  kp_calendar_view_get_date (cv, &d, &m, &y);
  g_return_if_fail (g_date_valid_dmy (d, m, y));

  mark = kp_calendar_view_get_current_mark (cv);
  entry = kp_training_log_get_entry (p_data->log, d, m, y, mark);
  g_return_if_fail (entry != NULL);
  
  dialog = kp_calendar_entry_info_dialog_new (entry);
  gtk_dialog_run (GTK_DIALOG (dialog));
  gtk_widget_destroy (dialog);
}

static void
cv_popup_delete (GtkMenuItem *item, KPCalendarView *cv)
{
  KPCalendarViewPrivateData *p_data;
  KPCalendarEntry *entry;
  const gchar *mark;
  guint d, m, y;
  
  p_data = KP_CALENDAR_VIEW_PRIVATE_DATA (cv);
  
  kp_calendar_view_get_date (cv, &d, &m, &y);
  g_return_if_fail (g_date_valid_dmy (d, m, y));

  mark = kp_calendar_view_get_current_mark (cv);
  entry = kp_training_log_get_entry (p_data->log, d, m, y, mark);
  g_return_if_fail (entry != NULL);

  kp_debug ("Removed from KPTrainingLog: %s\n",
            kp_calendar_entry_to_string (entry));

  kp_training_log_remove (p_data->log, entry);
}

static gboolean
kp_calendar_view_expose (GtkWidget *widget, GdkEventExpose *event)
{
  KPCalendarViewPrivateData *p_data;
  static GdkColor    *color = NULL;
  static GString     *buffer = NULL;
  GSList             *lines;
  GdkGC              *gc;
  PangoAttrList      *attrs;
  PangoAttribute     *attr_size;
  PangoLayout        *layout;
  PangoRectangle      rec;
  guint               w, h;
  guint               i, j, x, y, mod_x, mod_y, l;
  guint               height, cols, rows;
  guint               line_height;
  guint               m_left, m_right, m_top, m_bottom; /* marginals */
  /* These are gaps-related (in yearview, between the months for example) */
  guint               skip_p_x, skip_p_y, p_x, p_y;    
  guint               n_skips_x, n_skips_y;
  guint               skip_x, skip_y;
  guint               fix_x, fix_y; 
  guint               col,row;
  guint               w2, h2;

  if (!GTK_WIDGET_DRAWABLE (widget))
    return FALSE;

#if 0
  GTimer *timer;
  guint fuck;
  timer = g_timer_new ();
  for (fuck=0; fuck < 1; fuck++)
  {
#endif 

  if (!buffer)
    buffer = g_string_sized_new (128);
  
  gc = widget->style->fg_gc[GTK_WIDGET_STATE (widget)];

  p_data = KP_CALENDAR_VIEW_PRIVATE_DATA (KP_CALENDAR_VIEW (widget));

  gdk_window_begin_paint_region (widget->window, event->region);
  gdk_window_set_background (widget->window, get_color ("#FFFFFF", widget));
  attrs = pango_attr_list_new ();
  attr_size = pango_attr_size_new ((CVT (p_data).default_font_size * 10000.0));
  pango_attr_list_insert (attrs, attr_size);
  
  if (!color) {
    color = g_malloc (sizeof *color);
    gdk_color_parse ("black", color);
    gdk_color_alloc (gtk_widget_get_colormap (widget), color);
  }

  rows = CVT (p_data).rows;
  cols = CVT (p_data).cols;
  layout = gtk_widget_create_pango_layout (widget, "foo");
  pango_layout_get_pixel_extents (layout, &rec, NULL);
  line_height = rec.height;

  m_right = CVT (p_data).margin_right;
  m_bottom = CVT (p_data).margin_bottom;
  m_left = CVT (p_data).margin_left;
  m_top = CVT (p_data).margin_top;

  skip_p_x = CVT (p_data).skip_padding_n_x;
  skip_p_y = CVT (p_data).skip_padding_n_y;

  n_skips_x = (skip_p_x < 1) ? 0 : (cols / skip_p_x) - 1;
  n_skips_y = (skip_p_y < 1) ? 0 : (rows / skip_p_y) - 1;

  p_x = CVT (p_data).padding_x;
  p_y = CVT (p_data).padding_y;
  
  w = (widget->allocation.width - n_skips_x * p_x - m_right - m_left) / cols;
  h = (widget->allocation.height - n_skips_y * p_y - m_top - m_bottom) / rows;

  mod_x = (widget->allocation.width - n_skips_x * p_x) % cols;
  mod_y = (widget->allocation.height - n_skips_y * p_y) % rows;

  gdk_gc_set_foreground (gc, get_color ("#FFFFFF", widget));
  gdk_gc_set_background (gc, get_color ("#FFFFFF", widget));
  gdk_draw_rectangle (widget->window, gc, TRUE, 0, 0,
                      widget->allocation.width-1,
                      widget->allocation.height-1);
  gdk_gc_set_foreground (gc, get_color ("black", widget));

  pango_layout_set_width (layout, (guint)w * 1000);
  pango_layout_set_wrap (layout, PANGO_WRAP_CHAR);
  pango_layout_set_attributes (layout, attrs);

  pango_attr_list_unref (attrs);
  
  gdk_gc_set_foreground (gc, get_color ("#CCCCCC", widget));
 
  for (i=0, row=0, fix_y=0; i < rows; i++, row++) {

    row = (row == skip_p_y) ? 0 : row;
    skip_y = (n_skips_y > 0) ? (i / skip_p_y) * p_y : 0;
   
    h2 = h + (mod_y && i < mod_y);
    y = m_top + skip_y + i * h + fix_y;

    for (j=0, col=0, fix_x=0; j < cols; j++, col++) {
      col = (col == skip_p_x) ? 0 : col;
      if (CVT (p_data).mark_func)
        CVT (p_data).mark_func (KP_CALENDAR_VIEW (widget), buffer, j, i);
   
      /* In Year View it's too slow to use set_markup () */
      (CVT (p_data).type == KP_VIEW_MODEL_TYPE_YEAR)
        ? pango_layout_set_text (layout, buffer->str, -1)
        : pango_layout_set_markup (layout, buffer->str, -1);
      
      pango_layout_get_pixel_size (layout, NULL, &height);

      skip_x = (n_skips_x > 0) ? (j / skip_p_x) * p_x : 0;
      x = m_left + skip_x + j * w + fix_x;
     
      fix_x += (mod_x && j < mod_x);
      w2 = w + (mod_x && j < mod_x);
      
      /* Paint active day with different color */
      if (kp_calendar_view_get_active (KP_CALENDAR_VIEW (widget))
        && CVT (p_data).days[i][j] == p_data->day
        && CVT (p_data).months[i][j] == p_data->month
        && CVT (p_data).years[i][j] == p_data->year) {
        
        gdk_gc_set_foreground (gc, SELECTED_BG_COLOR (widget));
        gdk_draw_rectangle (widget->window, gc, TRUE, x+1, y+1, w2-1, h2-1);
      }
      gdk_gc_set_foreground (gc, get_color ("#CCCCCC", widget));

      /* top, left, bottom, right */
      if (row == 0) gdk_draw_line (widget->window, gc, x, y, x + w2, y);
      if (col == 0) gdk_draw_line (widget->window, gc, x, y, x, y + h2);
      gdk_draw_line (widget->window, gc, x+w2, y, x+w2, y+h2);
      gdk_draw_line (widget->window, gc, x, y+h2, x+w2, y+h2);

      gdk_gc_set_foreground (gc, get_color ("black", widget));

      lines = pango_layout_get_lines (layout);
      for (l=line_height; l < h && lines && lines->data; lines = lines->next) {
        gdk_draw_layout_line (widget->window, gc, x, y + l, lines->data);
        l += line_height + p_data->line_spacing;
      }
    }
    fix_y += (mod_y && i < mod_y);
  }
  gdk_window_end_paint (widget->window);

/* Testing stuff */
#if 0
  }
  g_timer_stop (timer);
  g_print ("Seconds: %.4f\n", g_timer_elapsed (timer, NULL));
#endif
  
  return TRUE;
}

static gboolean
show_popup_menu (KPCalendarView *cv, GdkEventButton *event)
{
  KPCalendarViewPrivateData *p_data;
  gboolean day_is_empty;
  GtkWidget *child;
  gchar date_str[32];
  GDate *date;
  guint d, m, y;
  
  kp_calendar_view_get_date (cv, &d, &m, &y);
  g_return_val_if_fail (g_date_valid_dmy (d, m, y), FALSE);
  date = g_date_new_dmy (d, m, y);
  (void) g_date_strftime (date_str, sizeof (date_str)-1, "%x", date);
 
  p_data = KP_CALENDAR_VIEW_PRIVATE_DATA (cv);
  
  g_return_val_if_fail (p_data != NULL, FALSE);

  if (p_data->popup_menu == NULL)
    return FALSE;

  /* if there are no workouts in the day, make properties and
   * delete buttons inactive */
  day_is_empty = CVT (p_data).marks[p_data->active_y][p_data->active_x] == NULL;

  gtk_widget_set_sensitive (p_data->popup_mi_properties, !day_is_empty);
  gtk_widget_set_sensitive (p_data->popup_mi_delete, !day_is_empty);

  if (p_data->popup_menu_title) {
    child = gtk_bin_get_child (GTK_BIN (p_data->popup_menu_title));
    if (GTK_IS_LABEL (child))
      gtk_label_set_text (GTK_LABEL (child), date_str);
    else
      g_return_val_if_reached (FALSE);
  }
  gtk_menu_popup (GTK_MENU (p_data->popup_menu),
                  NULL, NULL, NULL, NULL,
                 (event) ? event->button : 0,
                  gdk_event_get_time ((GdkEvent *) event));
  return TRUE;
}

static gboolean
kp_calendar_view_button_press (GtkWidget *widget, GdkEventButton *event)
{
  KPCalendarViewPrivateData *p_data;
  GSList *list_tmp;
  GSList *list = NULL;
  KPDate *date;
  guint key;
  gint row;
  gint col;
  
  g_return_val_if_fail (KP_IS_CALENDAR_VIEW (widget), TRUE);
  p_data = KP_CALENDAR_VIEW_PRIVATE_DATA (KP_CALENDAR_VIEW (widget));
  

  row = get_row_from_y (KP_CALENDAR_VIEW (widget), event->y);
  col = get_col_from_x (KP_CALENDAR_VIEW (widget), event->x);

  /* Clicked pixel isn't in any row or column */
  if (row < 0 || col < 0)
    return FALSE;

  p_data->active_x = col;
  p_data->active_y = row;

  if (!GTK_WIDGET_HAS_FOCUS (widget))
    gtk_widget_grab_focus (widget);

  if (event->button == 3) {
    update (KP_CALENDAR_VIEW (widget));
    return show_popup_menu (KP_CALENDAR_VIEW (widget), event);
  }
 
  /* In yearview, empty columns are marked as '0' in **days */
  if (CVT (p_data).days[row][col] > 0 && event->type == GDK_BUTTON_PRESS) {

    p_data->month = CVT (p_data).months[row][col];
    p_data->year = CVT (p_data).years[row][col];
    p_data->day = CVT (p_data).days[row][col];
    
    date = kp_date_new_dmy (p_data->day, p_data->month, p_data->year);
    g_signal_emit (G_OBJECT (widget),
                   kp_calendar_view_signals[DAY_SELECTED_SIGNAL], 0, date);
        
    key = get_hash_key_dmy (p_data->day, p_data->month, p_data->year);
    list_tmp = g_hash_table_lookup (marks_table, GUINT_TO_POINTER (key));
  
    if (list_tmp && CVT (p_data).marks[row][col])
      list = g_slist_nth (list_tmp, g_slist_index (list_tmp,
                                    CVT (p_data).marks[row][col]));
    
    /* Get next workout */
    if (list && list->next)
      CVT (p_data).marks[row][col] = list->next->data;
    else if (list_tmp)
      CVT (p_data).marks[row][col] = list_tmp->data;
    
    update (KP_CALENDAR_VIEW (widget));
  }
  

  return FALSE;
}

static gdouble
row_height (KPCalendarView * cv)
{
  KPCalendarViewPrivateData *p_data;
  guint               ah;
  guint               skip_p_y;

  p_data = KP_CALENDAR_VIEW_PRIVATE_DATA (cv);

  skip_p_y = (CVT (p_data).skip_padding_n_y < 1)
    ? 1 : CVT (p_data).skip_padding_n_y;

  ah = (GTK_WIDGET (cv)->allocation.height -
        - CVT (p_data).margin_top - CVT (p_data).margin_bottom
        - ((CVT (p_data).rows / skip_p_y) - 1) * CVT (p_data).padding_y);

  return ((gdouble) ah / (gdouble) CVT (p_data).rows);
}

/*
 * Returns the row which point is in.
 * If the point isn't in any row, returns -1.
 */
static              gint
get_row_from_y (KPCalendarView * cv, gdouble y)
{
  KPCalendarViewPrivateData *p_data;
  gdouble             y_row_start;
  gdouble             height;
  gdouble             skip_y;
  guint               i;

  p_data = KP_CALENDAR_VIEW_PRIVATE_DATA (cv);
  height = row_height (cv);

  for (i = 0; i < CVT (p_data).rows; i++) {
    skip_y = (CVT (p_data).skip_padding_n_y == 0) ? 0
      : (i / CVT (p_data).skip_padding_n_y) * CVT (p_data).padding_y;

    y_row_start = CVT (p_data).margin_top + i * height + skip_y;

    if ((y > y_row_start) && (y < (y_row_start + height))) {
      return i;
    }
  }
  return -1;
}

static              gdouble
col_width (KPCalendarView * cv)
{
  KPCalendarViewPrivateData *p_data;
  guint               aw;
  guint               skip_p_x;

  p_data = KP_CALENDAR_VIEW_PRIVATE_DATA (cv);

  skip_p_x = (CVT (p_data).skip_padding_n_x < 1)
    ? 1 : CVT (p_data).skip_padding_n_x;

  aw = (GTK_WIDGET (cv)->allocation.width
        - CVT (p_data).margin_left - CVT (p_data).margin_right
        - ((CVT (p_data).cols / skip_p_x) - 1) * CVT (p_data).padding_x);

  return ((gdouble) aw / (gdouble) CVT (p_data).cols);
}

/*
 * Returns the column which point is in.
 * If the point isn't in any column, returns -1.
 */
static              gint
get_col_from_x (KPCalendarView * cv, gdouble x)
{
  KPCalendarViewPrivateData *p_data;
  gdouble             x_col_start;
  gdouble             width;
  guint               skip_x;
  guint               i;

  p_data = KP_CALENDAR_VIEW_PRIVATE_DATA (cv);
  width = col_width (cv);

  for (i = 0; i < CVT (p_data).cols; i++) {
    skip_x = (CVT (p_data).skip_padding_n_x == 0) ? 0
      : (i / CVT (p_data).skip_padding_n_x) * CVT (p_data).padding_x;

    x_col_start = CVT (p_data).margin_left + i * width + skip_x;

    if ((x > x_col_start) && (x < (x_col_start + width))) {
      return i;
    }
  }
  return -1;
}

static gboolean
kp_calendar_view_key_press (GtkWidget * widget, GdkEventKey * event)
{
  KPCalendarViewPrivateData *p_data;
  guint x, y;
  
  g_return_val_if_fail (KP_IS_CALENDAR_VIEW (widget), TRUE);
  p_data = KP_CALENDAR_VIEW_PRIVATE_DATA (KP_CALENDAR_VIEW (widget));

  x = p_data->active_x;
  y = p_data->active_y;
  
  switch (event->keyval)
  {
    case GDK_KP_Left:
    case GDK_Left:
      if (x > 0) {
        x--; 
      } else {
        x = CVT (p_data).cols - 1;
      }
    break;

    case GDK_KP_Right:
    case GDK_Right:
      if (x < (CVT (p_data).cols - 1))
        x++;
      else
        x = 0;
    break;

    case GDK_KP_Up:
    case GDK_Up:
      if (y > 0)
        y--;
      else
        y = CVT (p_data).rows - 1;
    break;

    case GDK_KP_Down:
    case GDK_Down:
      if (y < (CVT (p_data).rows - 1))
        y++;
      else
        y = 0;
    break;

    case GDK_KP_Home:
    case GDK_Home:
      find_first (KP_CALENDAR_VIEW (widget), &x, &y);
    break;
    
    case GDK_KP_End:
    case GDK_End:
      find_last (KP_CALENDAR_VIEW (widget), &x, &y);
    break;
  }
  /* Not a day, just an empty cell */
  if (CVT (p_data).days[y][x] == 0) 
    return TRUE;

  p_data->active_x = x;
  p_data->active_y = y;
  p_data->day = CVT (p_data).days[p_data->active_y][p_data->active_x];
  p_data->month = CVT (p_data).months[p_data->active_y][p_data->active_x];
  p_data->year = CVT (p_data).years[p_data->active_y][p_data->active_x];

  update (KP_CALENDAR_VIEW (widget));
  
  return TRUE;
}


static void
find_last (KPCalendarView * cv, guint * x, guint * y)
{
  KPCalendarViewPrivateData *p_data;
  guint i, j;
  
  g_return_if_fail (KP_IS_CALENDAR_VIEW (cv));
  p_data = KP_CALENDAR_VIEW_PRIVATE_DATA (cv);

  for (i = CVT (p_data).rows; i > 0; i--)
    for (j = CVT (p_data).cols; j > 0; j--)
      if (CVT (p_data).days[i-1][j-1] > 0) {
        *x = j - 1;
        *y = i - 1;
        return;
      }
  g_assert_not_reached ();
}


static void
find_first (KPCalendarView *cv, guint * x, guint * y)
{
  KPCalendarViewPrivateData *p_data;
  guint i, j;
  
  g_return_if_fail (KP_IS_CALENDAR_VIEW (cv));
  p_data = KP_CALENDAR_VIEW_PRIVATE_DATA (cv);

  for (i = 0; i < CVT (p_data).rows; i++)
    for (j = 0; j < CVT (p_data).cols; j++)
      if (CVT (p_data).days[i][j] > 0) {
        *x = j;
        *y = i;
        return;
      }
  g_assert_not_reached ();
}

static void
find_index (KPCalendarView *cv, guint *x, guint *y)
{
  KPCalendarViewPrivateData *p_data;
  guint i, j;
  
  g_return_if_fail (KP_IS_CALENDAR_VIEW (cv));
  p_data = KP_CALENDAR_VIEW_PRIVATE_DATA (cv);
 
  for (i=0; i < CVT (p_data).rows; i++)
    for (j=0; j < CVT (p_data).cols; j++)
      if (CVT (p_data).days[i][j] == p_data->day
       && CVT (p_data).months[i][j] == p_data->month
       && CVT (p_data).years[i][j] == p_data->year) {

        *y = i;
        *x = j;
        return;
      }
  g_assert_not_reached ();
}

