/*
 * page_export_stills.cc Notebook Firewire/AVI/Still Frame Export Page Object
 * Copyright (C) 2001 Dan Dennedy <dan@dennedy.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <vector>
#include <iostream>
using std::cout;
using std::endl;

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <gnome.h>
#include <stdio.h>
#include <stdlib.h>

#include "page_export_audio.h"
#include "preferences.h"
#include "kino_common.h"
#include "frame.h"
#include "page_editor.h"
#include "message.h"

extern "C" {
	#include "support.h"
	#include "callbacks.h"
	#include "interface.h"
	#include "commands.h"

#include <string.h>

/** WAV File Production
*/

typedef struct {
	char riff[ 4 ];
	int length;
	char type[ 4 ];
}
RIFFChunkType, *RIFFChunk;

typedef struct {
	char format[ 4 ];
	int length;
	short filler;
	short channels;
	int rate;
	int bytespersecond;
	short bytespersample;
	short bitspersample;
}
FORMATChunkType, *FORMATChunk;

typedef struct {
	char data[ 4 ];
	int length;
}
DATAChunkType, *DATAChunk;

typedef struct {
	FILE *file;
	RIFFChunk riff;
	FORMATChunk format;
	DATAChunk data;
}
WAVStructType, *WAVStruct;

static RIFFChunk RIFFChunk_Init( ) {
	RIFFChunk chunk = (RIFFChunk)malloc( sizeof( RIFFChunkType ) );
	memcpy( chunk->riff, "RIFF", 4 );
	chunk->length = 4 + sizeof( FORMATChunkType ) + sizeof( DATAChunkType );
	memcpy( chunk->type, "WAVE", 4 );
	return chunk;
}

static FORMATChunk FORMATChunk_Init( short channels, int rate, int bytespersample ) {
	FORMATChunk format = (FORMATChunk)malloc( sizeof( FORMATChunkType ) );
	memset( format, 0, sizeof( FORMATChunkType ) );
	memcpy( format->format, "fmt ", 4 );
	format->length = 0x10;
	format->filler = 0x01;
	format->channels = channels;
	format->rate = rate;
	format->bytespersecond = rate * channels * bytespersample;
	format->bytespersample = bytespersample * channels;
	format->bitspersample = bytespersample * 8;
	return format;
}

static DATAChunk DATAChunk_Init( ) {
	DATAChunk data = (DATAChunk)malloc( sizeof( DATAChunkType ) );
	memset( data, 0, sizeof( DATAChunkType ) );
	memcpy( data->data, "data", 4 );
	return data;
}

static void WAVStruct_WriteHeader( WAVStruct wav ) {
	rewind( wav->file );
	fwrite( wav->riff, sizeof( RIFFChunkType ), 1, wav->file );
	fwrite( wav->format, sizeof( FORMATChunkType ), 1, wav->file );
	fwrite( wav->data, sizeof( DATAChunkType ), 1, wav->file );
}

static WAVStruct WAVStruct_Init( char *file, short channels, int frequency, 
						  short bytespersample ) {
	WAVStruct wav = (WAVStruct)malloc( sizeof( WAVStructType ) );
	wav->file = fopen( file, "w" );
	wav->riff = RIFFChunk_Init();
	wav->format = FORMATChunk_Init( channels, frequency, bytespersample );
	wav->data = DATAChunk_Init();
	WAVStruct_WriteHeader( wav );
	return wav;
}

static void WAVStruct_WriteData( WAVStruct wav, void *data, int length ) {
	wav->riff->length += length;
	wav->data->length += length;
	fwrite( data, length, 1, wav->file );
}

static void WAVStruct_Close( WAVStruct wav ) {
	WAVStruct_WriteHeader( wav );
	fclose( wav->file );
	free( wav->riff );
	free( wav->format );
	free( wav->data );
	free( wav );
}

}

/** Constructor for page.

  	\param common	common object to which this page belongs
*/

ExportAudio::ExportAudio( PageExport *exportPage, KinoCommon *common ) 
{
	cout << "> Creating ExportAudio Page" << endl;
	this->exportPage = exportPage;
	this->common = common;
	recordButton = GTK_TOGGLE_BUTTON( lookup_widget( common->getWidget(), "togglebutton_export_audio_record" ) );
	stopButton = GTK_TOGGLE_BUTTON( lookup_widget( common->getWidget(), "togglebutton_export_audio_stop" ) );

	fileEntry = GTK_ENTRY( gnome_file_entry_gtk_entry( GNOME_FILE_ENTRY( 
		lookup_widget( common->getWidget(), "gnome_fileentry_export_audio" ) ) ) );
	allToggle = GTK_TOGGLE_BUTTON( lookup_widget( common->getWidget(), "radiobutton_export_audio_all") );
	rangeToggle = GTK_TOGGLE_BUTTON( lookup_widget( common->getWidget(), "radiobutton_export_audio_range" ) );
	startSpin = GTK_ENTRY( lookup_widget( exportPage->getWidget(), "spinbutton_export_audio_start" ) );
	endSpin = GTK_ENTRY( lookup_widget( exportPage->getWidget(), "spinbutton_export_audio_end" ) );	

}

/** Destructor for page.
*/

ExportAudio::~ExportAudio() {
	cout << "> Destroying ExportAudio Page" << endl;
}

/** Start of page.
*/

void ExportAudio::start() {
	cout << ">> Starting ExportAudio" << endl;
	
}

/** Define active widgets.
*/

gulong ExportAudio::activate() {
	cout << ">> ExportAudio activate" << endl;
	return SCENE_LIST;
}

/** Leaving the page
*/

void ExportAudio::clean() {
	cout << ">> Leaving ExportAudio" << endl;
}


/** start exporting audio
*/
void ExportAudio::startExport() {
	char buffer[ 10240 ];
	int size;
	static Frame frame;
	int begin = 0;
	int end = 0;
	char file[512];
	char filename[512];

	gtk_toggle_button_set_active( recordButton, TRUE );
	gtk_toggle_button_set_active( stopButton, FALSE );
	exportPage->exportMutex = false;

	strcpy( file, gtk_entry_get_text( fileEntry ) );

	if ( strcmp(file, "") ) {

		if (gtk_toggle_button_get_active( allToggle )) {
			begin = 0;
			end = common->getPlayList()->GetNumFrames() - 1;
		} 
		else if (gtk_toggle_button_get_active( rangeToggle )) {
			begin = atoi( gtk_entry_get_text( startSpin ) );
			end = atoi( gtk_entry_get_text( endSpin ) );
		}
		exportPage->resetProgress();
	
		// parse the base filename and extension 
		strcpy( filename, file);

		cout << "Outputting wav to " << filename << " from " << begin << " to " << end << endl;

		// Get a sample frame to obtain recording info
		common->getPlayList()->GetFrame( begin, frame );

		short channels = (short)frame.decoder->audio->num_channels;
		int frequency = frame.decoder->audio->frequency;
		//short bytespersample = channels * frequency * 2;
		//size = frame.ExtractAudio( buffer );
		WAVStruct wav = WAVStruct_Init( filename, channels, frequency, 2 );

		for (int i = begin; i <= end && exportPage->isExporting; i++ ) {
			common->getPlayList()->GetFrame( i, frame );
		    size = frame.ExtractAudio( buffer );
			WAVStruct_WriteData( wav, buffer, size );
			exportPage->updateProgress( (gfloat) (i - begin) / (gfloat) (end - begin) );
		}
		
		WAVStruct_Close( wav );

		exportPage->isExporting = false;
		exportPage->exportMutex = true;
		gtk_toggle_button_set_active( recordButton, FALSE );
		gtk_toggle_button_set_active( stopButton, TRUE );
		exportPage->exportMutex = false;

	} else {
		modal_message( "You must enter a filename." );

		exportPage->isExporting = false;
		exportPage->exportMutex = true;
		gtk_toggle_button_set_active( recordButton, FALSE );
		gtk_toggle_button_set_active( stopButton, TRUE );
		exportPage->exportMutex = false;
	}
}


/** stop the exporting still frames
*/
void ExportAudio::stopExport() {
	/* do nothing, let the export page clear the export flag
	   to fall out of the loop.
	*/
}

/** put the scene begin and end frame numbers into spinners

    \param i the numerical index position of the scene in the playlist1
*/
void ExportAudio::selectScene( int i ) {
	int begin = 0;
	int end = 0;
	GtkAdjustment *adjust = gtk_spin_button_get_adjustment( GTK_SPIN_BUTTON( startSpin ) );
	
	gtk_toggle_button_set_active( rangeToggle, TRUE );
	
	cout << ">>> ExportAudio selectScene " << i << endl;
	vector <int> scene = common->getPageEditor()->GetScene();

	begin = i == 0 ? 0 : scene[ i - 1 ];
	adjust->lower = 0;
	adjust->upper = scene[ scene.size() - 1 ];
	gtk_spin_button_set_value( GTK_SPIN_BUTTON( startSpin ), begin );
	gtk_signal_emit_by_name( GTK_OBJECT( adjust ), "changed" );

	adjust = gtk_spin_button_get_adjustment( GTK_SPIN_BUTTON( endSpin ) );
	adjust->lower = 0;
	adjust->upper = scene[ scene.size() - 1 ];
	end = scene[i];	
	gtk_spin_button_set_value( GTK_SPIN_BUTTON( endSpin ), end );
	gtk_signal_emit_by_name( GTK_OBJECT( adjust ), "changed" );
}

// callbacks
extern "C" {
	extern KinoCommon *common;

	void
	on_togglebutton_export_audio_stop_toggled
	                                        (GtkToggleButton *togglebutton,
	                                        gpointer         user_data)
	{
		stopExport();
	}
	
	
	void
	on_togglebutton_export_audio_record_toggled
	                                        (GtkToggleButton *togglebutton,
	                                        gpointer         user_data)
	{
		startExport();
	}
	
	
	gboolean
	on_spinbutton_export_audio_range_start_focus_in_event
	                                        (GtkWidget       *widget,
	                                        GdkEventFocus   *event,
	                                        gpointer         user_data)
	{
		gtk_toggle_button_set_active( 
			GTK_TOGGLE_BUTTON( lookup_widget( widget, "radiobutton_export_stills_range" ) ),
			TRUE );
		return FALSE;
	}
	
	
	gboolean
	on_spinbutton_export_audio_range_end_focus_in_event
	                                        (GtkWidget       *widget,
	                                        GdkEventFocus   *event,
	                                        gpointer         user_data)
	{
		gtk_toggle_button_set_active( 
			GTK_TOGGLE_BUTTON( lookup_widget( widget, "radiobutton_export_stills_range" ) ),
			TRUE );
		return FALSE;
	}

}
