// $Id: jogshuttle.cc,v 1.5 2001/11/18 21:13:25 ddennedy Exp $
/*
 * Copyright (C) 2001 Tomoaki Hayasaka <hayasakas@postman.riken.go.jp>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
 
 /// Support for special USB Jog/Shuttle input controllers

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "jogshuttle.h"
#include <fcntl.h>
#include <unistd.h>
#include <errno.h>
#include <linux/input.h>
#include "preferences.h"
#include "commands.h"

extern "C" {
#include "support.h"
	extern struct navigate_control g_nav_ctl;
}

/// JogShuttle

extern "C" void JogShuttle_new(GtkWidget *mainWindow);

/// A C wrapper for the constructor so object can be created in main.c
void JogShuttle_new(GtkWidget *mainWindow)
{
  new JogShuttle(mainWindow);
}

/// A C wrappper for the GDK input handler callback
void JogShuttle_inputCallback(gpointer data, gint source, GdkInputCondition condition)
{
  JogShuttle *js = static_cast<JogShuttle *>(data);
  g_return_if_fail(js != NULL);

  js->inputCallback(source, condition);
}

/** Constructor

    If enabled in preferences, open the user-specified device and
    hook our callback function into the GDK input handler.
    
    \param mainWindow A GTK widget pointer to our window.
*/
JogShuttle::JogShuttle(GtkWidget *mainWindow) :
  mainWindow_(mainWindow),
  input_(-1),
  monitorTag_(-1)
{
  Preferences &prefs = Preferences::getInstance();
  if (prefs.enableJogShuttle) {
      char *filename = prefs.jogShuttleDevice;
      input_ = open(filename, O_RDONLY);
      if (input_ < 0) {
        g_print("JogShuttle::JogShuttle: open %s: %s\n", filename, g_strerror(errno));
        return;
      }
    
      monitorTag_ = gdk_input_add(input_, GDK_INPUT_READ,
    			      JogShuttle_inputCallback, (gpointer)this);
    }
}

/** Destructor

    Remove the callback function from the GDK input handler.
*/
JogShuttle::~JogShuttle()
{
  if (input_ >= 0) {
    gdk_input_remove(monitorTag_);

    close(input_);
    input_ = -1;
  }
}


/** Handle movement on the jog dial.

    \param dir A number from -x to x to specify the offset from the current frame.
               A negative number moves backward. Typically, x is 1 to step frame-by-frame.
*/
void JogShuttle::jog(int dir)
{
	gdk_threads_enter();
	if (dir < 0) videoBack();
	else videoForward();
	gdk_threads_leave();
}

/** Handle movement of the shuttle ring.

    \param angle A number from -8 to 8 that speccifies a direction and speed.
*/
void JogShuttle::shuttle(int angle)
{
	if (angle > 0) angle--;
	if (angle < 0) angle++;
	gdk_threads_enter();
	videoShuttle( angle );
	gdk_threads_leave();
}

/** Handle a button press.
    
    This executes commands directly instead of using the key_press callback.
    
    \param code The keybaord code to which a button is mapped.
    \todo consolidate commands with keyboard input
*/
void JogShuttle::button(int code)
{
  switch (code) {

  case 0xf1:  // Paste before
	gdk_threads_enter();
	processMenuCommand( "P" );
	gdk_threads_leave();
    break;
    
  case 0xf2:  // set in point (d0)
	gdk_threads_enter();
  	processMenuCommand( "d0" );
	gdk_threads_leave();
    break;

  case 0xf3:  // set out point (d$)
	gdk_threads_enter();
  	processMenuCommand( "d$" );
	gdk_threads_leave();
    break;

  case 0xf4:  // paste after
	gdk_threads_enter();
  	processMenuCommand( "p" );
	gdk_threads_leave();
    break;

  case 0xc2:  // jump to start of movie
	gdk_threads_enter();
  	videoStart();
	gdk_threads_leave();
    break;
    
  case 0xc3:  // prev sequence
	gdk_threads_enter();
  	videoPreviousScene();
	gdk_threads_leave();
    break;
    
  case 0xc1:  // play
    if (g_nav_ctl.active == FALSE) {
		gdk_threads_enter();
		videoPlay();
		gdk_threads_leave();
    } else {
		gdk_threads_enter();
        videoPause();
		gdk_threads_leave();
    }
    break;
    
  case 0xc4:  // next sequence
	gdk_threads_enter();
  	videoNextScene();
	gdk_threads_leave();
    break;
    
  case 0xc5:  // jump to end of movie
	gdk_threads_enter();
  	videoEndOfMovie();
	gdk_threads_leave();
    break;

  default:
    break; //do nothing

  }
}

/** The GDK input callback function.

    GDK calls this whenever input is received. It is hooked into the system
    during object construction.
*/
void JogShuttle::inputCallback(gint source, GdkInputCondition condition)
{
  g_return_if_fail(this != NULL);
  g_return_if_fail(input_ >= 0);
  g_return_if_fail(input_ == source);
  g_return_if_fail(condition == GDK_INPUT_READ);

  struct input_event ev;
  ssize_t n = read(input_, &ev, sizeof(ev));
  if (n != sizeof(ev)) {
    g_print("JogShuttle::inputCallback: read: %s\n", g_strerror(errno));
    return;
  }

#if 0
  printf("JogShuttle:");
  for (int i = 0; i < n; i++) {
    printf(" %02x", ((unsigned char *)(&ev))[i]);
  }
  putchar('\n');
#endif

  if (ev.type != EV_KEY || ev.value == 0)
    return;

  switch (ev.code) {
  case  2: this->jog(-1); break;
  case  3: this->jog( 1); break;
  case 30: this->shuttle(1); break;
  case 48: this->shuttle(2); break;
  case 46: this->shuttle(3); break;
  case 32: this->shuttle(4); break;
  case 18: this->shuttle(5); break;
  case 33: this->shuttle(6); break;
  case 34: this->shuttle(7); break;
  case 35: this->shuttle(8); break;
  case 23: this->shuttle(-1); break;
  case 36: this->shuttle(-2); break;
  case 37: this->shuttle(-3); break;
  case 38: this->shuttle(-4); break;
  case 50: this->shuttle(-5); break;
  case 13: this->shuttle(-6); break;
  case 27: this->shuttle(-7); break;
  case 43: this->shuttle(-8); break;
  
  // DRD> I do not know why Tomoaki chose these input codes.
  case 19: this->button(0xb1); break;
  case 49: this->button(0xf1); break;
  case 24: this->button(0xf2); break;
  case 25: this->button(0xf3); break;
  case 16: this->button(0xf4); break;
  case 21: this->button(0xc1); break;
  case 47: this->button(0xc2); break;
  case 45: this->button(0xc3); break;
  case 31: this->button(0xc4); break;
  case 44: this->button(0xc5); break;
  case 17: this->button(0xc6); break;
  case 20: this->button(0xe1); break;
  case 22: this->button(0xe2); break;
  default:
    g_print("JogShuttle::inputCallback: unknown button code 0x%02x\n", ev.code);
    break;
  };

  return;
}
