/*
  Copyright Mission Critical Linux, 2000

  Kimberlite is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2, or (at your option) any
  later version.

  Kimberlite is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with Kimberlite; see the file COPYING.  If not, write to the
  Free Software Foundation, Inc.,  675 Mass Ave, Cambridge, 
  MA 02139, USA.
*/

/* "none" power switch library.

   This module provides a stub interface to allow configuration of a cluster
   without requiring an actual remote power switch.  Basically it just
   returns success to all calls without doing anything.
 
   WARNING: operating a cluster without a remote power switch can result
   in data corruption in the event of service takeover from a hung node
   which becomes unhung.

   author: Tim Burke <burke@missioncriticallinux.com>
   Cloned from power_rps10.c
*/

#include <power.h>
#include <parseconf.h>
#include <clusterdefs.h>
#include "plock.h"

#include <unistd.h>

#include <stdlib.h>
#include <sys/syslog.h>
#include <logger.h>
#include <clusterdefs.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <termios.h>
#include <stdio.h>
#include <sys/time.h>
#include <string.h>
#include <errno.h>
#include <regex.h>
#include <signal.h>

#include <clucfg.h>

static const char *version __attribute__ ((unused)) = "$Id: power_none.c,v 1.4 2000/12/06 03:39:05 burke Exp $";

#ifndef NDEBUG
#define NDEBUG
#endif

#ifndef NDEBUG
#define PR(format, args...) fprintf(stderr, format, ## args)
#else
#define PR(format, args...)
#endif 


enum PWR_boolean PWR_NONE_configure(char *config_file_name);
enum PWR_boolean PWR_NONE_set(char* option, char* value);
enum PWR_boolean PWR_NONE_get(char* option, char *dflt, char** value);
enum PWR_boolean PWR_NONE_init(PWR_result *status);
enum PWR_boolean PWR_NONE_release(void);
PWR_result  PWR_NONE_status(void);
PWR_result  PWR_NONE_reboot(void);
PWR_result  PWR_NONE_off(void);
PWR_result  PWR_NONE_on(void);


/* Implementation Details */

static int initialized = 0;

/* Implementation Variables */

static PWR_result status;

/* Private functions */

static PWR_result internal_status(void);

#define SET(where, what)   (where |= (what))
#define RESET(where, what) (where &= ~(what))
#define ISSET(where, what) (where & (what))

/* Private: Read a response from the switch. 
*/
static int read_onoff(PWR_result *result) {
  
  if(initialized)
    SET(*result, PWR_INIT);
  else
    RESET(*result, PWR_INIT);

  RESET(*result, PWR_TIMEOUT);
  RESET(*result, PWR_ERROR);
  return 1;
}

/* Private: Get the status from the switch, constrained by the given
   timeout. 
*/
static PWR_result internal_status() {
  PWR_result result = 0;

  read_onoff(&result);
  PR( "read onoff, result: %d\n",result);
  return result;
}

/* Private: send a command to the switch.  Return the result. 
*/
static PWR_result internal_command(void) {
  PWR_result result=0;

  if(initialized) {
    SET(result, PWR_INIT);
  }
  else {
    PR(__FUNCTION__ " not initialised");
    RESET(result, PWR_INIT);
    SET(result, PWR_ERROR);
  }
  return result;
}
/****************************************************************************/

/* Public Interface Functions */

/* Read a configuration file for settings and parameters. 
*/
enum PWR_boolean PWR_NONE_configure(char *config_file_name __attribute__((unused))) {
  return(PWR_TRUE);
}

/* Set or query the configuration parameters of the switch
   interface. 
*/
enum PWR_boolean PWR_NONE_set(char* option, char* value) {
  CFG_status rc;
  rc = CFG_Set(option, value);
  if(CFG_OK == rc) {
    return PWR_TRUE;
  }
  else {
    return PWR_FALSE;
  }
}

enum PWR_boolean PWR_NONE_get(char* option, char* dflt, char** value) {
  CFG_status rc;
  rc = CFG_Get(option, dflt, value);
  if(CFG_OK == rc || CFG_DEFAULT == rc) {
    return PWR_TRUE;
  }
  else {
    return PWR_FALSE;
  }
}

#define STREQ(s1,s2) (strcmp((s1),(s2)) == 0)

/* Initialize the power switch, and the communication paths to it. 
*/
enum PWR_boolean PWR_NONE_init(PWR_result *s) {

  clulog(LOG_NOTICE, "PWR_NONE_init: Operating a cluster without a remote "
  		"power switch.\n");

  status = internal_status();
  initialized = 1;
  SET(status,   PWR_INIT);
  *s = status;
  return PWR_TRUE;
}

/* Give back resources to the system.  This is only needed if the
   power switch is loaded occasionally in some long running program.
   For the use model scenarios under which this module was designed,
   there is not normally a need to call this function. 
*/
enum PWR_boolean PWR_NONE_release(void) {
  initialized = 0;
  return 1;
}

/* Query the switch to determine its current status, and its readiness
   to carry out additional commands.  Returns a PWR_result that
   contains flags that tell whether the switch is open or closed.
*/
PWR_result PWR_NONE_status(void) {
  return internal_status();
}

/* Send a command to the switch that will cause the system powered by
   the switch to reboot.  Normally, this means that the switch will be
   opened for some period of time, after which it will be closed
   again.  
*/
PWR_result PWR_NONE_reboot(void) {
  clulog(LOG_CRIT, "PWR_NONE_reboot: falsely claiming to reboot other cluster member.\n");
  clulog(LOG_CRIT, "PWR_NONE_reboot: data integrity may be compromised.\n");
  return internal_command();
}

/* Send a command to the switch that will cause the switch to open,
   securing power to the effected system. 
*/
PWR_result PWR_NONE_off(void) {
  clulog(LOG_CRIT, "PWR_NONE_off: falsely claiming to power off other cluster member.\n");
  clulog(LOG_CRIT, "PWR_NONE_off: data integrity may be compromised.\n");
  return internal_command();
}

/* Send a command to the switch that will cause the switch to close,
   supplying power to eht effected system. 
*/
PWR_result PWR_NONE_on(void) {
  return internal_command();
}

