/*
  Copyright Mission Critical Linux, 2000

  Kimberlite is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2, or (at your option) any
  later version.

  Kimberlite is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with Kimberlite; see the file COPYING.  If not, write to the
  Free Software Foundation, Inc.,  675 Mass Ave, Cambridge, 
  MA 02139, USA.
*/
/* plock.c

   Routines to place and remove the lock file for the serial device in
   /var/lock.

   author: Ron Lawrence <lawrence@missioncriticallinux.com> */

#include "plock.h"
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <termios.h>
#include <stdio.h>
#include <sys/time.h>
#include <string.h>
#include <errno.h>
#include <regex.h>
#include <signal.h>

static const char *version __attribute__ ((unused)) = "$Id: plock.c,v 1.3 2000/08/15 18:40:00 moyer Exp $";

#ifndef NDEBUG
#define PR(format, args...) fprintf(stderr, format, ## args)
#else
#define PR(format, args...)
#endif 

#define BUFFER_LEN 255
/* the name of the serial device file. */ 
#define DFLT_DEVICE "/dev/ttyS0"

/* Serial device modes. Things like baud, parity, stop bits, etc. */

#define TTY_LOCK_D "/var/lock"

/* Private: Create a lock file for the serial device.  Copied, with
   some slight modification from heartbeat. 
*/
int pwr_lock_serial_device(char *devicename) {
  char *serial_device = devicename;
  char lf_name[256], tf_name[256], buf[12];
  int fd;
  pid_t pid, mypid;
  int rc, ret;
  struct stat sbuf;

  PR( "locking serial device\n");

  mypid = getpid();
  snprintf(lf_name, 256, "%s/LCK..%s", TTY_LOCK_D,
           serial_device + sizeof("/dev/") - 1);
  snprintf(tf_name, 256, "%s/tmp%d-%s", TTY_LOCK_D, 
           mypid, serial_device + sizeof("/dev/") - 1);

  PR("lf_name = %s, tf_name = %s\n", lf_name, tf_name);


  if ((fd = open(lf_name, O_RDONLY)) >= 0) {
    sleep(1); /* if someone was about to create on, give'm a
               * sec to do so */
    if (read(fd, buf, 12) < 1) {
      PR("/* lockfile empty -> rm it and go on */\n");
    } else {
      if (sscanf(buf, "%d", &pid) < 1) {
        PR("/* lockfile screwed up -> rm it and go on */\n");
      } else {
        if (0 != kill(pid, 0)) {
#ifndef NDEBUG
          perror(__FUNCTION__);
#endif
          PR("errno = %d\n", errno);
          if(errno != ESRCH) {
            PR("/* tty is locked by existing (not\n"
               " * necessarily running) process with pid: %d\n"
               " * -> give up */\n", pid);
            close(fd);
            return -1;
          } else {
            PR("/* stale lockfile -> rm it and go on */\n");
            unlink(lf_name);
          }
        }
      }
    }
  }
  if ((fd = open(tf_name, O_CREAT | O_WRONLY, 0660)) < 0) {
    PR("/* Hmmh, why did we fail? Anyway, nothing we can do about */\n");
    return -3;
  }
  sprintf(buf, "%10d\n", mypid);
  if(write(fd, buf, 12) < 12) {
    PR("/* again, nothing we can do about */\n");
    return -3;
  }
  close(fd);
  ret = link(tf_name, lf_name);
  switch(ret) {
    case 0:
      if(stat(tf_name, &sbuf) < 0) {
        PR("/* something weird happened */\n");
        rc = -3;
        break;
      }
      if(sbuf.st_nlink < 2) {
        PR("/* somehow, it didn't get through - NFS trouble? */\n");
        rc = -2;
        break;
      }
      rc = 0;
      break;
    case -1:
      if (errno == EEXIST) {
	PR("EEXIST\n");
	rc = -1;
	break;
      }
    default:
      PR("DEFAULT\n");
      rc = -3;
  }
  unlink(tf_name);
  return rc;
}

/* Private: Release the lock file for the serial device.  Copied, with
   ever so slight modifications, from heartbeat. 
*/
int pwr_unlock_serial_device(char *devicename) {
  char *serial_device = devicename;
  char lf_name[256];
        
  snprintf(lf_name, 256, "%s/LCK..%s", TTY_LOCK_D,
           serial_device + sizeof("/dev/") - 1);
  return unlink(lf_name);
}
