/*
  Copyright Mission Critical Linux, 2000

  Kimberlite is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2, or (at your option) any
  later version.

  Kimberlite is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with Kimberlite; see the file COPYING.  If not, write to the
  Free Software Foundation, Inc.,  675 Mass Ave, Cambridge, 
  MA 02139, USA.
*/
/*
 *  $Id: lock.c,v 1.16 2000/09/13 19:39:03 burke Exp $
 *
 *  Copyright (C) 2000 Mission Critical Linux, LLC
 *
 *  author: Dave Winchell <winchell@missioncriticallinux.com>
 *  description: Upper layer in the locking synchronization primitives.
 */

/* TODO:
 * Test multithreaded version.
 */

#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/param.h>
#include <sys/file.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <sys/time.h>
#include <sys/syslog.h>
#include <sys/mman.h>
#include <signal.h>
#include <stdlib.h>
#include <errno.h>

#include <logger.h>
#include "diskstate.h"
#include "disk_proto.h" 
#include <power.h>
#include <clucfg.h>

/* Note that multithreaded code is untested. */ 
#ifdef CLU_LOCK_MULTITHREADED
#include <pthread.h>
#endif
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

int _clu_powerCyclePartner_direct(void);
int _clu_powerCyclePartner_msg(void);
int _clear_partner_state(void);
int _clu_process_lock(void);
int _clu_process_try_lock(void);
void _clu_process_unlock(void);
void _clu_lock_node(void);
int _clu_try_lock_node(void);
void _clu_unlock_node(void);
void _clu_lock_init(void);
void _clu_lock_1(void *pc);
int _clu_try_lock_1(void *pc);
void _clu_un_lock_1(void *pc);
void _clu_write_lock(int node, int data, void *pc);
int _clu_read_lock(int node);
int _clu_kill_other_node(void);
void _clu_lock_init_if_needed(void);
int _clu_process_lock_depth(void);
#ifdef CLU_LOCK_MULTITHREADED
void _clu_lock_atfork_prepare(void);
void _clu_lock_atfork_parent(void);
void _clu_lock_atfork_child(void);
#endif
int _clu_check_lock_for_errors(int node);
int _setPartnerNodeStatusDown(void);
void node_status_init_state(NodeStatusBlock *statb);

static const char *version __attribute__ ((unused)) = "$Revision: 1.16 $";

/* CLU_LOCK_TIME_OUT_SECS is set high for now, since I just enabled the 
 * power cycle partner bit.  We'll set it by experience.  I think we will 
 * end up with something like 10 seconds.
 */
 
#define USEC_PER_SEC 1000000
#define FAULT() *(int *)0 = 0;

int _clu_node_id = -1;
int _clu_lock_fd = -1;
int delay_shift = 16;
pid_t _clu_init_pid = -1;
int _clu_lock_count = 0;
NodeStatusBlock *_clu_lk_node_status;
int clu_msg_based_shoot = 1;

#ifdef CLU_LOCK_MULTITHREADED
pthread_t _clu_lock_holder;
pthread_mutex_t _clu_lk = PTHREAD_MUTEX_INITIALIZER;
pthread_cond_t _clu_cv = PTHREAD_COND_INITIALIZER;
pthread_once_t once_control = PTHREAD_ONCE_INIT;
#endif

void clu_lock(void)
{
	int depth;
	void *pc;

	pc = __builtin_return_address(0);
	_clu_lock_init_if_needed();
	depth = _clu_process_lock();
	if(depth == 1) {
		_clu_lock_node();
		_clu_lock_1(pc);
	}
}
int clu_try_lock(void)
{
	int depth;
	void *pc;

	pc = __builtin_return_address(0);
	_clu_lock_init_if_needed();
	depth = _clu_process_try_lock();
	if(!depth)
		return LOCK_FAILURE;

	if(depth == 1) {
		if(_clu_try_lock_node() == LOCK_FAILURE) {
			_clu_process_unlock();
			return LOCK_FAILURE;
		}
		if(_clu_try_lock_1(pc) == LOCK_FAILURE) {
			_clu_unlock_node();
			_clu_process_unlock();
			return LOCK_FAILURE;
		}
	}
	return LOCK_SUCCESS;
}
void clu_un_lock(void)
{
	void *pc;

	_clu_lock_init_if_needed();
	if(_clu_process_lock_depth() == 1) {
		_clu_un_lock_1(pc);
		_clu_unlock_node();
	}
	_clu_process_unlock();
}

#ifdef CLU_LOCK_MULTITHREADED /* Multithreaded support untested. */

int _clu_process_lock(void)
{
	pthread_t tid = pthread_self();

	if(_clu_lock_count && pthread_equal(_clu_lock_holder, tid)) {
		_clu_lock_count++;
		return _clu_lock_count;
	}
	pthread_mutex_lock(&_clu_lk);

	while(_clu_lock_count)
		pthread_cond_wait(&_clu_cv, &_clu_lk);

	_clu_lock_holder = tid;
	_clu_lock_count = 1;
	pthread_mutex_unlock(&_clu_lk);
	return _clu_lock_count;
}

int _clu_process_try_lock(void)
{
	pthread_t tid = pthread_self();
	int ret;

	if(_clu_lock_count && pthread_equal(_clu_lock_holder, tid)) {
		_clu_lock_count++;
		return _clu_lock_count;
	}
	ret = pthread_mutex_trylock(&_clu_lk);
	if(ret == EBUSY)
		return 0;
	else if (ret)
		FAULT();
	if(_clu_lock_count) {
		pthread_mutex_unlock(&_clu_lk);
		return 0;
	}
	_clu_lock_holder = tid;
	_clu_lock_count = 1;
	pthread_mutex_unlock(&_clu_lk);
	return _clu_lock_count;
}

void _clu_process_unlock(void)
{
	assert_clu_lock_held("_clu_process_unlock");
	pthread_mutex_lock(&_clu_lk);
	_clu_lock_count--;
	if(!_clu_lock_count)
		pthread_cond_signal(&_clu_cv);
	pthread_mutex_unlock(&_clu_lk);

}
int test_clu_lock_held(void)
{
	pthread_t tid = pthread_self();

	if((_clu_lock_count > 0) && pthread_equal(_clu_lock_holder, tid))
		return 1;
	else
		return 0;
}

#else /* !CLU_LOCK_MULTITHREADED  */

int _clu_process_lock(void)
{
	_clu_lock_count++;
	return _clu_lock_count;
}
int _clu_process_try_lock(void)
{
	_clu_lock_count++;
	return _clu_lock_count;
}

void _clu_process_unlock()
{
	_clu_lock_count--;
	if(_clu_lock_count < 0) {
		clulog(LOG_CRIT, "_clu_process_unlock: cluster lock not held\n");
		FAULT();
	}
}
int test_clu_lock_held(void)
{
	/* The init call is needed here, but not in the multithreaded version
	 * beacuse of the fork check done in _clu_lock_init_if_needed() 
	 * which is handled automatically by pthread_atfork().
	 */

	_clu_lock_init_if_needed();

	if(_clu_lock_count < 0) {
		clulog(LOG_CRIT, "test_clu_lock_held: invalid lock value\n");
		FAULT();
	}
	if(_clu_lock_count == 0)
		return 0;
	else
		return 1;
}



#endif

extern int in_initializePartition;

void assert_clu_lock_held(char *s)
{
	if(!in_initializePartition && !test_clu_lock_held()) {
		clulog(LOG_CRIT, "assert_clu_lock_held: %s: cluster lock n"
		       "ot held\n", s);
		FAULT();
	}
}

int _clu_process_lock_depth(void)
{
	return _clu_lock_count;
}

void _clu_lock_node(void)
{
	u_long msecs = 0;
	struct flock lock;

	memset(&lock, 0, sizeof(lock));
	lock.l_type = F_WRLCK;
	while(fcntl(_clu_lock_fd, F_SETLK, &lock)) {
		usleep(10000);
		msecs += 10;
		if(msecs > (CLU_NODE_LOCK_TIME_OUT_SECS * 1000)) {
			clulog(LOG_EMERG,"_clu_lock_node: unable to obtain "
			       "node lock\n");
			FAULT();
		}
	}
	return;
}
int _clu_try_lock_node(void)
{
	struct flock lock;

	memset(&lock, 0, sizeof(lock));
	lock.l_type = F_WRLCK;
	if(fcntl(_clu_lock_fd, F_SETLK, &lock))
		return LOCK_FAILURE;
	else
		return LOCK_SUCCESS;
}
void _clu_unlock_node(void)
{
	struct flock lock;

	memset(&lock, 0, sizeof(lock));
	lock.l_type = F_UNLCK;
	if(fcntl(_clu_lock_fd, F_SETLK, &lock)) {
		clulog(LOG_EMERG,"_clu_unlock_node:  error from flock unlock,"
		       " errno = %d\n", errno);
		FAULT();
	}
	return;
}


void _clu_lock_1(void *pc)
{
	int this_node, other_node;
	int wait_shoot = 0;
	int wait_cur;

	this_node = _clu_node_id;
	other_node = _clu_node_id ^ 1;

  top:
	_clu_write_lock(this_node, 1, pc);    /* bid (synchronous) */
	if(!_clu_read_lock(other_node))
		return;
	
	_clu_write_lock(this_node, 0, pc);    /* clear bid (synchronous) */
	wait_cur = random() & ((1<<delay_shift)-1);
	usleep(wait_cur);
	wait_shoot += wait_cur;

	if(wait_shoot > (USEC_PER_SEC * CLU_GLOBAL_LOCK_TIME_OUT_SECS)) {
		DiskLockBlock lock_block;

		lockRead(other_node, &lock_block);
		clulog(LOG_CRIT, "_clu_lock_1: node %d stuck with lock, "
		       "pid = 0x%lx pc = 0x%lx data = %d\n", other_node, 
		       lock_block.holder_pid, lock_block.holder_pc,
		       lock_block.lockData);
		       
		if(clu_powerCyclePartner()) {
			clulog(LOG_EMERG,"_clu_lock_1: failure from "
			       "clu_powerCyclePartner\n");
			FAULT();
		}
		wait_shoot = 0;
	}
	goto top;
}
int _clu_try_lock_1(void *pc)
{
	int this_node, other_node;

	this_node = _clu_node_id;
	other_node = _clu_node_id ^ 1;

	_clu_write_lock(this_node, 1, pc);    /* bid (synchronous) */
	if(!_clu_read_lock(other_node))
		return LOCK_SUCCESS;              /* have lock */
	
	_clu_write_lock(this_node, 0, pc);    /* clear bid (synchronous) */

	return LOCK_FAILURE;
}



void _clu_un_lock_1(void *pc)
{
	_clu_write_lock(_clu_node_id, 0, pc);      /* clear ownership (synchronous) */
	usleep(random() & ((1<<delay_shift)-1));  /* avoids lock starvation */
}


void _clu_write_lock(int node, int data, void *pc)
{
	int ret;
	DiskLockBlock lock_block;
	pid_t pid;

	pid = getpid();

	lock_block.lockData = data;
	lock_block.holder_pc = *(ulong *)&pc;
	lock_block.holder_pid = pid;

	ret = lockWrite(node, &lock_block);
	if(ret) {
		clulog(LOG_EMERG,"_clu_write_lock: bad return from lockWrite,"
		       " ret = %d\n", ret);
		FAULT();
	}
}
int _clu_read_lock(int node)
{
	int ret;
	DiskLockBlock lock_block;

	ret = lockRead(node, &lock_block);
	if(ret < 0) {
		clulog(LOG_EMERG,"_clu_read_lock: bad return from lockRead,"
		       " ret = %d\n", ret);
		FAULT();
	}
	return lock_block.lockData;
}	
/* repairs corruption for locks of downed nodes */
void clu_lock_repair(void)
{
	int node, ret;
	NodeStatusBlock statb;

	clu_lock();
	for(node = 0; node < MAX_NODES; node++) {
		if(node == _clu_node_id)
			continue;
		ret = readStatusBlock(OFFSET_FIRST_STATUS_BLOCK + (node * SPACE_PER_STATUS_BLOCK),
				      &statb,  0);
		if(ret) {
			clulog(LOG_EMERG, "clu_lock_repair: unable to read status block, ret = %d\n", ret);
			clu_un_lock();
			return;
		}
		if(statb.state == NODE_DOWN)
			if(_clu_check_lock_for_errors(node)) {
				/* Its always dangerous to write another node's lock cell.
				 * Its only done at cluster init time, right after power cycling a node,
				 * and here.
				 * Here, the danger is that the other node is waiting to get the
				 * cluster lock (to mark the node state UP) and has written a bid (1).
				 * Our clear here, clears that node's bid.
				 * The delay alows that node to reassert the bid.
				 */
				_clu_write_lock(node, 0, 0);
				usleep(1<<(delay_shift+2));  /* 4 times max lock delay */
				clulog(LOG_EMERG, "clu_lock_repair: repaired lock for node %d\n", node);
			}
	}
	clu_un_lock();
}

int _clu_check_lock_for_errors(int node)
{
	int ret;
	DiskLockBlock lock_block;
	int part;

	for(part = 0; part < 2; part++) {
		ret = diskLseekRawReadChecksum(part,
					       OFFSET_FIRST_LOCK_BLOCK + (node * SPACE_PER_LOCK_BLOCK),
					       (char *)&lock_block, sizeof(DiskLockBlock),
					       (ulong)&((DiskLockBlock *)0)->check_sum);
		if(ret != SHADOW_SUCCESS)
			return 1;
	}
	return 0;
}

	
/* init code.  */
void _clu_lock_init(void)
{
	int ret;
	int fd;

	if(_clu_lock_fd != -1)
		return;
	fd = open(CLUSTER_LOCKFILE, O_RDWR | O_CREAT, S_IRWXU);
	if(fd == -1) {
		clulog(LOG_EMERG,"_clu_lock_init: open failure, error: %s "
		       "file %s\n", strerror(errno), CLUSTER_LOCKFILE);
		FAULT();
	}
	_clu_node_id = cluGetLocalNodeId();
	if(_clu_node_id < 0) {
		clulog(LOG_EMERG, "_clu_lock_init: unable to get node ID\n");
		FAULT();
	}		
	ret = initLockSubsys();
	if(ret) {
		clulog(LOG_EMERG, "_clu_lock_init: unable to init LockSubsys\n");
		FAULT();
	}		
	_clu_lk_node_status = (NodeStatusBlock *)allocAlignedBuf();
	if (_clu_lk_node_status == MAP_FAILED) {
		clulog(LOG_EMERG, "_clu_lock_init: unable to allocate aligned"
		       " node status buffer.\n");
		FAULT();
	}
#ifdef  CLU_LOCK_MULTITHREADED
	pthread_atfork(_clu_lock_atfork_prepare, _clu_lock_atfork_parent, 
		       _clu_lock_atfork_child);
#endif
	_clu_lock_fd = fd;
}

#ifdef  CLU_LOCK_MULTITHREADED
void _clu_lock_init_if_needed(void)
{
	if(_clu_lock_fd == -1)
		pthread_once(&once_control, _clu_lock_init);
}
void _clu_lock_atfork_prepare(void)
{
	pthread_mutex_lock(&_clu_lk);
}
void _clu_lock_atfork_parent(void)
{
	pthread_mutex_unlock(&_clu_lk);
}
void _clu_lock_atfork_child(void)
{
	_clu_lock_count = 0;
	pthread_mutex_unlock(&_clu_lk);
}
#else
void _clu_lock_init_if_needed(void)
{
	pid_t pid;

	if(_clu_lock_fd == -1)
		_clu_lock_init();

	pid = getpid();
	/* lock is stripped in the child */
	if(pid != _clu_init_pid) {
		_clu_lock_count = 0;
		_clu_init_pid = pid;
	}
}
#endif

/* checksum code */
ulong clu_long_check_sum(ulong *start, int num_longs)
{
	ulong sum = 0;

	for(;num_longs--; start++)
		sum += *start;

	/* The entire block written to zero would be detected by
	 * the missing magic.  So this is not strictly necessary.
	 */

	if(sum == 0)
		sum = 1;
	return sum;
}
ulong clu_byte_check_sum(char *start, int bytes)
{
	ulong sum = 0;

	for(;bytes--; start++)
		sum += ((ulong)*start) & 0xff;

	/* The entire block written to zero would be detected by
	 * the missing magic.  So this is not strictly necessary.
	 */

	if(sum == 0)
		sum = 1;
	return sum;
}
int clu_write_checksum(char *alignedBuf, int len, ulong chksum_off)
{
	ulong *chksum;
	int chksumlongs;

	
	if(((ulong)alignedBuf & (ulong)(sizeof(ulong) - 1)) ||
	   (len & (sizeof(ulong) - 1)) ||
	   (chksum_off & (sizeof(ulong) - 1))) {
		clulog(LOG_CRIT, "clu_write_checksum: alignment error\n");
		FAULT();
		return -1;
	}

	chksum = (ulong *)((ulong)alignedBuf + chksum_off);
	*chksum = 0;
	chksumlongs = len/sizeof(ulong);
	*chksum = clu_long_check_sum((ulong *)alignedBuf, chksumlongs);
	return 0;
}	
int clu_check_checksum(char *alignedBuf, int len, ulong chksum_off)
{
	ulong *chksum;
	ulong check_sum;
	ulong data_check_sum;

	
	if(((ulong)alignedBuf & (ulong)(sizeof(ulong) - 1)) ||
	   (len & (sizeof(ulong) - 1)) ||
	   (chksum_off & (sizeof(ulong) - 1))) {
		clulog(LOG_CRIT, "clu_check_checksum: alignment error\n");
		FAULT();
	}

	chksum = (ulong *)((ulong)alignedBuf + chksum_off);
	check_sum = *chksum;
	*chksum = 0;
	data_check_sum = clu_long_check_sum((ulong *)alignedBuf, len/sizeof(ulong));
	if(check_sum != data_check_sum) {
		clulog(LOG_EMERG,"clu_check_checksum: expected = 0x%lx observed = 0x%lx \n",
		       check_sum, data_check_sum);
		return -1;
	}
	else
		return 0;
}

/* The reason this power switch code is in the lock file is that in clu_powerCyclePartner()
 * the is a direct call to lockClear().  This should be the only place lockClear is called.
 * Any calls in quorumd should be removed.
 */

int clu_powerCyclePartner(void)
{
	int ret;

	if(clu_msg_based_shoot) {
		ret = _clu_powerCyclePartner_msg();
		if(ret) {
			clulog(LOG_CRIT, "clu_powerCyclePartner: bad ret from"
			       " clu_powerCyclePartner_msg\n");
			return(ret);
		}
	}
	else {
		ret = _clu_powerCyclePartner_direct();
		if (ret) {
			clulog(LOG_CRIT, "clu_powerCyclePartner: bad ret from"
			       " clu_powerCyclePartner_direct\n");
			return(ret);
		}
	}
	if(clu_clear_partner_state()) {
		clulog(LOG_CRIT, "clu_powerCyclePartner: bad ret from clu_clear_partner_state\n");
		shut_myself_down("clu_powerCyclePartner: bad ret from clu_clear_partner_state\n");
	} 
	return 0;
}
int clu_clear_partner_state(void)
{
	_clu_lock_init_if_needed();

	if(lockClear(_clu_node_id ^ 1)) {
		clulog(LOG_CRIT, "clu_powerCyclePartner: bad ret from "
		       "lockClear\n");
		return -1;
	}
	if(_setPartnerNodeStatusDown()) {
		clulog(LOG_CRIT, "clu_powerCyclePartner: bad ret from "
		       "setPartnerNodeStatusDown\n");
		return -1;
	}
		
	return 0;
}
int _clu_powerCyclePartner_direct(void)
{
	PWR_result status=0;

	/*
	 * First we need to open the device...
	 */
	if (PWR_init(&status) == PWR_FALSE) {
		(void)PWR_release();

		if (status & PWR_ERROR)
			errno = EIO;
		else 
			errno = ETIMEDOUT;

		clulog(LOG_CRIT, "_clu_powerCyclePartner_direct: Unable to "
		       "initialize power switch.");
		return -1;
	}

	/*
	 * Now we can power cycle the bugger
	 */
	status = PWR_reboot();
	if (status & PWR_ERROR) {
		clulog(LOG_CRIT, "_clu_powerCyclePartner_direct: "
		       "power_cycle_partner: error while talking to "
		       "power switch\n");
		(void)PWR_release();
		errno = EIO;
		return -1;
	} else if (status & PWR_TIMEOUT) {
		clulog(LOG_CRIT, "_clu_powerCyclePartner_direct: "
		       "power_cycle_partner: timeout"
		       " while talking to power switch\n");
		(void)PWR_release();
		errno = ETIMEDOUT;
		return -1;
	}

	/*
	 * And finally, relinquish control of the device.
	 */
	(void)PWR_release();
	return 0;
}
/*
 * Send message to power daemon to power cycle partner node.
 */

int _clu_powerCyclePartner_msg(void)
{
	int               len, retval;
	int               auth=0;
	msg_handle_t      connfd;
	generic_msg_hdr   msghdr;
	PswitchStatusMsg  msg;

        memset(&msghdr, 0, sizeof(msghdr));
        memset(&msg, 0, sizeof(msg));

	msghdr.magic = GENERIC_HDR_MAGIC;
	msghdr.command = PSWITCH_RESET;
	msghdr.length = 0;

	connfd = msg_open(PROCID_POWERD, _clu_node_id);
	if (connfd < 0) {
		clulog(LOG_ERR, "clu_powerCyclePartner: can't "
		       "connect to powerd.\n");
		return -2;
	}

	retval = msg_send(connfd, &msghdr, sizeof(msghdr));
	if (retval < 0) {
		clulog(LOG_ERR, "clu_powerCyclePartner: can't "
		       "send to powerd.\n");
		msg_close(connfd);
		return -2;
	}
	len = msg_receive_timeout(connfd, &msg, sizeof(msg), &auth, 30);
	if (len <= 0) {
		clulog(LOG_ERR, "clu_powerCyclePartner: no response "
		       "from powerd.\n");
		msg_close(connfd);
		return -2;
	}
	msg_close(connfd);
	
	if (msg.status != 0) {
		clulog(LOG_ERR, "clu_powerCyclePartner: powerd error "
		       "status %d.\n", msg.status);
	}
	return(msg.status);
}

/*
 * Writes the partner's node status block to indicate that the node is down.
 * This is only done when the partner has failed and we intend to trigger
 * the service manager to takeover services. To do this we could read in
 * the block, just modify the state field and write out the block. But to
 * keep the amount of time the shootdown procedure can take down to a 
 * minimum, it just blasts out the full structure.
 * Returns: 0 - success
 *	    -1 - write failure
 */
int _setPartnerNodeStatusDown(void)
{
	NodeStatusBlock statb;
	int offset;

	offset = (OFFSET_FIRST_STATUS_BLOCK + 
		             ((_clu_node_id ^ 1) * SPACE_PER_STATUS_BLOCK));
	node_status_init_state(&statb);
	strcpy(statb.nodename, "takeover");
	if (writeStatusBlock(offset, &statb,0) != 0) {
		clulog(LOG_ERR, "_setPartnerNodeStatusDown: unable to update "
		       "status block.\n");
		return(-1);
	}
	return(0);
}
/*
 * Called to initialize the memory resident version of the disk status block.
 * During the course of normal operation, various fields within this block
 * will be updated accordingly.
 */
void
node_status_init_state(NodeStatusBlock *statb)
{
	memset(statb, 0, sizeof(NodeStatusBlock));
	statb->magic_number = STATUS_BLOCK_MAGIC_NUMBER;
	statb->version = STATUS_BLOCK_LATEST_VERSION;
	/*
	 * Establish this node's incarnation number.  This is used to detect
	 * any IOs initiated from a formerly hung node.
	 */
	statb->incarnationNumber = time(NULL);
	/*
	 * Just filling in nodename via `gethostname` for trivia purposes.
	 * This doesn't hold up when the interface associated with a cluster
	 * interconnect doesn't match whats used in the hostname command.
	 * However since this field really isn't used its harmless.
	 */
	gethostname(statb->nodename, (size_t)MAXHOSTNAMELEN);
	/* statb->timestamp = filled in later */
	statb->updateNodenum = _clu_node_id; 
	statb->state = NODE_DOWN; 
    	statb->configTimestamp = (time_t)0;
}
