/*
  Copyright Mission Critical Linux, 2000

  Kimberlite is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2, or (at your option) any
  later version.

  Kimberlite is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with Kimberlite; see the file COPYING.  If not, write to the
  Free Software Foundation, Inc.,  675 Mass Ave, Cambridge, 
  MA 02139, USA.
*/

/* pallocate.c

   A pool allocator for small blocks of memory.  Trades off some extra
   storage for improved performance by allocating small requests all
   of the same size.  Some tuning may be required.  This allocation of
   small blocks avoids fragmentation of the malloc allocated heap.

   There is a need to linearly search for free space in a bit map of
   allocated blocks.  It may be useful to have a higher order bitmap
   that reduces the number of bits that must be examined during each
   search.  It mightt also be interesting to experiment with the
   effect of adding a full marker to the map, so that, when the map
   has been entirely filled up, it isn't necessary to check it again
   until the next region is nearing exhaustion.

   Other optimizations are, most likely, possible.

   This is a rewrite of an earlier allocator that had bit off more
   than it could chew.

   Mon Nov 13 13:08:45 2000  RAL 

   Author: Ron Lawrence <lawrence@missioncriticallinux.com>
   Version: $Revision: 1.4 $
*/

#include "allocate.h"
#include <stdlib.h>
#include <memory.h>

#define REGIONSIZE (512*1024)
#define BLOCKSIZE 32
#define BITSPERWORD 32
#define MAPSIZE (REGIONSIZE / BLOCKSIZE / BITSPERWORD)

struct bitmap {
  int words[MAPSIZE];
};
struct region {
  char memory[REGIONSIZE];
};
struct MEM_pool {
  int initialized;
  int region_count;
  int region_size;
  struct bitmap *bitmaps;
  struct region **regions;
};

struct MEM_pool* gp = NULL;
/* Set up all of the data structures. */

static int initialize(struct MEM_pool* b) {
  /* N.B. FIXME, need checks for malloc returning null. */
  b->region_count = 1;
  b->region_size = REGIONSIZE;
  b->bitmaps = (struct bitmap*)malloc(sizeof(struct bitmap));
  b->regions = (struct region**)malloc(sizeof(struct region*));
  b->regions[0] = (struct region*)malloc(sizeof(struct region));
  memset(b->bitmaps, 0, sizeof(struct bitmap));
  memset(b->regions[0], 0, sizeof(struct region));
  b->initialized = 1;
  return 1;
}
/* Given a pointer, p, find the region in which it resides.  Return -1
   if the pointer doesn't reside in any region in the allocator.

   Technically, it is not well defined to do pointer arithmetic
   between pointers that are not known to be pointing into a
   contiguous region of memory, but, well... */

static int find_pointer(struct MEM_pool* b, void* p) {
  int i, offset;

  int found = -1;
  for(i=0; i < b->region_count; i++) {
    if((offset = (int)p - (int)b->regions[i]) < 0)
      continue;
    if(offset < REGIONSIZE) {
      found = i;
      break;
    }
  }
  return  found;
}
/* Given a bitmap, and an index, return the value of the bit at that
   index. */

static int bit_get(struct bitmap* map, int idx) {
  int wordnumber;
  int bitnumber;

  wordnumber = idx / BITSPERWORD;
  bitnumber  = idx % BITSPERWORD;

  return (map->words[wordnumber] & (1 << bitnumber));
}
/* Set the bit at idx in bitmap, map, to have the given value. */

static void bit_set(struct bitmap* map, int idx, int value) {
  int wordnumber;
  int bitnumber;

  wordnumber = idx / BITSPERWORD;
  bitnumber  = idx % BITSPERWORD;
  if (1==value)
    map->words[wordnumber] |=  (1 << bitnumber);
  else
    map->words[wordnumber] &= ~(1 << bitnumber);
}
/* Given a bit position, as in a bitmap, find the actual offset that
   corresponds to that position in a memory buffer. */

static int bit2offset(int bit) {
  return bit * BITSPERWORD;
}
/* Given an offset into a memory region, return the bit idx in the
   bitmap that corresponds to it.  Note that the offset may not
   exactly map to a bit position, in which case the nearest lower bit
   idx is returned. */

static int offset2bit(int offset) {
  return offset / BITSPERWORD;
}
/* Find a free idx in the bitmap given by the idx map. */

static int find_free(struct MEM_pool* b, int map) {
  int i,j;
  for(i = 0; i < MAPSIZE; i++) {
    if(-1 == b->bitmaps[map].words[i]) 
      continue;                 /* all bits used */
    for(j = 0; j < BITSPERWORD; j++) {
      if(0 == bit_get(&(b->bitmaps[map]), i * BITSPERWORD + j)) {
        return i * BITSPERWORD + j;
      }                   
    }
  }
  return -1;
}
/* Allocate a new block of storage for small allocations.  Update the
   MEM_pool data structures.  note that the bitmaps are realloc'ed,
   because nothing points into them, but that the regions themselves
   cannot be realloc'ed.  Reallocation would cause items to move their
   addresses, which cannot be done. */

static int allocate_new_region(struct MEM_pool* b) {
  /* N.B.  FIXME, need check for malloc returning NULL */
  b->region_count++;
  b->bitmaps = 
    (struct bitmap*)realloc(b->bitmaps,
                            b->region_count * sizeof(struct bitmap));
  b->regions = 
    (struct region**)realloc(b->regions,
                             b->region_count * sizeof(struct region*));
  b->regions[b->region_count-1] = 
    (struct region*)malloc(sizeof(struct region));
  memset(b->bitmaps[b->region_count-1].words,0,sizeof(struct bitmap));
  memset(b->regions[b->region_count-1]->memory,0,sizeof(struct region));
  return 1;
}
/************************************************************************/
/* Public */

/* allocate a region of memory of at least size, size.  Return a
   pointer to the allocated memory.  

   Internally, the requested memory is taken from a large,
   preallocated, pool of bytes, unless the request is larger than
   BLOCKSIZE bytes.  If the request is larger, then the memory is
   allocated using malloc.

  */

struct MEM_pool* MEM_create_pool(__attribute__((unused))size_t ignored) {
  struct MEM_pool* b;

  b = (struct MEM_pool*)malloc(sizeof(struct MEM_pool));
  b->initialized=0;
  return b;
}

void MEM_destroy_pool(struct MEM_pool* p) {
  int i;
  if(p->initialized) {
    for(i=0; i < p->region_count; i++) {
      free(p->regions[i]);
    }
    free(p->bitmaps);
    free(p->regions);
  }
  free(p);
}

char* MEM_pool_allocate_string(struct MEM_pool* b, size_t size) {
  if(!(b->initialized)) initialize(b);
  if(size > BLOCKSIZE) {
    return malloc(size);
  }
  else {
    int idx = -1;
    int map;
    for(map = 0; map < b->region_count; map++) {
      if(-1 == (idx = find_free(b,map)))
        continue;
      else {
        bit_set(&(b->bitmaps[map]), idx, 1);
        return ((char*)b->regions[map]) + bit2offset(idx);
      }
    }
    /* if we are here, there are no more blocks left to allocate in
       any regions. */
    allocate_new_region(b);
    return MEM_pool_allocate_string(b,size);
  }
}

/* Free a block of memory pointed to by p.  This memory must be memory
   previously returned by pallocate.  The pointer p may safely be
   null. */

char* MEM_pool_release_string(struct MEM_pool* b, char *p) {
  int region =0;
  if(NULL==p) return NULL;
  if(-1 == (region = find_pointer(b,p))) {
    free(p);
  } 
  else {
    int offset, bit;
    offset = (int)p - (int)b->regions[region];
    bit = offset2bit(offset);
    bit_set(&(b->bitmaps[region]), bit, 0);
    /* It would be faster to skip this next step. */
    memset(p,0,BLOCKSIZE);
  }
  return NULL;
}

/* It seems reasonable that a prealloc could be defined here as
   well. */

void* MEM_pool_realloc_string(struct MEM_pool *b, void*p, int newsize) {
  int region =0;
  /* newsize must be >= 0 */
  if(-1 == (region = find_pointer(b,p))) {
    return realloc(p,newsize);
  } 
  else {
    if(newsize <= BLOCKSIZE) {
      return p;                 /* No need to reallocate. */
    } 
    else {
      char* p2;
      p2 = malloc(newsize);
      memcpy(p2,p,BLOCKSIZE);
      MEM_pool_release_string(b,p);
      return p2;
    }
  }
  return NULL;
}

/* Allocate from a global pool. */

char* MEM_allocate_string(size_t n) {
  if(NULL == gp) {
    gp = MEM_create_pool(99);
  }
  return MEM_pool_allocate_string(gp,n);
}

/* reallocate a previously allocated buffer */

char* MEM_reallocate_string(char*str, size_t newsize) {
  return MEM_pool_realloc_string(gp,str,newsize);
}
/* free  from a global pool */

char* MEM_release_string(char* str) {
  MEM_pool_release_string(gp,str);
  return NULL;
}

/* Destroy the global pool */

void MEM_release_all(void) {
  MEM_destroy_pool(gp);
  gp = NULL;
}

#ifndef NDEBUG
void MEM_dump(void) {}
void MEM_pool_dump(__attribute__((unused))struct MEM_pool* pool) {}
#endif
