// ForthCompiler.cpp
//
// FORTH compiler to generate FORTH Byte Code (FBC) from expressions
//   or programs
//
// Copyright (c) 1998--2000 Krishna Myneni and David P. Wallace, 
//   Creative Consulting for Research and Education
//
// This software is provided under the terms of the General Public License.
//
// Revisions:
// 	9-12-1998
//	9-15-1998 added SP@, RP@, +!
//      9-16-1998 added -ROT, PICK, ROLL, A@ 
//	9-18-1998 error checking for incomplete structures at end of definition
//	10-6-1998 added ?DUP
//	10-14-1998 fixed COUNT
//	10-19-1998 added 0<, 0=, 0>, TRUE, FALSE, INVERT
//      02-09-1999 added EXECUTE, ' (tick)
//      03-01-1999 added OPEN, LSEEK, CLOSE, READ, WRITE
//      03-02-1999 added IOCTL
//      03-03-1999 added USLEEP
//      03-07-1999 added FILL, CMOVE
//      03-27-1999 added +LOOP, UNLOOP
//      03-31-1999 added CMOVE>, KEY
//      05-06-1999 added FLOOR, FROUND
//      05-24-1999 added FATAN2, LSHIFT, RSHIFT
//      05-27-1999 added ACCEPT
//      05-29-1999 added QUIT, BASE, BINARY, DECIMAL, HEX, U<, U.
//      06-02-1999 added */, */MOD, NUMBER?
//      06-05-1999 added CHAR (ASCII)
//      06-09-1999 function IsInt now calls Forth's NUMBER? 
//      06-16-1999 fixed to allow multiple LEAVEs within single DO-LOOP
//      07-18-1999 added FIND
//      08-24-1999 compiler reports redefinition of words
//      09-06-1999 added use of global ptr pTIB to permit implemetation of TICK, WORD, etc.
//      09-12-1999 added SYSTEM
//      10-2-1999  used precedence byte to determine execution of non-deferred words
//      10-4-1999 added CREATE, VARIABLE, FVARIABLE as intrinsic words
//      10-6-1999 added CONSTANT, FCONSTANT as intrinsic words
//      10-7-1999 added CHDIR
//      10-8-1999 added ERASE, [']
//      10-9-1999 added TIME&DATE, MS, ?, 2@, 2!, BL
//      10-20-1999 moved global input and output stream pointers into
//                   this module from ForthVM.cpp; added >FILE, CONSOLE
//      10-28-1999 added KEY?
//      11-16-1999 added RECURSE
//      12-14-1999 fixed ExtractName for case of null string
//      12-24-1999 added U>, F0=, F0<, S>D, D>F, F>D
//      12-25-1999 added CELLS, CELL+, CHAR+, DFLOATS, DFLOAT+, SFLOATS, SFLOAT+
//      12-27-1999 added BYE
//      1-13-2000  added ?ALLOT
//      1-23-2000  added 0<>, .R, U.R, [CHAR]; removed ASCII
//      1-24-2000  added LITERAL, made '"' and '."' immediate words
//      2-14-2000  added M*, UM*, FM/MOD
//      2-26-2000  added SM/REM, UM/MOD
//      3-1-2000   display VM errors
//      3-5-2000   changed DO, LEAVE, BEGIN, WHILE, REPEAT, UNTIL, AGAIN,
//                   IF, ELSE, THEN, RECURSE, and '(' from compiler directives
//                   to actual words.
//      3-7-2000   ensure control stacks are cleared after VM error.
//      5-17-2000  added DOES>
//      6-11-2000  added CASE, ENDCASE, OF, ENDOF
//      6-15-2000  added ?DO, ABORT"
//      8-08-2000  added default directory search for include files  DPW
//  
#include <fstream.h>
#include <string.h>
#include <ctype.h>
#include <stdlib.h>
#include "fbc.h"
#include "ForthCompiler.h"

extern int debug;
extern void ClearControlStacks();
extern void PrintVM_Error (int);
extern int ForthVM (vector<byte>*, int**, byte**);
extern vector<DictionaryEntry> Dictionary;
extern vector<DictionaryEntry>::iterator LocateWord (char*);
extern void RemoveLastWord();
extern vector<char*> StringTable;

extern "C" int* GlobalSp;
extern "C" byte* GlobalTp;
extern "C" int* JumpTable;
extern "C" int Base;
extern "C" int State;  // TRUE = compile, FALSE = interpret
extern "C" char* pTIB; 
extern "C" char TIB[];  // contains current line of input
extern "C" int C_numberquery();
extern "C" int L_abort();

void strupr (char*);

// stacks for keeping track of nested control structures

vector<int> ifstack;	// stack for if-then constructs
vector<int> beginstack;	// stack for begin ... constructs
vector<int> whilestack;	// stack for while jump holders
vector<int> dostack;    // stack for do loops
vector<int> leavestack; // stack for leave jumps
vector<int> recursestack; // stack for recursion
vector<int> casestack;  // stack for case jumps
vector<int> ofstack;   // stack for of...endof constructs

// The global input and output streams

istream* pInStream ;
ostream* pOutStream ;

// Global ptr to current opcode vector

vector<byte>* pCurrentOps;

// The last word interpreted (needs to be global)

char WordToken[256];

char* WordNames[] =
    {
        "WORD", "WORDS", "FIND", 
	"'", "[']",
	"CREATE", "DOES>", "FORGET", 
	"COLD", "ALLOT", "?ALLOT",
	"LITERAL",
	"CONSTANT", "FCONSTANT",
	"VARIABLE", "FVARIABLE",
	"CELLS", "CELL+", "CHAR+",
	"DFLOATS", "DFLOAT+", "SFLOATS", "SFLOAT+",
	"?", "@", "!", 
	"2@", "2!", "A@",
        "C@", "C!",
        "W@", "W!",
        "F@", "F!",
        "DF@", "DF!",
        "SF@", "SF!",
	"SP@", "RP@",
        ">R", "R>", "R@", "?DUP",
        "DUP", "DROP", "SWAP",
        "OVER", "ROT", "-ROT", 
	"NIP", "TUCK", "PICK", "ROLL",
        "2DUP", "2DROP", "2SWAP",
        "2OVER", "2ROT",
        "DEPTH",
	"BASE", "BINARY", "DECIMAL", "HEX",
        "1+", "1-", "2+", "2-", 
	"2*", "2/",
        "DO", "?DO", 
	"LOOP", "+LOOP", 
	"LEAVE", "UNLOOP", 
	"I", "J",
	"BEGIN", "WHILE", "REPEAT",
	"UNTIL", "AGAIN",
	"IF", "ELSE", "THEN",
	"CASE", "ENDCASE", "OF", "ENDOF",
	"RECURSE", "BYE",
        "EXIT", "QUIT", "ABORT",
	"ABORT\x22",
        "EXECUTE", "CALL", "SYSTEM",
	"TIME&DATE", "MS", "CHDIR",
	">FILE", "CONSOLE",
	"(",
	"\x22", "C\x22", "S\x22", 
	"COUNT", "NUMBER?",
        ".", ".R", "U.", "U.R",
	"F.", ".\x22", ".S",
        "CR", "SPACES", "EMIT", "TYPE",
	"BL", "[CHAR]", "CHAR",
	"KEY", "KEY?", "ACCEPT",
        "=", "<>", "<", ">", "<=", ">=",
	"U<", "U>",
	"0<", "0=", "0<>", "0>", 
	"FALSE", "TRUE",
        "AND", "OR", "XOR", "NOT", "INVERT",
	"LSHIFT", "RSHIFT",
        "+", "-", "*", "/",
	"MOD", "/MOD", 
	"*/", "*/MOD", "+!",
	"M*", "UM*", "UM/MOD", 
	"FM/MOD", "SM/REM",
        "ABS", "NEGATE", "MIN", "MAX",
	"OPEN", "LSEEK", "CLOSE", 
	"READ", "WRITE", "IOCTL",
	"USLEEP", "FILL", "ERASE", 
	"CMOVE", "CMOVE>",
        "FDUP", "FDROP", "FSWAP",
        "FOVER", "FROT",
        "F=", "F<>", "F<", "F>", "F<=", "F>=",
	"F0=", "F0<",
        "F+", "F-", "F*", "F/", "F**", "FSQRT",
        "FABS", "FNEGATE",
	"FLOOR", "FROUND",
	"FMIN", "FMAX",
        "FSIN", "FCOS", "FTAN",
        "FACOS", "FASIN", "FATAN",
	"FATAN2",
        "FLOG", "FLN", "FEXP",
        "DEG>RAD", "RAD>DEG",
        "S>D", "S>F", "F>S", "D>F", "F>D"
    };

byte WordCodes[] =
    {
        OP_WORD, OP_WORDS, OP_FIND, 
	OP_TICK, OP_BRACKETTICK,
	OP_CREATE, OP_DOES, OP_FORGET, 
	OP_COLD, OP_ALLOT, OP_QUERYALLOT,
	OP_LITERAL,
	OP_CONSTANT, OP_FCONSTANT,
	OP_VARIABLE, OP_FVARIABLE,
	OP_CELLS, OP_CELLPLUS, OP_INC,
	OP_DFLOATS, OP_DFLOATPLUS, OP_CELLS, OP_CELLPLUS,
	OP_QUESTION, OP_FETCH, OP_STORE, 
	OP_DFFETCH, OP_DFSTORE, OP_AFETCH,
        OP_CFETCH, OP_CSTORE,
        OP_WFETCH, OP_WSTORE,
        OP_DFFETCH, OP_DFSTORE,
        OP_DFFETCH, OP_DFSTORE,
        OP_SFFETCH, OP_SFSTORE,
	OP_SPFETCH, OP_RPFETCH,
        OP_PUSH, OP_POP, OP_RFETCH, OP_QUERYDUP,
        OP_DUP, OP_DROP, OP_SWAP,
        OP_OVER, OP_ROT, OP_MINUSROT, 
	OP_NIP, OP_TUCK, OP_PICK, OP_ROLL,
        OP_2DUP, OP_2DROP, OP_2SWAP,
        OP_2OVER, OP_2ROT,
        OP_DEPTH,
	OP_BASE, OP_BINARY, OP_DECIMAL, OP_HEX,
        OP_INC, OP_DEC, OP_TWOPLUS, OP_TWOMINUS,
	OP_TWOSTAR, OP_TWODIV,
        OP_DO, OP_QUERYDO, 
	OP_LOOP, OP_PLUSLOOP, 
	OP_LEAVE, OP_UNLOOP,
	OP_I, OP_J,
	OP_BEGIN, OP_WHILE, OP_REPEAT,
	OP_UNTIL, OP_AGAIN,
	OP_IF, OP_ELSE, OP_THEN,
	OP_CASE, OP_ENDCASE, OP_OF, OP_ENDOF,
	OP_RECURSE, OP_BYE,
        OP_RET, OP_QUIT, OP_ABORT,
	OP_ABORTQUOTE,
        OP_EXECUTE, OP_CALL, OP_SYSTEM,
	OP_TIMEANDDATE, OP_MS, OP_CHDIR,
	OP_TOFILE, OP_CONSOLE,
	OP_LPAREN,
	OP_CQUOTE, OP_CQUOTE, OP_SQUOTE,
	OP_COUNT, OP_NUMBERQUERY,
        OP_DOT, OP_DOTR, OP_UDOT, OP_UDOTR,
	OP_FDOT, OP_DOTQUOTE, OP_DOTS,
        OP_CR, OP_SPACES, OP_EMIT, OP_TYPE,
	OP_BL, OP_BRACKETCHAR, OP_CHAR,
	OP_KEY, OP_KEYQUERY, OP_ACCEPT,
        OP_EQ, OP_NE, OP_LT, OP_GT, OP_LE, OP_GE,
	OP_ULT, OP_UGT,
	OP_ZEROLT, OP_ZEROEQ, OP_ZERONE, OP_ZEROGT, 
	OP_FALSE, OP_TRUE,
        OP_AND, OP_OR, OP_XOR, OP_NOT, OP_NOT,
	OP_LSHIFT, OP_RSHIFT,
        OP_ADD, OP_SUB, OP_MUL, OP_DIV, 
	OP_MOD, OP_SLASHMOD,
	OP_STARSLASH, OP_STARSLASHMOD, OP_PLUSSTORE,
	OP_MSTAR, OP_UMSTAR, OP_UMSLASHMOD, 
	OP_FMSLASHMOD, OP_SMSLASHREM,
        OP_ABS, OP_NEG, OP_MIN, OP_MAX,
	OP_OPEN, OP_LSEEK, OP_CLOSE, 
	OP_READ, OP_WRITE, OP_IOCTL,
	OP_USLEEP, OP_FILL, OP_ERASE, 
	OP_CMOVE, OP_CMOVEFROM,
        OP_2DUP, OP_2DROP, OP_2SWAP,
        OP_2OVER, OP_2ROT,
        OP_FEQ, OP_FNE, OP_FLT, OP_FGT, OP_FLE, OP_FGE,
	OP_FZEROEQ, OP_FZEROLT,
        OP_FADD, OP_FSUB, OP_FMUL, OP_FDIV, OP_FPOW, OP_FSQRT,
        OP_FABS, OP_FNEG,
	OP_FLOOR, OP_FROUND,
	OP_FMIN, OP_FMAX,
        OP_FSIN, OP_FCOS, OP_FTAN,
        OP_FACOS, OP_FASIN, OP_FATAN,
	OP_FATAN2,
        OP_FLOG, OP_FLN, OP_FEXP,
        OP_DEGTORAD, OP_RADTODEG,
        OP_STOD, OP_STOF, OP_FTOS, OP_DTOF, OP_FTOD
    };

// Non-deferred words are executed immediately by
//   the interpreter in the non-compiling state.
 
byte NondeferredWords[] = 
{
  OP_BINARY, 
  OP_DECIMAL, 
  OP_HEX, 
  OP_WORD, 
  OP_TICK,
  OP_CREATE,
  OP_FORGET,
  OP_COLD,
  OP_ALLOT,
  OP_QUERYALLOT,
  OP_CONSTANT,
  OP_FCONSTANT,
  OP_VARIABLE,
  OP_FVARIABLE,
  OP_CHAR,
  OP_TOFILE,
  OP_CONSOLE
};

byte ImmediateWords[] =
{
  OP_LPAREN,
  OP_BRACKETCHAR,
  OP_BRACKETTICK,
  OP_LITERAL,
  OP_CQUOTE,
  OP_SQUOTE,
  OP_DOTQUOTE,
  OP_DO,
  OP_QUERYDO,
  OP_LEAVE,
  OP_ABORTQUOTE,
  OP_BEGIN,
  OP_WHILE,
  OP_REPEAT,
  OP_UNTIL,
  OP_AGAIN,
  OP_IF,
  OP_ELSE,
  OP_THEN,
  OP_CASE,
  OP_ENDCASE,
  OP_OF,
  OP_ENDOF,
  OP_RECURSE
};


char* C_ErrorMessages[] =
{
	"",
	"",
	"End of definition with no beginning",
	"End of string",	 
        "Not allowed inside colon definition",
	"Error opening file",
	"Incomplete IF...THEN structure",
	"Incomplete BEGIN structure",
	"Unknown word",
	"No matching DO",
	"Incomplete DO loop",
	"Incomplete CASE structure"
};


//---------------------------------------------------------------

char* ExtractName (char* str, char* name)
{
// Starting at ptr str, extract the non delimiter text into
//   a buffer starting at name with null terminator appended
//   at the end. Return a pointer to the next position in
//   str.

    char* delim = "\n\r\t ";
    char *pStr = str, *pName = name;

    if (*pStr)
      {
	while (strchr(delim, *pStr)) ++pStr;
	while (*pStr && (strchr(delim, *pStr) == NULL))
	  {
	    *pName = *pStr;
	    ++pName;
	    ++pStr;
	  }
      }
    *pName = 0;
    return pStr;
}
//---------------------------------------------------------------
	
int IsForthWord (char* name, DictionaryEntry* pE)
{
// Locate and Return a copy of the dictionary entry
//   with the specified name.  Return True if found,
//   False otherwise. A copy of the entry is returned
//   in *pE.

    vector<DictionaryEntry>::iterator i = LocateWord (name);

    if (i)
    {
        *pE = *i;
        return TRUE;
    }
    else
        return FALSE;
}
//---------------------------------------------------------------

int IsFloat (char* token, double* p)
{
// Check the string token to see if it is an LMI style floating point
//   number; if so set the value of *p and return True, otherwise
//   return False.

    char *pStr = token;

//    cout << "\nIsFloat: token = " << token;

    if (strchr(pStr, 'E'))
    {
        while ((isdigit(*pStr)) || (*pStr == '-')
          || (*pStr == 'E') || (*pStr == '+') || (*pStr == '.'))
        {
            ++pStr;
//            cout << ' ' << ((int) *pStr);
        }
        if (*pStr == 0)
        {
            // LMI Forth style

            --pStr;
            if (*pStr == 'E') *pStr = '\0';
            *p = atof(token);
            return TRUE;
        }
    }

    return FALSE;
}
//----------------------------------------------------------------

int IsInt (char* token, int* p)
{
// Check the string token to see if it is an integer number;
//   if so set the value of *p and return True, otherwise return False.

  char s[256];
  *s = (unsigned char) strlen(token);
  strcpy (s+1, token);
  *GlobalSp-- = (int) s;
  *GlobalTp-- = OP_ADDR;
  
  int err = C_numberquery();
  if (err)
    {
      // stack has probably become corrupted -- call abort

      cout << "Stack error during compilation.\n";
      L_abort();
      return FALSE;
    }

  ++GlobalSp; ++GlobalTp;
  int b = *GlobalSp++; ++GlobalTp;
  ++GlobalSp; ++GlobalTp;
  *p = *GlobalSp;

  return b;
}
//---------------------------------------------------------------

void OutputForthByteCode (vector<byte>* pFBC)
{
// Output opcode vector to an output stream for use in
//   debugging the compiler.

    int i, n = pFBC->size();
    byte* bp = pFBC->begin();

    *pOutStream << "\nOpcodes:\n";
    for (i = 0; i < n; i++)
    {
        *pOutStream << ((int) *bp) << ' ';
        if (((i + 1) % 8) == 0) *pOutStream << '\n';
        ++bp;
    }
    *pOutStream << '\n';
    return;
}
//---------------------------------------------------------------

void SetForthInputStream (istream& SourceStream)
{
  // Set the input stream for the Forth Compiler and Virtual Machine

  pInStream = &SourceStream;
}
//--------------------------------------------------------------

void SetForthOutputStream (ostream& OutStream)
{
  // Set the output stream for the Forth Compiler and Virtual Machine

  pOutStream = &OutStream;
}
//---------------------------------------------------------------

int ForthCompiler (vector<byte>* pOpCodes, int* pLc)
{
// The FORTH Compiler
//
// Reads and compile the source statements from the input stream
//   into a vector of FORTH Byte Codes.
//
// Return value:
//
//  0   no error
//  other --- see ForthCompiler.h

  int ecode = 0, opcount = 0;
  char s[256], *begin_string, *end_string;
  double fval;
  int i, j, ival, *sp;
  vector<byte>::iterator ib1, ib2;
  vector<int>::iterator iI;
  DictionaryEntry d;
  vector<DictionaryEntry>::iterator id;
  byte opval, *fp, *ip, *bp, *tp;


  static int linecount = 0;
  static DictionaryEntry NewWord;

  fp = (byte *) &fval;
  ip = (byte *) &ival;

  if (! State) linecount = 0;
  pCurrentOps = pOpCodes;

  while (TRUE)
    {
      // Read each line and parse

      pInStream->getline(TIB, 255);
      if (debug) (*pOutStream) << TIB << '\n';

      if (pInStream->fail())
	{
	  if (State)
	    {
	      ecode = E_C_ENDOFSTREAM;  // reached end of stream before end of definition
	      break;
	    }
	  pOpCodes->push_back(OP_RET);
	  break;    // end of stream reached
	}
      ++linecount;
      pTIB = TIB;
      while (*pTIB && (pTIB < (TIB + 255)))
	{
	  if (*pTIB == ' ' || *pTIB == '\t')
	    ++pTIB;
	  else if (*pTIB == '\\' && (*(pTIB + 1) == ' ' ||
				   *(pTIB+1) == '\0' || *(pTIB+1) == '\t'))
	    break;  // ignore rest of the line
	  else if (*pTIB == ':')
	    {
	      if (pOpCodes->size())
		{
		  // Execute the code outside of a definition

		  pOpCodes->push_back(OP_RET);
		  ival = ForthVM (pOpCodes, &sp, &tp);
		  pOpCodes->erase(pOpCodes->begin(), pOpCodes->end());
		  if (ival) 
		    {
		      PrintVM_Error(ival);
		      goto endcompile;
		    }
		}

	      State = TRUE;
	      ++pTIB;
	      pTIB = ExtractName (pTIB, WordToken);
	      strupr(WordToken);
	      strcpy (NewWord.WordName, WordToken);
	      NewWord.WordCode = OP_DEFINITION;
	      NewWord.Precedence = 0;
	      NewWord.Pfa = NULL;
	      NewWord.Cfa = NULL;

	      recursestack.erase(recursestack.begin(), recursestack.end());
	    }
	  else if (*pTIB == ';')
	    {
	      pOpCodes->push_back(OP_RET);

	      if (State)
		{
		  // Check for incomplete control structures
		    
		  if (ifstack.size())
		    {
		      ecode = E_C_INCOMPLETEIF;
		      goto endcompile;
		    }
		  if (beginstack.size() || whilestack.size())
		    {
		      ecode = E_C_INCOMPLETEBEGIN;
		      goto endcompile;
		    }
		  if (dostack.size() || leavestack.size())
		    {
		      ecode = E_C_INCOMPLETELOOP;
		      goto endcompile;
		    }
		  if (casestack.size() || ofstack.size())
		    {
		      ecode = E_C_INCOMPLETECASE;
		      goto endcompile;
		    }

		  // Add a new entry into the dictionary

		  if (debug) OutputForthByteCode (pOpCodes);
 		  
		  NewWord.Pfa = new byte[pOpCodes->size()];
		  NewWord.Cfa = NewWord.Pfa;

		  // Resolve any self references (recursion)

		  bp = (byte*) &NewWord.Pfa;
		  while (recursestack.size())
		    {
		      i = recursestack[recursestack.size() - 1];
		      ib1 = pOpCodes->begin() + i;
		      for (i = 0; i < sizeof(void*); i++) *ib1++ = *(bp + i);
		      recursestack.pop_back();
		    }

		  byte* dest = (byte*) NewWord.Pfa;
		  bp = pOpCodes->begin();
		  while (bp < pOpCodes->end()) *dest++ = *bp++;
		  if (IsForthWord(NewWord.WordName, &d))
		    *pOutStream << NewWord.WordName << " is redefined\n";
		  Dictionary.push_back(NewWord);
		  pOpCodes->erase(pOpCodes->begin(), pOpCodes->end());
		  State = FALSE;
		}
	      else
		{
		  ecode = E_C_ENDOFDEF;
		  goto endcompile;
		}
	      ++pTIB;
	    }
	  else
	    {
	      pTIB = ExtractName (pTIB, WordToken);
	      strupr(WordToken);

	      if (IsForthWord(WordToken, &d))
		{
		  pOpCodes->push_back(d.WordCode);
		  

		  if (d.WordCode == OP_DEFINITION)
		    {
		      pOpCodes->pop_back();
		      pOpCodes->push_back(OP_ADDR);
		      bp = (byte*) &d.Cfa;
		      for (i = 0; i < sizeof(byte*); i++)
			pOpCodes->push_back(*(bp + i));
		      pOpCodes->push_back(OP_EXECUTE);
		    }
		  else if (d.WordCode == OP_ADDR)
		    {
		      // push address into the byte code vector

		      bp = (byte*) &d.Pfa;

		      for (i = 0; i < sizeof(byte*); i++)
			pOpCodes->push_back(*(bp + i));
		    }
		  else if (d.WordCode == OP_IVAL)
		    {
		      // push value into the byte code vector

		      bp = (byte*) d.Pfa;			
		      for (i = 0; i < sizeof(int); i++)
			pOpCodes->push_back(*(bp + i));
		    }
		  else if (d.WordCode == OP_FVAL)
		    {
		      // push float value into the vector

		      bp = (byte*) d.Pfa;
		      for (i = 0; i < sizeof(double); i++)
			pOpCodes->push_back(*(bp + i));
		    }
		  else if (d.WordCode == OP_UNLOOP)
		    {
		      if (dostack.empty())
			{
			  ecode = E_C_NODO;
			  goto endcompile;
			}
		    }
		  else if (d.WordCode == OP_LOOP || d.WordCode == OP_PLUSLOOP)
		    {
		      if (dostack.empty())
			{
			  ecode = E_C_NODO;
			  goto endcompile;
			}
		      if (leavestack.size())
			{
			  i = dostack[dostack.size() - 1];
			  do
			    {
			      j = leavestack[leavestack.size() - 1];
			      if (j > i)
				{
				  ival = pOpCodes->size() - j + 1;
				  ib1 = pOpCodes->begin() + j;
				  *ib1++ = *ip;       // write the relative jump count
				  *ib1++ = *(ip + 1);
				  *ib1++ = *(ip + 2);
				  *ib1 = *(ip + 3);
				  leavestack.pop_back();
				}
			    } while ((j > i) && (leavestack.size())) ;
			}
		      dostack.pop_back();
		    }
		  else
		    {
		      ;
		    }
		  if (d.Precedence & PRECEDENCE_NON_DEFERRED)
		    {
		      if (State)
			{
			  // A defined word automatically becomes non-deferred
			  //  if it's definition contains a non-deferred word

			  NewWord.Precedence |= PRECEDENCE_NON_DEFERRED ;
			}
		      else
			{
			  // Execute the opcode vector immediately for
			  // non-deferred word in interpreter mode

			  pOpCodes->push_back(OP_RET);
			  if (debug) OutputForthByteCode (pOpCodes);
			  ival = ForthVM (pOpCodes, &sp, &tp);
			  pOpCodes->erase(pOpCodes->begin(), pOpCodes->end());
			  if (ival) 
			    {
			      PrintVM_Error(ival);
			      goto endcompile;
			    }
			}
		    }
		  if (d.Precedence & PRECEDENCE_IMMEDIATE)
		    {
		      // Execute an immediate word during compilation;
		      
		      pOpCodes->pop_back();
		      vector<byte> SingleOp;
		      SingleOp.push_back(d.WordCode);
		      SingleOp.push_back(OP_RET);
		      ival = ForthVM (&SingleOp, &sp, &tp);
		      SingleOp.erase(SingleOp.begin(), SingleOp.end());
		      if (ival) 
			{
			  PrintVM_Error(ival);
			  goto endcompile;
			}		      
		    }
		}
	      else if (IsInt(WordToken, &ival))
		{
		  pOpCodes->push_back(OP_IVAL);
		  for (i = 0; i < sizeof(int); i++)
		    pOpCodes->push_back(*(ip + i)); // store in proper order
		}
	      else if (IsFloat(WordToken, &fval))
		{
		  pOpCodes->push_back(OP_FVAL);
		  for (i = 0; i < sizeof(double); i++)
		    pOpCodes->push_back(*(fp + i)); // store in proper order
		}
	      else if (strcmp(WordToken, "INCLUDE") == 0)
		{
		  if (State)
		    {
		      ecode = E_C_NOTINDEF;
		      goto endcompile;
		    }
		  ++pTIB;
		  pTIB = ExtractName (pTIB, WordToken);
		  strcpy (s, pTIB);  // save remaining part of input line in TIB
		  if (!strchr(WordToken, '.')) strcat(WordToken, ".4th");

		  ifstream f(WordToken);
		  if(!f)
		  {
		      //*pOutStream << '\n' << WordToken << " does not exist in current directory.
		      if(!strchr(WordToken,'/'))
		      {
			char temp[256];
			if (getenv("KFORTH_DIR"))
			  strncpy(temp, getenv("KFORTH_DIR"), 255);
			strcat(temp, "/");
			strcat(temp, WordToken);
			f.close();
			f.open(temp);
			if (f)
			  {
			    strcpy(WordToken, temp);
			    *pOutStream << '\n' << WordToken << '\n';
			  }
		      }
		  }
		  f.close();
		  f.open(WordToken);

		  if (f.fail())
		    {
		      *pOutStream << '\n' << WordToken << '\n';
		      ecode = E_C_OPENFILE;
		      goto endcompile;
		    }
		  istream* pTempIn = pInStream;  // save input stream ptr
		  SetForthInputStream(f);  // set the new input stream
		  ecode = ForthCompiler (pOpCodes, &linecount);
		  f.close();
		  pInStream = pTempIn;  // restore the input stream
		  if (ecode) goto endcompile;

		  // Execute the code immediately
		      
		  ival = ForthVM (pOpCodes, &sp, &tp);
		  pOpCodes->erase(pOpCodes->begin(), pOpCodes->end());
		  if (ival) 
		    {
		      PrintVM_Error(ival);
		      goto endcompile;
		    }
		  strcpy(TIB, s);  // restore TIB with remaining input line
		  pTIB = TIB;      // restore ptr
		}
	      else
		{
		  *pOutStream << '\n' << WordToken << '\n';
		  ecode = E_C_UNKNOWNWORD;  // unknown keyword
		  goto endcompile;
		}
	    }
	}
    }

endcompile:
    
  if (ecode != E_C_ENDOFSTREAM)
    {
      // A compiler error occurred; reset to interpreter mode and
      //   clear all flow control stacks.
 
      State = FALSE;
      ClearControlStacks();
    }
  if (debug) *pOutStream << "Error: " << ecode << " State: " << State << '\n';
  *pLc = linecount;
  return ecode;
}

void strupr (char* p)
{
// convert string to upper case

  while (*p) {*p = toupper(*p); ++p;}
}

