// ForthVM-fast.cpp
//
// The FORTH Virtual Machine (FVM) to execute Forth byte code.
//
// Copyright (c) 1996--2004 Krishna Myneni, Creative Consulting for
//   Research & Education
//
// This software is provided under the General Public License.
//
// Created: 2-24-96
// Revisions: 
//       10-14-1998
//       4-28-1999  increased stack size to 32768 and ret stack to 4096  KM
//       5-29-1999  moved ABORT to vm.s; added abort/quit error code;
//       6-06-1999  created C++ functions which can be called from vm  KM
//       9-06-1999  implemented CPP_word  KM
//       9-07-1999  initialize State in OpenForth  KM
//       10-2-1999  initialize precedence byte for non-deferred words  KM
//       10-4-1999  added CPP_create, CPP_variable, CPP_fvariable  KM
//       10-6-1999  added CPP_constant, CPP_fconstant  KM
//       10-10-1999 initialize precedence for immediate words;
//                    added CPP_char, CPP_brackettick, CPP_forget, CPP_cold  KM
//       10-20-1999 moved global input and output stream pointers to
//                    ForthCompiler.cpp; added CPP_tofile and CPP_console  KM
//       12-14-1999 modified CPP_tofile to use default file when not specified  KM
//       1-13-2000  added CPP_queryallot  KM
//       1-23-2000  added CPP_dotr, CPP_udotr, CPP_bracketchar, and changed 
//                    behavior of CPP_char for ANSI compatible behavior  KM
//       1-24-2000  added CPP_literal, CPP_quote, CPP_dotquote  KM
//       3-5-2000   added CPP_lparen, CPP_do, CPP_leave, CPP_begin,
//                    CPP_while, CPP_repeat, CPP_until, CPP_again,
//                    CPP_if, CPP_else, CPP_then, CPP_recurse  KM
//       3-7-2000   added ClearControlStacks; perform after VM error  KM
//       5-17-2000  added CPP_does  KM
//       6-12-2000  added CPP_case, CPP_endcase, CPP_of, CPP_endof  KM
//       6-15-2000  added CPP_abortquote  KM
//       9-05-2000  added CPP_ddot KM
//       4-01-2001  cast pointers for delete [] in RemoveLastWord  KM
//       4-22-2001  modified CPP_lparen to handle multiline comments  KM
//       5-13-2001  added CPP_dotparen KM
//       5-20-2001  added CPP_bracketsharp, CPP_sharp, CPP_sharps, CPP_hold,
//                    CPP_sign, CPP_sharpbracket, and CPP_uddot  KM
//       5-29-2001  wrote CPP_querydo  KM
//       9-02-2001  fixed CPP_find to return the code pointer  KM
//       9-03-2001  added CPP_immediate and CPP_nondeferred  KM
//       9-21-2001  modified CPP_word to skip initial space  KM
//       9-26-2001  fixed CPP_dotr and CPP_udotr to not print trailing space,
//                    using new fundamental word CPP_udot0  KM
//      12-10-2001  added CPP_evaluate, [ and ]  km
//      07-30-2002  fixed CPP_evaluate problems; added CPP_backslash;
//                    incr. line counters in CPP_lparen and CPP_dotparen  km
//      09-09-2002  fixed CPP_evaluate for VM reentrancy problem  km
//      09-26-2002  revised include statements and added "using"
//                    declarations to resolve C++ std namespace defs;
//                    fixed g++ 3.2 complaints about iterator usage; 
//                    use C linkage for CPP_x functions; replace istrstream
//                    class with istringstream class; modified CPP_dots to
//                    check for stack underflow; also check for stack underflow
//                    on return from VM   km
//      09-29-2002  fixed CPP_word to remove delimiter from input stream  km
//      04-11-2003  fixed problems with CPP_word and added CPP_source and
//                    CPP_refill  km
//      01-31-2004  added CPP_include, CPP_state, CPP_nondeferred  km
//      02-09-2004  added CPP_allocate and CPP_free  km
//      03-17-2004  modified OpenForth() to initialize from WordTemplate array km
//      03-18-2004  added CPP_spstore  km
//      03-27-2004  changed LocateWord to pre-increment iterator as suggested
//                    by BK; improves efficiency by 24%; also use 2-byte
//                    precomparison to gain another 12% improvement  km
//      04-08-2004  added CPP_postpone(), CPP_compilecomma(), and skeleton for
//                    CPP_bracketcompile(), and auxiliary words OpsCompileByte(),
//                    and OpsCompileInt()  km
//      04-09-2004  revised CPP_postpone() to handle variables, constants, and
//                    2constants. Added auxiliary word OpsCompileDouble().
//                    Added CPP_semicolon()  km
//      04-23-2004  changed variable "debug" to type "bool"  km
//      04-26-2004  fixed problem with CPP_sharp() -- needed to use unsigned 
//                    arithmetic. Import L_udmstar and L_utmslash from vm.s.
//                    Now passes John Hayes' ANS core tests for <# # #S #>  km 
//      06-19-2004  modified PrintVM_Error() to handle both kinds of errors km
#include <string.h>
#include <stdlib.h>
#include <math.h>
#include <iostream>
#include <fstream>
#include <sstream>
using std::istringstream;
using std::cout;
using std::endl;
using std::istream;
using std::ostream;
using std::ifstream;
using std::ofstream;
#include "fbc.h"
#include <vector>
using std::vector;
#include "ForthCompiler.h"
#include "ForthVM.h"

#define STACK_SIZE 32768
#define RETURN_STACK_SIZE 4096

extern bool debug;

// Provided by ForthCompiler.cpp

void strupr (char*);
void SetForthInputStream (istream&);
void SetForthOutputStream (ostream&);
extern WordTemplate IntrinsicWords[];
extern char* C_ErrorMessages[];
extern int linecount;
extern istream* pInStream ;    // global input stream
extern ostream* pOutStream ;   // global output stream
extern vector<byte>* pCurrentOps;
extern vector<int> ifstack;
extern vector<int> beginstack;
extern vector<int> whilestack;
extern vector<int> dostack;
extern vector<int> querydostack;
extern vector<int> leavestack;
extern vector<int> recursestack;
extern vector<int> casestack;
extern vector<int> ofstack;
extern DictionaryEntry NewWord;

extern "C" {

  // functions exported FROM vmc.c

  void set_start_time(void);

  // vm functions exported FROM vm.s

  int L_depth();
  int L_tick();
  int L_abort();
  int L_ret();
  int L_dabs();
  int L_dminus();
  int L_mstarslash();
  int L_udmstar();
  int L_utmslash();
  int vm (byte*);     // the machine code virtual machine


  // global pointers and state variables exported TO other modules

  int* GlobalSp;      // the global stack pointer

  byte* GlobalIp;     // the global instruction pointer
  int* GlobalRp;      // the global return stack pointer

  int* BottomOfStack;
  int* BottomOfReturnStack;

  int* vmEntryRp;
  int Base;
  int State;
  char* pTIB;
  int NumberCount;
  char WordBuf[256];
  char TIB[256];
  char NumberBuf[256];
}

extern "C" int JumpTable[];

// The Dictionary

vector<DictionaryEntry> Dictionary;

// Tables

vector<char*> StringTable;

// stacks; these are global to this module

int ForthStack[STACK_SIZE];                  // the stack

int ForthReturnStack[RETURN_STACK_SIZE];     // the return stack

bool FileOutput = FALSE;
vector<byte>* pPreviousOps;    // copy of ptr to old opcode vector for [ and ]
vector<byte> tempOps;          // temporary opcode vector for [ and ]

char* V_ErrorMessages[] =
{
	"",
	"Not data type ADDR",
	"Not data type IVAL",
	"Invalid data type",	
	"Divide by zero",
	"Return stack corrupt",
	"Invalid opcode", 
        "Stack underflow",
	"",
	"Allot failed --- cannot reassign pfa",
	"Cannot create word",
	"End of string not found",
	"No matching DO",
	"No matching BEGIN",
	"ELSE without matching IF",
	"THEN without matching IF",
	"ENDOF without matching OF",
	"ENDCASE without matching CASE",
	"Cannot open file",
	"Address outside of stack space"
};


//---------------------------------------------------------------

int OpenForth ()
{
// Initialize the FORTH dictionary; return the size of
//   the dictionary.

    int i, wcode;
    DictionaryEntry d;

    set_start_time();

    for (i = 0; i < NUMBER_OF_INTRINSIC_WORDS; i++)
    {
        strcpy(d.WordName, IntrinsicWords[i].WordName);
        wcode = IntrinsicWords[i].WordCode;
	d.WordCode = wcode;
	d.Precedence = IntrinsicWords[i].Precedence;
        d.Pfa = new byte[8];
	d.Cfa = d.Pfa;
	byte* bp = (byte*) d.Pfa;
	if (wcode >> 8)
	{
	    bp[0] = OP_ADDR;
	    *((int*) (bp+1)) = (int) JumpTable[wcode];
	    bp[5] = OP_CALL;
	    bp[6] = OP_RET;
	}
	else
	{
	    bp[0] = wcode;
	    bp[1] = OP_RET;
	}
	
        Dictionary.push_back(d);
    }

    // Initialize the global stack pointers

    BottomOfStack = ForthStack + STACK_SIZE - 1;
    BottomOfReturnStack = ForthReturnStack + RETURN_STACK_SIZE - 1;
   
    GlobalSp = BottomOfStack;

    GlobalRp = BottomOfReturnStack;


    vmEntryRp = BottomOfReturnStack;

    Base = 10;
    State = FALSE;

    return Dictionary.size();
}
//---------------------------------------------------------------

void CloseForth ()
{
    // Clean up the compiled words

    while (Dictionary.size())
    {
        RemoveLastWord();
    }

    // Clean up the string table

    vector<char*>::iterator j = StringTable.begin();

    while (j < StringTable.end())
    {
        if (*j) delete [] *j;
        ++j;
    }
    StringTable.erase(StringTable.begin(), StringTable.end());
}

//---------------------------------------------------------------

void RemoveLastWord ()
{
// Remove the last dictionary entry

	vector<DictionaryEntry>::iterator i = Dictionary.end() - 1;
	delete [] (byte*) i->Pfa;	// free memory
	if (i->Pfa != i->Cfa) delete [] (byte*) i->Cfa;
	Dictionary.pop_back(); 
}
//---------------------------------------------------------------

vector<DictionaryEntry>::iterator LocateWord (char* name)
{
// Search the dictionary from end to beginning for an entry
//   with the specified name. Return the iterator to the word
//   or NULL if not found.

	vector<DictionaryEntry>::iterator i;
    
	for (i = Dictionary.end()-1; i >= Dictionary.begin(); --i)
	{
	    if (*((word*) name) == *((word*) i->WordName)) // pre-compare
        	if (strcmp(name, i->WordName) == 0) break;
	}

	if (i >= Dictionary.begin())
        	return i;
	else
		return ((vector<DictionaryEntry>::iterator) NULL);
}
//---------------------------------------------------------------

void ClearControlStacks ()
{
  // Clear the flow control stacks

  if (debug) cout << "Clearing all flow control stacks" << endl; 
  ifstack.erase(ifstack.begin(), ifstack.end());
  beginstack.erase(beginstack.begin(),beginstack.end());
  whilestack.erase(whilestack.begin(),whilestack.end());
  dostack.erase(dostack.begin(), dostack.end());
  querydostack.erase(querydostack.begin(), querydostack.end());
  leavestack.erase(leavestack.begin(), leavestack.end());
  ofstack.erase(ofstack.begin(), ofstack.end());
  casestack.erase(casestack.begin(), casestack.end());
}
//---------------------------------------------------------------

void OpsCopyInt (int offset, int i)
{
  // Copy integer into the current opcode vector at the specified offset 

  vector<byte>::iterator ib = pCurrentOps->begin() + offset;
  byte* ip = (byte*) &i;
  *ib++ = *ip; *ib++ = *(ip + 1); *ib++ = *(ip + 2); *ib = *(ip + 3);

}
//---------------------------------------------------------------

void OpsPushInt (int i)
{
  // push an integer into the current opcode vector

  byte* ip = (byte*) &i;
  for (int j = 0; j < sizeof(int); j++) pCurrentOps->push_back(*(ip + j));
}
//---------------------------------------------------------------

int OpsCompileByte ()
{
  // push a byte value from the stack into the current opcode vector

  ++GlobalSp;
  byte* ip = (byte*) GlobalSp;
  pCurrentOps->push_back(*ip);
  return 0;
}
//---------------------------------------------------------------

int OpsCompileInt ()
{
  // push an int value from the stack into the current opcode vector

  ++GlobalSp;
  OpsPushInt(*GlobalSp);
  return 0;
}
//---------------------------------------------------------------

int OpsCompileDouble ()
{
  // push a double value from the stack into the current opcode vector

  ++GlobalSp;
  OpsPushInt(*GlobalSp++);
  OpsPushInt(*GlobalSp);
  return 0;
}
//----------------------------------------------------------------

void PrintVM_Error (int ec)
{
    int ei = ec & 0xFF;
    int imax = (ec >> 8) ? MAX_C_ERR_MESSAGES : MAX_V_ERR_MESSAGES;
    char *pMsg, elabel[12];
    
    if ((ei >= 0) * (ei < imax))
    {
	pMsg = (ec >> 8) ? C_ErrorMessages[ei] : V_ErrorMessages[ei];
	if (ec >> 8)  strcpy(elabel, "Compiler"); 
	else strcpy(elabel, "VM");
	*pOutStream << elabel << " Error(" << ei << "): " <<
	    pMsg << endl;
    }
}
//---------------------------------------------------------------

int ForthVM (vector<byte>* pFBC, int** pStackPtr, byte** pTypePtr)
{
// The FORTH Virtual Machine
//
// Arguments:
//
//      pFBC        pointer to vector of Forth byte codes
//      pStackPtr   receives pointer to the top item on the stack at exit
//      pTypePtr    receives pointer to the top item on the type stack at exit
//
// Return value: error code (see ForthVM.h)
//
if (debug)  cout << ">ForthVM Sp: " << GlobalSp << " Rp: " << GlobalRp << endl;
  if (pFBC->size() == 0) return 0;  // null opcode vector

  // Initialize the instruction ptr and error code

  // byte *ip = (byte*) pFBC->begin();
  byte *ip = (byte *) &(*pFBC)[0];
  int ecode = 0;

  // Execute the virtual machine; return when error occurs or
  //   the return stack is exhausted.

  ecode = vm (ip);

  if (ecode)
    {
      if (debug) cout << "vm Error: " << ecode << "  Offending OpCode: " << ((int) *(GlobalIp-1)) << endl;
      ClearControlStacks();
      GlobalRp = BottomOfReturnStack;        // reset the return stack ptrs
 
    }
  else if (GlobalSp > BottomOfStack)
  {
      ecode = E_V_STK_UNDERFLOW;
  }
  else if (GlobalRp > BottomOfReturnStack)
  {
      ecode = E_V_RET_STK_CORRUPT;
  }
  else
      ;

  // On stack underflow, update the global stack pointers.

  if ((ecode == E_V_STK_UNDERFLOW) || (ecode == E_V_RET_STK_CORRUPT))
  {
      L_abort();
  }

  // Set up return information

  *pStackPtr = GlobalSp + 1;

if (debug)  cout << "<ForthVM Sp: " << GlobalSp << " Rp: " << GlobalRp << 
	      "  vmEntryRp: " << vmEntryRp << endl;
  return ecode;
}
//---------------------------------------------------------------

// Use C linkage for all of the VM functions

extern "C" {

int CPP_colon()
{
    // stack: ( -- | the colon compiler )

    char WordToken[256];
    State = TRUE;
    ++pTIB;
    pTIB = ExtractName (pTIB, WordToken);
    strupr(WordToken);
    strcpy (NewWord.WordName, WordToken);
    NewWord.WordCode = OP_DEFINITION;
    NewWord.Precedence = PRECEDENCE_NONE;
    NewWord.Pfa = NULL;
    NewWord.Cfa = NULL;
    recursestack.erase(recursestack.begin(), recursestack.end());

    return 0;
}
//----------------------------------------------------------------

int CPP_semicolon()
{
  // stack: ( -- | terminate compilation of word )

  int ecode = 0;

  pCurrentOps->push_back(OP_RET);

  if (State)
    {
      // Check for incomplete control structures
		    
      if (ifstack.size())                          ecode = E_C_INCOMPLETEIF;

      if (beginstack.size() || whilestack.size())  ecode = E_C_INCOMPLETEBEGIN;
      
      if (dostack.size() || leavestack.size())     ecode = E_C_INCOMPLETELOOP;

      if (casestack.size() || ofstack.size())      ecode = E_C_INCOMPLETECASE;
      
      if (ecode) return ecode;

      // Add a new entry into the dictionary

      if (debug) OutputForthByteCode (pCurrentOps);
 		  
      NewWord.Pfa = new byte[pCurrentOps->size()];
      NewWord.Cfa = NewWord.Pfa;

      // Resolve any self references (recursion)

      byte *bp, *dest;
      int i;
      vector<byte>::iterator ib;
      DictionaryEntry d;


      bp = (byte*) &NewWord.Pfa;
      while (recursestack.size())
	{
	  i = recursestack[recursestack.size() - 1];
	  ib = pCurrentOps->begin() + i;
	  for (i = 0; i < sizeof(void*); i++) *ib++ = *(bp + i);
	  recursestack.pop_back();
	}

      dest = (byte*) NewWord.Pfa;
      bp = (byte*) &(*pCurrentOps)[0]; // ->begin();
      while ((vector<byte>::iterator) bp < pCurrentOps->end()) *dest++ = *bp++;
      if (IsForthWord(NewWord.WordName, &d))
	*pOutStream << NewWord.WordName << " is redefined\n";
      Dictionary.push_back(NewWord);
      pCurrentOps->erase(pCurrentOps->begin(), pCurrentOps->end());
      State = FALSE;
    }
  else
    {
      ecode = E_C_ENDOFDEF;
      // goto endcompile;
    }
    
  return ecode;
}
//-----------------------------------------------------------------

int CPP_backslash()
{
  // stack: ( -- | advance pTIB to end of line )

  while (*pTIB) ++pTIB;
  return 0;
}
// --------------------------------------------------------------

int CPP_lparen()
{
  // stack: ( -- | advance pTIB past end of comment )

  while (TRUE)
    {
      while ((pTIB < (TIB + 255)) && (! (*pTIB == ')')) && *pTIB) ++pTIB;
      if (*pTIB == ')')
	{
	  ++pTIB;
	  break;
	}
      else
	{
	  pInStream->getline(TIB, 255);
	  if (pInStream->fail()) return E_V_NO_EOS;
	  ++linecount;
	  pTIB = TIB;
	}
    }

  return 0;
}
//---------------------------------------------------------------

int CPP_dotparen()
{
  // stack: ( -- | display comment and advance pTIB past end of comment )

  ++pTIB;

  while (TRUE)
    {
      while ((pTIB < (TIB + 255)) && (! (*pTIB == ')')) && *pTIB) 
	{
	  *pOutStream << *pTIB;
	  ++pTIB; 
	}

      if (*pTIB == ')')
	{
	  pOutStream->flush();
	  ++pTIB;
	  break;
	}
      else
	{
	  *pOutStream << endl;
	  pInStream->getline(TIB, 255);
	  if (pInStream->fail()) return E_V_NO_EOS;
	  ++linecount;
	  pTIB = TIB;
	}
    }

  return 0;
}
//---------------------------------------------------------------

int CPP_bracketsharp()
{
  // stack: ( -- | initialize for number conversion )

  NumberCount = 0;
  NumberBuf[255] = 0;
  return 0;
}
//----------------------------------------------------------------

int CPP_sharp()
{
  // stack: ( ud1 -- ud2 | convert one digit of ud1 )

  unsigned int u1, u2, rem;
  char ch;

  *GlobalSp-- = *(GlobalSp+2); *GlobalSp-- = *(GlobalSp+2); // 2dup
  *GlobalSp-- = 0;  // pad to triple length
  *GlobalSp-- = Base;
  L_utmslash();
  u1 = *(GlobalSp + 1);  // quotient
  u2 = *(GlobalSp + 2);

  // quotient is on the stack; we need the remainder

  *GlobalSp-- = Base;
  L_udmstar();
  ++GlobalSp;  // drop
  L_dminus();
  rem = *(GlobalSp + 2);  // get the remainder
  *(GlobalSp + 1) = u1;   // replace rem with quotient on the stack
  *(GlobalSp + 2) = u2;
  ch = (rem < 10) ? (rem + 48) : (rem + 55);
  ++NumberCount;
  NumberBuf[255 - NumberCount] = ch;

  return 0;
}
//----------------------------------------------------------------

int CPP_sharps()
{
  // stack: ( ud -- 0 0 | finish converting all digits of ud )

  unsigned int u1, u2;
  u1 = 1; u2 = 0;

  while (u1 | u2)
    {
      CPP_sharp();
      u1 = *(GlobalSp + 1);
      u2 = *(GlobalSp + 2);
    }
  return 0;
}
//----------------------------------------------------------------

int CPP_hold()
{
  // stack: ( n -- | insert character into number string )
  char ch = *(++GlobalSp);
  ++NumberCount;
  NumberBuf[255-NumberCount] = ch;
  return 0;
}
//----------------------------------------------------------------

int CPP_sign()
{
  // stack: ( n -- | insert sign into number string if n < 0 )
  int n = *(++GlobalSp);
  if (n < 0)
    {
      ++NumberCount;
      NumberBuf[255-NumberCount] = '-';
    }
  return 0;
}
//----------------------------------------------------------------

int CPP_sharpbracket()
{
  // stack: ( 0 0 -- | complete number conversion )

  ++GlobalSp; ++GlobalSp; // 2drop 
  *GlobalSp-- = (int) (NumberBuf + 255 - NumberCount);

  *GlobalSp-- = NumberCount;

  return 0;
}
//----------------------------------------------------------------

int CPP_dot ()
{
  // stack: ( n -- | print n in current base ) 
  
  ++GlobalSp;
  if (GlobalSp > BottomOfStack) 
    return E_V_STK_UNDERFLOW;
  else
    {
      int n = *GlobalSp;
      if (n < 0)
	{
	  *pOutStream << '-';
	  *GlobalSp = abs(n);
	}
      --GlobalSp;
      return CPP_udot();
    }
  return 0;
}
//--------------------------------------------------------------

int CPP_dotr ()
{
  // stack: ( n1 n2 -- | print n1 in field n2 wide )

  ++GlobalSp;
  if (GlobalSp > BottomOfStack) return E_V_STK_UNDERFLOW;
  
  int i, n, ndig, nfield, nchar;
  unsigned int u, utemp, uscale;

  nfield = *GlobalSp++;
  if (GlobalSp > BottomOfStack) return E_V_STK_UNDERFLOW;

  if (nfield <= 0) return 0;  // don't print anything if field with <= 0

  n = *GlobalSp;
  u = abs(n);
  ndig = 1;
  uscale = 1;
  utemp = u;

  while (utemp /= Base) {++ndig; uscale *= Base;}
  int ntot = (n < 0) ? ndig + 1 : ndig;

  if (ntot <= nfield)
    {
      for (i = 0; i < (nfield - ntot); i++) *pOutStream << ' ';
    }

  if (n < 0) *pOutStream << '-';
  *GlobalSp-- = u;
  i = CPP_udot0();
  pOutStream->flush();
  return i;
}
//---------------------------------------------------------------

int CPP_udotr ()
{
  // stack: ( u n -- | print unsigned in field width n )

  ++GlobalSp;
  if (GlobalSp > BottomOfStack) return E_V_STK_UNDERFLOW;
  
  int i, ndig, nfield, nchar;
  unsigned int u, utemp, uscale;

  nfield = *GlobalSp++;
  if (GlobalSp > BottomOfStack) return E_V_STK_UNDERFLOW;

  if (nfield <= 0) return 0;  // don't print anything if field with <= 0

  u = *GlobalSp;
  ndig = 1;
  uscale = 1;
  utemp = u;

  while (utemp /= Base) {++ndig; uscale *= Base;}

  if (ndig <= nfield)
    {
      for (i = 0; i < (nfield - ndig); i++) *pOutStream << ' ';
    }
  *GlobalSp-- = u;
  i = CPP_udot0();
  pOutStream->flush();
  return i;
}
//---------------------------------------------------------------

int CPP_udot0 ()
{
  // stack: ( u -- | print unsigned single in current base )

  ++GlobalSp;
  if (GlobalSp > BottomOfStack) return E_V_STK_UNDERFLOW;
  
  int i, ndig, nchar;
  unsigned int u, utemp, uscale;

  u = *GlobalSp;
  ndig = 1;
  uscale = 1;
  utemp = u;

  while (utemp /= Base) {++ndig; uscale *= Base;}

  for (i = 0; i < ndig; i++) 
    {
      utemp = u/uscale;
      nchar = (utemp < 10) ? (utemp + 48) : (utemp + 55);
      *pOutStream << (char) nchar;
      u -= utemp*uscale;
      uscale /= Base;
    }
  return 0;
}
//--------------------------------------------------------------

int CPP_udot ()
{
  // stack: ( u -- | print unsigned single in current base followed by space )

  int e = CPP_udot0();
  if (e)
    return e;
  else
    {
      *pOutStream << ' ';
      pOutStream->flush();
    }
  return 0;
}
//---------------------------------------------------------------

int CPP_uddot ()
{
  // stack: ( ud -- | print unsigned double in current base )

  if ((GlobalSp + 2) > BottomOfStack) return E_V_STK_UNDERFLOW;
  
  unsigned int u1;

  u1 = *(GlobalSp + 1);
  if (u1 == 0)
    {
      ++GlobalSp;
      return CPP_udot();
    }
  else
    {
      CPP_bracketsharp();
      CPP_sharps();
      CPP_sharpbracket();
      CPP_type();
      *pOutStream << ' ';
      pOutStream->flush();
    }
  
  return 0;
}
//---------------------------------------------------------------

int CPP_ddot ()
{
  // stack: ( d -- | print signed double length number )

  if ((GlobalSp + 2) > BottomOfStack) 
    return E_V_STK_UNDERFLOW;
  else
    {
      int n = *(GlobalSp+1);
      if (n < 0)
	{
	  *pOutStream << '-';
	  L_dabs();
	}
      return CPP_uddot();
    }
  return 0;
}
//---------------------------------------------------------------

int CPP_fdot ()
{
  // stack: ( f -- | print floating point number )

  ++GlobalSp; ++GlobalSp;
  if (GlobalSp > BottomOfStack)
    return E_V_STK_UNDERFLOW;
  else
    {
      --GlobalSp;
      *pOutStream << *((double*) GlobalSp) << ' ';
      ++GlobalSp;
      (*pOutStream).flush();
    }
  return 0;
}
//---------------------------------------------------------------

int CPP_dots ()
{
  if (GlobalSp > BottomOfStack) return E_V_STK_UNDERFLOW;

  L_depth();  
  ++GlobalSp;
  int depth = *GlobalSp;
  ++GlobalSp;

  if (debug)
    {
      *pOutStream << "\nTop of Stack = " << ((int)ForthStack);
      *pOutStream << "\nBottom of Stack = " << ((int)BottomOfStack);
      *pOutStream << "\nStack ptr = " << ((int)GlobalSp);
      *pOutStream << "\nDepth = " << depth;
    }
 
  if (depth > 0)
    {
      int i;
      byte* bptr;

      for (i = 0; i < depth; i++)
        {
	  
                *pOutStream << "\n\t\t" << *(GlobalSp + i);
 
        }
    }
  else
    {
        *pOutStream << "<empty>";
    }
  *pOutStream << '\n';
  --GlobalSp;
  return 0;
}
//---------------------------------------------------------------

int CPP_find ()
{
  // stack: ( ^str -- ^str 0 | xt_addr 1 | xt_addr -1 )

  ++GlobalSp; 
  unsigned char* s = *((unsigned char**) GlobalSp);
  char name [128];
  int len = *s;
  strncpy (name, (char*) s+1, len);
  name[len] = 0;
  strupr(name);
  vector<DictionaryEntry>::iterator i = LocateWord (name);  
  if (i != (vector<DictionaryEntry>::iterator) NULL)
    {
      *GlobalSp-- = (int) i->Cfa;
 
      *GlobalSp-- =  (i->Precedence & PRECEDENCE_IMMEDIATE) ? 1 : -1 ;
 
    }
  else
    {
      --GlobalSp; 
      *GlobalSp-- = 0;

    }
  return 0;
}
//---------------------------------------------------------------

int CPP_emit ()
{
  // stack: ( n -- | display character with ascii code n )

  ++GlobalSp;
  if (GlobalSp > BottomOfStack)
    return E_V_STK_UNDERFLOW;
  else
    {
      *pOutStream << (char)(*GlobalSp);
      (*pOutStream).flush();
    }
  return 0;
}
//---------------------------------------------------------------

int CPP_cr ()
{
  *pOutStream << '\n';
  return 0;
}
//---------------------------------------------------------------

int CPP_spaces ()
{
  ++GlobalSp;
  if (GlobalSp > BottomOfStack) 
    return E_V_STK_UNDERFLOW;
  else
    {
      int n = *GlobalSp;
      if (n > 0)
	for (int i = 0; i < n; i++) *pOutStream << ' ';
      (*pOutStream).flush();
    }
  return 0;
}
//---------------------------------------------------------------

int CPP_type ()
{
  ++GlobalSp;
  if (GlobalSp > BottomOfStack) 
    return E_V_STK_UNDERFLOW;
  else
    {
      int n = *GlobalSp++; 
      if (GlobalSp > BottomOfStack) 
	return E_V_STK_UNDERFLOW;

      char* cp = *((char**) GlobalSp);
      for (int i = 0; i  < n; i++) *pOutStream << *cp++;
      (*pOutStream).flush();
    }
  return 0;
}
//---------------------------------------------------------------

int CPP_words ()
{
  char *cp, field[16];
  int nc;

  for (int i = 0; i < Dictionary.size(); i++)
    {
      memset (field, 32, 16);
      field[15] = '\0';
      cp = Dictionary[i].WordName;
      nc = strlen(cp);
      strncpy (field, cp, (nc > 15) ? 15 : nc);
      *pOutStream << field;
      if ((i+1) % 5 == 0) *pOutStream << '\n';
    }
  return 0;
}
//---------------------------------------------------------------

int CPP_allocate()
{
    // stack: ( u -- a ior | allocate u bytes and return address and success)
  ++GlobalSp;
  if (GlobalSp > BottomOfStack) 
    return E_V_STK_UNDERFLOW;

  unsigned int requested = *GlobalSp;
  byte *p = new byte[requested];
  *GlobalSp-- = (int) p;
  *GlobalSp-- = p ? 0 : -1;

  return 0;
}
//----------------------------------------------------------------

int CPP_free()
{
    // stack: ( a -- ior | free the allocated region at address a )
    ++GlobalSp;

    byte *p = (byte*) *GlobalSp; 
    delete [] p;
    *GlobalSp-- = 0;
    return 0;
}
//----------------------------------------------------------------

int CPP_allot ()
{
  ++GlobalSp;
  if (GlobalSp > BottomOfStack) 
    return E_V_STK_UNDERFLOW;


  vector<DictionaryEntry>::iterator id = Dictionary.end() - 1;
  int n = *GlobalSp;
  if (n > 0)
    {
      if (id->Pfa == NULL)
	{ 
	  id->Pfa = new byte[n];
	  if (id->Pfa) memset (id->Pfa, 0, n); 
	}
      else 
	return E_V_REALLOT;
    }
  else
    id->Pfa = NULL;

  return 0;
}
//--------------------------------------------------------------

int CPP_queryallot ()
{
  // stack: ( n -- a | allot n bytes and leave starting address on the stack )

  int e = CPP_allot();
  if (!e)
    {
      // Get last word's Pfa and leave on the stack

      vector<DictionaryEntry>::iterator id = Dictionary.end() - 1;
      *GlobalSp-- = (int) id->Pfa;

    }
  return e;
}
//---------------------------------------------------------------

int CPP_word ()
{
  // stack: ( n -- ^str | parse next word in input stream )
  // n is the delimiting character and ^str is a counted string.

  char delim = *(++GlobalSp); 
  char *dp = WordBuf + 1;
  
  if (*pTIB == ' ') ++pTIB;

  while (*pTIB)
    {
      if (*pTIB != delim) break;
      ++pTIB;
    }
  if (*pTIB)
    {
      int count = 0;
      while (*pTIB)
	{
	    // cout << '[' << *pTIB << ']';
	  if (*pTIB == delim) break;
	  *dp++ = *pTIB++;
	  ++count;
	}
      if (*pTIB) ++pTIB;  // consume the delimiter
      *WordBuf = count;
      *dp = ' ';
    }
  else
    {
      *WordBuf = 0;
    }
  *GlobalSp-- = (int) WordBuf;
  return 0;
}
//----------------------------------------------------------------

int CPP_create ()
{

  // stack: ( -- | create dictionary entry using next word in input stream )

  char token[128];
  pTIB = ExtractName(pTIB, token);
  int nc = strlen(token);

  if (nc)
    {
      DictionaryEntry NewWord;
      strupr(token);
      strcpy (NewWord.WordName, token);
      NewWord.WordCode = OP_ADDR;
      NewWord.Pfa = NULL;
      NewWord.Cfa = NULL;
      NewWord.Precedence = 0;

      Dictionary.push_back(NewWord);
      return 0;
    }
  else
    {
      return E_V_CREATE;  // create failed
    }
}
//-----------------------------------------------------------------

int CPP_variable ()
{
  // stack: ( -- | create dictionary entry and allot space )

  if (CPP_create()) return E_V_CREATE;  
  *GlobalSp-- = sizeof(int);
  int e = CPP_allot();
  if (e) return e;
  vector<DictionaryEntry>::iterator id = Dictionary.end() - 1;
  byte *bp = new byte[6];
  id->Cfa = bp;
  bp[0] = OP_ADDR;
  *((int*) &bp[1]) = (int) id->Pfa;
  bp[5] = OP_RET;
  return 0;
}
//-----------------------------------------------------------------

int CPP_fvariable ()
{
  // stack: ( -- | create dictionary entry and allot space )

  if (CPP_create()) return E_V_CREATE;  
  *GlobalSp-- = sizeof(double);

  int e = CPP_allot();
  if (e) return e;
  vector<DictionaryEntry>::iterator id = Dictionary.end() - 1;
  byte *bp = new byte[6];
  id->Cfa = bp;
  bp[0] = OP_ADDR;
  *((int*) &bp[1]) = (int) id->Pfa;
  bp[5] = OP_RET;
  return 0;  
}
//------------------------------------------------------------------

int CPP_constant ()
{
  // stack: ( n -- | create dictionary entry and store n as constant )

  if (CPP_create()) return E_V_CREATE;
  vector<DictionaryEntry>::iterator id = Dictionary.end() - 1;
  id->WordCode = OP_IVAL;
  id->Pfa = new int[1];
  ++GlobalSp;
  *((int*) (id->Pfa)) = *GlobalSp;
  byte *bp = new byte[7];
  id->Cfa = bp;
  bp[0] = OP_ADDR;
  *((int*) &bp[1]) = (int) id->Pfa;
  bp[5] = OP_FETCH;
  bp[6] = OP_RET;
  return 0;
}
//------------------------------------------------------------------

int CPP_fconstant ()
{
  // stack: ( f -- | create dictionary entry and store f )

  if (CPP_create()) return E_V_CREATE;
  vector<DictionaryEntry>::iterator id = Dictionary.end() - 1;
  id->WordCode = OP_FVAL;
  id->Pfa = new double[1];
  ++GlobalSp;
  *((double*) (id->Pfa)) = *((double*)GlobalSp);
  ++GlobalSp;
  byte *bp = new byte[7];
  id->Cfa = bp;
  bp[0] = OP_ADDR;
  *((int*) &bp[1]) = (int) id->Pfa;
  bp[5] = OP_DFFETCH;
  bp[6] = OP_RET;
  return 0;
}
//------------------------------------------------------------------

int CPP_char ()
{
  // stack: ( -- n | parse next word in input stream and return first char )

  *GlobalSp-- = 32;

  CPP_word();
  char* cp = *((char**) ++GlobalSp) + 1;
  *GlobalSp-- = *cp;

  return 0;
}
//-----------------------------------------------------------------

int CPP_bracketchar ()
{
  CPP_char();
  CPP_literal();
  return 0;
}
//------------------------------------------------------------------

int CPP_brackettick ()
{
  L_tick ();
  return CPP_literal();  
}
//-------------------------------------------------------------------

int CPP_compilecomma ()
{
  // stack: xt --
  CPP_literal();
  pCurrentOps->push_back(OP_EXECUTE);
  return 0;  
}
// -----------------------------------------------------------------

int CPP_bracketcompile ()
{
  return 0;
}
// -----------------------------------------------------------------

int CPP_postpone ()
{
  char token[128];
  ++pTIB;
  pTIB = ExtractName (pTIB, token);
  strupr(token);

  vector<DictionaryEntry>::iterator id = LocateWord(token);
  if (id != (vector<DictionaryEntry>::iterator) NULL)
    {
      if (id->Precedence & PRECEDENCE_IMMEDIATE)
	{
	  pCurrentOps->push_back(id->WordCode);
	  if (id->WordCode == OP_DEFINITION)
	    OpsPushInt((int) id->Cfa);
	}
      else
	{
	  if (id->WordCode == OP_IVAL)
	    {
	      pCurrentOps->push_back(OP_IVAL);
	      OpsPushInt (*((int*) id->Pfa));
	      pCurrentOps->push_back(OP_LITERAL);
	    }
	  else if (id->WordCode == OP_ADDR)
	    {
	      pCurrentOps->push_back(OP_ADDR);
	      OpsPushInt ((int) id->Pfa);
	      pCurrentOps->push_back(OP_LITERAL);
	    }
	  else
	    {
	      pCurrentOps->push_back(OP_IVAL);
	      OpsPushInt((int) id->WordCode);
	      pCurrentOps->push_back(OP_ADDR);
	      OpsPushInt((int) OpsCompileByte);
	      pCurrentOps->push_back(OP_CALL);
	      if (id->WordCode == OP_DEFINITION)
		{
		  pCurrentOps->push_back(OP_ADDR);
		  OpsPushInt((int) id->Cfa);
		  pCurrentOps->push_back(OP_ADDR);
		  OpsPushInt((int) OpsCompileInt);
		  pCurrentOps->push_back(OP_CALL);
		}
	      else if (id->WordCode == OP_FVAL)
		{
	          pCurrentOps->push_back(OP_FVAL);
	          int* pVal = (int*) id->Pfa;
	          OpsPushInt (*pVal);
	          OpsPushInt (*(pVal+1));
	          pCurrentOps->push_back(OP_ADDR);
	          OpsPushInt((int) OpsCompileDouble);
	          pCurrentOps->push_back(OP_CALL);
		}	  
	    } 
	}
      if (State && (id->Precedence & PRECEDENCE_NON_DEFERRED)) 
	NewWord.Precedence |= PRECEDENCE_NON_DEFERRED;

    }


  return 0;
  
}
// -----------------------------------------------------------------

int CPP_forget ()
{
  char token[128];

  ++pTIB;
  pTIB = ExtractName (pTIB, token);
  strupr(token);

  vector<DictionaryEntry>::iterator id = LocateWord (token);
  if (id != (vector<DictionaryEntry>::iterator) NULL)
    {
      while (Dictionary.end() > id) 
	RemoveLastWord();
    }
  else
    {
      *pOutStream << "No such word: " << token << '\n';
    }
  return 0;
}
//-------------------------------------------------------------------

int CPP_cold ()
{
  // stack: ( -- | restart the Forth environment )

  CloseForth();
  OpenForth();

  return 0;
}
//--------------------------------------------------------------------

int CPP_bye ()
{
  // stack: ( -- | close Forth and exit the process )

  CloseForth();
  *pOutStream << "Goodbye.\n";
  exit(0);

  return 0;
}
//--------------------------------------------------------------------

int CPP_tofile ()
{
  char filename[128];
  *filename = 0;

  pTIB = ExtractName (pTIB, filename);
  if (*filename == 0)
    {
      strcpy (filename, DEFAULT_OUTPUT_FILENAME);
      // cout << "Output redirected to " << filename << '\n';
    }
  ofstream *pFile = new ofstream (filename);
  if (! pFile->fail())
    {
      if (FileOutput)
	{ 
	  (*((ofstream*) pOutStream)).close();  // close current file output stream
	  delete pOutStream;
	} 
      pOutStream = pFile;
      FileOutput = TRUE;
    }
  else
    {
      *pOutStream << "Failed to open output file stream.\n";
    }
  return 0;  
}
//--------------------------------------------------------------------

int CPP_console ()
{
  if (FileOutput)
    {
      (*((ofstream*) pOutStream)).close();  // close the current file output stream
      delete pOutStream;
    }     
  pOutStream = &cout;  // make console the new output stream
  FileOutput = FALSE;

  return 0;
}
//--------------------------------------------------------------------

int CPP_literal ()
{
  // stack: ( n -- | remove item from the stack and place in compiled opcodes )

  pCurrentOps->push_back(OP_IVAL);
  byte* bp = (byte*)(++GlobalSp);
  for (int i = 0; i < sizeof(int); i++) pCurrentOps->push_back(*bp++);  
  return 0;
}
//-------------------------------------------------------------------

int CPP_cquote ()
{
  // compilation stack: ( -- | compile a counted string into the string table )
  // runtime stack: ( -- ^str | place address of counted string on stack )

  char* begin_string = pTIB + 1;
  char* end_string = strchr(begin_string, '"');
  if (end_string == NULL)
    {
      return E_V_NO_EOS;
    }
  pTIB = end_string + 1;
  int nc = (int) (end_string - begin_string);
  char* str = new char[nc + 2];
  *((byte*)str) = (byte) nc;
  strncpy(str+1, begin_string, nc);
  str[nc+1] = '\0';
  StringTable.push_back(str);
  byte* bp = (byte*) &str;

  pCurrentOps->push_back(OP_ADDR);
  for (int i = 0; i < sizeof(byte*); i++)
    pCurrentOps->push_back(*(bp + i));

  return 0;
}
//-------------------------------------------------------------------

int CPP_squote ()
{
  // compilation stack: ( -- | compile a string into the string table )
  // runtime stack: ( -- a count )

  int e = CPP_cquote();
  if (e) return e;
  char* s = *(StringTable.end() - 1);
  int v = s[0];
  pCurrentOps->push_back(OP_INC);
  pCurrentOps->push_back(OP_IVAL);
  OpsPushInt(v);

  return 0;
}
//-------------------------------------------------------------------

int CPP_dotquote ()
{
  // stack: ( -- | display a string delimited by quote from the input stream)

  int e = CPP_cquote();
  if (e) return e;

  pCurrentOps->push_back(OP_COUNT);
  pCurrentOps->push_back(OP_TYPE);

  return 0;
}
//------------------------------------------------------------------

int CPP_do ()
{
  // stack: ( -- | generate opcodes for beginning of loop structure )

  pCurrentOps->push_back(OP_PUSH);
  pCurrentOps->push_back(OP_PUSH);
  pCurrentOps->push_back(OP_PUSHIP);

  dostack.push_back(pCurrentOps->size());
  return 0;
}
//------------------------------------------------------------------

int CPP_querydo ()
{
  // stack: ( -- | generate opcodes for beginning of conditional loop )
  
  pCurrentOps->push_back(OP_2DUP);
  pCurrentOps->push_back(OP_EQ);
  CPP_if();
  pCurrentOps->push_back(OP_2DROP);
  CPP_else();
  CPP_do();

  querydostack.push_back(pCurrentOps->size());
  return 0;
}
//------------------------------------------------------------------

int CPP_leave ()
{
  // stack: ( -- | generate opcodes to jump out of the current loop )

  if (dostack.empty()) return E_V_NO_DO;
  pCurrentOps->push_back(OP_UNLOOP);
  pCurrentOps->push_back(OP_JMP);
  leavestack.push_back(pCurrentOps->size());
  OpsPushInt(0);
  return 0;
}
//------------------------------------------------------------------

int CPP_abortquote ()
{
  // stack: ( -- | generate opcodes to print message and abort )

  int nc = strlen(NewWord.WordName);;
  char* str = new char[nc + 3];
  strcpy(str, NewWord.WordName);
  strcat(str, ": ");
  StringTable.push_back(str);

  pCurrentOps->push_back(OP_JZ);
  OpsPushInt(25);   // relative jump count                       

// the relative jump count (above) must be modified if the 
// instructions below are updated!

  pCurrentOps->push_back(OP_ADDR);
  OpsPushInt((int) str);
  pCurrentOps->push_back(OP_IVAL);
  OpsPushInt(nc+2);
  pCurrentOps->push_back(OP_TYPE);
  int e = CPP_dotquote();
  pCurrentOps->push_back(OP_CR);
  pCurrentOps->push_back(OP_ABORT);
  return e;

}
//------------------------------------------------------------------

int CPP_begin()
{
  // stack: ( -- | mark the start of a begin ... structure )

  beginstack.push_back(pCurrentOps->size());
  return 0;
}
//------------------------------------------------------------------

int CPP_while()
{
  // stack: ( -- | build the begin ... while ... repeat structure )	      

  if (beginstack.empty()) return E_V_NO_BEGIN;
  pCurrentOps->push_back(OP_JZ);
  whilestack.push_back(pCurrentOps->size());
  OpsPushInt(0);
  return 0;
}
//------------------------------------------------------------------

int CPP_repeat()
{
  // stack: ( -- | complete begin ... while ... repeat block )

  if (beginstack.empty()) return E_V_NO_BEGIN;  // no matching BEGIN

  int i = beginstack[beginstack.size()-1];
  beginstack.pop_back();

  int ival;

  if (whilestack.size())
    {
      int j = whilestack[whilestack.size()-1];
      if (j > i)
	{
	  whilestack.pop_back();
	  ival = pCurrentOps->size() - j + 6;
	  OpsCopyInt (j, ival);  // write the relative jump count
	}
    }

  ival = i - pCurrentOps->size();
  pCurrentOps->push_back(OP_JMP);
  OpsPushInt(ival);   // write the relative jump count

  return 0;
}
//-------------------------------------------------------------------

int CPP_until()
{
  // stack: ( -- | complete begin ... until block )

  if (beginstack.empty()) return E_V_NO_BEGIN;  // no matching BEGIN

  int i = beginstack[beginstack.size()-1];
  beginstack.pop_back();
  int ival = i - pCurrentOps->size();
  pCurrentOps->push_back(OP_JZ);
  OpsPushInt(ival);   // write the relative jump count

  return 0;
}
//-------------------------------------------------------------------

int CPP_again()
{
  // stack: ( -- | complete begin ... again block )

  if (beginstack.empty()) return E_V_NO_BEGIN;  // no matching BEGIN

  int i = beginstack[beginstack.size()-1];
  beginstack.pop_back();
  int ival = i - pCurrentOps->size();
  pCurrentOps->push_back(OP_JMP);
  OpsPushInt(ival);   // write the relative jump count

  return 0;
}
//--------------------------------------------------------------------

int CPP_if()
{
  // stack: ( -- | generate start of an if-then or if-else-then block )

  pCurrentOps->push_back(OP_JZ);
  ifstack.push_back(pCurrentOps->size());
  OpsPushInt(0);   // placeholder for jump count
  return 0;
}
//------------------------------------------------------------------

int CPP_else()
{
  // stack: ( -- | build the if-else-then block )

  pCurrentOps->push_back(OP_JMP);
  OpsPushInt(0);  // placeholder for jump count

  if (ifstack.empty()) return E_V_ELSE_NO_IF;  // ELSE without matching IF
  int i = ifstack[ifstack.size()-1];
  ifstack.pop_back();
  ifstack.push_back(pCurrentOps->size() - sizeof(int));
  int ival = pCurrentOps->size() - i + 1;
  OpsCopyInt (i, ival);  // write the relative jump count

  return 0;
}
//-------------------------------------------------------------------

int CPP_then()
{
  // stack: ( -- | complete the if-then or if-else-then block )

  if (ifstack.empty()) 
    return E_V_THEN_NO_IF;  // THEN without matching IF or IF-ELSE

  int i = ifstack[ifstack.size()-1];
  ifstack.pop_back();
  int ival = (int) (pCurrentOps->size() - i) + 1;
  OpsCopyInt (i, ival);   // write the relative jump count

  return 0;
}
//-------------------------------------------------------------------

int CPP_case()
{
  // stack: ( n -- | mark the beginning of a case...endcase structure)

  casestack.push_back(-1);
  return 0;
}
//-----------------------------------------------------------------

int CPP_endcase()
{
  // stack: ( -- | terminate the case...endcase structure)

  if (casestack.size() == 0) return E_V_NO_CASE;  // ENDCASE without matching CASE
  pCurrentOps->push_back(OP_DROP);

  // fix up all absolute jumps

  int i, ival;
  do
    {
      i = casestack[casestack.size()-1];
      casestack.pop_back();
      if (i == -1) break;
      ival = (int) (pCurrentOps->size() - i) + 1;
      OpsCopyInt (i, ival);   // write the relative jump count
    } while (casestack.size()) ;

  return 0;
}
//----------------------------------------------------------------

int CPP_of()
{
  // stack: ( -- | generate start of an of...endof block)

  pCurrentOps->push_back(OP_OVER);
  pCurrentOps->push_back(OP_EQ);
  pCurrentOps->push_back(OP_JZ);
  ofstack.push_back(pCurrentOps->size());
  OpsPushInt(0);   // placeholder for jump count
  pCurrentOps->push_back(OP_DROP);
  return 0;
}
//-----------------------------------------------------------------

int CPP_endof()
{
  // stack: ( -- | complete an of...endof block)

  pCurrentOps->push_back(OP_JMP);
  casestack.push_back(pCurrentOps->size());
  OpsPushInt(0);   // placeholder for jump count

  if (ofstack.empty())
    return E_V_ENDOF_NO_OF;  // ENDOF without matching OF

  int i = ofstack[ofstack.size()-1];
  ofstack.pop_back();
  int ival = (int) (pCurrentOps->size() - i) + 1;
  OpsCopyInt (i, ival);   // write the relative jump count

  return 0;
}
//-----------------------------------------------------------------

int CPP_recurse()
{
  pCurrentOps->push_back(OP_ADDR);
  if (State)
    {
      recursestack.push_back(pCurrentOps->size());
      OpsPushInt(0);
    }
  else
    {
      int ival = (int) &(*pCurrentOps)[0]; // ->begin();
      OpsPushInt(ival);
    }
  pCurrentOps->push_back(OP_EXECUTE);
  return 0;
}
//---------------------------------------------------------------------

int CPP_lbracket()
{
  State = FALSE;
  pPreviousOps = pCurrentOps;
  tempOps.erase(tempOps.begin(), tempOps.end());
  pCurrentOps = &tempOps;
  return 0;
}
//--------------------------------------------------------------------

int CPP_rbracket()
{
  pCurrentOps->push_back(OP_RET);
  if (debug) OutputForthByteCode(pCurrentOps);
  byte* pIp = GlobalIp;
  int e = vm((byte*) &(*pCurrentOps)[0]);
  pCurrentOps->erase(pCurrentOps->begin(), pCurrentOps->end());
  GlobalIp = pIp;
  State = TRUE;
  pCurrentOps = pPreviousOps;
  return e;
}
//-------------------------------------------------------------------

int CPP_does()
{
  // Allocate new opcode array

  byte* p = new byte[12];

  // Insert pfa of last word in dictionary

  p[0] = OP_ADDR;
  vector<DictionaryEntry>::iterator id = Dictionary.end() - 1;
  *((int*)(p+1)) = (int) id->Pfa;

  // Insert current instruction ptr 

  p[5] = OP_ADDR;
  *((int*)(p+6)) = (int)(GlobalIp + 1);

  p[10] = OP_EXECUTE;
  p[11] = OP_RET;

  id->Cfa = (void*) p;
  id->WordCode = OP_DEFINITION;

  L_ret();
  return 0;
}
//-------------------------------------------------------------------

int CPP_immediate ()
{
  // Mark the most recently defined word as immediate.
  // stack: ( -- )

  vector<DictionaryEntry>::iterator id = Dictionary.end() - 1;
  id->Precedence |= PRECEDENCE_IMMEDIATE;
  return 0;
}
//-------------------------------------------------------------------

int CPP_nondeferred ()
{
  // Mark the most recently defined word as non-deferred.
  // stack: ( -- )

  vector<DictionaryEntry>::iterator id = Dictionary.end() - 1;
  id->Precedence |= PRECEDENCE_NON_DEFERRED;
  return 0;
}
//-------------------------------------------------------------------

int CPP_evaluate ()
{
  // Compile a string
  // ( ... a u -- ? )

  char s[256], s2[256];
  int nc = *(++GlobalSp);
  char *cp = (char*) (*(++GlobalSp));

  if (nc < 256)
    {
      memcpy (s, cp, nc);
      s[nc] = 0;
      if (*s) 
	{
	  istringstream* pSS = NULL;
	  istream* pOldStream = pInStream;  // save old input stream
	  strcpy (s2, pTIB);  // save remaining part of input line in TIB
	  pSS = new istringstream(s);
	  SetForthInputStream(*pSS);
	  vector<byte> op, *pOps, *pOldOps;
	  int e;
	  pOldOps = pCurrentOps;
	  pOps = State ? pCurrentOps : &op;

	  --linecount;
	  e = ForthCompiler(pOps, &linecount);

	  // Restore the opcode vector, the input stream, and the input buffer

	  pCurrentOps = pOldOps;
	  SetForthInputStream(*pOldStream);  // restore old input stream
	  strcpy(TIB, s2);  // restore TIB with remaining input line
	  pTIB = TIB;      // restore ptr
	  delete pSS;

	}
    }
  return 0;
  
}
//-------------------------------------------------------------------

int CPP_include()
{
    char WordToken[256], filename[256], s[256];
    int ecode;
    vector<byte> ops, *pOldOps;

    ++pTIB;
    pTIB = ExtractName (pTIB, WordToken);
    strcpy (s, pTIB);  // save remaining part of input line in TIB
    if (!strchr(WordToken, '.')) strcat(WordToken, ".4th");
    strcpy (filename, WordToken);
    ifstream f(filename);
    if (!f)
    {
	if(!strchr(filename,'/'))
	{
	    char temp[256];
	    if (getenv("KFORTH_DIR"))
	    { 
		strcpy(temp, getenv("KFORTH_DIR"));
	        strcat(temp, "/");
	        strcat(temp, filename);
		strcpy(filename, temp);
		f.open(filename);
		if (f) 
		{
		    *pOutStream << endl << filename << endl;
		}
	    }
	}
    }

    if (f.fail()) 
    {
	*pOutStream << endl << filename << endl;
	return (E_V_OPENFILE);
    }
	
    istream* pTempIn = pInStream;  // save input stream ptr
    SetForthInputStream(f);  // set the new input stream
    int oldlc = linecount; linecount = 0;
    pOldOps = pCurrentOps;
    ecode = ForthCompiler (&ops, &linecount);
    f.close();
    pInStream = pTempIn;  // restore the input stream
    pCurrentOps = pOldOps; 
    if (ecode) 
    {
	*pOutStream << filename << "  " ;
	return (ecode);
    }
    linecount = oldlc;

    // Execute the code immediately
		      
    int *sp;
    byte *tp;
    ecode = ForthVM (&ops, &sp, &tp);
    ops.erase(ops.begin(), ops.end());
    if (ecode) return(ecode);

    strcpy(TIB, s);  // restore TIB with remaining input line
    pTIB = TIB;      // restore ptr
    
    return 0;
}
//-------------------------------------------------------------------

int CPP_source()
{
    *GlobalSp-- = (int) TIB;

    *GlobalSp-- = strlen(TIB);

    return 0;
}
//-------------------------------------------------------------------

int CPP_refill()
{
    pInStream->getline(TIB, 255);
    *GlobalSp-- = (pInStream->fail()) ? FALSE : TRUE;

    pTIB = TIB;
    return 0;
}
//-------------------------------------------------------------------

int CPP_state()
{
    // pCurrentOps->push_back(OP_ADDR);
    // OpsPushInt((int) &State);
    *GlobalSp-- = (int)(&State);

    return 0;
}
//-------------------------------------------------------------------

int CPP_spstore()
{
    // stack: ( addr -- | make the stack ptr point to a new address)

    ++GlobalSp;

    int* p = (int*)(*GlobalSp); --p;
    if ((p > BottomOfStack) || (p < ForthStack))
	return E_V_BADSTACKADDR;  // new SP must be within its stack space

    int n = (int) (p - ForthStack);
    GlobalSp = ForthStack + n;


    return 0;
}
//--------------------------------------------------------------------

int CPP_rpstore()
{
    // stack: ( addr -- | make the stack ptr point to a new address)

    ++GlobalSp;

    int* p = (int*)(*GlobalSp); --p;
    if ((p > BottomOfReturnStack) || (p < ForthReturnStack))
	return E_V_BADSTACKADDR;  // new RP must be within its stack space

    int n = (int) (p - ForthReturnStack);
    GlobalRp = ForthReturnStack + n;

    return 0;
}

}
