/***************************************************************************
 *   Copyright (C) 2004 by Roberto Virga                                   *
 *   rvirga@users.sf.net                                                   *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include <kbssetiprojectmonitor.h>

#include "kbssetistarmaplog.h"

const QString KBSSETIStarMapLog::s_filename = "starmap.csv";

KBSSETIStarMapLog::KBSSETIStarMapLog(const KURL &url, QObject *parent, const char *name)
                 : KBSLogMonitor(url, parent, name)
{
  initKeys();
  
  addLogFile(s_filename);

  connect(this, SIGNAL(fileUpdated(const QString &)), this, SLOT(updateFile(const QString &)));
}

QStringList KBSSETIStarMapLog::keys() const
{
  return m_keys;
}

bool KBSSETIStarMapLog::parseFile(KBSFileInfo *file, const QString &fileName)
{
  qDebug("Parsing file %s...", file->fileName.latin1());
  
  QStringList lines;
  if(!readFile(fileName, lines)) return false;
  
  if(s_filename == file->fileName)
    return parseStarMapLogDocument(lines);
  else
    return false;
}

KBSLogDatum KBSSETIStarMapLog::formatWorkunitDatum(KBSSETIProjectMonitor *projectMonitor,
                                                   const QString &workunit) const
{
  const SETIResult *datum = projectMonitor->result(workunit);
  if(NULL == datum) return KBSLogDatum();
    
  KBSLogDatum out;
   
  out["wu_name"] = datum->workunit_header.name;
  out["start_ra"] = datum->workunit_header.group_info.data_desc.start.ra;
  out["start_dec"] = datum->workunit_header.group_info.data_desc.start.dec;
  out["tape_version"] = datum->workunit_header.group_info.recorder_cfg.version;
  out["subband_base"] = datum->workunit_header.subband_desc.base;
  out["receiver"] = datum->workunit_header.group_info.receiver_cfg.name;
  out["time_recorded"] = formatSETIClassicDate(datum->workunit_header.group_info.data_desc.time_recorded_jd);
  out["comment"] = "generated by KBoincSpy";
  if(datum->state.best_gaussian.gaussian.time_jd > 0) {
    out["bg_score"] = datum->state.best_gaussian.bg.score;
    out["bg_chisq"] = datum->state.best_gaussian.gaussian.chisqr;
    out["bg_power"] = datum->state.best_gaussian.gaussian.peak_power;
  }
  if(datum->state.best_spike.spike.time_jd > 0) {
    out["bs_score"] = datum->state.best_spike.bs.score;
    out["bs_power"] = datum->state.best_spike.spike.peak_power;
  }
  
  return out;
}

QMap<QString,KBSLogData> KBSSETIStarMapLog::formatWorkunit(KBSProjectMonitor *monitor,
                                                           const QString &workunit) const
{
  KBSSETIProjectMonitor *projectMonitor = static_cast<KBSSETIProjectMonitor*>(monitor);
  QMap<QString,KBSLogData> out;
  
  out[s_filename] << formatWorkunitDatum(projectMonitor, workunit);
  
  return out;
}

void KBSSETIStarMapLog::appendHeader(const KBSFileInfo *, QIODevice *)
{
}

void KBSSETIStarMapLog::appendWorkunit(const KBSFileInfo *info, QIODevice *io, const KBSLogDatum &datum)
{
  QTextStream text(io);
  
  if(info->fileName == s_filename)
    text << formatCSVDatum(datum, m_keys, ';').remove('"') << "\r\n";
}

void KBSSETIStarMapLog::initKeys()
{
  m_keys.clear(); m_keys 
    << "wu_name" << "start_ra" << "start_dec" << "tape_version"
    << "subband_base" << "receiver" << "time_recorded" << "comment"
    << "bg_chisq" << "bg_score" << "bg_power" << "bs_score"
    << "bs_power";
}

bool KBSSETIStarMapLog::parseStarMapLogDocument(const QStringList &lines)
{
  if(lines.isEmpty()) return true;
  
  QStringList::const_iterator line = lines.begin();
  
  for(unsigned i = 0; i < m_workunits.count(); ++i)
    if(lines.end() != line) ++line;
    else return true;
    
  while(lines.end() != line) {
    KBSLogDatum datum = parseCSVDatum(*line, m_keys, ';');
    ++line;
    
    datum["time_recorded"] = parseSETIClassicDate(datum["time_recorded"].toString());
    
    m_workunits << datum;
  }
  
  qDebug("... parse OK");
    
  return true;
}

void KBSSETIStarMapLog::updateFile(const QString &)
{
  emit workunitsUpdated();
}

#include "kbssetistarmaplog.moc"
