// K-3D
// Copyright (c) 2006, Romain Behar
//
// Contact: romainbehar@yahoo.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\author Romain Behar (romainbehar@yahoo.com)
*/

#include <k3dsdk/gl.h>
#include <k3dsdk/i18n.h>
#include <k3dsdk/measurement.h>
#include <k3dsdk/mesh_modifier.h>
#include <k3dsdk/module.h>
#include <k3dsdk/node.h>
#include <k3dsdk/persistent.h>
#include <k3dsdk/transformable.h>

#include "helpers.h"

namespace libk3dmesh
{

/////////////////////////////////////////////////////////////////////////////
// show_face_orientations

class show_face_orientations :
	public k3d::gl::drawable<k3d::transformable<k3d::mesh_modifier<k3d::persistent<k3d::node> > > >
{
	typedef k3d::gl::drawable<k3d::transformable<k3d::mesh_modifier<k3d::persistent<k3d::node> > > > base;

public:
	show_face_orientations(k3d::idocument& Document) :
		base(Document),
		m_color(init_owner(*this) + init_name("color") + init_label(_("Color")) + init_description(_("Orientation arrows color")) + init_value(k3d::color(0, 0, 1)))
	{
		m_color.changed_signal().connect(make_async_redraw_slot());
	}

	/** \todo Improve the implementation so we don't have to do this */
	k3d::iunknown* on_rewrite_hint(iunknown* const Hint)
	{
		// Force updates to re-allocate our mesh, for simplicity
		return 0;
	}

	void on_create_mesh(const k3d::mesh& InputMesh, k3d::mesh& Mesh)
	{
		k3d::deep_copy(InputMesh, Mesh);
	}

	void on_update_mesh(const k3d::mesh& InputMesh, k3d::mesh& Mesh)
	{
	}

	void on_gl_draw(const k3d::gl::render_state& State)
	{
		// Get input mesh
		k3d::mesh* const input = m_input_mesh.value();
		if(!input)
			return;

		k3d::gl::store_attributes attributes;

		glDisable(GL_LIGHTING);
		k3d::gl::color3d(m_color.value());

		for(k3d::mesh::polyhedra_t::iterator polyhedron = input->polyhedra.begin(); polyhedron != input->polyhedra.end(); ++polyhedron)
		{
			for(k3d::polyhedron::faces_t::const_iterator face = (*polyhedron)->faces.begin(); face != (*polyhedron)->faces.end(); ++face)
			{
				const k3d::split_edge* const edge1 = (*face)->first_edge;
				if(!edge1)
					continue;

				const k3d::split_edge* const edge2 = edge1->face_clockwise;
				if(!edge2)
					continue;

				k3d::point3 center = helpers::center_point(**face);
				k3d::point3 point1 = k3d::mix(center, edge1->vertex->position, 0.8);
				k3d::point3 point2 = k3d::mix(center, edge2->vertex->position, 0.8);

				glBegin(GL_LINES);
					k3d::gl::vertex3d(point1);
					k3d::gl::vertex3d(point2);
				glEnd();

				glBegin(GL_POINTS);
					k3d::gl::vertex3d(point1);
				glEnd();
			}
		}
	}

	void on_gl_select(const k3d::gl::render_state& State, const k3d::gl::select_state& SelectState)
	{
	}

	k3d::iplugin_factory& factory()
	{
		return get_factory();
	}

	static k3d::iplugin_factory& get_factory()
	{
		static k3d::plugin_factory<
			k3d::document_plugin<show_face_orientations>,
			k3d::interface_list<k3d::imesh_source,
			k3d::interface_list<k3d::imesh_sink> > > factory(
				k3d::uuid(0xf42986d3, 0x275e421f, 0x9378ab70, 0x39c170e8),
				"ShowFaceOrientations",
				"Show face orientations (start from first edge)",
				"Utility",
				k3d::iplugin_factory::STABLE);

		return factory;
	}

private:
	k3d_data(k3d::color, immutable_name, change_signal, with_undo, local_storage, no_constraint, writable_property, with_serialization) m_color;
};

/////////////////////////////////////////////////////////////////////////////
// face_normals_factory

k3d::iplugin_factory& show_face_orientations_factory()
{
	return show_face_orientations::get_factory();
}

} // namespace libk3dmesh

