// K-3D
// Copyright (c) 1995-2004, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\author Tim Shead (tshead@k-3d.com)
*/

#include <k3dsdk/geometry.h>
#include <k3dsdk/imaterial.h>
#include <k3dsdk/object.h>
#include <k3dsdk/persistence.h>
#include <k3dsdk/material.h>
#include <k3dsdk/material_collection.h>
#include <k3dsdk/measurement.h>
#include <k3dsdk/mesh_source.h>
#include <k3dsdk/module.h>
#include <k3dsdk/plugins.h>
#include <k3dsdk/transform.h>

namespace libk3dnurbs
{

/////////////////////////////////////////////////////////////////////////////
// nurbs_hyperboloid_implementation

class nurbs_hyperboloid_implementation :
	public k3d::material_collection<k3d::mesh_source<k3d::persistent<k3d::object> > >
{
	typedef k3d::material_collection<k3d::mesh_source<k3d::persistent<k3d::object> > > base;

public:
	nurbs_hyperboloid_implementation(k3d::idocument& Document) :
		base(Document),
		m_x1(k3d::init_name("x1") + k3d::init_description("x1 [distance]") + k3d::init_value(5.0) + k3d::init_document(Document) + k3d::init_precision( 2) + k3d::init_step_increment(0.1) + k3d::init_units(typeid(k3d::measurement::distance))),
		m_y1(k3d::init_name("y1") + k3d::init_description("y1 [distance]") + k3d::init_value(-5.0) + k3d::init_document(Document) + k3d::init_precision( 2) + k3d::init_step_increment(0.1) + k3d::init_units(typeid(k3d::measurement::distance))),
		m_z1(k3d::init_name("z1") + k3d::init_description("z1 [distance]") + k3d::init_value(-5.0) + k3d::init_document(Document) + k3d::init_precision( 2) + k3d::init_step_increment(0.1) + k3d::init_units(typeid(k3d::measurement::distance))),
		m_x2(k3d::init_name("x2") + k3d::init_description("x2 [distance]") + k3d::init_value(5.0) + k3d::init_document(Document) + k3d::init_precision( 2) + k3d::init_step_increment(0.1) + k3d::init_units(typeid(k3d::measurement::distance))),
		m_y2(k3d::init_name("y2") + k3d::init_description("y2 [distance]") + k3d::init_value(5.0) + k3d::init_document(Document) + k3d::init_precision( 2) + k3d::init_step_increment(0.1) + k3d::init_units(typeid(k3d::measurement::distance))),
		m_z2(k3d::init_name("z2") + k3d::init_description("z2 [distance]") + k3d::init_value(5.0) + k3d::init_document(Document) + k3d::init_precision( 2) + k3d::init_step_increment(0.1) + k3d::init_units(typeid(k3d::measurement::distance))),
		m_thetamax(k3d::init_name("thetamax") + k3d::init_description("thetamax [angle]") + k3d::init_value(k3d::radians(360.0)) + k3d::init_document(Document) + k3d::init_precision( 2) + k3d::init_step_increment(k3d::radians(1.0)) + k3d::init_units(typeid(k3d::measurement::angle)))
	{
		enable_serialization(k3d::persistence::proxy(m_x1));
		enable_serialization(k3d::persistence::proxy(m_y1));
		enable_serialization(k3d::persistence::proxy(m_z1));
		enable_serialization(k3d::persistence::proxy(m_x2));
		enable_serialization(k3d::persistence::proxy(m_y2));
		enable_serialization(k3d::persistence::proxy(m_z2));
		enable_serialization(k3d::persistence::proxy(m_thetamax));

		register_property(m_x1);
		register_property(m_y1);
		register_property(m_z1);
		register_property(m_x2);
		register_property(m_y2);
		register_property(m_z2);
		register_property(m_thetamax);

		m_material.changed_signal().connect(SigC::slot(*this, &nurbs_hyperboloid_implementation::on_reset_geometry));
		m_x1.changed_signal().connect(SigC::slot(*this, &nurbs_hyperboloid_implementation::on_reset_geometry));
		m_y1.changed_signal().connect(SigC::slot(*this, &nurbs_hyperboloid_implementation::on_reset_geometry));
		m_z1.changed_signal().connect(SigC::slot(*this, &nurbs_hyperboloid_implementation::on_reset_geometry));
		m_x2.changed_signal().connect(SigC::slot(*this, &nurbs_hyperboloid_implementation::on_reset_geometry));
		m_y2.changed_signal().connect(SigC::slot(*this, &nurbs_hyperboloid_implementation::on_reset_geometry));
		m_z2.changed_signal().connect(SigC::slot(*this, &nurbs_hyperboloid_implementation::on_reset_geometry));
		m_thetamax.changed_signal().connect(SigC::slot(*this, &nurbs_hyperboloid_implementation::on_reset_geometry));

		m_output_mesh.need_data_signal().connect(SigC::slot(*this, &nurbs_hyperboloid_implementation::on_create_geometry));
	}

	void on_reset_geometry()
	{
		m_output_mesh.reset();
	}

	k3d::mesh* on_create_geometry()
	{
		const double x1 = m_x1.property_value();
		const double y1 = m_y1.property_value();
		const double z1 = m_z1.property_value();
		const double x2 = m_x2.property_value();
		const double y2 = m_y2.property_value();
		const double z2 = m_z2.property_value();
		const double thetamax = m_thetamax.property_value();

		const double thetamin1 = atan2(y1, x1);
		const double thetamin2 = atan2(y2, x2);

		const double radius1 = sqrt(x1 * x1 + y1 * y1);
		const double radius2 = sqrt(x2 * x2 + y2 * y2);

		const k3d::vector3 offset1 = z1 * k3d::vector3(0, 0, 1);
		const k3d::vector3 offset2 = z2 * k3d::vector3(0, 0, 1);

		k3d::mesh* const mesh = new k3d::mesh();
		k3d::nupatch* const nupatch = new k3d::nupatch();
		mesh->nupatches.push_back(nupatch);

		nupatch->material = m_material.interface();
		nupatch->u_order = 3;
		nupatch->v_order = 2;

		std::vector<double> weights1;
		std::vector<k3d::vector3> arc_points1;
		k3d::nurbs_arc(k3d::vector3(1, 0, 0), k3d::vector3(0, 1, 0), thetamin1, thetamin1 + thetamax, 4, nupatch->u_knots, weights1, arc_points1);

		std::vector<double> weights2;
		std::vector<k3d::vector3> arc_points2;
		k3d::nurbs_arc(k3d::vector3(1, 0, 0), k3d::vector3(0, 1, 0), thetamin2, thetamin2 + thetamax, 4, nupatch->u_knots, weights2, arc_points2);

		nupatch->v_knots.insert(nupatch->v_knots.end(), 2, 0);
		nupatch->v_knots.insert(nupatch->v_knots.end(), 2, 1);

		for(unsigned long j = 0; j != arc_points1.size(); ++j)
			{
				mesh->points.push_back(new k3d::point(radius1 * arc_points1[j] + offset1));
				nupatch->control_points.push_back(k3d::nupatch::control_point(mesh->points.back(), weights1[j]));
			}

		for(unsigned long j = 0; j != arc_points2.size(); ++j)
			{
				mesh->points.push_back(new k3d::point(radius2 * arc_points2[j] + offset2));
				nupatch->control_points.push_back(k3d::nupatch::control_point(mesh->points.back(), weights2[j]));
			}

		assert_warning(is_valid(*nupatch));
		return mesh;
	}

	k3d::iplugin_factory& factory()
	{
		return get_factory();
	}

	static k3d::iplugin_factory& get_factory()
	{
		static k3d::plugin_factory<k3d::document_plugin<nurbs_hyperboloid_implementation>, k3d::interface_list<k3d::imesh_source > > factory(
		k3d::uuid(0xb653277c, 0x7bba4505, 0xbabcd65d, 0x591ecedf),
			"NurbsHyperboloid",
			"Generates a NURBS hyperboloid",
			"Objects",
			k3d::iplugin_factory::STABLE);

		return factory;
	}

private:
	k3d_measurement_property(double, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_x1;
	k3d_measurement_property(double, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_y1;
	k3d_measurement_property(double, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_z1;
	k3d_measurement_property(double, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_x2;
	k3d_measurement_property(double, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_y2;
	k3d_measurement_property(double, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_z2;
	k3d_measurement_property(double, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_thetamax;
};

/////////////////////////////////////////////////////////////////////////////
// nurbs_hyperboloid_factory

k3d::iplugin_factory& nurbs_hyperboloid_factory()
{
	return nurbs_hyperboloid_implementation::get_factory();
}

} // namespace libk3dnurbs


