// K-3D
// Copyright (c) 1995-2004, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\author Timothy M. Shead (tshead@k-3d.com)
*/

#include <k3dsdk/bitmap_source.h>
#include <k3dsdk/persistence.h>
#include <k3dsdk/measurement.h>
#include <k3dsdk/module.h>
#include <k3dsdk/object.h>
#include <k3dsdk/plugins.h>
#include <k3dsdk/transform.h>

#include <iterator>

namespace libk3dbitmap
{

/////////////////////////////////////////////////////////////////////////////
// bitmap_checker_implementation

class bitmap_checker_implementation :
	public k3d::bitmap_source<k3d::persistent<k3d::object> >
{
	typedef k3d::bitmap_source<k3d::persistent<k3d::object> > base;

public:
	bitmap_checker_implementation(k3d::idocument& Document) :
		base(Document),
		m_width(k3d::init_name("width") + k3d::init_description("Width [number]") + k3d::init_document(Document) + k3d::init_value(64UL) + k3d::init_precision(0) + k3d::init_step_increment(1) + k3d::init_units(typeid(k3d::measurement::scalar)) + k3d::init_constraint(k3d::constraint::minimum(1UL))),
		m_height(k3d::init_name("height") + k3d::init_description("Height [number]") + k3d::init_document(Document) + k3d::init_value(64UL) + k3d::init_precision(0) + k3d::init_step_increment(1) + k3d::init_units(typeid(k3d::measurement::scalar)) + k3d::init_constraint(k3d::constraint::minimum(1UL))),
		m_check_width(k3d::init_name("check_width") + k3d::init_description("Check Width [number]") + k3d::init_document(Document) + k3d::init_value(8UL) + k3d::init_precision(0) + k3d::init_step_increment(1) + k3d::init_units(typeid(k3d::measurement::scalar)) + k3d::init_constraint(k3d::constraint::minimum(1UL))),
		m_check_height(k3d::init_name("check_height") + k3d::init_description("Check Height [number]") + k3d::init_document(Document) + k3d::init_value(8UL) + k3d::init_precision(0) + k3d::init_step_increment(1) + k3d::init_units(typeid(k3d::measurement::scalar)) + k3d::init_constraint(k3d::constraint::minimum(1UL))),
		m_color1(k3d::init_name("color1") + k3d::init_description("Color [color]") + k3d::init_document(Document) + k3d::init_value(k3d::color(1, 1, 1))),
		m_color2(k3d::init_name("color2") + k3d::init_description("Color [color]") + k3d::init_document(Document) + k3d::init_value(k3d::color(0, 0, 0)))
	{
		enable_serialization(k3d::persistence::proxy(m_width));
		enable_serialization(k3d::persistence::proxy(m_height));
		enable_serialization(k3d::persistence::proxy(m_check_width));
		enable_serialization(k3d::persistence::proxy(m_check_height));
		enable_serialization(k3d::persistence::proxy(m_color1));
		enable_serialization(k3d::persistence::proxy(m_color2));

		register_property(m_width);
		register_property(m_height);
		register_property(m_check_width);
		register_property(m_check_height);
		register_property(m_color1);
		register_property(m_color2);

		m_width.changed_signal().connect(SigC::slot(*this, &bitmap_checker_implementation::on_reset_bitmap));
		m_height.changed_signal().connect(SigC::slot(*this, &bitmap_checker_implementation::on_reset_bitmap));
		m_check_width.changed_signal().connect(SigC::slot(*this, &bitmap_checker_implementation::on_reset_bitmap));
		m_check_height.changed_signal().connect(SigC::slot(*this, &bitmap_checker_implementation::on_reset_bitmap));
		m_color1.changed_signal().connect(SigC::slot(*this, &bitmap_checker_implementation::on_reset_bitmap));
		m_color2.changed_signal().connect(SigC::slot(*this, &bitmap_checker_implementation::on_reset_bitmap));
	
		m_output.need_data_signal().connect(SigC::slot(*this, &bitmap_checker_implementation::on_create_bitmap));
	}

	void on_reset_bitmap()
	{
		m_output.reset();
	}
	
	k3d::bitmap* on_create_bitmap()
	{
		const k3d::pixel_size_t width = m_width.property_value();
		const k3d::pixel_size_t height = m_height.property_value();
		const k3d::pixel_size_t check_width = m_check_width.property_value();
		const k3d::pixel_size_t check_height = m_check_height.property_value();
		const k3d::color color1 = m_color1.property_value();
		const k3d::color color2 = m_color2.property_value();

		std::auto_ptr<k3d::bitmap> bitmap(new k3d::bitmap(width, height));
	
		k3d::checkerboard_fill(*bitmap, check_width, check_height, color1, color2);
	
		return bitmap.release();
	}
	
	k3d::iplugin_factory& factory()
	{
		return get_factory();
	}

	static k3d::iplugin_factory& get_factory()
	{
		static k3d::plugin_factory<
			k3d::document_plugin<bitmap_checker_implementation>,
			k3d::interface_list<k3d::ibitmap_source> > factory(
				k3d::uuid(0x8d65a65f, 0x853e4460, 0xbc2ecbad, 0xce65b404),
				"BitmapChecker",
				"Generates a checker-pattern bitmap",
				"Objects",
				k3d::iplugin_factory::STABLE);

		return factory;
	}

private:
	k3d_measurement_property(k3d::pixel_size_t, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::with_constraint) m_width;
	k3d_measurement_property(k3d::pixel_size_t, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::with_constraint) m_height;
	k3d_measurement_property(k3d::pixel_size_t, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::with_constraint) m_check_width;
	k3d_measurement_property(k3d::pixel_size_t, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::with_constraint) m_check_height;
	k3d_data_property(k3d::color, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_color1;
	k3d_data_property(k3d::color, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_color2;
};

/////////////////////////////////////////////////////////////////////////////
// bitmap_checker_factory

k3d::iplugin_factory& bitmap_checker_factory()
{
	return bitmap_checker_implementation::get_factory();
}

} // namespace libk3dbitmap

