

/*
 * The JTS Topology Suite is a collection of Java classes that
 * implement the fundamental operations required to validate a given
 * geo-spatial data set to a known topological specification.
 *
 * Copyright (C) 2001 Vivid Solutions
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * For more information, contact:
 *
 *     Vivid Solutions
 *     Suite #1A
 *     2328 Government Street
 *     Victoria BC  V8T 5G5
 *     Canada
 *
 *     (250)385-6040
 *     www.vividsolutions.com
 */
package com.vividsolutions.jts.operation.overlay;

import java.util.*;
import com.vividsolutions.jts.geom.*;
import com.vividsolutions.jts.algorithm.*;
import com.vividsolutions.jts.geomgraph.*;

/**
 * Constructs {@link Point}s from the nodes of an overlay graph.
 * @version 1.6
 */
public class PointBuilder {
  private OverlayOp op;
  private GeometryFactory geometryFactory;
  private PointLocator ptLocator;

  public PointBuilder(OverlayOp op, GeometryFactory geometryFactory, PointLocator ptLocator) {
    this.op = op;
    this.geometryFactory = geometryFactory;
    this.ptLocator = ptLocator;
  }
  /**
   * @return a list of the Points in the result of the specified overlay operation
   */
  public List build(int opCode)
  {
    List nodeList = collectNodes(opCode);
    List resultPointList = simplifyPoints(nodeList);
    return resultPointList;
  }

  private List collectNodes(int opCode)
  {
    List resultNodeList = new ArrayList();
    // add nodes from edge intersections which have not already been included in the result
    for (Iterator nodeit = op.getGraph().getNodes().iterator(); nodeit.hasNext(); ) {
      Node n = (Node) nodeit.next();
      if (! n.isInResult()) {
        Label label = n.getLabel();
        if (OverlayOp.isResultOfOp(label, opCode)) {
          resultNodeList.add(n);
        }
      }
    }
    return resultNodeList;
  }
  /**
   * This method simplifies the resultant Geometry by finding and eliminating
   * "covered" points.
   * A point is covered if it is contained in another element Geometry
   * with higher dimension (e.g. a point might be contained in a polygon,
   * in which case the point can be eliminated from the resultant).
   */
  private List simplifyPoints(List resultNodeList)
  {
    List nonCoveredPointList = new ArrayList();
    for (Iterator it = resultNodeList.iterator(); it.hasNext(); ) {
      Node n = (Node) it.next();
      Coordinate coord = n.getCoordinate();
      if (! op.isCoveredByLA(coord)) {
        Point pt = geometryFactory.createPoint(coord);
        nonCoveredPointList.add(pt);
      }
    }
    return nonCoveredPointList;
  }
}
