/****************************************************************************
 *                            WaveWidgetEvent.cc
 *
 * Author: Matthew Ballance
 * Desc:   Core event management code for the wave widget
 * <Copyright> (c) 2001-2003 Matthew Ballance (mballance@users.sourceforge.net)
 *
 *    This source code is free software; you can redistribute it
 *    and/or modify it in source code form under the terms of the GNU
 *    General Public License as published by the Free Software
 *    Foundation; either version 2 of the License, or (at your option)
 *    any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 *
 * </Copyright>
 ****************************************************************************/
#include "WaveWidget.h"
#include "WidgetManager.h"
#include "WidgetCommon.h"
#include "SdbReader.h"
#include "SdbrCursor.h"
#include "CallbackMgr.h"
#include "PsPixmapObj.h"
#include "BitVector.h"
#include "BitVectorABVal.h"

#include "SigDB.h"
#include "SdbSignal.h"
#include "DFIO.h"
#include "DFIOTrace.h"

#include "LogInstance.h"
#include "LogDestination.h"
#include "LogMgr.h"

#include "TreeWidget.h"
#include "TreeWidgetNode.h"


/******************************************************************
 * SetupBindings()
 ******************************************************************/
void WaveWidget::SetupBindings()
{
    Bind("<ButtonPress>",   "%b 1 %x %y 0", ButtonEventProc, this);
    Bind("<ButtonRelease>", "%b 0 %x %y 0", ButtonEventProc, this);
    Bind("<<Cursor-Add-Select>>", "%b 1 %x %y 1", ButtonEventProc, this);
}

/******************************************************************
 * ButtonEventProc()
 *
 * objv[0] = cmd
 * objv[1] = buttonNum
 * objv[2] = buttonDown    (1 if Press, 0 if Release)
 * objv[3] = x
 * objv[4] = y
 ******************************************************************/
int WaveWidget::ButtonEventProc(
        int                   objc,
        Tcl_Obj              *const objv[])
{
    int   buttonDown, buttonNum;
    int   x, y, mod;
    
    Tcl_GetIntFromObj(d_interp, objv[1], &buttonNum);
    Tcl_GetIntFromObj(d_interp, objv[2], &buttonDown);
    Tcl_GetIntFromObj(d_interp, objv[3], &x);
    Tcl_GetIntFromObj(d_interp, objv[4], &y);
    Tcl_GetIntFromObj(d_interp, objv[5], &mod);

    if (buttonDown) {
        if (buttonNum == 1) {
            SET_FLAG(RepaintB1Down);

            if (!strcmp(d_config.cursorMode, "select")) {
                /**** Handle cursor-b1 action ****/
                CursorB1Action(x, y, mod);
            }
        } else if (buttonNum == 2) {
            SET_FLAG(RepaintB2Down);
            button_down = 1;
        } else if (buttonNum == 3) {
            SET_FLAG(RepaintB3Down);
        }

        sel_xi = x;
        sel_yi = y;
        sel_xf = sel_xi;
        sel_yf = sel_yi;
    } else {
        if ((IS_FLAG_SET(RepaintB2Down) ||
                    (IS_FLAG_SET(RepaintB1Down) &&
                         !strcmp(d_config.cursorMode, "zoom"))
                    && d_config.selectBoxCmd)) {
            SendSelBoxCmd(sel_xi, sel_yi, sel_xf, sel_yf);
        } else {
            /**** Does cursor-mgmt code need to know about release? ****/
        }

        CLR_FLAG(RepaintB1Down|RepaintB2Down|RepaintB3Down);
        SetupRedraw(RepaintBRelease);
        grabbedCursor = 0;
        button_down = 0;
    }

    return TCL_OK;
}

/******************************************************************
 * EventProc()
 ******************************************************************/
void WaveWidget::EventProc(XEvent *eventPtr)
{
    Uint32            redraw = 0;
    Int32             tmpx, tmpy;
    XRectangle        rect;
    char              buf[64];
    Tk_Window         tkwin = getWin();

    switch (eventPtr->type) {

        /************************************************************
         * Expose
         ************************************************************/
        case Expose:
            log_high("Expose(%d, %d, %d, %d, %d)\n",
                    eventPtr->xexpose.count,
                    eventPtr->xexpose.x,
                    eventPtr->xexpose.y,
                    eventPtr->xexpose.width,
                    eventPtr->xexpose.height);

            if (d_backingPixmapValid && (widgetWidth == width()) &&
                    (widgetHeight == height())) {

                d_WinPixmap(eventPtr->xexpose.x, eventPtr->xexpose.y) =
                    d_displayPixmap(eventPtr->xexpose.x, eventPtr->xexpose.y,
                            eventPtr->xexpose.width, eventPtr->xexpose.height);
            } else if (eventPtr->xexpose.count == 0) {
                log_high("Request redraw\n");
                SetupRedraw(RepaintExpose);
            } 
        break;

        case FocusIn:
        break;

        case FocusOut:
        break;

#if 0
        /************************************************************
         * ButtonPress
         ************************************************************/
        case ButtonPress:
            if (eventPtr->xbutton.button == 1) {
                SET_FLAG(RepaintB1Down);
                if (!strcmp(d_config.cursorMode, "select")) {
                    if (sdbr) {
                        grabbedCursor = sdbr->cursorMgr->nearestCursor(
                            PixelsToTime(eventPtr->xbutton.x)+displayStartTime,
                            displayStartTime, 
                            displayStartTime+PixelsToTime(widgetWidth),
                            PixelsToTime(10));
                    }
                }
            } else if (eventPtr->xbutton.button == 2) {
                SET_FLAG(RepaintB2Down);
                button_down = 1;
            } else if (eventPtr->xbutton.button == 3) {
                SET_FLAG(RepaintB3Down);
           }

            sel_xi = eventPtr->xbutton.x;
            sel_yi = eventPtr->xbutton.y;
            sel_xf = sel_xi;
            sel_yf = sel_yi;
        break;

        /************************************************************
         * ButtonRelease:
         ************************************************************/
        case ButtonRelease:

        /*** If the middle-button was relesed and there is a 
         *** command bound to it, then eval the command\
         ***/
            if ((IS_FLAG_SET(RepaintB2Down) ||
                        (IS_FLAG_SET(RepaintB1Down) && 
                             !strcmp(d_config.cursorMode, "zoom")) 
                        && d_config.selectBoxCmd)) {
                SendSelBoxCmd(sel_xi, sel_yi, sel_xf, sel_yf);
            }

            CLR_FLAG(RepaintB1Down|RepaintB2Down|RepaintB3Down);
            SetupRedraw(RepaintBRelease);
            grabbedCursor = 0;
            button_down = 0;
        break;
#endif

        case MotionNotify:
            if (IS_FLAG_SET(RepaintB2Down) ||
                    (IS_FLAG_SET(RepaintB1Down) && 
                         !strcmp(d_config.cursorMode, "zoom"))) {
                tmpx = eventPtr->xmotion.x;
                tmpy = eventPtr->xmotion.y;

                if (tmpx >= 0 && tmpx <= (int)width()) {
                    sel_xf = tmpx;
                } else if (tmpx > (int)width()) {
                    sel_xf = width();
                } else {
                    sel_xf = 0;
                }

                if (tmpy >= 0 && (tmpy <= (int)height())) {
                    sel_yf = tmpy;
                } else if (tmpy < 0) {
                    sel_yf = 0;
                } else {
                    sel_yf = height();
                }

                SetupRedraw(RepaintMouseDrag);
            } else if (IS_FLAG_SET(RepaintB1Down) && grabbedCursor) {
                Uint32     newTime;

                newTime = PixelsToTime(eventPtr->xmotion.x)+displayStartTime;
                grabbedCursor->setTime(newTime);

                if (d_config.cursorMoveCmd) {
                    sprintf(buf, " %d ", newTime);
                    if (Tcl_VarEval(d_interp,
                                d_config.cursorMoveCmd,buf,0) != TCL_OK) {
                        fprintf(stderr, "ERROR :: CursorMove failed\n");
                        fprintf(stderr, "\t%s\n", 
                                Tcl_GetStringResult(d_interp));
                    }

                    {
                        d_sigValTime.low  = newTime;
                        d_sigValTime.high = 0;
                        d_sigValNeedUpdate = true;
//                        UpdateSigValTree(d_sigValTime);
                    }
                }

                SetupRedraw(RepaintMouseDrag);
            }
        break;
    }
}

