/****************************************************************************
 *                              IviOpts.cc
 * Author: Matthew Ballance
 * Desc:   This class describes the command-line options that ivi supports.
 *
 * <Copyright> (c) 2001-2003 Matthew Ballance (mballance@users.sourceforge.net)
 *
 *    This source code is free software; you can redistribute it
 *    and/or modify it in source code form under the terms of the GNU
 *    General Public License as published by the Free Software
 *    Foundation; either version 2 of the License, or (at your option)
 *    any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 *
 * </Copyright>
 *
 ****************************************************************************/
#include "IviOpts.h"

/********************************************************************
 * DoFile
 ********************************************************************/
DoFile::DoFile(const char *handler, const char *file) :
    d_handler(handler), d_file(file)
{
    ;
}

/********************************************************************
 * ViewFile
 ********************************************************************/
ViewFile::ViewFile(const char *handler, const char *file) :
    d_handler(handler), d_file(file)
{
    ;
}

/***************************************************
 * IviOpts()
 ***************************************************/
IviOpts::IviOpts(Uint32 argc, Char **argv)
{
    Uint32 i;
    int    e_code = 1;
    int    arg_len;

    /*** Setup defaults ***/
    commandLine = false;
    doFiles     = new Vector<DoFile>; 
    viewFiles   = new Vector<ViewFile>; 
    preDoFiles  = new Vector<String>; 
    modulePath  = new Vector<String>; 
    modules     = new Vector<String>; 
    design      = 0;
    logFile     = 0;
    needHelp    = false;
    versionCheck = false;
    console      = false;

    for (i=1; i<argc; i++) {

        if (argv[i][0] == '-') {
            arg_len = strlen(argv[i]);
            if (strcmp(argv[i], "-c") == 0) {
                commandLine = true;
            } else if (!strcmp(argv[i], "-console")) {
                console = true;
            } else if (!strcmp(argv[i], "-do") || !strcmp(argv[i], "-inc")) {
                if (++i >= argc) {
                    fprintf(stdout, "ERROR: No dofile/incfile given!!\n");
                    goto help_exit;
                }
                doFiles->append(new DoFile("ivi", argv[i])); 
            } else if (!strncmp(argv[i], "-view", 5)) {
                /**** Accept two formats here:
                 **** -view <file>
                 **** -view_<name> <file>
                 ****/
                if ((i+1) >= argc) {
                    fprintf(stdout, "ERROR: No viewfile given!!\n");
                    goto help_exit;
                }

                if (!strcmp(argv[i], "-view")) {
                    i++;
                    viewFiles->append(new ViewFile(0, argv[i]));
                } else {
                    char *sp;

                    if (!(sp = strchr(argv[i], '_'))) {
                        fprintf(stderr, "ERROR: expecting view_<handler>\n");
                        goto help_exit;
                    }

                    sp++;
                    i++;

                    viewFiles->append(new ViewFile(sp, argv[i]));
                }
            } else if ((arg_len>3) && !strcmp("do", &argv[i][arg_len-2])) {
                char do_buf[32], *s;
                /**** foreign do command ****/
                strcpy(do_buf, &argv[i][1]);
                do_buf[arg_len-1-3] = 0;

                if (++i >= argc) {
                    fprintf(stdout, "ERROR: No dofile/incfile given!!\n");
                    goto help_exit;
                }

                doFiles->append(new DoFile(do_buf, argv[i]));
            } else if (!strcmp(argv[i], "-pre_do") == 0) {
                if (++i >= argc) {
                    fprintf(stdout, "ERROR: No pre-dofile specified!!\n");
                    goto help_exit;
                }
                preDoFiles->append(new String(argv[i])); 
            } else if (strcmp(argv[i], "-l") == 0) {
                if (++i >= argc) {
                    fprintf(stdout, "ERROR: No logfile specified!!\n");
                    goto help_exit;
                }
                if (logFile) {
                    fprintf(stdout, "ERROR: Only one logfile may "
                            "be specified\n");
                    goto help_exit;
                }
                logFile = new String(argv[i]);
            } else if (strcmp(argv[i], "-m") == 0) {
                if (++i >= argc) {
                    fprintf(stdout, "ERROR: No module specified!!\n");
                    goto help_exit;
                }
                modules->append(new String(argv[i]));
            } else if (strcmp(argv[i], "-M") == 0) {
                if (++i >= argc) {
                    fprintf(stdout, "ERROR: No module-path specified!!\n");
                    goto help_exit;
                }
                modulePath->append(new String(argv[i]));
            } else if (!strcmp(argv[i], "-help") ||
                    !strcmp(argv[i], "-h") || !strcmp(argv[i], "--help")) {
                e_code = 0;
                goto help_exit;
            } else if (!strcmp(argv[i], "--version")) {
                versionCheck = true;
            } else {
                fprintf(stdout, "ERROR: Unrecognized option \"%s\"\n",
                        argv[i]);
                goto help_exit;
            }
        } else {
            /* Design file to load */
            design = new String(argv[i]); 
        }
    }

    return;

help_exit:
    printHelp();
    exit(e_code);
}

/********************************************************************
 * printHelp()
 ********************************************************************/
void IviOpts::printHelp()
{
    static char *help[] = {
        "ivi [options] [design_file]\n\n",
        "    options:\n",
        "        -do <file>\n",
        "        -inc <file>\n",
        "            IVI executes the TCL commands from <file> after loading\n",
        "            any specified design\n",
        "\n",
        "        -l <file>\n",
        "            Causes a transcript of the simulation session to be\n",
        "            saved to the specified file.\n",
        "\n",
	"        -view <file>\n",
	"        -view_<fmt> <file>\n",
	"            Causes the specified files to be loaded by IVI as \n",
	"            waveform data. Waveform data-files are loaded before\n",
	"            include files.\n",
	"\n",
	"            Currently, the following formats are supported:\n",
	"            - vcd\n",
	"              Open a VCD (not eVCD) file. Use -view_vcd\n",
	"\n",
	"            - stub\n",
	"              Open an IVI stub-format data file. Use -view_stub\n",
	"              The file extension should be 'sdd'\n",
	"\n",
        "        -help\n",
        "        --help\n",
        "        -h\n",
        "        -?\n",
        "            Print this message\n",
        ""
    };
    Uint32 idx=0;

    while (help[idx][0]) {
        fprintf(stdout, "%s", help[idx]);
        idx++;
    }
}


