#****************************************************************************
#*                              SdbPane.tcl
#*
#* Author: Matthew Ballance
#* Desc:   Implements a collection of SDBs. Automatically updates when
#*         a new SDB is added... Uses DdbStruct to render each SDB's 
#*         content.
#* <Copyright> (c) 2001-2003 Matthew Ballance (mballance@users.sourceforge.net)
#*
#*    This source code is free software; you can redistribute it
#*    and/or modify it in source code form under the terms of the GNU
#*    General Public License as published by the Free Software
#*    Foundation; either version 2 of the License, or (at your option)
#*    any later version.
#*
#*    This program is distributed in the hope that it will be useful,
#*    but WITHOUT ANY WARRANTY; without even the implied warranty of
#*    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#*    GNU General Public License for more details.
#*
#*    You should have received a copy of the GNU General Public License
#*    along with this program; if not, write to the Free Software
#*    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
#*
#* </Copyright>
#****************************************************************************

namespace eval SdbPane {
    variable configspec {
        {"-sb_width"        12 }
        {"-font"            "-*-arial-medium-r-normal-*-*-*-*-*-*-*"}
        {"-browse_cmd"      "" }
        {"-sepchar"         "."}
        {"-sdb_rejfilter"      "" }
        {"-sdb_acceptfilter"   "" }
    }
}

#********************************************************************
#* SdbPane
#********************************************************************
proc SdbPane::SdbPane {path args} {
    global     CallbackTypes
    variable   configspec

    set path [frame $path]

    bind $path <Destroy> "SdbPane::Destroy $path"
 
    rename $path ::$path:cmd
    proc ::$path {cmd args} "return \[eval SdbPane::cmd $path \$cmd \$args\]"

    array set $path {_dummy _dummy}
    upvar #0 $path data

    #**** Setup options...
    foreach varr $configspec {
        set data([lindex $varr 0]) [lindex $varr 1]
    }

    eval configure $path $args

    set data(w:sdb_nb) [NoteBook $path.nb -bg "grey80"]

    set sdb_list [sdb_mgr list]

    foreach sdb $sdb_list {

        if {![isSdbValid $path [$sdb cget -sdb_id]]} {
            continue
        }

        set iname_list [split $sdb {.}]
        set inst [lindex $iname_list end]

        set page [$data(w:sdb_nb) insert end $inst -text [$sdb cget -sdb_id] \
            -raisecmd "SdbPane::RaiseTab $path"]
         
        set m [DdbStruct::DdbStruct $page.struct \
            -sepchar $data(-sepchar) -browsecmd  \
            "SdbPane::PrvBrowseModule $path" \
            -ddb [$sdb cget -ddb] -sdb $sdb -sb_width $data(-sb_width) ]

        pack $page.struct -expand yes -fill both
    }

    foreach sdb $sdb_list {
        if {![isSdbValid $path [$sdb cget -sdb_id]]} {
            continue
        }

        set iname_list [split $sdb {.}]
        set inst [lindex $iname_list end]
        $data(w:sdb_nb) raise $inst
        break
    }

    set data(sdb_cb) [callback add $CallbackTypes(SDB_MGR_UPDATE) \
            null [list SdbPane::UpdateSdbList $path]]

    pack $data(w:sdb_nb) -expand yes -fill both

    return $path
}

#********************************************************************
#* Destroy
#********************************************************************
proc SdbPane::Destroy {path} {
    upvar #0 $path data

    callback destroy $data(sdb_cb)
}

#********************************************************************
#* isSdbValid
#********************************************************************
proc SdbPane::isSdbValid {path name} {
    upvar #0 $path data

    if {$data(-sdb_rejfilter) != ""} {
        if {[lsearch $data(-sdb_rejfilter) $name] > -1} {
            #**** Invalid...
            return 0
        }
    }

    if {$data(-sdb_acceptfilter) != ""} {
        if {[lsearch $data(-sdb_acceptfilter) $name] == -1} {
            #**** Doesn't match...
            return 0
        }
    }
    
    return 1
}

#********************************************************************
#* configure
#********************************************************************
proc SdbPane::configure {path args} {
    upvar #0 $path data

    while {[llength $args] > 0} {
        set arg [lindex $args 0]

        set val [array get data $arg]
        if {$val == ""} {
            error "no option \"$arg\""
        }

        if {[info commands SdbPane::config$arg] != ""} {
            set data($arg) [SdbPane::config$arg $path [lindex $args 1]]
        } else {
            set data($arg) [lindex $args 1]
        }

        set args [lrange $args 2 end]
    }

}

#********************************************************************
#* cmd
#********************************************************************
proc SdbPane::cmd {path cmd args} {
    upvar #0 $path data

    switch $cmd {
        config -
        configure {
            eval configure $args
        }

        curr_page {
            return "$data(w:sdb_nb).f[$data(w:sdb_nb) raise].struct"
        }

        default {
            error "unknown SdbPane sub-cmd $cmd"
        }
    }
}

#********************************************************************
#* PrvBrowseModule
#********************************************************************
proc SdbPane::PrvBrowseModule {w entry} {
    upvar #0 $w data

    if {$data(-browse_cmd) != ""} {
        if {$entry == ""} {
            set entry "@@none@@"
        }
        eval $data(-browse_cmd) $entry
    }
}

#*********************************************************
#* UpdateSdbList
#*
#*
#* Want to look at the current list of SDB's and the
#* current list of pages...
#*
#* NOTE :: Expect only _ONE_ change per callback
#*********************************************************
proc SdbPane::UpdateSdbList {w} {
    upvar #0 $w data

    set sdbl [sdb_mgr list]
    set plst [$data(w:sdb_nb) pages]

    set slen [llength $sdbl]
    set plen [llength $plst]

    if {$slen > $plen} {
        #**** Adding an SDB (should be on the list-end) *****
        set new_sdb [lindex $sdbl end]

        if {$data(-sdb_rejfilter) != ""} {
            if {[lsearch $data(-sdb_rejfilter) [$new_sdb cget -sdb_id]] > -1} {
                #**** Skip - this SDB is on the exclusion list
                return
            }
        }

        if {$data(-sdb_acceptfilter) != ""} {
            if {[lsearch $data(-sdb_acceptfilter)  \
                    [$new_sdb cget -sdb_id]] == -1} {
                #**** Skip - this SDB is not on the accept list
                return
            }
        }

        set iname_list [split $new_sdb {.}]
        set inst [lindex $iname_list end]

        set new_page [$data(w:sdb_nb) insert end  $inst \
            -text [$new_sdb cget -sdb_id] \
            -raisecmd "SdbPane::RaiseTab $w"]

        set str [DdbStruct::DdbStruct $new_page.struct \
            -sepchar $data(-sepchar) -browsecmd "SdbPane::PrvBrowseModule $w" \
            -ddb [$new_sdb cget -ddb] -sdb $new_sdb -sb_width $data(-sb_width)]

        pack $str -expand yes -fill both
    } else {
        #**** We expect to remove a pane. Therefore, we scan through each
        #**** page in the notebook and compare it to the list of SDBs

        foreach page $plst {
            set found 0

            foreach sdb $sdbl {
                set iname_list [split $sdb {.}]
                set inst [lindex $iname_list end]

                if {$inst == $page} {
                    set found 1
                    break
                }
            }

            if {$found == 0} {
                $data(w:sdb_nb) delete $page 1
            }

        }
    }

    if {$plen == 0} {
        set page1 [lindex [$data(w:sdb_nb) pages] 0]
        $data(w:sdb_nb) raise $page1
    }
}

#*********************************************************
#* RaiseTab
#*********************************************************
proc SdbPane::RaiseTab {w} {
    upvar #0 $w data

    set struct "$data(w:sdb_nb).f[$data(w:sdb_nb) raise].struct"
    set ddb [$struct cget -ddb]
    set sdb [$struct cget -sdb]

    PrvBrowseModule $w [$struct selected]
}

