/*****************************************************************************
 *                             TclListObj.cc
 *
 * Author: Matthew Ballance
 * Desc:   Implements c++ wrapper around a TCL list object
 * <Copyright> (c) 2001-2003 Matthew Ballance (mballance@users.sourceforge.net)
 *
 *    This source code is free software; you can redistribute it
 *    and/or modify it in source code form under the terms of the GNU
 *    General Public License as published by the Free Software
 *    Foundation; either version 2 of the License, or (at your option)
 *    any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 *
 * </Copyright>
 *****************************************************************************/
#include "TclListObj.h"
#include "ivi_String.h"

/********************************************************************
 * TclListObj()
 ********************************************************************/
TclListObj::TclListObj(Tcl_Interp *interp)
{
    if (interp) {
        this->interp = interp;
        obj = Tcl_NewListObj(0, 0);
    }
}

/********************************************************************
 * operator ()
 ********************************************************************/
void TclListObj::operator () (Tcl_Interp *interp)
{
    if (interp) {
        this->interp = interp;
        obj = Tcl_NewListObj(0, 0);
    }
}

/********************************************************************
 * TclListObj(interp, list)
 ********************************************************************/
TclListObj::TclListObj(Tcl_Interp *interp, Tcl_Obj *list)
{
    this->interp = interp;
    obj = list;
}

/********************************************************************
 * operator << (Uint32)
 ********************************************************************/
TclListObj TclListObj::operator << (Uint32 val)
{
    if (!obj) {
        obj = Tcl_NewListObj(0, 0);
    }
    Tcl_ListObjAppendElement(interp, obj, Tcl_NewIntObj(val));
    return TclListObj(interp, obj);
}

/********************************************************************
 * operator << (Int32)
 ********************************************************************/
TclListObj TclListObj::operator << (Int32 val)
{
    if (!obj) {
        obj = Tcl_NewListObj(0, 0);
    }
    Tcl_ListObjAppendElement(interp, obj, Tcl_NewIntObj(val));
    return TclListObj(interp, obj);
}

/********************************************************************
 * operator << (Char *)
 ********************************************************************/
TclListObj TclListObj::operator << (const Char *val)
{
    if (!obj) {
        obj = Tcl_NewListObj(0, 0);
    }
    Tcl_ListObjAppendElement(interp, obj, Tcl_NewStringObj(val, -1));
    return TclListObj(interp, obj);
}

/********************************************************************
 * operator << (String &)
 ********************************************************************/
TclListObj TclListObj::operator << (const String &val)
{
    if (!obj) {
        obj = Tcl_NewListObj(0, 0);
    }
    Tcl_ListObjAppendElement(interp, obj, 
            Tcl_NewStringObj(val.value(), val.length()));
    return TclListObj(interp, obj);
}

/********************************************************************
 * operator << (TclListObj)
 ********************************************************************/
TclListObj TclListObj::operator << (const TclListObj &val)
{
    if (!obj) {
        obj = Tcl_NewListObj(0, 0);
    }
    Tcl_ListObjAppendElement(interp, obj, Tcl_DuplicateObj(val.obj));
    return TclListObj(interp, obj);
}

/********************************************************************
 * operator << (Tcl_Obj)
 ********************************************************************/
TclListObj TclListObj::operator << (Tcl_Obj *const val)
{
    if (!obj) {
        obj = Tcl_NewListObj(0, 0);
    }
    Tcl_ListObjAppendElement(interp, obj, Tcl_DuplicateObj(val));
    return TclListObj(interp, obj);
}

/********************************************************************
 * operator []
 ********************************************************************/
Tcl_Obj *TclListObj::operator [] (Uint32 idx)
{
    Tcl_Obj *ret;
    if (!obj) {
        return 0;
    }

    Tcl_ListObjIndex(interp, obj, idx, &ret);
    return ret;
}

/********************************************************************
 * length()
 ********************************************************************/
Uint32 TclListObj::length()
{
    int len = 0;

    if (obj) {
        Tcl_ListObjLength(interp, obj, &len);
    }

    return len;
}

/********************************************************************
 * append_elems()
 ********************************************************************/
void TclListObj::append_elems(Tcl_Obj *list)
{
    Tcl_ListObjAppendList(interp, obj, list);
}

/********************************************************************
 * append_elems(int, list)
 ********************************************************************/
void TclListObj::append_elems(int objc, Tcl_Obj *const objv[])
{
    for (int i=0; i<objc; i++) {
        Tcl_ListObjAppendElement(interp, obj, objv[i]);
    }
}

/********************************************************************
 * end()
 ********************************************************************/
Tcl_Obj *TclListObj::end(void)
{
    Tcl_Obj *tmp = obj;
    obj = 0;
    return tmp;
}


