/****************************************************************************
 *                             TclChannelObj.h
 * 
 * Author: Matthew Ballance
 * Desc:   Implements a basic character-oriented channel. This basic channel
 *         may be extended.
 * <Copyright> (c) 2001-2003 Matthew Ballance (mballance@users.sourceforge.net)
 *
 *    This source code is free software; you can redistribute it
 *    and/or modify it in source code form under the terms of the GNU
 *    General Public License as published by the Free Software
 *    Foundation; either version 2 of the License, or (at your option)
 *    any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 *
 * </Copyright>
 ****************************************************************************/
#ifndef INCLUDED_TCL_CHANNEL_OBJ_H
#define INCLUDED_TCL_CHANNEL_OBJ_H
#include "types.h"
#include "Fifo.h"
#include "ivi_String.h"
#include <tcl.h>

class TclChannelObj {

    /****************************************************************
     * Public Methods
     ****************************************************************/
    public:
        /************************************************************
         * TclChannelObj()
         ************************************************************/
        TclChannelObj(
                Tcl_Interp    *interp,
                const char    *channelName,
                const char    *channelTypeName,
                Int32          flags);

        /************************************************************
         * ~TclChannelObj()
         ************************************************************/
        virtual ~TclChannelObj();

        /************************************************************
         * write()
         ************************************************************/
        virtual void write(Char *msg);

        /************************************************************
         * InputProc()
         ************************************************************/
        virtual int InputProc(
                char         *buf,
                int           bufSize,
                int          *errorCodePtr);

        /************************************************************
         * OutputProc()
         ************************************************************/
        virtual int OutputProc(
                const char   *buf,
                int           toWrite,
                int          *errorCodePtr);

        /************************************************************
         * SeekProc()
         ************************************************************/
        virtual int SeekProc(
                long          offset,
                int           seekMode,
                int          *errorCodePtr);

        /************************************************************
         * SetOptionProc()
         ************************************************************/
        virtual int SetOptionProc(
                Tcl_Interp   *interp,
                const char   *optionName,
                const char   *optionValue);

        /************************************************************
         * GetOptionProc()
         ************************************************************/
        virtual int GetOptionProc(
                Tcl_Interp   *interp,
                const char   *optionName,
                Tcl_DString  *dsPtr);

        /************************************************************
         * WatchProc()
         ************************************************************/
        virtual void WatchProc(int mask);

        /************************************************************
         * GetHandleProc()
         ************************************************************/
        virtual int GetHandleProc(
                int           direction,
                ClientData   *handlePtr);

        /************************************************************
         * BlockModeProc()
         ************************************************************/
        virtual int BlockModeProc(
                int           mode);

        /************************************************************
         * FlushProc()
         ************************************************************/
        virtual int FlushProc();

        /************************************************************
         * HandlerProc()
         ************************************************************/
        virtual int HandlerProc(
                int           interestMask);

        /************************************************************
         * Close2Proc()
         ************************************************************/
        virtual int Close2Proc(
                Tcl_Interp   *interp,
                int           flags);

        /************************************************************
         * CloseProc()
         ************************************************************/
        virtual int CloseProc(
                Tcl_Interp   *interp);

        /************************************************************
         * getChannelName()
         ************************************************************/
        const char *getChannelName() { return d_channelName.value(); }

    /****************************************************************
     * Public Data
     ****************************************************************/
    public:
        Uint32        ok;

    /****************************************************************
     * Protected Data
     ****************************************************************/
    protected:

    /****************************************************************
     * Private Methods
     ****************************************************************/
    private:

        /************************************************************
         * InputProc_Static()
         ************************************************************/
        static int InputProc_Static(
                ClientData    clientData,
                char         *buf,
                int           bufSize,
                int          *errorCodePtr);

        /************************************************************
         * OutputProc_Static()
         ************************************************************/
        static int OutputProc_Static(
                ClientData    clientData,
                const char   *buf,
                int           toWrite,
                int          *errorCodePtr);

        /************************************************************
         * SeekProc_Static()
         ************************************************************/
        static int SeekProc_Static(
                ClientData    clientData,
                long          offset,
                int           seekMode,
                int          *errorCodePtr);

        /************************************************************
         * SetOptionProc_Static()
         ************************************************************/
        static int SetOptionProc_Static(
                ClientData    clientData,
                Tcl_Interp   *interp,
                const char   *optionName,
                const char   *optionValue);

        /************************************************************
         * GetOptionProc_Static()
         ************************************************************/
        static int GetOptionProc_Static(
                ClientData    clientData,
                Tcl_Interp   *interp,
                const char   *optionName,
                Tcl_DString  *dsPtr);

        /************************************************************
         * WatchProc_Static()
         ************************************************************/
        static void WatchProc_Static(
                ClientData    clientData,
                int           mask);

        /************************************************************
         * GetHandleProc_Static()
         ************************************************************/
        static int GetHandleProc_Static(
                ClientData    clientData,
                int           direction,
                ClientData   *handlePtr);

        /************************************************************
         * BlockModeProc_Static()
         ************************************************************/
        static int BlockModeProc_Static(
                ClientData    clientData,
                int           mode);

        /************************************************************
         * FlushProc_Static()
         ************************************************************/
        static int FlushProc_Static(
                ClientData    clientData);

        /************************************************************
         * HandlerProc_Static()
         ************************************************************/
        static int HandlerProc_Static(
                ClientData    clientData,
                int           interestMask);

        /************************************************************
         * Close2Proc_Static()
         ************************************************************/
        static int Close2Proc_Static(
                ClientData    clientData,
                Tcl_Interp   *interp,
                int           flags);

        /************************************************************
         * CloseProc_Static()
         ************************************************************/
        static int CloseProc_Static(
                ClientData    clientData,
                Tcl_Interp   *interp);

        /************************************************************
         * InstCmd()
         ************************************************************/
        static int InstCmd(
                ClientData                  clientData,
                Tcl_Interp                 *interp,
                int                         argc,
                const char                **argv) {
            Tcl_Preserve(clientData);
            int ret = ((TclChannelObj *)clientData)->InstCmd(argc, argv);
            Tcl_Release(clientData);
            return ret;
        }

        /************************************************************
         * InstCmd()
         ************************************************************/
        int InstCmd(int argc, const char **argv);

        /************************************************************
         * Delete()
         ************************************************************/
        static void Delete(char *ptr) {
            delete (TclChannelObj *)ptr;
        }

        /************************************************************
         * Execute()
         ************************************************************/
        int Execute();

    /****************************************************************
     * Private Data
     ****************************************************************/
    private:
        Fifo                       d_fifo;
        Int32                      d_mask;
        Tcl_Channel                d_channel;
        String                     d_channelName;
        Tcl_Interp                *d_interp;
        Tcl_ChannelType            d_ChannelProcs;

        static Tcl_ChannelType     d_DefaultChannelProcs;
};

#endif /* INCLUDED_TCL_CHANNEL_OBJ_H */
