/****************************************************************************
 *                       RemoteAppTclCmdListener.cc
 *
 * Author: Matthew Ballance
 * Desc: 
 * <Copyright> (c) 2001-2003 Matthew Ballance (mballance@users.sourceforge.net)
 *
 *    This source code is free software; you can redistribute it
 *    and/or modify it in source code form under the terms of the GNU
 *    General Public License as published by the Free Software
 *    Foundation; either version 2 of the License, or (at your option)
 *    any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 *
 * </Copyright>
 ****************************************************************************/
#include "RemoteAppTclCmdListener.h"
#include "RemoteAppConnection.h"
#include "IviRemoteAppIdx.h"
#include <string.h>


class RemoteAppTclCmdResp : public RemoteAppConnectionListener {

    /****************************************************************
     * Public Methods
     ****************************************************************/
    public:

        /************************************************************
         * RemoteAppTclCmdResp
         ************************************************************/
        RemoteAppTclCmdResp(RemoteAppTclCmdListener *tcmd) : 
            RemoteAppConnectionListener(IVI_CMD_CH_IDX_TCL_CMD_RESP) 
        { 
            d_tcmd = tcmd; 
        }

        /************************************************************
         * ~RemoteAppTclCmdResp()
         ************************************************************/
        virtual ~RemoteAppTclCmdResp() { ; }

        /************************************************************
         * Receive()
         ************************************************************/
        virtual void Receive(Uint32 chanIdx, Uint32 len, Uchar *data) {
            if (d_tcmd) {
                d_tcmd->CmdResp(len, data);
            } else {
                fprintf(stderr, "ERROR: d_tcmd = NULL\n");
            }
        }

    /****************************************************************
     * Private Data
     ****************************************************************/
    private:
        RemoteAppTclCmdListener     *d_tcmd;
};


/********************************************************************
 * RemoteAppTclCmdListener()
 ********************************************************************/
RemoteAppTclCmdListener::RemoteAppTclCmdListener(Tcl_Interp *interp) :
     RemoteAppConnectionListener(IVI_CMD_CH_IDX_TCL_CMD)
{
    d_respBufMax = 0;
    d_respBufLen = 0;
    d_respBuf    = 0;
    d_resp       = 0;

    d_interp = interp;
}

/********************************************************************
 * ~RemoteAppTclCmdListener()
 ********************************************************************/
RemoteAppTclCmdListener::~RemoteAppTclCmdListener()
{

}

/********************************************************************
 * Command()
 ********************************************************************/
int RemoteAppTclCmdListener::Command(Tcl_Interp *interp, Tcl_Obj *obj)
{
    char *cmd = Tcl_GetString(obj);

    return Command(interp, cmd);
}

/********************************************************************
 * Command()
 ********************************************************************/
int RemoteAppTclCmdListener::Command(Tcl_Interp *interp, int argc, char **argv)
{
    Tcl_Obj *list = Tcl_NewListObj(0, 0);

    Tcl_IncrRefCount(list);

    for (int i=0; i<argc; i++)  {
        Tcl_ListObjAppendElement(interp, list, Tcl_NewStringObj(argv[i], -1));
    }

    Command(interp, list);

    Tcl_DecrRefCount(list);
}

/********************************************************************
 * Command()
 ********************************************************************/
int RemoteAppTclCmdListener::Command(Tcl_Interp *interp, 
        int objc, Tcl_Obj *const objv[])
{
    Tcl_Obj *list;
    int      ret;

    list = Tcl_NewListObj(objc, objv);
    Tcl_IncrRefCount(list);

    ret = Command(interp, list);

    Tcl_DecrRefCount(list);

    return ret;
}

/********************************************************************
 * Command()
 *
 * Sends a command to the remote listener for execution. Result is 
 * placed in 'interp'
 ********************************************************************/
int RemoteAppTclCmdListener::Command(Tcl_Interp *interp, char *command)
{
    int ret = 0;

    Send(IVI_CMD_CH_IDX_TCL_CMD, strlen(command)+1, (Uchar *)command);

 
    /**** Now, wait for command completion... ****/
    d_resp = 0;
    do {
        if (d_connection->Poll(REMOTE_POLL_FOREVER) < 0) {
            d_resp = -1;
        }
    } while (!d_resp);

    if (d_resp > 0) {
        ret = d_respBuf[0];

        Tcl_SetObjResult(interp, 
                Tcl_NewStringObj((const char *)&d_respBuf[1], -1));
    } else {
        ret = TCL_ERROR;

        fprintf(stderr, "ERROR: Communication Error\n");

        Tcl_SetResult(interp, "Communication Error", TCL_STATIC);
    }

    return ret;
}

/********************************************************************
 * Receive()
 ********************************************************************/
void RemoteAppTclCmdListener::Receive(Uint32 chanIdx, Uint32 len, Uchar *data)
{
    char       *rbuf; 
    const char *resp;
    int         ret, rlen;

    ret = Tcl_EvalEx(d_interp, (char *)data, -1, TCL_EVAL_GLOBAL);

    if (ret != TCL_OK) {
        fprintf(stderr, "\tERROR: %s\n", Tcl_GetStringResult(d_interp));
    } 

    resp = Tcl_GetStringResult(d_interp);

    rlen = strlen(resp);
    rbuf = new char[rlen+2];
    strcpy(&rbuf[1], resp);

    rbuf[0] = ret;

    Send(IVI_CMD_CH_IDX_TCL_CMD_RESP, rlen+2, (Uchar *)rbuf);

    delete [] rbuf;
}

/********************************************************************
 * Connect()
 ********************************************************************/
void RemoteAppTclCmdListener::Connect(RemoteAppConnection *conn)
{
    RemoteAppTclCmdResp *respl;

    d_connection = conn;

    /**** Create a new response listener and attach it to the channel ****/
    respl = new RemoteAppTclCmdResp(this);
    d_connection->AddListener(respl);

}

/********************************************************************
 * CmdResp()
 ********************************************************************/
void RemoteAppTclCmdListener::CmdResp(Uint32 len, Uchar *data)
{
    if (len > d_respBufMax) {

        if (d_respBufMax) {
            delete [] d_respBuf;
        }

        d_respBuf    = new Uchar[len];
        d_respBufMax = len;
    }

    d_respBufLen = len;
    memcpy(d_respBuf, data, len);

    d_resp = 1;
}


