/*****************************************************************************
 * $Id: mod_save.c,v 1.20 2002/01/17 19:35:48 roberto Exp $
 *
 *      Saves last played song or playlist (with song within that playlist)
 *      and position (time) in that song, so that when IRMP3 is restarted
 *      it'll start playing the last played song (or song withing playlist)
 *      and the exact time where it finished.
 *      Useful for those using IRMP3 in car jukeboxes.
 *
 * Copyright (C) 2001,2002 by Roberto Mello <rmello@fslc.usu.edu>
 * Licensed under the GPL version 2 or later.
 *
 * Thanks goes to Ray, Jon and Louis for their invaluable help with this.
 * Also thanks to the IRMP3 authors, for this great piece of software with
 * great documentation for developers.
 *
 * Note: Although I had C++ experience, this was my first C program larger
 * than 5 lines, so I'm sure there are things that could be done better, and
 * I'd be more than happy to hear about them.
 */

#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <string.h>
#include <errno.h>

#include "tools.h"
#include "log.h"
#include "mod.h"
#include "config.h"
#include "mod_save.h"

/*************************************************************************
 * GLOBALS
 */
extern int daemonize;     // set by irmp3.c if we are a daemon
int   MOD_SAVE_ENABLED_P = 0;
int   MOD_SAVE_LOADDIR = 0;
int   MOD_SAVE_INIT = 0;
char  *MOD_SAVE_COMMAND_0;
char  *MOD_SAVE_COMMAND_1;
char  *MOD_SAVE_FILE = NULL;

/* ************************************************************************
 * MODULE INFO
 *   This structure is used to pass to mod_register to register
 *   our module with the main program.
 */
mod_t mod_save = {
	mod_save_deinit,  // our deinit function
	mod_save_reload,  // called when got SIGHUP
	0,
	NULL,
	NULL,             // we don't need to be called periodically
	mod_save_message, // our message handler
	NULL,             // SIGCHLD handler
};


/* *********************************************************************
 * OPEN FILE
 *   Opens a file descriptor with some error checking and reporting.
 */
int eopen(char *pathname, int write_mode) {
	int fd;
	
	if (write_mode == 0) {
		fd = open(pathname, O_RDONLY);
	} else {
		fd = open(pathname, O_WRONLY|O_CREAT|O_TRUNC,
		          S_IRUSR|S_IWUSR|S_IRGRP|S_IROTH);
	}

	if (fd < 0) {
		log_printf(LOG_ERROR, "mod_save: could not open %s: %s\n",
		           pathname, strerror(errno));
	}
	return fd;
}


/* ************************************************************************
 * RECEIVE MESSAGE
 *   Whenever another module sends out a message via mod_sendmsg(), this
 *   function of all active modules will be called to notify them about the new
 *   message. msgtype shows the type of the message, msg contains the message.
 */
void mod_save_message (int msgtype, char *msg)
{
	char *timec = NULL;
	char type[512];
	int  fd, rc, time;

	log_printf(LOG_DEBUG,
 		   "mod_save_message(init = %d): got message type %d: '%s'\n", MOD_SAVE_INIT, msgtype, msg);

	/*
	 * First let's check if we are enabled.
	 * If we aren't, don't do anything.
	 */
	if (!MOD_SAVE_ENABLED_P) {
		return;
	}

	if (strncmp("mod_save init ", msg, 14) == 0 && MOD_SAVE_INIT == 0) {
		/*
		 * We should not react to any other messages while we are in 
		 * initialization mode, for they are _our_ messages.
		 */
		MOD_SAVE_INIT = 1; 
		return;
	}

	/*
	 * If we are in init mode, we need to wait for the player's "play" message to
	 * execute the other commands. When we do receive the "play" message, send one 
	 * command at a time. We do this because if we send all commands at once, the 
	 * player either doesn't catch them or gets confused and dies.
	 */
	if (MOD_SAVE_INIT == 1) { 
		if (msgtype == 2 && strncmp("play", msg, 4) == 0) {
			if (MOD_SAVE_COMMAND_0 != NULL) {
				log_printf(LOG_DEBUG, "mod_save_message(init): freeing command 0 '%s'\n", MOD_SAVE_COMMAND_0);
				mod_sendmsgf(MSGTYPE_INPUT, MOD_SAVE_COMMAND_0);
				MOD_SAVE_COMMAND_0 = NULL;
				return;
			}
			
			if (MOD_SAVE_COMMAND_1 != NULL) {
				log_printf(LOG_DEBUG, "mod_save_message(init): freeing command 1 '%s'\n", MOD_SAVE_COMMAND_1);
				mod_sendmsgf(MSGTYPE_INPUT, MOD_SAVE_COMMAND_1);
				MOD_SAVE_COMMAND_1 = NULL;
				return;
			}
		}
		/*
		 * No more commands left. Set init to 0.
		 */
		
		if (MOD_SAVE_COMMAND_0 == NULL && MOD_SAVE_COMMAND_1 == NULL) {
			MOD_SAVE_INIT = 0;
		}
		return;
	}

	if(strncmp("time ", msg, 5) == 0 ) {
		/*
		 * Update time position info
		 */
		if( (fd = eopen(MOD_SAVE_FILE, 0)) < 0) {
			return;
		}

		/*
		 * Find out the time
		 */
		if (timec == NULL && (timec = malloc(20)) == NULL) {
			/*
			 * Could not allocate memory. Bail out.
			 */
			return;
		}
				
		strcpy(timec, msg);
		time = atoi(strtok(timec + 5, " "));

		rc = readline(fd, type, sizeof(type));

		if (rc > 0 && strncmp("list", type, 4) == 0) {
			/* a list */
			log_printf(LOG_NOISYDEBUG, "mod_save_message(): updating time (list) %d \n", time);
			mod_save_write_list(fd, msg, time, 1);
			close(fd);
		} else {
			/*a song */
			mod_save_write_song(fd, msg, time);
			close(fd);
		}

		free(timec);
		close(fd);
		return;

	} else if(strncmp("play ", msg, 5) == 0 ) {
		/*
		 * Song being played. Check if this is within a loaddir.
		 * If so, write it as a new song. 
		 */
		if(MOD_SAVE_LOADDIR) {
			if( (fd = eopen(MOD_SAVE_FILE, 1)) < 0) {
				return;
			}

			log_printf(LOG_DEBUG, "mod_save_message(): saving 'song \%s 0' to file\n", msg + 5);
			sendtext(fd, "song\n%s\n0", msg + 5);
			close(fd);
		}

	} else if(strncmp("playlist loaddir ", msg, 17) == 0 ) {
		/*
		 * Heads up! It's a "playlist loaddir". This will turn back 
		 * to 0 when a "playlist load" is issued.
		 */
		MOD_SAVE_LOADDIR = 1;

	} else if(strncmp("playlist load ", msg, 14) == 0) {
		/*
		 * playlist load
		 */

		if( (fd = eopen(MOD_SAVE_FILE, 1)) < 0) {
			return;
		}

		MOD_SAVE_LOADDIR = 0;
		log_printf(LOG_DEBUG, "mod_save_message(): saving 'list %s 0 0' to file\n", msg + 14);
		sendtext(fd, "list\n%s\n0\n0", msg + 14);
		close(fd);

	} else if(MOD_SAVE_LOADDIR == 0 && strncmp("playlist jump ", msg, 14) == 0) {
		/*
		 * playlist jump. If this is in a loaddir, it'll be caught by "play".
		 */
		log_printf(LOG_DEBUG, "mod_save_message(): playlist jump detected.\n");

		if( (fd = eopen(MOD_SAVE_FILE, 0)) < 0) {
			return;
		}

		rc = readline(fd, type, sizeof(type));
		if (rc > 0) {
			log_printf(LOG_DEBUG, "mod_save_message(): type is '%s'\n", type);
			if (strncmp("list", type, 4) == 0) {
				/* a list */
				log_printf(LOG_DEBUG, "mod_save_message(): calling mod_save_write_list fd=%d, msg=%s, 0, 0\n", fd, msg);
				mod_save_write_list(fd, msg, 0, 0);
				close(fd);
				return;
			} else {
				/* a song */
				log_printf(LOG_DEBUG, "mod_save_message(): a song. doing nothing.\n");
				close(fd);
				return;
			}
		}
	} 			
}	

/* ************************************************************************
 * WRITE LIST TO FILE
 *   This will write/update list information to a file
 */

void mod_save_write_list(int fd, char *msg, int position, int update_time) {
	char pathname[512], positionc[10], jumpc[10];
	int  jump = 0, positionb = 0;

	/*
	 * Read pathname, then position, then jump.
	 */
	readline(fd, pathname, sizeof(pathname));
	readline(fd, positionc, sizeof(positionc));
	readline(fd, jumpc, sizeof(jumpc));

	close(fd);
		
	/*
	 * If we are NOT updating the time, calculate how many to jump and save
	 * to file. Otherwise, see if it's a repeated "time" message from mpg123,
	 * if not, save info to file.
	 */
	if(update_time == 0) {
		if( (fd = eopen(MOD_SAVE_FILE, 1)) < 0) {
			return;
		}

		jump = atoi(jumpc) + atoi( msg + 14 ); 
		log_printf(LOG_NOISYDEBUG, 
			   "mod_save_write_list(): calculated jump=%d . jumpc=%s, atoijc=%d, atoi2=%d \n", jump, jumpc, atoi(jumpc), atoi(msg+14));
		sendtext(fd, "list\n%s\n%d\n%d", trim(pathname), position, jump); 
		close(fd);

	} else {

		log_printf(LOG_NOISYDEBUG, "mod_save_write_list(): updating time. jumpc=%s, atoijc=%d, jump=%d \n", jumpc, atoi(jumpc), jump);
		jump = atoi(jumpc);
		positionb = atoi(positionc);

		if (position == positionb) {
			/* repeated time message. No need to save to file.*/
			log_printf(LOG_NOISYDEBUG, 
		 		   "mod_save_write_list(): repeated time %d. doing nothing.\n", positionb);
		} else {
			/* non-repeated time message. Save to file. */
			if( (fd = eopen(MOD_SAVE_FILE, 1)) < 0) {
				return;
			}

			log_printf(LOG_NOISYDEBUG, 
				   "mod_save_write_list(): saving 'list %s %d %d' to file\n", trim(pathname), position, jump);
			sendtext(fd, "list\n%s\n%d\n%d", trim(pathname), position, jump); 
			close(fd);
		}
	}
}

/* ************************************************************************
 * WRITE SONG TO FILE
 *   This will write/update song information to a file
 */

void mod_save_write_song(int fd, char *msg, int position) {
	char pathname[512];

	readline(fd, pathname, sizeof(pathname));

	close(fd);
	if( (fd = eopen(MOD_SAVE_FILE, 1)) < 0) {
		return;
	}
		
	log_printf(LOG_NOISYDEBUG, 
		   "mod_save_write_song(): saving 'song %s %d' to file\n", trim(pathname), position);
	sendtext(fd, "song\n%s\n%d", trim(pathname), position); 

}


/* ************************************************************************
 * MODULE INIT FUNCTION
 *   This is called right after starting IRMP3 and should set up global
 *   variables and register your module via mod_register().  It should return
 *   NULL if everything went ok, or an error string with the error description.
 */
char *mod_save_init (void)
{
	int fd, rc;

	log_printf(LOG_DEBUG, "mod_save_init(): initializing\n");

	/* Register our module with the main program so that it knows what we
	 * want to handle. We must provide a pointer to a mod_t variable which
	 * is valid during the whole runtime, so don't specify a local variable
	 * here.
	 */
	mod_register(&mod_save);

	// Get values from config file
	MOD_SAVE_ENABLED_P = config_getnum("mod_save_enable", 1);
	MOD_SAVE_FILE = config_getstr("mod_save_file", NULL);
	/*
	 * Send a message saying that we are in init mode. We will
	 * intercept this message and set the global variable MOD_SAVE_INIT
	 * to the appropriate value. We do this because apparently this function
	 * ends before the messages we send arrive back to us.
	 */
	mod_sendmsgf(MSGTYPE_GENERIC, "mod_save init 1");
	log_printf(LOG_NOISYDEBUG, 
		   "mod_save_init(): Enabled? %d, File: %s\n", MOD_SAVE_ENABLED_P, MOD_SAVE_FILE);

	/* 
	 * First let's check if we are enabled. If we are, try to start playing
	 * the last song.
	 */
	if (!MOD_SAVE_ENABLED_P) {
		return NULL;
	}

	if ( (fd = eopen(MOD_SAVE_FILE, 0)) >= 0) {
		char toplay[512], positionc[100], jumpc[10];
		int  position = 0, jump = 0;

		rc = readline(fd, toplay, sizeof(toplay));

		if (rc >= 0) {
			if (strncmp(toplay, "list", 4) == 0) {
				readline(fd, toplay, sizeof(toplay));
				readline(fd, positionc, sizeof(positionc));
				readline(fd, jumpc, sizeof(jumpc));

				jump = atoi(jumpc);
				position = atoi(positionc);

				log_printf(LOG_DEBUG, "mod_save_init: trying to play list %s, at position %d, and jump %d\n", toplay, position, jump);
				mod_sendmsgf(MSGTYPE_INPUT, "playlist load %s", toplay);

				/*
				 * Any other commands in this block will wait for the "play" message
				 * from the player to be executed. See mod_save_message.
				 */
				if (jump > 0) {
					if((MOD_SAVE_COMMAND_0 = malloc (1024)) == NULL) {
						printf ("error\n");
						exit(1);
					}
					sprintf(MOD_SAVE_COMMAND_0, "playlist jump +%d", jump);
				} else {
					if((MOD_SAVE_COMMAND_0 = malloc (1024)) == NULL) {
						printf ("error\n");
						exit(1);
					}
					sprintf(MOD_SAVE_COMMAND_0, "playlist jump -%d", jump);
				}

				if (position != 0) {
					if((MOD_SAVE_COMMAND_1 = malloc (1024)) == NULL) {
						printf ("error\n");
						exit(1);
					}
					sprintf(MOD_SAVE_COMMAND_1, "seek +%d", position);
				}

			} else {
				readline(fd, toplay, sizeof(toplay));
				readline(fd, positionc, sizeof(positionc));

				position = atoi(positionc);

				log_printf(LOG_DEBUG, "mod_save_init: trying to play song %s, at position %d\n", toplay, position);
				mod_sendmsgf(MSGTYPE_INPUT, "play %s", toplay);
				/*
				 * The other command in this block will wait for the "play" message
				 * from the player to be executed. See mod_save_message.
				 */

				if(position != 0) {
					if((MOD_SAVE_COMMAND_1 = malloc (1024)) == NULL) {
						printf ("error\n");
						exit(1);
					}
					sprintf(MOD_SAVE_COMMAND_1, "seek +%d", position);
				}
			}
		}
	}

	close(fd);
	/*
	 * Now that we are done, send a message saying we're not in init anymore
	 */
	mod_sendmsgf(MSGTYPE_GENERIC, "mod_save init 0");
	return NULL;
}


/* ************************************************************************
 * MODULE DEINIT FUNCTION
 *   This is called right before IRMP3 shuts down. You should clean up all your
 *   used data here to prepare for a clean exit.
 */
void mod_save_deinit (void)
{
	MOD_SAVE_ENABLED_P = 0;
	MOD_SAVE_FILE = NULL;
	MOD_SAVE_COMMAND_0 = NULL;
	MOD_SAVE_COMMAND_1 = NULL;

	log_printf(LOG_DEBUG, "mod_save_deinit(): deinitialized\n");
}


/* ************************************************************************
 * MODULE RELOAD FUNCTION
 *   This is called whenever IRMP3 needs to reload all configs (usually when
 *   receiving a SIGHUP). You should reinit your module or check for changed
 *   configuration here so that changes take effect.
 */
char *mod_save_reload (void)
{
	log_printf(LOG_DEBUG, "mod_save_reload(): reloading\n");
	MOD_SAVE_ENABLED_P = config_getnum("MOD_SAVE_ENABLE", 1);
	MOD_SAVE_FILE = config_getstr("MOD_SAVE_FILE", NULL);

	return NULL;
}


/* ************************************************************************
 * EOF
 */
