/*************************************************************************
 * $Id: mod_playlist.c,v 1.16 2001/05/17 16:49:50 dpotter Exp $
 *
 * mod_playlist.c -- Playlist support
 *
 * Copyright (C) by Andreas Neuhaus <andy@fasta.fh-dortmund.de>
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <fcntl.h>
#include <time.h>
#include <dirent.h>
#include <fnmatch.h>
#include <sys/types.h>
#include <sys/stat.h>

#include "tools.h"
#include "log.h"
#include "mod.h"
#include "mod_playlist.h"
#include "config.h"

/*************************************************************************
 * GLOBALS
 */
mod_playlist_t *mod_playlist_list = NULL;
mod_playlist_t *mod_playlist_shufflelist = NULL;
mod_playlist_t *mod_playlist_current = NULL;
int mod_playlist_shuffle = 0;
int mod_playlist_repeat = 2;
int mod_playlist_skip_back_timeout = 2;

/*************************************************************************
 * MODULE INFO
 */
mod_t mod_playlist = {
	mod_playlist_deinit,	// deinit
	NULL,			// reload
	&blank_fd,		// watchfd
	NULL,			// poll
	NULL,			// update
	mod_playlist_message,	// message
	NULL,			// SIGCHLD handler
};


/*************************************************************************
 * CLEAR PLAYLIST
 */
void mod_playlist_free (void)
{
	mod_playlist_t *p,*n;

	if(mod_playlist_list)
	{
	  for (p=mod_playlist_list; p; p=n)
	  {
	     n=p->next;    
	     free(p);
	  }
    }
	mod_playlist_list = NULL;
	mod_playlist_shufflelist = NULL;
	mod_playlist_current = NULL;
	log_printf(LOG_DEBUG, "mod_playlist_free(): cleared the playlist\n");
}


/*************************************************************************
 * ADD TO PLAYLIST
 */
int mod_playlist_add (char *song)
{
	mod_playlist_t *lastentry, *newentry;

	// find last playlist entry
	lastentry = mod_playlist_list;
	while (lastentry && lastentry->next)
		lastentry = lastentry->next;

	// setup new playlist entry
	newentry = malloc(sizeof(mod_playlist_t));
	if (!newentry)
		return -1;
	strncpy(newentry->song, song, sizeof(newentry->song)-1);
	newentry->song[sizeof(newentry->song)-1] = 0;
	newentry->next = NULL;
	newentry->prev = lastentry;
	newentry->shufflenext = NULL;
	newentry->shuffleprev = NULL;

	// link new entry into playlist chain
	lastentry ? lastentry->next : mod_playlist_list = newentry;
	if (!mod_playlist_current)
		mod_playlist_current = mod_playlist_list;
	log_printf(LOG_NOISYDEBUG, "mod_playlist_add(): added '%s'\n", newentry->song);

	return 0;
}


/*************************************************************************
 * LOAD PLAYLIST
 */
int mod_playlist_load (char *name)
{
	char *c, *basename, root[512], song[512], buf[512];
	int fd, count;

	if (!name || !*name)
		return -1;
	
	// get root path of playlist
	strncpy(root, name, sizeof(root)-1);
	c = strrchr(root, '/');
	if (c)
		*c = 0;
	else
		getcwd(root, sizeof(root)-1);

	// read playlist file
	fd = open(name, O_RDONLY);
	if (fd < 0) {
		log_printf(LOG_DEBUG, "mod_playlist_load(): unable to open playlist '%s'\n", name);
		return -2;
	}
	count = 0;
	while (readline(fd, song, sizeof(song)) >= 0) {
		trim(song);
		log_printf(LOG_DEBUG, "mod_playlist_load(): song is now '%s'\n", song);
		if (*song != '/') {
			snprintf(buf, sizeof(buf)-1, "%s/%s", root, song);
			mod_playlist_add(buf);
		} else
			mod_playlist_add(song);
		count++;
	}
	close(fd);

	log_printf(LOG_DEBUG, "mod_playlist_load(): added %d songs\n", count);

/*	basename=(void *) malloc (sizeof *name);
	if((basename=rindex(name,(int) '/'))) basename++;
		else strcpy(basename,name);
*/
	if ((basename = rindex(name, (int) '/'))) mod_sendmsgf(MSGTYPE_INPUT, "loadplaylist %s", (basename++));
				             else mod_sendmsgf(MSGTYPE_INPUT, "loadplaylist %s", name);
	return count;
}


/*************************************************************************
 * LOAD PLAYLIST FROM DIRECTORY
 */
int mod_playlist_loaddir_scan (char *root, char *subdir, char *pattern)
{
	DIR *dir;
	struct dirent *dirent;
	struct stat st;
	char buf[512], dirs[100][512];
	int count, dircount, i;

	// Scan the directory
	snprintf(buf, sizeof(buf)-1, "%s%s", root, subdir);
	log_printf(LOG_DEBUG, "mod_playlist_loaddir_scan(): scanning dir '%s'\n", buf);
	dir = opendir(buf);
	if (!dir) {
		log_printf(LOG_DEBUG, "mod_playlist_loaddir_scan(): unable to read '%s'!\n", buf);
		return 0;
	}
	count = 0;
	dircount = 0;
	while (1) {
		dirent = readdir(dir);
		if (!dirent)
			break;
		if (!strcmp(dirent->d_name, ".") || !strcmp(dirent->d_name, ".."))
			continue;
		snprintf(buf, sizeof(buf)-1, "%s%s%s", root, subdir, dirent->d_name);
		if (stat(buf, &st)) {
			log_printf(LOG_DEBUG, "mod_playlist_loaddir_scan(): cannot stat '%s', skipping!\n", buf);
			continue;
		}
		if (S_ISDIR(st.st_mode)) {
			strcpy(dirs[dircount], dirent->d_name);
			dircount++;
		}
		// Add song if is matches our search string
		snprintf(buf, sizeof(buf)-1, "%s%s", subdir, dirent->d_name);
		if (!fnmatch(pattern, buf, FNM_NOESCAPE)) {
			snprintf(buf, sizeof(buf)-1, "%s%s%s", root, subdir, dirent->d_name);
			count++;
			mod_playlist_add(buf);
		}
	}
	closedir(dir);

	// process subdirs
	for (i=0; i<dircount; i++) {
		snprintf(buf, sizeof(buf)-1, "%s%s/", subdir, dirs[i]);
		count += mod_playlist_loaddir_scan(root, buf, pattern);
	}

	return count;
}

int mod_playlist_loaddir (char *pattern)
{
	char root[512], search[512], buf[512];
	char *c;
	int count;

	if (!pattern || !*pattern)
		return -2;

	strncpy(root, pattern, sizeof(root)-1);
	c = strchr(root, '*');
	if (!c) {
		log_printf(LOG_DEBUG, "mod_playlist_loaddir(): no wildcard specified!\n");
		return -1;
	}
	strcpy(buf, c);
	*c = 0;
	c = strrchr(root, '/');
	if (!c) {
		log_printf(LOG_DEBUG, "mod_playlist_loaddir(): no root path!\n");
		return -1;
	}
	strncpy(search, c+1, sizeof(search)-1);
	strncat(search, buf, sizeof(search)-1);
	*c = 0;
	log_printf(LOG_DEBUG, "mod_playlist_loaddir(): Searching for '%s' in '%s'\n", search, root);

	count = mod_playlist_loaddir_scan(root, "/", search);
	log_printf(LOG_DEBUG, "mod_playlist_loaddir(): Added %d songs\n", count);
	return count;
}


/*************************************************************************
 * SKIP TO SONG (relative)
 */
int mod_playlist_skip (int offset, int quiet)
{
	int i, count;

	if (!offset || !mod_playlist_current || mod_playlist_repeat==1)
		return 0;
	count = 0;

	// skip forward
	if (offset > 0)
		for (i=0; i<offset; i++) {
			if (!(mod_playlist_shuffle ? mod_playlist_current->shufflenext : mod_playlist_current->next) && mod_playlist_repeat==2)
				mod_playlist_current = mod_playlist_shuffle ? mod_playlist_shufflelist : mod_playlist_list;
			else
				mod_playlist_current = mod_playlist_shuffle ? mod_playlist_current->shufflenext : mod_playlist_current->next;
			count++;
			if (!mod_playlist_current)
				break;
		}

	// skip backwards
	else if (offset < 0)
		for (i=0; i<-offset; i++) {
			if (!(mod_playlist_shuffle ? mod_playlist_current->shuffleprev : mod_playlist_current->prev) && mod_playlist_repeat==2) {
				mod_playlist_current = mod_playlist_shuffle ? mod_playlist_shufflelist : mod_playlist_list;
				while (mod_playlist_current && (mod_playlist_shuffle ? mod_playlist_current->shufflenext : mod_playlist_current->next))
					mod_playlist_current = (mod_playlist_shuffle ? mod_playlist_current->shufflenext : mod_playlist_current->next);
			} else
				mod_playlist_current = mod_playlist_shuffle ? mod_playlist_current->shuffleprev : mod_playlist_current->prev;
			count--;
			if (!mod_playlist_current)
				break;
		}

	if (!quiet)
		log_printf(LOG_NORMAL, "Playlist skipped %d songs\n", count);
	return count;
}


/*************************************************************************
 * SKIP TO DIRECTORY IN PLAYLIST
 */ 
int mod_playlist_skipdir (int offset)
{
	int i, count, dircount;
	mod_playlist_t *current;
	char dir[512], buf[512], *p;

	if (!offset || !mod_playlist_current || mod_playlist_repeat==1)
		return 0;
	count = 0;
	dircount = 0;

	// remember current song
	current = mod_playlist_current;

	// skip forward
	if (offset > 0)
		for (i=0; i<offset; i++) {
			strncpy(dir, mod_playlist_current->song, sizeof(dir)-1);
			p = strrchr(dir, '/');
			if (!p)
				p = dir;
			*p = 0;
			do {
				count += mod_playlist_skip(1, 1);
				if (!mod_playlist_current || mod_playlist_current == current)
					break;
				strncpy(buf, mod_playlist_current->song, sizeof(buf)-1);
				p = strrchr(buf, '/');
				if (!p)
					p = buf;
				*p = 0;
			} while (!strcmp(buf, dir));
			dircount++;
			if (!mod_playlist_current)
				break;
		}

	// skip backwards
	else if (offset < 0)
	{
		for (i=-1; i<-offset; i++) {
			strncpy(dir, mod_playlist_current->song, sizeof(dir)-1);
			p = strrchr(dir, '/');
			if (!p)
				p = dir;
			*p = 0;
			do {
				count += mod_playlist_skip(-1, 1);
				if (!mod_playlist_current || mod_playlist_current == current)
					break;
				strncpy(buf, mod_playlist_current->song, sizeof(buf)-1);
				p = strrchr(buf, '/');
				if (!p)
					p = buf;
				*p = 0;
			} while (!strcmp(buf, dir));
			dircount--;
			if (!mod_playlist_current)
				break;
		}
		dircount++;
		count += mod_playlist_skip(1, 1);
	}

	log_printf(LOG_NORMAL, "Playlist skipped %d songs (%d directories)\n", count, dircount);
	return count;
}


/*************************************************************************
 * SETUP SHUFFLED PLAYLIST
 */
void mod_playlist_shufflecalc (int modifycurrent)
{
	mod_playlist_t *p;
	int i, j, count;

	if (!mod_playlist_shuffle)
		return;

	// use normal pointers first
	count = 0;
	mod_playlist_shufflelist = mod_playlist_list;
	for (p=mod_playlist_list; p; p=p->next) {
		p->shufflenext = p->next;
		p->shuffleprev = p->prev;
		count++;
	}

	// setup shuffle pointers
	srand(time(NULL));
	p = mod_playlist_shufflelist;
	for (i=0; i<count*5; i++) {
		// select a random entry
		for (j=(float)rand()*count/RAND_MAX+1; j>=0; j--)
			p = p->next ? p->next : mod_playlist_list;
		// move entry to beginning
		if (p != mod_playlist_shufflelist) {
			if (p->shuffleprev)
				p->shuffleprev->shufflenext = p->shufflenext;
			if (p->shufflenext)
				p->shufflenext->shuffleprev = p->shuffleprev;
			p->shuffleprev = NULL;
			mod_playlist_shufflelist->shuffleprev = p;
			p->shufflenext = mod_playlist_shufflelist;
			mod_playlist_shufflelist = p;
		}
	}

	// set current song if desired
	if (modifycurrent)
		mod_playlist_current = mod_playlist_shufflelist;

	log_printf(LOG_DEBUG, "mod_playlist_shuffle(): set up shuffled playlist\n");
}


/*************************************************************************
 * RECEIVE MESSAGE
 */
void mod_playlist_message (int msgtype, char *rawmsg)
{
	static int secpos, secremain;
	char *c1, *c2, *c3,msg[512];
	mod_playlist_t *n;
	int i,c;

	strncpy(msg,rawmsg,sizeof(msg));	// pad msg with nulls

	// handle input messages
	if (msgtype == MSGTYPE_INPUT) {
		c1 = msg ? strtok(msg, " \t") : NULL;
		c2 = c1 ? strtok(NULL, " \t") : NULL;
		c3 = c2 ? c2 + strlen(c2)+1 : NULL;
		if (c3 && c3[0] == 0) c3 = NULL;

		// process playlist commands
		if (c1 && !strcasecmp(c1, "playlist")) {
			// load playlist from file
			if (c2 && !strcasecmp(c2, "load") && c3) {
				mod_playlist_free();
				log_printf(LOG_NORMAL, "Loading playlist '%s'\n", c3);
				if (mod_playlist_load(c3) < 0)
					log_printf(LOG_ERROR, "Unable to load playlist!\n");
				else {
					mod_playlist_shufflecalc(1);
					mod_sendmsgf(MSGTYPE_INPUT, "play %s", mod_playlist_current->song);
				}

			// load playlist by scanning dirs
			} else if (c2 && !strcasecmp(c2, "loaddir") && c3) {
				mod_playlist_free();
				log_printf(LOG_NORMAL, "Loading playlist from pattern '%s'\n", c3);
				if (mod_playlist_loaddir(c3) < 0)
					log_printf(LOG_ERROR, "Unable to load playlist!\n");
				else {
					mod_playlist_shufflecalc(1);
					mod_sendmsgf(MSGTYPE_INPUT, "play %s", mod_playlist_current->song);
				}

                        // add song to playlist
 			} else if (c2 && !strcasecmp(c2, "add") && c3) {
 				log_printf(LOG_NORMAL, "add song '%s'\n", c3);
 				if (mod_playlist_add(c3) < 0)
 					log_printf(LOG_ERROR, "Unable to add!\n");
 			} else if (c2 && !strcasecmp(c2, "done")) {
 				log_printf(LOG_NORMAL, "done\n");
 				mod_playlist_shufflecalc(1);
 				mod_sendmsgf(MSGTYPE_INPUT, "play %s", mod_playlist_current->song);

			// clear playlist
			} else if (c2 && !strcasecmp(c2, "clear")) {
				mod_sendmsg(MSGTYPE_INPUT, "stop");
				mod_playlist_free();

			// jump/skip within the playlist
			} else if (c2 && !strcasecmp(c2, "jump") && c3) {
				if (*c3!='+' && *c3!='-')
					mod_playlist_current = mod_playlist_list;
				if 		(*c3 == '-') 	i=atoi(c3);
				else if (*c3 == '+') 	i=atoi(c3+1);
				else 					i=atoi(c3);
				if (i != -1 || secpos <= mod_playlist_skip_back_timeout || mod_playlist_skip_back_timeout == -1)
					mod_playlist_skip(i, 0);
				if (mod_playlist_current)
					mod_sendmsgf(MSGTYPE_INPUT, "play %s", mod_playlist_current->song);
				else
					mod_sendmsg(MSGTYPE_INPUT, "stop");

			// jump/skip within playlist directories
			} else if (c2 && !strcasecmp(c2, "jumpdir") && c3) {
				if (*c3!='+' && *c3!='-')
					mod_playlist_current = mod_playlist_list;
				if 		(*c3 == '-') 	i=atoi(c3);
				else if (*c3 == '+') 	i=atoi(c3+1);
				else 					i=atoi(c3);
				mod_playlist_skipdir(i);
				if (mod_playlist_current)
					mod_sendmsgf(MSGTYPE_INPUT, "play %s", mod_playlist_current->song);
				else
					mod_sendmsg(MSGTYPE_INPUT, "stop");

			// toggle shuffle mode
			} else if (c2 && !strcasecmp(c2, "shuffle")) {
				if (!c3)
					mod_playlist_shuffle = mod_playlist_shuffle ? 0 : 1;
				else
					mod_playlist_shuffle = atoi(c3);
				mod_playlist_shuffle %= 2;
				if (mod_playlist_shuffle)
					mod_playlist_shufflecalc(0);
				mod_sendmsgf(MSGTYPE_GENERIC, "shuffle %d", mod_playlist_shuffle);
				log_printf(LOG_NORMAL, "Shuffle %s\n", mod_playlist_shuffle ? "on" : "off");

			// toggle repeat mode
			} else if (c2 && !strcasecmp(c2, "repeat")) {
				if (!c3)
					mod_playlist_repeat++;
				else
					mod_playlist_repeat = atoi(c3);
				mod_playlist_repeat %= 3;
				mod_sendmsgf(MSGTYPE_GENERIC, "repeat %d", mod_playlist_repeat);
				log_printf(LOG_NORMAL, "Repeat %s\n", mod_playlist_repeat==0 ? "off" : (mod_playlist_repeat==1 ? "current" : "all"));
			}
		} else if (c1 && !strcasecmp(c1, "query")) {
			if (c2 && !strcasecmp(c2,"plpos")) {
				if (!mod_playlist_list) {
					mod_sendmsgf(MSGTYPE_PLAYER,"info plpos: 0 0");
				} else {
					i=1;
					c=0;
					n=mod_playlist_list;
					while(n) {
						if (n==mod_playlist_current) c=i;
						i++;
						n=n->next;
					}
					mod_sendmsgf(MSGTYPE_PLAYER,"info plpos: %d %d",c,i-1);
				}
			}
		}

	// handle player messages
	} else if (msgtype == MSGTYPE_PLAYER) {
		c1 = msg ? strtok(msg, " \t") : NULL;
		c2 = c1 ? strtok(NULL, " \t") : NULL;
		c3 = c2 ? strtok(NULL, " \t") : NULL;

		// reset times
		if (c1 && !strcasecmp(c1, "stop")) {
			secpos = 0;
			secremain = 0;

		// skip to next song when the current song ends
		} else if (c1 && !strcasecmp(c1, "endofsong")) {
			mod_playlist_skip(1, 0);
			if (mod_playlist_current)
				mod_sendmsgf(MSGTYPE_INPUT, "play %s", mod_playlist_current->song);
			else
				mod_sendmsgf(MSGTYPE_INPUT, "stop");

		// we need to keep track of the song time for
		// playlist skip -1
		} else if (c1 && !strcasecmp(c1, "time")) {
			secpos = c2 ? atoi(c2) : 0;
			secremain = c3 ? atoi(c3) : 0;
		}
	}
}


/*************************************************************************
 * MODULE INIT FUNCTION
 */
char *mod_playlist_init (void)
{
	log_printf(LOG_DEBUG, "mod_playlist_init(): initializing\n");

	// register our module
	mod_register(&mod_playlist);

	// get repeat-mode from config file. defaulting to 2 = all
	mod_playlist_repeat = 
	  strcasecmp("off",config_getstr("playlist_repeat", "all"))==0?0:
	  strcasecmp("current",config_getstr("playlist_repeat", "all"))==0?1:2;

	// after playlist_skip_back_timeout seconds "jump -1" will 
	// jump to the start of the current song instead of the previous song.
	// set the variable to -1 to disable this behaviour.
	mod_playlist_skip_back_timeout = atoi(config_getstr("playlist_skip_back_timeout", "2"));

	// tell other modules what repeat-mode we are in.
	mod_sendmsgf(MSGTYPE_GENERIC, "repeat %d", mod_playlist_repeat);

	return NULL;
}


/*************************************************************************
 * MODULE DEINIT FUNCTION
 */
void mod_playlist_deinit (void)
{
	// clear up playlist
	mod_playlist_free();
	log_printf(LOG_DEBUG, "mod_playlist_deinit(): deinitialized\n");
}


/*************************************************************************
 * EOF
 */
