/*************************************************************************
 * $Id: mod_mixer.c,v 1.8 2001/04/13 01:16:54 dpotter Exp $
 *
 * mod_mixer.h -- IRMP3 module for soundcard mixer
 *
 * Copyright (C) by Andreas Neuhaus <andy@fasta.fh-dortmund.de>
 *
 */
 
#include <stdio.h>
#include <stdlib.h>
#include <fcntl.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>
#include <sys/ioctl.h>
#include <linux/soundcard.h>

#include "tools.h"
#include "log.h"
#include "config.h"
#include "mod.h"
#include "mod_mixer.h"


/*************************************************************************
 * GLOBALS
 */
int mod_mixer_balance = 0;		// balance level (-50..50)
char mod_mixer_vol_channel[10];	// mixer channel for volume ("vol")


/*************************************************************************
 * MODULE INFO
 */
mod_t mod_mixer = {
	mod_mixer_deinit,	// deinit
	NULL,			// reload
	&blank_fd,		// watchfd
	NULL,			// poll
	NULL,			// update
	mod_mixer_message,	// message
	NULL,			// SIGCHLD handler
};


/*************************************************************************
 * FIND A MIXER DEVICE
 */
int mod_mixer_finddev (char *chan)
{
	char *devs[] = SOUND_DEVICE_NAMES;
	int i;

	// search the number of a named mixer device
	for (i=0; i<SOUND_MIXER_NRDEVICES; i++)
		if (!strcasecmp(devs[i], chan))
			return i;
	return -1;
}


/*************************************************************************
 * GET A VALUE FROM MIXER DEVICE
 */
int mod_mixer_get (char *chan, int *llevel, int *rlevel)
{
	int fd, dev;
	int i, stereo;

	// search the device
	dev = mod_mixer_finddev(chan);
	if (dev < 0) {
		log_printf(LOG_DEBUG, "mod_mixer_get(): Mixer device '%s' doesn't exist\n", chan);
		return -1;
	}

	// open mixer
	fd = open(MOD_MIXER_DEV, O_RDONLY);
	if (fd < 0) {
		log_printf(LOG_DEBUG, "mod_mixer_get(): Error opening mixer: %s\n", strerror(errno));
		return -1;
	}

	// check channel
	if (ioctl(fd, SOUND_MIXER_READ_DEVMASK, &i) < 0) {
		log_printf(LOG_DEBUG, "mod_mixer_get(): read devmask failed: %s\n", strerror(errno));
		return -2;
	}
	if (!(i & (1<<dev))) {
		log_printf(LOG_DEBUG, "mod_mixer_get(): Mixer device %s is not available!\n", chan);
		return -3;
	}
	if (ioctl(fd, SOUND_MIXER_READ_STEREODEVS, &i) < 0) {
		log_printf(LOG_DEBUG, "mod_mixer_get(): read stereodevs failed: %s\n", strerror(errno));
		return -2;
	}
	stereo = (i & (1<<dev)) ? 1 : 0;

	// read level
	if (ioctl(fd, MIXER_READ(dev), &i) < 0) {
		log_printf(LOG_DEBUG, "mod_mixer_get(): mixer read failed: %s\n", strerror(errno));
		return -2;
	}
	if (stereo) {
		*llevel = i & 0xff;
		*rlevel = (i & 0xff00) >> 8;
	} else {
		*llevel = i & 0xff;
		*rlevel = i & 0xff;
	}

	// close mixer
	close(fd);

	// okay
	log_printf(LOG_NOISYDEBUG, "mod_mixer_get(): '%s' reads %d/%d\n", chan, *llevel, *rlevel);
	return 0;
}


/*************************************************************************
 * SET A VALUE TO MIXER DEVICE
 */
int mod_mixer_set (char *chan, int llevel, int rlevel)
{
	int fd, dev;
	int i, stereo;

	// search the device
	dev = mod_mixer_finddev(chan);
	if (dev < 0) {
		log_printf(LOG_DEBUG, "mod_mixer_set(): Mixer device '%s' doesn't exist\n", chan);
		return -1;
	}

	// open mixer
	fd = open(MOD_MIXER_DEV, O_RDONLY);
	if (fd < 0) {
		log_printf(LOG_DEBUG, "mod_mixer_set(): Error opening mixer: %s\n", strerror(errno));
		return -1;
	}

	// check channel
	if (ioctl(fd, SOUND_MIXER_READ_DEVMASK, &i) < 0) {
		log_printf(LOG_DEBUG, "mod_mixer_set(): read devmask failed: %s\n", strerror(errno));
		return -2;
	}
	if (!(i & (1<<dev))) {
		log_printf(LOG_DEBUG, "mod_mixer_set(): Mixer device %s is not available!\n", chan);
		return -3;
	}
	if (ioctl(fd, SOUND_MIXER_READ_STEREODEVS, &i) < 0) {
		log_printf(LOG_DEBUG, "mod_mixer_set(): read stereodevs failed: %s\n", strerror(errno));
		return -2;
	}
	stereo = (i & (1<<dev)) ? 1 : 0;

	// check level
	llevel = MIN(MAX(llevel, 0), 100);
	rlevel = MIN(MAX(rlevel, 0), 100);

	// set level
	if (stereo)
		i = (rlevel << 8) + llevel;
	else
		i = llevel;
	if (ioctl(fd, MIXER_WRITE(dev), &i) < 0) {
		log_printf(LOG_DEBUG, "mod_mixer_set(): mixer write failed: %s\n", strerror(errno));
		return -2;
	}

	// close mixer
	close(fd);

	// okay
	log_printf(LOG_DEBUG, "mod_mixer_set(): '%s' set to %d/%d\n", chan, llevel, rlevel);
	return 0;
}


/*************************************************************************
 * SET THE VOLUME MIXER CHANNEL
 */
int mod_mixer_setvolume (int value, int balance)
{
	int llevel, rlevel;


	// get old levels
	if (value < 0) {
		if (mod_mixer_get(mod_mixer_vol_channel, &llevel, &rlevel))
			return -1;
		value = MAX(llevel, rlevel);
	}

	// check levels
	value = MIN(MAX(value, 0), 100);
	balance = MIN(MAX(balance, -50), 50);
	log_printf(LOG_NOISYDEBUG, "mod_mixer_setvolume(): set vol %d bal %d\n", value, balance);

	// calculate new levels
	llevel = (balance > 0) ? (value * (50-abs(balance)) / 50) : value;
	rlevel = (balance < 0) ? (value * (50-abs(balance)) / 50) : value;

	// set new levels
	if (mod_mixer_set(mod_mixer_vol_channel, llevel, rlevel))
		return -1;

	return 0;
}


/*************************************************************************
 * SET DEFAULT SETTINGS
 */
void mod_mixer_default (void)
{
	int val;

	val = config_getnum("mixer_volume", -1);
	mod_mixer_balance = config_getnum("mixer_balance", 0);
	mod_mixer_setvolume(val, mod_mixer_balance);
	if ((val = config_getnum("mixer_bass", -1)) >= 0)
		mod_mixer_set("bass", val, val);
	if ((val = config_getnum("mixer_treble", -1)) >= 0)
		mod_mixer_set("treble", val, val);
}


/*************************************************************************
 * RECEIVE MESSAGE
 */
void mod_mixer_message (int msgtype, char *rawmsg)
{
	char *c1, *c2, *c3;
	int llevel, rlevel, level;
	char msg[512];
	int percent;
	static int mute=0,beforemutevol=0;
	
	strncpy(msg,rawmsg,sizeof(msg));	// pad msg with nulls
	
	// handle input messages
	if (msgtype == MSGTYPE_INPUT) {
		c1 = msg ? strtok(msg, " \t") : NULL;
		c2 = c1 ? strtok(NULL, " \t") : NULL;
		c3 = c2 ? c2 + strlen(c2)+1 : NULL;
		if (c3 && c3[0] == 0) c3 = NULL;
		
		if (c1 && !strcasecmp(c1, "mixer")) {
			if (c2 && !strcasecmp(c2, "volume") && c3) {
				log_printf(LOG_DEBUG, "mod_mixer_message(): got volume setting '%s'\n", c3);
				if (mute) {
					mute=0;
					mod_sendmsg(MSGTYPE_MIXER,"mute 0");
					if (beforemutevol) {
						level=beforemutevol;
						level = MIN(MAX(level, 0), 100);
						mod_mixer_setvolume(level, mod_mixer_balance);
						beforemutevol=0;
					}
				}
				percent = (strlen(c3) && c3[strlen(c3)-1]=='%') ? 1 : 0;
				if (*c3=='+' || *c3=='-') {
					mod_mixer_get(mod_mixer_vol_channel, &llevel, &rlevel);
					level = MAX(llevel, rlevel);
					if (percent)
						level += level * ((*c3=='-') ? atoi(c3) : atoi(c3+1)) / 100;
					else
						level += (*c3=='-') ? atoi(c3) : atoi(c3+1);
				} else
					level = atoi(c3);
				level = MIN(MAX(level, 0), 100);
				mod_mixer_setvolume(level, mod_mixer_balance);
				mod_sendmsgf(MSGTYPE_MIXER, "volume %d", level);
			} else if (c2 && !strcasecmp(c2,"mute")) {
				log_printf(LOG_DEBUG,"mod_mixer_message(): received mute command.\n");
				if (c3) mute=atoi(c3);
				else mute ? mute-- : mute++;
				
				if (mute == 1 ) {
					if (!beforemutevol) {
						mod_mixer_get(mod_mixer_vol_channel, &llevel, &rlevel);
						beforemutevol=MAX(llevel,rlevel);
					}
					level = 0;
					mod_sendmsg(MSGTYPE_MIXER, "mute 1");
				} else {
					if (beforemutevol) {
						level=beforemutevol;
						beforemutevol=0;
					}
					else {
						mod_mixer_get(mod_mixer_vol_channel, &llevel, &rlevel);
						level=MAX(llevel,rlevel);
					}
					mod_sendmsg(MSGTYPE_MIXER, "mute 0");
				}
				level = MIN(MAX(level, 0), 100);
				mod_mixer_setvolume(level, mod_mixer_balance);
				mod_sendmsgf(MSGTYPE_MIXER, "volume %d", level); 

			} else if (c2 && !strcasecmp(c2, "balance") && c3) {
				log_printf(LOG_DEBUG, "mod_mixer_message(): got balance setting '%s'\n", c3);
				if (*c3=='+' || *c3=='-')
					mod_mixer_balance += (*c3=='-') ? atoi(c3) : atoi(c3+1);
				else 
					mod_mixer_balance = atoi(c3);
				mod_mixer_balance = MIN(MAX(mod_mixer_balance, -50), 50);
				mod_mixer_setvolume(-1, mod_mixer_balance);
				mod_sendmsgf(MSGTYPE_MIXER, "balance %d", mod_mixer_balance);

			} else if (c2 && !strcasecmp(c2, "bass") && c3) {
				log_printf(LOG_DEBUG, "mod_mixer_message(): got bass setting '%s'\n", c3);
				if (*c3=='+' || *c3=='-') {
					mod_mixer_get("bass", &llevel, &rlevel);
					level = MAX(llevel, rlevel);
					level += (*c3=='-') ? atoi(c3) : atoi(c3+1);
				} else
					level = atoi(c3);
				level = MIN(MAX(level, 0), 100);
				mod_mixer_set("bass", level, level);
				mod_sendmsgf(MSGTYPE_MIXER, "bass %d", level);

			} else if (c2 && !strcasecmp(c2, "treble") && c3) {
				log_printf(LOG_DEBUG, "mod_mixer_message(): got treble setting '%s'\n", c3);
				if (*c3=='+' || *c3=='-') {
					mod_mixer_get("treble", &llevel, &rlevel);
					level = MAX(llevel, rlevel);
					level += (*c3=='-') ? atoi(c3) : atoi(c3+1);
				} else
					level = atoi(c3);
				level = MIN(MAX(level, 0), 100);
				mod_mixer_set("treble", level, level);
				mod_sendmsgf(MSGTYPE_MIXER, "treble %d", level);

			} else if (c2 && !strcasecmp(c2, "default")) {
				mod_mixer_default();
			}
		} else if (c1 && !strcasecmp(c1,"query")) {
			if (c2 && !strcasecmp(c2,"volume")) {
				mod_mixer_get(mod_mixer_vol_channel, &llevel, &rlevel);
				level = MAX(llevel, rlevel);
				mod_sendmsgf(MSGTYPE_MIXER,"info volume %d",level);
			} else if (c2 && !strcasecmp(c2,"treble")) {
				if (0 <= mod_mixer_get("treble", &llevel, &rlevel)) {
					level = MAX(llevel, rlevel);
					mod_sendmsgf(MSGTYPE_MIXER,"info treble %d",level);
				}
			} else if (c2 && !strcasecmp(c2,"bass")) {
				if (0 <= mod_mixer_get("bass", &llevel, &rlevel)) {
					level = MAX(llevel, rlevel);
					mod_sendmsgf(MSGTYPE_MIXER,"info bass %d",level);
				}
			}
		}
	}
}
	

/*************************************************************************
 * MODULE INIT FUNCTION
 */
char *mod_mixer_init (void)
{
	log_printf(LOG_DEBUG,"mod_mixer_init(): initializing\n");

	// set standard mixer settings
	mod_mixer_default();

	// register our module
	mod_register(&mod_mixer);

	// find the mixer channel
	strcpy(mod_mixer_vol_channel,config_getstr("mixer_vol_channel","vol"));

	return NULL;
}


/*************************************************************************
 * MODULE DEINIT FUNCTION
 */
void mod_mixer_deinit (void)
{
	log_printf(LOG_DEBUG,"mod_mixer_deinit(): deinitialized\n");
}


/*************************************************************************
 * EOF
 */
