/*************************************************************************
 * $Id: mod.c,v 1.8 2000/04/04 23:37:33 dpotter Exp $
 *
 * mod.c -- The module handling functions
 *
 * Copyright (C) by Andreas Neuhaus <andy@fasta.fh-dortmund.de>
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <signal.h>
#include <string.h>
#include <sys/types.h>
#include <sys/wait.h>

#include "log.h"
#include "modules.inc"
#include "mod.h"

/*************************************************************************
 * GLOBALS
 */

fd_set	blank_fd;
/*************************************************************************
 * LIST OF ALL MODULES
 */
mod_t *mod_list = NULL;
mod_message_t *mod_msgs = NULL;


/*************************************************************************
 * REGISTER A NEW MODULE
 */
void mod_register (mod_t *newmod)
{
	newmod->next = mod_list;
	mod_list = newmod;
}


/*************************************************************************
 * SEND MESSAGE TO ALL MODULES
 */
void mod_sendmsg (int msgtype, char *msg)
{
	mod_message_t *newmsg, *m;

	// create new message entry
	newmsg = malloc(sizeof(mod_message_t));
	if (!newmsg)
		return;
	newmsg->msg = malloc(strlen(msg)+1);
	if (!newmsg->msg) {
		free(newmsg);
		return;
	}
	newmsg->msgtype = msgtype;
	strcpy(newmsg->msg, msg);
	newmsg->next = NULL;

	// append message to queue
	log_printf(LOG_NOISYDEBUG, "mod_sendmsg(): %d '%s'\n", newmsg->msgtype, newmsg->msg);
	for (m=mod_msgs; m && m->next; m=m->next);
	m ? m->next : mod_msgs = newmsg;
}

void mod_sendmsgf (int msgtype, char *msg, ...)
{
	va_list ap;
	char buf[1024];
	va_start(ap, msg);
	vsnprintf(buf, sizeof(buf)-1, msg, ap);
	buf[sizeof(buf)-1] = 0;
	va_end(ap);
	mod_sendmsg(msgtype, buf);
}


/*************************************************************************
 * INIT ALL MODULES
 */
void mod_init (void)
{
	mod_initfunc_t *mi;
	char *s;

	// call init function of all modules
	for (mi=mod_list_init; *mi; mi++) {
		s = (*mi)();
		if (s)
			log_printf(LOG_ERROR, "Error initializing module: %s\n", s);
	}
}


/*************************************************************************
 * DEINIT ALL MODULES
 */
void mod_deinit (void)
{
	mod_t *m;

	// call deinit function of all modules
	// and clean up module list
	for (m=mod_list; m; m=m->next)
		if (m->deinit)
			m->deinit();
	mod_list = NULL;
}


/*************************************************************************
 * RELOAD ALL MODULES
 */
void mod_reload (void)
{
	mod_t *m;
	char *s;

	// call reload function of all modules
	for (m=mod_list; m; m=m->next)
		if (m->reload) {
			s = m->reload();
			if (s)
				log_printf(LOG_ERROR, "Error reloading module: %s\n", s);
		}
}


/*************************************************************************
 * DISPERSE MESSAGES AND UPDATE ALL MODULES
 */
void mod_update (void)
{
	mod_t *m;
	mod_message_t *msg;
	mod_message_t msgdup;

	// process message queue
	while (mod_msgs) {
		msg = mod_msgs;

		// global messages we handle
		if (msg->msgtype == MSGTYPE_INPUT && !strcasecmp(msg->msg, "reload")) {
				raise(SIGHUP);

		// disperse message to modules, make a copy of the message
		// so that modules may alter it without affecting
		// other modules (think of strtok)
		} else {
			for (m=mod_list; m; m=m->next)
				if (m->message) {
					msgdup.msgtype = msg->msgtype;
					msgdup.msg = malloc(strlen(msg->msg)+1);
					strcpy(msgdup.msg, msg->msg);
					m->message(msgdup.msgtype, msgdup.msg);
					free(msgdup.msg);
				}
		}

		// remove old message from queue
		mod_msgs = mod_msgs->next;
		free(msg);
	}

	// call update function of all modules
	for (m=mod_list; m; m=m->next)
		if (m->update)
			m->update();
}

/*************************************************************************
 * HANDLE SIGCHLD CALLS FOR ALL MODULES
 */

void mod_sigchld(int signal)
{
	pid_t pid;
	mod_t *m;
	int status;


	// this process was called because a child of IRMP3 just died and
	// we received a SIGCHLD signal.  wait() won't block.
	pid = wait(&status);
	
	log_printf(LOG_DEBUG, "mod_sigchld(): notified that child %d died.\n", pid);

	// call child function of all modules
	for (m=mod_list;m;m=m->next)
		if (m->chld)
			m->chld(pid,status);
}



/*************************************************************************
 * EOF
 */
