#include <stdio.h>
#include <sys/types.h>
#include <errno.h>
#include <fcntl.h>
#include <linux/ipmi_ioctls.h>
#include <ipmi_ctl.h>
#include <sdr.h>
#include <sensor_events.h>

static void decode_string(unsigned char type,
                          unsigned char language_code,
                          unsigned char *source,
                          char          *target,
                          int           size);
int	report_device_id(int fd);
int	enable_watchdog(int fd);
int	disable_watchdog(int fd);
int	pet_watchdog(int fd);
int	query_watchdog(int fd);
int load_fru(int fd);
int driver_info(int fd);
int set_asset(int fd, char *asset_tag);
int set_blink(int fd, int state);
void display_fru(void);
void usage(char *progname);
void display_sensor_list();

DEVICE_ID_RESPONSE	dev_id;
extern char					*optarg;
extern int					opterr;
extern int					optopt;

extern SENSOR	*sensor_head;

int main(argc, argv)
char **argv;
{
	int	fd, rc;
	int	mode = 0;

	if ((fd = open("/dev/ipmi/kcs", O_RDWR))<0)
		{
		perror("open");
		exit(-1);
		}
	if (argc < 2)
		{
		usage(argv[0]);
		exit(-1);
		}
	printf("IPMI CTL Version 2.0, (c) 2001 San Mehat (nettwerk@valinux.com)\n");
	report_device_id(fd);
	while((rc = getopt(argc, argv, "rspdqvfa:b:iR:DC"))!=EOF)
		{
		if ((char) rc == 'a')
			{
			char	*asset_tag;
		
			asset_tag = (char *) strdup(optarg);
			set_asset(fd, asset_tag);
			free(asset_tag);
			exit(0);
			}
		if ((char) rc == 'R')
			{
			unsigned char	*snum;

			snum = (char *) strdup(optarg);
			if (dl_sdr(fd)<0)
				{
				printf("Error: Unable to download sensor list\n");
				exit(1);
				}
			if (atoi(snum) == 0)
				{
				char	str[255];

				printf("Entering Sensor-shell mode\n");
				while(1)
					{
					printf("Sensor #, 'l' to list, 'q' to quit ->");
					fflush(stdout);
					if (!fgets(str,sizeof(str),stdin))
						exit(0);
					str[(strlen(str)-1)] = 0;
					if ((str[0] == 'q') || (str[0] == 'Q'))
						exit(0);
					if ((str[0] == 'l') || (str[0] == 'L'))
						{
						display_sensor_list();
						continue;
						}
					if (!(atoi(str)))
						{
						printf("Invalid sensor number\n");
						continue;
						}
					read_sensor(fd, (unsigned char) atoi(str));
					}
				exit(0);
				}
			else
				{
				read_sensor(fd, (unsigned char) atoi(snum));
				exit(0);
				}
			}
		if ((char) rc == 'r')
			{
			}
		else if ((char) rc == 'i')
			{
			if (dl_sdr(fd)<0)
				{
				printf("Error: Unable to download sensor list\n");
				exit(1);
				}
			display_sensor_list();
			exit(0);
			}
		else if ((char) rc == 'D')
			{
			if (dl_sdr(fd)<0)
				{
				printf("Error: Unable to download sensor list\n");
				exit(1);
				}
			if (dl_sel(fd)<0)
				{
				printf("Error: Unable to download SEL\n");
				exit(1);
				}
			}
		else if ((char) rc == 'C')
			{
			// clear log
			clear_sel(fd);
			exit(0);
			}
		else if ((char) rc == 's')
			{
			enable_watchdog(fd);
			exit(0);
			}
		else if ((char) rc == 'p')
			{
			pet_watchdog(fd);
			exit(0);
			}
		else if ((char) rc == 'd')
			{
			disable_watchdog(fd);
			exit(0);
			}
		else if ((char) rc == 'q')
			{
			query_watchdog(fd);
			exit(0);
			}
		else if ((char) rc == 'b')
			{
			char	*state;
			state = (char *) strdup(optarg);
			if (state[0] == '1')
				set_blink(fd, 1);
			else if (state[0] == '0')
				set_blink(fd, 0);
			else
				{
				printf("Blink state must be 1 or 0\n");
				exit(-1);
				}
			free(state);
			}
		else if ((char) rc == 'v')
			{
			driver_info(fd);
			exit(0);
			}
		else if ((char) rc == 'f')
			{
			load_fru(fd);
			display_fru();
			exit(0);
			}
		else
			{
			usage(argv[0]);
			exit(-1);
			}
		}

	close(fd);
	exit(0);
}

int set_blink(int fd, int state)
{
	struct ipmi_driver_info	drv_inf;
	IPMI_XFER								xfer;
	IPMI_XFER								*xferp = &xfer;
	unsigned char						cc;
	int											rc;

	memset(&drv_inf, 0, sizeof(drv_inf));
	if (ioctl(fd, IOCTL_DRIVER_INFO, (void *) &drv_inf)<0)
		{
		printf("set_blink(): failed (%m)\n");
		return(-1);
		}
	
	if (!strcasecmp(drv_inf.driver_name, "ipmi_kcs"))
		{
		if (!(drv_inf.flags & KCS_FLAG_BLINKY))
			{
			printf("blink not available on this hardware\n");
			return(-1);
			}
		}
	else
		{
		printf("blink support not implimented for this driver\n");
		return(-1);
		}
	INIT_XFER(xferp);
	SET_REQUEST_LUN(xferp, 0);
	SET_REQUEST_NETFN(xferp, APP_REQUEST);
	if (state == 1)
		{
		SET_REQUEST_CMD(xferp, 0x55);
		}
	else
		{
		SET_REQUEST_CMD(xferp, 0x56);
		}
	if ((rc = ioctl(fd, IOCTL_IPMI_XFER, (void *) &xfer))<0)
		{
		printf("set_blink(): failed (%m)\n");
		return(-1);
		}
	GET_RESPONSE_CC(xferp, cc);

	if (cc != 0x00)
		{
		printf("set_blink(): failed (cc = 0x%.2x)\n",cc);
		return(-1);
		}
	return(0);
}

int driver_info(int fd)
{
	struct ipmi_driver_info	drv_inf;

	memset(&drv_inf, 0, sizeof(drv_inf));
	if (ioctl(fd, IOCTL_DRIVER_INFO, (void *) &drv_inf)<0)
		{
		printf("driver_info(): failed (%m)\n");
		return(-1);
		}
	printf("IPMI driver name : %s\n",drv_inf.driver_name);
	printf("Driver Version   : %d.%d\n",drv_inf.major_ver, drv_inf.minor_ver);
	printf("MB Chipset Vendor: 0x%.4x\n",drv_inf.mb_chipset_vendor);
	printf("MB Chipset Device: 0x%.4x\n",drv_inf.mb_chipset_device);
	printf("Driver Flags     : ");
	if (!strcasecmp(drv_inf.driver_name, "ipmi_kcs"))
		{
		if (drv_inf.flags & KCS_FLAG_BLINKY)
			printf("BLINKY ");
		if (drv_inf.flags & KCS_FLAG_LEGACY)
			printf("LEGACY_MACHINE ");
		else
			printf("NEW_MACHINE ");
		printf("\n");
		}
	else
		printf("(not implimented for this driver)\n");
	return(0);
}

/*
 * Watchdog related code
 */
int	enable_watchdog(int fd)
{
	int						rc;
	IPMI_XFER			xfer;
	IPMI_XFER			*xferp = &xfer;
	SET_WATCHDOG	set;
	unsigned char	cc;
	int						fixup = 0;

	while(1)
		{
		memset(&set, 0, sizeof(set));

		set.timer_use = 0x04;
		set.timeout_action = 0x03;
		if (fixup)
			{
			set.pre_irq = 0x00;
			set.pretimeout_interval = 0;
			}
		else
			{
			set.pre_irq = 0x03;
			set.pretimeout_interval = 10;
			}
		set.tuefc_smsos= 0x01;
		set.initial_count = (30*10);

		INIT_XFER(xferp);
		SET_REQUEST_LUN(xferp, 0);
		SET_REQUEST_NETFN(xferp, APP_REQUEST);
		SET_REQUEST_CMD(xferp, CMD_WATCHDOG_SET);
		SET_REQUEST_DATA(xferp, (unsigned char *) &set, sizeof(set));

		if ((rc = ioctl(fd, IOCTL_IPMI_XFER, (void *) &xfer))<0)
			{
			printf("enable_watchdog(): failed (%m)\n");
			return(-1);
			}
		GET_RESPONSE_CC(xferp, cc);
		if (cc == 0xcc)
			{
			fixup++;
			if (fixup ==2)
				{
				printf("Flakey NMI fixup failed\n");
				return(-1);
				}
			printf("Flakey NMI fixup enabled\n");
			}
		else if (cc != 0x00)
			{
			printf("enable_watchdog(): failed (cc = 0x%.2x)\n",cc);
			return(-1);
			}
		break;
		}
	return(0);
}

int	disable_watchdog(int fd)
{
	int						rc;
	IPMI_XFER			xfer;
	IPMI_XFER			*xferp = &xfer;
	SET_WATCHDOG	set;
	unsigned char	cc;

	memset(&set, 0, sizeof(set));

	set.timer_use = 0x04;
	set.timeout_action = 0x00;
	set.pre_irq = 0x00;
	set.pretimeout_interval = 10;
	set.tuefc_smsos= 0x01;
	set.initial_count = (30*10);

	INIT_XFER(xferp);
	SET_REQUEST_LUN(xferp, 0);
	SET_REQUEST_NETFN(xferp, APP_REQUEST);
	SET_REQUEST_CMD(xferp, CMD_WATCHDOG_SET);
	SET_REQUEST_DATA(xferp, (unsigned char *) &set, sizeof(set));

	if ((rc = ioctl(fd, IOCTL_IPMI_XFER, (void *) &xfer))<0)
		{
		printf("disable_watchdog(): failed (%m)\n");
		return(-1);
		}
	GET_RESPONSE_CC(xferp, cc);
	if (cc != 0x00)
		{
		printf("disable_watchdog(): failed (cc = 0x%.2x)\n",cc);
		return(-1);
		}
	return(0);
}

int query_watchdog(int fd)
{
	GET_WATCHDOG_RESPONSE	resp;
	IPMI_XFER							xfer;
	IPMI_XFER							*xferp = &xfer;
	int										rc;
	unsigned char					cc;

	INIT_XFER(xferp);
	SET_REQUEST_LUN(xferp, 0);
	SET_REQUEST_NETFN(xferp, APP_REQUEST);
	SET_REQUEST_CMD(xferp, CMD_WATCHDOG_GET);

	if ((rc = ioctl(fd, IOCTL_IPMI_XFER, (void *) &xfer))<0)
		{
		printf("query_watchdog(): failed (%m)\n");
		return(-1);
		}

	GET_RESPONSE_CC(xferp, cc);
	if (cc != 0x00)
		{
		printf("enable_watchdog(): failed (cc = 0x%.2x)\n",cc);
		return(-1);
		}
	memset(&resp, 0, sizeof(resp));

	GET_RESPONSE_DATA(xferp, (unsigned char *) &resp);
	printf("Watchdog initial count : %d\n",resp.init_count);
	printf("Watchdog current count : %d\n",resp.current_count);
	
	printf("Watchdog timeout action: %d (%s)\n",resp.timeout_act,
					(	(resp.timeout_act == 0x00) ? "Disabled" :
						(resp.timeout_act == 0x01) ? "Hard Reset" :
						(resp.timeout_act == 0x02) ? "Power Down" :
						(resp.timeout_act == 0x03) ? "Power Cycle" : "Unknown"));
	return(0);
}

int	pet_watchdog(int fd)
{
	IPMI_XFER			xfer;
	IPMI_XFER			*xferp = &xfer;
	int						rc;
	unsigned char	cc;

	INIT_XFER(xferp);
	SET_REQUEST_LUN(xferp, 0);
	SET_REQUEST_NETFN(xferp, APP_REQUEST);
	SET_REQUEST_CMD(xferp, CMD_WATCHDOG_RESET);

	if ((rc = ioctl(fd, IOCTL_IPMI_XFER, (void *) &xfer))<0)
		{
		printf("report_device_id(): failed (%m)\n");
		return(-1);
		}

	GET_RESPONSE_CC(xferp, cc);
	if (cc != 0x00)
		{
		printf("pet_watchdog(): failed (cc = 0x%.2x)\n",cc);
		return(-1);
		}
	return(0);
}

/*
 * Device ID related code
 */
int	report_device_id(int fd)
{
	IPMI_XFER						xfer;
	IPMI_XFER						*xferp = &xfer;
	int									rc;
	unsigned char				cc;

	INIT_XFER(xferp);
	SET_REQUEST_LUN(xferp, 0);
	SET_REQUEST_NETFN(xferp, APP_REQUEST);
	SET_REQUEST_CMD(xferp, CMD_GET_DEVICE_ID);

	if ((rc = ioctl(fd, IOCTL_IPMI_XFER, (void *) &xfer))<0)
		{
		printf("report_device_id(): failed (%m)\n");
		return(-1);
		}

	GET_RESPONSE_CC(xferp, cc);
	if (cc != 0x00)
		{
		printf("report_device_id(): failed (cc = 0x%.2x)\n",cc);
		return(-1);
		}
	memset(&dev_id, 0, sizeof(dev_id));
	GET_RESPONSE_DATA(xferp, (unsigned char *) &dev_id);
	printf("BMC version %x.%x, IPMI version %d.%d\n",
					dev_id.major_firmware_revision,
					dev_id.minor_firmware_revision,
					dev_id.ipmi_version_major,
					dev_id.ipmi_version_minor);
	return(0);
}

/*
 * FRU related code
 */
char					board_area[BOARD_AREA_NUM_FIELDS][64];
char					product_area[PRODUCT_AREA_NUM_FIELDS][64];
int						asset_offset = -1;
int						asset_length = 0;
int						asset_lang = 0;
int						fru_size = 0;

void display_fru()
{
	printf("Mainboard FRU Size  : %d\n",fru_size);
	printf("Board Manufacturer  : %s\n",(board_area[BOARD_MANUFACTURER][0] ? board_area[BOARD_MANUFACTURER] : "Not Available"));
	printf("Board Product Name  : %s\n",(board_area[BOARD_PRODUCT_NAME][0] ? board_area[BOARD_PRODUCT_NAME] : "Not Available"));
	printf("Board Part Number   : %s\n",(board_area[BOARD_PART][0] ? board_area[BOARD_PART] : "Not Available"));
	printf("Board Serial #      : %s\n",(board_area[BOARD_SERIAL][0] ? board_area[BOARD_SERIAL] : "Not Available"));

	printf("Product Manufacturer: %s\n",(product_area[PRODUCT_MANUFACTURER][0] ? product_area[PRODUCT_MANUFACTURER] : "Not Available"));
	printf("Product Name        : %s\n",(product_area[PRODUCT_NAME][0] ? product_area[PRODUCT_NAME] : "Not Available"));
	printf("Product Part Number : %s\n",(product_area[PRODUCT_PART][0] ? product_area[PRODUCT_PART] : "Not Available"));
	printf("Product Version     : %s\n",(product_area[PRODUCT_VERSION][0] ? product_area[PRODUCT_VERSION] : "Not Available"));
	printf("Product Serial #    : %s\n",(product_area[PRODUCT_SERIAL][0] ? product_area[PRODUCT_SERIAL] : "Not Available"));
	printf("Product Asset Tag   : %s\n",(product_area[PRODUCT_ASSET][0] ? product_area[PRODUCT_ASSET] : "Not Available"));
	printf("Asset Tag Max Length: %d\n",asset_length);
}

int load_fru(int fd)
{
	FRU_COMMON_HEADER			*cmn_hdr;
	FRU_AREA_INFO					inv_info_resp;
	FRU_DATA_REQ_IPMI_10	data_req_10;
	FRU_DATA_REQ_IPMI_09	data_req_09;
	FRU_DATA_CHUNK				chunk;
	IPMI_XFER							xfer;
	IPMI_XFER							*xferp = &xfer;
	unsigned char					fru_dev_id = 0x00, cc, *p, num, lang, *fru_buffer;
	int										rc, fru_pos, brtl,i;
	TL										*tl;

	INIT_XFER(xferp);
	SET_REQUEST_LUN(xferp, 0);
	SET_REQUEST_NETFN(xferp, STORAGE_REQUEST);
	SET_REQUEST_CMD(xferp,CMD_GET_FRU_INV_AREA_INFO);

	if (dev_id.ipmi_version_major >= 1)
		SET_REQUEST_DATA(xferp, &fru_dev_id, sizeof(fru_dev_id));

	if ((rc = ioctl(fd, IOCTL_IPMI_XFER, (void *) &xfer))<0)
		{
		printf("load_fru(): failed (%m)\n");
		return(-1);
		}

	GET_RESPONSE_CC(xferp, cc);
	if (cc != 0x00)
		{
		printf("load_fru(): failed (cc = 0x%.2x)\n",cc);
		return(-1);
		}
	memset(&inv_info_resp, 0, sizeof(inv_info_resp));
	GET_RESPONSE_DATA(xferp, (unsigned char *) &inv_info_resp);
	
	fru_buffer = (unsigned char *) malloc(inv_info_resp.size);
	memset(fru_buffer, 0, inv_info_resp.size);

	brtl = inv_info_resp.size;
	fru_pos = 0;
	fru_size = inv_info_resp.size;
	
	while(brtl)
		{
		INIT_XFER(xferp);
		SET_REQUEST_LUN(xferp, 0);
		SET_REQUEST_NETFN(xferp, STORAGE_REQUEST);
		SET_REQUEST_CMD(xferp, CMD_READ_FRU_INV_DATA);

		if (dev_id.ipmi_version_major >= 1)
			{
			data_req_10.devid = 0x00;
			data_req_10.offset = fru_pos;
			if (brtl <=16)
				data_req_10.count = brtl;
			else
				data_req_10.count = 16;
			SET_REQUEST_DATA(xferp, &data_req_10, sizeof(data_req_10));
			}
		else
			{
			data_req_09.offset = fru_pos;
			if (brtl <=16)
				data_req_09.count = brtl;
			else
				data_req_09.count = 16;
			SET_REQUEST_DATA(xferp, &data_req_09, sizeof(data_req_09));
			}

		if ((rc = ioctl(fd, IOCTL_IPMI_XFER, (void *) &xfer))<0)
			{
			printf("load_fru(): failed (%m)\n");
			free(fru_buffer);
			return(-1);
			}
		GET_RESPONSE_CC(xferp, cc);
		if (cc != 0x00)
			{
			printf("load_fru(): failed (cc = 0x%.2x)\n",cc);
			free(fru_buffer);
			return(-1);
			}
		GET_RESPONSE_DATA(xferp, (unsigned char *) &chunk);
		memcpy(&fru_buffer[fru_pos], &chunk.data[0], chunk.count);
		brtl -=chunk.count;
		fru_pos+=chunk.count;
		}

	cmn_hdr = (FRU_COMMON_HEADER *) fru_buffer;
	if (cmn_hdr->board_area_offset)
		{
    p = &fru_buffer[(cmn_hdr->board_area_offset * 8)];
    lang = p[2];
    p +=6;
    for (i=0; i < BOARD_AREA_NUM_FIELDS; i++)
      {
      tl = (TL *) p;
      p++;
      decode_string(tl->type_code, lang, p, board_area[i], tl->num_bytes);
      board_area[i][tl->num_bytes] = 0;
      p += tl->num_bytes;
      }
		}
	if (cmn_hdr->product_info_offset)
		{
    p = &fru_buffer[(cmn_hdr->product_info_offset * 8)];
    lang = p[2];
    p +=3;

    for (i=0; i < PRODUCT_AREA_NUM_FIELDS; i++)
      {
      tl = (TL *) p;
      if (i == PRODUCT_ASSET)
        {
        asset_offset = (p - fru_buffer);
        asset_length = tl->num_bytes;
        asset_lang = lang;
        }
      p++;
      decode_string(tl->type_code, lang, p, product_area[i], tl->num_bytes);
      product_area[i][tl->num_bytes] = 0;
      p += tl->num_bytes;
      }
		}
#if 0
  for (i=0; i < BOARD_AREA_NUM_FIELDS; i++)
    printf("Field %d = %s\n",i,board_area[i]);
  for (i=0; i < PRODUCT_AREA_NUM_FIELDS; i++)
    printf("Field %d = %s\n",i,product_area[i]);
#endif

	free(fru_buffer);
	return(0);
}

int set_asset(int fd, char *asset_tag)
{
	FRU_DATA_WRITE_REQ_IPMI_09	*wr_rq_09;
	FRU_DATA_WRITE_REQ_IPMI_10	*wr_rq_10;
	IPMI_XFER										xfer;
	IPMI_XFER										*xferp = &xfer;
	unsigned char								buffer[255], cc;
	TL													*tl;
	int													rc;

	memset(buffer, 0, sizeof(buffer));

	wr_rq_09 = (FRU_DATA_WRITE_REQ_IPMI_09 *) buffer;
	wr_rq_10 = (FRU_DATA_WRITE_REQ_IPMI_10 *) buffer;

	load_fru(fd);
	if (asset_offset < 0)
		return(-ELIBBAD);
	if (strlen(asset_tag) > asset_length)
		return (-EOVERFLOW);
	INIT_XFER(xferp);
	SET_REQUEST_LUN(xferp, 0);
	SET_REQUEST_NETFN(xferp, STORAGE_REQUEST);
	SET_REQUEST_CMD(xferp, CMD_WRITE_FRU_INV_DATA);
	
	if (dev_id.ipmi_version_major >= 1)
		{
		wr_rq_10->dev_id = 0x00;
		wr_rq_10->offset = asset_offset;
		tl = (TL *) &wr_rq_10->data[0];
		tl->type_code = 0x03;
		tl->num_bytes = asset_length;
		memcpy(&wr_rq_10->data[1], asset_tag, strlen(asset_tag));
		SET_REQUEST_DATA(xferp, wr_rq_10, (sizeof(FRU_DATA_WRITE_REQ_IPMI_09) + sizeof(TL) + asset_length));
		}
	else
		{
		wr_rq_09->offset = asset_offset;
		tl = (TL *) &wr_rq_09->data[0];
		tl->type_code = 0x03;
		tl->num_bytes = asset_length;
		memcpy(&wr_rq_09->data[1], asset_tag, strlen(asset_tag));
		SET_REQUEST_DATA(xferp, wr_rq_09, (sizeof(FRU_DATA_WRITE_REQ_IPMI_09) + sizeof(TL) + asset_length));
		}
	if ((rc = ioctl(fd, IOCTL_IPMI_XFER, (void *) &xfer))<0)
		{
		printf("set_asset(): failed (%m)\n");
		return(-1);
		}
	GET_RESPONSE_CC(xferp, cc);
	if (cc != 0x00)
		{
		printf("set_asset(): failed (cc = 0x%.2x)\n",cc);
		return(-1);
		}
	return(0);
}

#define STRING_DATA_TYPE_BINARY         0x00
#define STRING_DATA_TYPE_BCD_PLUS       0x01
#define STRING_DATA_TYPE_SIX_BIT_ASCII  0x02
#define STRING_DATA_TYPE_LANG_DEPENDANT 0x03

static void decode_string(unsigned char type,
                          unsigned char language_code,
                          unsigned char *source,
                          char          *target,
                          int           size)
{
  unsigned char *s = &source[0];
  unsigned char *d = &target[0];
 
  memset(target, 0, size);
  if (type == STRING_DATA_TYPE_BCD_PLUS)
    {
    while(size)
      {
      if ((*s >= (unsigned char) 0x00) && (*s <= (unsigned char) 0x09))
        *d++ = (unsigned char) 0x30 + (unsigned char) *s++;
      else
        {
        if (*s == (unsigned char) 0x0A)
          *d++ = (unsigned char) ' ';
        else if (*s == (unsigned char) 0x0B)
          *d++ = (unsigned char) '-';
        else if (*s == (unsigned char) 0x0C)
          *d++ = (unsigned char) '.';
        else if ((*s <= (unsigned char) 0x0D) && (*s <= (unsigned char) 0x0F))
          {
          *d++ = (unsigned char) '*';
          }
        s++;
        }
      size --;
      }
    }
  else if (type == STRING_DATA_TYPE_SIX_BIT_ASCII)
    {
    printf("Six bit ASCII decode not supported\n");
    }
  else if (type == STRING_DATA_TYPE_LANG_DEPENDANT)
    {
    if ((language_code == 0x00) || (language_code == 0x25) || (language_code == 0x19))
      {
      strncpy(target, source, size);
      target[size] = 0x0;
      }
    else
      {
      printf("Language 0x%x dependant decode not supported\n",
            language_code);
      return;
      }
    }
  else
    {
    printf("Unable to decode type 0x%.2x\n",type);
    return;
    }
}


/*
 * General Purpose stuff
 */
void usage(char *progname)
{
	printf("Usage: %s [-r] [-s] [-p] [-d]\n",progname);
	printf("       -r         -- report mainboard device ID\n");
	printf("       -s         -- enable watchdog timer\n");
	printf("       -p         -- pet watchdog timer\n");
	printf("       -d         -- disable watchdog timer\n");
	printf("       -q         -- query watchdog timer\n");
	printf("       -f         -- display FRU \n");
	printf("       -v         -- verbose driver info\n");
	printf("       -a <tag>   -- set asset tag\n");
	printf("       -b <state> -- set blink state (if supported)\n");
	printf("       -i         -- Retrieve sensor information\n");
	printf("       -R <num>   -- Display realtime sensor reading\n");
	printf("                  -- (0 == Interactive shell)\n");
	printf("       -D         -- Download System Event Log\n");
	printf("       -C         -- Clear System Event Log\n");
}

void display_sensor_list()
{
	SENSOR  *list;
  char    tmp[128];
  char    sensor_class[128];

	if (!sensor_head)
		{
		printf("No sensor list available\n");
		return;
		}
  list = sensor_head;
  while(list)
    {
    if (list->rec_header.record_type == 0x01)
      {
      decode_string(list->sensor_type.type_01_sensor.id_string_type_code,
                  0,
                  &list->sensor_type.type_01_sensor.id_string[0],
                  &tmp[0],
                  list->sensor_type.type_01_sensor.id_string_num_bytes);
      if (list->sensor_type.type_01_sensor.sens_capable_reading_type == ANALOG)
        strcpy(sensor_class,"ANALOG");
      else
        strcpy(sensor_class,"UNSUPPORTED");
      printf("Sensor %-2d, Name %-25s, Class %s\n",
            list->sensor_type.type_01_sensor.sensor_number,
            tmp,
            sensor_class);
      }
    else if (list->rec_header.record_type == 0x02)
      {
      int class;
      decode_string(list->sensor_type.type_02_sensor.id_string_type_code,
                  0,
                  &list->sensor_type.type_02_sensor.id_string[0],
                  &tmp[0],
                  list->sensor_type.type_02_sensor.id_string_num_bytes);
      class = sensor_get_class(list->sensor_type.type_02_sensor.event_reading_type_code);
      if (class == SENSOR_CLASS_DIGITAL)
        strcpy(sensor_class,"DIGITAL");
      else if (class == SENSOR_CLASS_DISCRETE)
        strcpy(sensor_class,"UNSUPPORTED");
      else
        strcpy(sensor_class,"UNSUPPORTED");
      printf("Sensor %-2d, Name %-25s, Class %s\n",
            list->sensor_type.type_01_sensor.sensor_number,
            tmp,
            sensor_class);
      }
    else
      {
      list = list->next;
      continue;
      }
    list = list->next;
    }
}

