/* IIWU Synth  A soundfont synthesizer
 *
 * Copyright (C)  2001 Peter Hanappe
 * Author: Peter Hanappe, peter@hanappe.com
 *
 * This file is part of the Varese program. 
 * Varese is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307,
 * USA.
 *
 */
/* iiwu_alsa.c
 *
 * Driver for the Advanced Linux Sound Architecture
 *
 * Based on jMax's alsaaudioport.c by Francois Dechelle, Norbert
 * Schnell, Maurizio De Cecco, e.a. that is in turn based on Ardour's
 * alsa_device.cc by Paul Barton-Davis. The midi device uses code from
 * jMax's alsarawmidi.c file and from Smurf's midi_alsaraw.c by Josh
 * Green.  */

#include "iiwu_auport.h"
#include "iiwu_midi.h"

#if ALSA_SUPPORT
#include <sys/asoundlib.h>
#include <pthread.h>
#include <fcntl.h>
#include <unistd.h>
#include <errno.h>
#include <sys/poll.h>

#define IIWU_ALSA_DEFAULT_AUDIO_DEVICE "hw:0,0"
#define IIWU_ALSA_DEFAULT_MIDI_DEVICE  "hw:0,0"
#define OPEN_IN_THREAD 0
#define BUFFER_LENGTH 512

typedef struct {
  snd_pcm_t *handle;
  iiwu_auport_t* auport;
  void* buffer;
  pthread_t thread;
  int cont;
  int buffer_size;
  int format;
} iiwu_alsa_audio_driver_t;

static void* iiwu_alsa_audio_run(void* d);
static int iiwu_alsa_audio_start(iiwu_alsa_audio_driver_t* d);
static int iiwu_alsa_audio_open(iiwu_alsa_audio_driver_t* d);
int delete_iiwu_alsa_audio_driver(iiwu_audio_driver_t* p);

typedef struct {
  snd_rawmidi_t *rawmidi_in;
  struct pollfd *pfd;
  int npfd;
  pthread_t thread;
  int status;
  unsigned char buffer[BUFFER_LENGTH];
  iiwu_midi_parser_t* parser;
  iiwu_midi_handler_t* handler;
} iiwu_alsa_midi_driver_t;

iiwu_midi_driver_t* new_iiwu_alsa_midi_driver(iiwu_midi_handler_t* handler);
int delete_iiwu_alsa_midi_driver(iiwu_midi_driver_t* p);
int iiwu_alsa_midi_driver_join(iiwu_midi_driver_t* p);
int iiwu_alsa_midi_driver_status(iiwu_midi_driver_t* p);
static void* iiwu_alsa_midi_run(void* d);

/*
 * new_iiwu_alsa_audio_driver
 */
iiwu_audio_driver_t* new_iiwu_alsa_audio_driver(iiwu_auport_t* port)
{
  char* devname;
  iiwu_alsa_audio_driver_t* dev = NULL;
  iiwu_pcm_data_t* dev_format;
  int buffer_byte_size;
  dev = IIWU_NEW(iiwu_alsa_audio_driver_t);
  if (dev == NULL) {
    IIWU_LOG(ERR, "Out of memory");
    return NULL;    
  }
  IIWU_MEMSET(dev, 0, sizeof(iiwu_alsa_audio_driver_t));
  dev->auport = port;
  dev->cont = 1;
  /* allocate buffer */
  dev_format = iiwu_auport_get_dev_format(port);
  dev->buffer_size = iiwu_auport_get_buffer_size(port);
  buffer_byte_size = dev->buffer_size * iiwu_pcm_data_framesize(dev_format);
  dev->buffer = IIWU_MALLOC(buffer_byte_size);
  if (dev->buffer == NULL) {
    IIWU_LOG(ERR, "Out of memory");
    goto error_recovery;
  }
  /* check the device name */
  devname = iiwu_auport_get_device_name(port);
  if ((devname == NULL) || (IIWU_STRCMP(devname, "default") == 0)) {
    iiwu_auport_set_device_name(port, IIWU_ALSA_DEFAULT_AUDIO_DEVICE);
    devname = IIWU_ALSA_DEFAULT_AUDIO_DEVICE;
  }

#if !OPEN_IN_THREAD 
  if (iiwu_alsa_audio_open(dev) != IIWU_OK) {
    goto error_recovery;
  }
#endif

  /* this spins of a new thread that calls the audio application in a
     loop thru a callback mechanism */
  if (iiwu_alsa_audio_start(dev)) {
    IIWU_LOG(ERR, "Can't start the audio thread");    
    goto error_recovery;
  }
  return (iiwu_audio_driver_t*) dev;

error_recovery:
  delete_iiwu_alsa_audio_driver((iiwu_audio_driver_t*) dev);
  return NULL;  
}

/*
 * iiwu_alsa_audio_open
 */
static int iiwu_alsa_audio_open(iiwu_alsa_audio_driver_t* dev)
{
  int err;
  iiwu_auport_t* port = dev->auport;
  iiwu_pcm_data_t* dev_format = iiwu_auport_get_dev_format(port);
  char* devname = iiwu_auport_get_device_name(port);
  int format, channels, sample_rate, periods;
  snd_pcm_hw_params_t *hwparams;
  int open_mode = 0;
  int dir; /* ? */

  /* reserve some space for the hw_params */
  snd_pcm_hw_params_alloca(&hwparams);

  /* open device */
  if ((err = snd_pcm_open(&dev->handle, (char *) devname, SND_PCM_STREAM_PLAYBACK, open_mode)) < 0) {
    IIWU_LOG(ERR, "Failed to open the sound device");
    goto error_recovery;
  }

  /* set the audio format */
  if ((err = snd_pcm_hw_params_any(dev->handle, hwparams)) < 0) {
    IIWU_LOG(ERR, "Failed to initialize the audio settings");
    goto error_recovery;
  }
  if ((err = snd_pcm_hw_params_set_access(dev->handle, hwparams, SND_PCM_ACCESS_RW_INTERLEAVED)) < 0) {
    IIWU_LOG(ERR, "Failed to initialize the audio access");
    goto error_recovery;
  }
  dev->format = 0;
  switch (iiwu_pcm_data_get_format(dev_format)) {
  case IIWU_SAMPLE_S8: dev->format = SND_PCM_FORMAT_S8; break;
  case IIWU_SAMPLE_U8: dev->format = SND_PCM_FORMAT_U8; break;
  case IIWU_SAMPLE_S16HE: dev->format = SND_PCM_FORMAT_S16; break;
  case IIWU_SAMPLE_S16BE: dev->format = SND_PCM_FORMAT_S16_BE; break;
  case IIWU_SAMPLE_S16LE: dev->format = SND_PCM_FORMAT_S16_LE; break;
  case IIWU_SAMPLE_U16HE: dev->format = SND_PCM_FORMAT_U16; break;
  case IIWU_SAMPLE_U16BE: dev->format = SND_PCM_FORMAT_U16_BE; break;
  case IIWU_SAMPLE_U16LE: dev->format = SND_PCM_FORMAT_U16_LE; break;
  }
  if (dev->format == 0) {
    IIWU_LOG(ERR, "Unknown sample format");
    goto error_recovery;
  }
  format = dev->format;
  if ((err = snd_pcm_hw_params_set_format(dev->handle, hwparams, SND_PCM_FORMAT_S16)) < 0) {
    IIWU_LOG(ERR, "Failed to initialize the audio format");
    goto error_recovery;
  }
  channels = iiwu_pcm_data_get_channels(dev_format);
  if ((err = snd_pcm_hw_params_set_channels(dev->handle, hwparams, channels)) < 0) {
    IIWU_LOG(ERR, "Failed to initialize the audio channels");
    goto error_recovery;
  }
  sample_rate = iiwu_pcm_data_get_sample_rate(dev_format);
  if ((err = snd_pcm_hw_params_set_rate_near(dev->handle, hwparams, sample_rate, 0)) < 0) {
    IIWU_LOG(ERR, "Failed to initialize the sampling rate");
    goto error_recovery;
  }

  /* Set the queue and buffer size */
  periods = iiwu_auport_get_queue_size(port) / iiwu_auport_get_buffer_size(port);
  if ((err = snd_pcm_hw_params_set_periods(dev->handle, hwparams, periods, 0) < 0)) {
    IIWU_LOG(ERR, "Failed to initialize the period");
    goto error_recovery;
  }
  if (periods != snd_pcm_hw_params_get_periods(hwparams, 0)) {
    IIWU_LOG(ERR, "Failed to initialize the period");
    goto error_recovery;
  }
  if ((err = snd_pcm_hw_params_set_period_size(dev->handle, hwparams, iiwu_auport_get_buffer_size(port), 0)) < 0) {
    IIWU_LOG(ERR, "Failed to initialize the buffer size");
    goto error_recovery;
  }
  if (iiwu_auport_get_buffer_size(port) != snd_pcm_hw_params_get_period_size(hwparams, &dir)) {
    IIWU_LOG(ERR, "Failed to initialize the buffer size");
    goto error_recovery;
  }
  if ((err = snd_pcm_hw_params_set_buffer_size(dev->handle, hwparams, iiwu_auport_get_queue_size(port))) < 0) {
    IIWU_LOG(ERR, "Failed to initialize the queue size");
    goto error_recovery;
  }
  if ((err = snd_pcm_hw_params(dev->handle, hwparams)) < 0) {
    IIWU_LOG(ERR, "Failed to initialize the audio settings");
    goto error_recovery;
  }

  return IIWU_OK;  
  
error_recovery:
  return IIWU_FAILED;  
}

/*
 * delete_iiwu_alsa_audio_driver
 */
int delete_iiwu_alsa_audio_driver(iiwu_audio_driver_t* p) 
{
  iiwu_alsa_audio_driver_t* dev = (iiwu_alsa_audio_driver_t*) p;  
  if (dev == NULL) {
    return IIWU_OK;
  }
  dev->cont = 0;
  if (dev->thread) {
    if (pthread_join(dev->thread, NULL)) {
      IIWU_LOG(ERR, "Failed to join the audio thread");
      return IIWU_FAILED;
    }
  }
  if (dev->handle) {
    snd_pcm_drop(dev->handle);
  }
  if (dev->buffer != NULL) {
    IIWU_FREE(dev->buffer);
  }
  IIWU_FREE(dev);
  return IIWU_OK;
}

/*
 * iiwu_alsa_audio_run
 */
void* iiwu_alsa_audio_run(void* d)
{
  iiwu_alsa_audio_driver_t* dev = (iiwu_alsa_audio_driver_t*) d;
  iiwu_auport_t* port = dev->auport;
  void* buffer = dev->buffer;
  int len = dev->buffer_size;
  int err;

#if OPEN_IN_THREAD 
  if (iiwu_alsa_audio_open(dev) != IIWU_OK) {
    pthread_exit(NULL);
  }
#endif

  /* push the start button */
  if ((err = snd_pcm_prepare(dev->handle)) < 0) {
    IIWU_LOG(ERR, "Failed to prepare the audio device");
    pthread_exit(NULL);
  }

  /* i'm not sure whether i have to call snd_pcm_start */
/*    if (snd_pcm_start(dev->handle) < 0) { */
/*      IIWU_LOG(ERR, "Failed to start the audio device"); */
/*      pthread_exit(NULL); */
/*    } */

  /* loop in the fast track */
  while ((iiwu_auport_get_state(port) == IIWU_AUPORT_PLAYING) && dev->cont) {
    iiwu_auport_write(port, buffer, len);
    err = snd_pcm_writei(dev->handle, buffer, len);
    if (err == -EPIPE ) {
      if ((err = snd_pcm_prepare(dev->handle)) < 0) {
	IIWU_LOG(ERR, "Failed to prepare the audio device after underrun");
	dev->cont = 0;
      }
    } else if (err < 0) {
      dev->cont = 0;
      IIWU_LOG(ERR, "Failed to write to the audio device");
    }
  }
  IIWU_LOG(DBG, "Audio thread finished");
  pthread_exit(NULL);
}

/*
 * iiwu_alsa_audio_start
 */
int iiwu_alsa_audio_start(iiwu_alsa_audio_driver_t* d) 
{
  pthread_attr_t attr;
  int err;
  int sched = SCHED_FIFO;
  if (pthread_attr_init(&attr)) {
    IIWU_LOG(ERR, "Couldn't initialize audio thread attributes");
    return IIWU_FAILED;
  }
  /* the pthread_create man page explains that
     pthread_attr_setschedpolicy returns an error if the user is not
     permitted the set SCHED_FIFO. it seems however that no error is
     returned but pthread_create fails instead. that's why i try to
     create the thread twice in a while loop. */
  while (1) {
    err = pthread_attr_setschedpolicy(&attr, sched);
    if (err) {
      IIWU_LOG(WARN, "Couldn't set high priority scheduling for the audio output");
      if (sched == SCHED_FIFO) {
	sched = SCHED_OTHER;
	continue;
      } else {
	IIWU_LOG(ERR, "Couldn't set scheduling policy");
	return err;
      }
    }
    err = pthread_create(&d->thread, &attr, iiwu_alsa_audio_run, (void*) d);
    if (err) {
      IIWU_LOG(WARN, "Couldn't set high priority scheduling for the audio output");
      if (sched == SCHED_FIFO) {
	sched = SCHED_OTHER;
	continue;
      } else {
	IIWU_LOG(PANIC, "Couldn't create the audio thread");
	return err;
      }
    }
    return 0;
  }
}

/*
 * new_iiwu_alsa_midi_driver
 */
iiwu_midi_driver_t* new_iiwu_alsa_midi_driver(iiwu_midi_handler_t* handler)
{
  int i, err;
  iiwu_alsa_midi_driver_t* dev;
  pthread_attr_t attr;
  int sched = SCHED_FIFO;
  int count;
  struct pollfd *pfd = NULL;
  char* device = NULL;

  /* not much use doing anything */
  if (handler == NULL) {
    IIWU_LOG(ERR, "Invalid argument");
    return NULL;
  }

  /* allocate the device */
  dev = IIWU_NEW(iiwu_alsa_midi_driver_t);
  if (dev == NULL) {
    IIWU_LOG(ERR, "Out of memory");
    return NULL;
  }
  IIWU_MEMSET(dev, 0, sizeof(iiwu_alsa_midi_driver_t));

  dev->handler = handler;

  /* allocate one event to store the input data */
  dev->parser = new_iiwu_midi_parser();
  if (dev->parser == NULL) {
    IIWU_LOG(ERR, "Out of memory");
    goto error_recovery;
  }

  /* get the device name. if none is specified, use the default device. */
  device = iiwu_midi_handler_get_device_name(handler);
  if (device == NULL) {
    iiwu_midi_handler_set_device_name(handler, IIWU_ALSA_DEFAULT_MIDI_DEVICE);
    device = iiwu_midi_handler_get_device_name(handler);
  }

  /* open the hardware device. only use midi in. */
  if ((err = snd_rawmidi_open(&dev->rawmidi_in, NULL, device, SND_RAWMIDI_NONBLOCK)) < 0) {
    IIWU_LOG(ERR, "Error opening ALSA raw MIDI port");
    goto error_recovery;
  }

  /* get # of MIDI file descriptors */
  count = snd_rawmidi_poll_descriptors_count(dev->rawmidi_in);
  if (count > 0) {		/* make sure there are some */
    pfd = IIWU_MALLOC(sizeof (struct pollfd) * count);
    dev->pfd = IIWU_MALLOC(sizeof (struct pollfd) * count);
    /* grab file descriptor POLL info structures */
    count = snd_rawmidi_poll_descriptors(dev->rawmidi_in, pfd, count);
  }

  /* copy the input FDs */
  for (i = 0; i < count; i++) {		/* loop over file descriptors */
    if (pfd[i].events & POLLIN) { /* use only the input FDs */
      dev->pfd[dev->npfd].fd = pfd[i].fd;
      dev->pfd[dev->npfd].events = POLLIN; 
      dev->pfd[dev->npfd].revents = 0; 
      dev->npfd++;
    }
  }
  IIWU_FREE(pfd);

  dev->status = IIWU_MIDI_READY;

  /* create the midi thread */
  if (pthread_attr_init(&attr)) {
    IIWU_LOG(ERR, "Couldn't initialize midi thread attributes");
    goto error_recovery;
  }
  /* use fifo scheduling. if it fails, use default scheduling. */
  while (1) {
    err = pthread_attr_setschedpolicy(&attr, sched);
    if (err) {
      IIWU_LOG(WARN, "Couldn't set high priority scheduling for the MIDI input");
      if (sched == SCHED_FIFO) {
	sched = SCHED_OTHER;
	continue;
      } else {
	IIWU_LOG(ERR, "Couldn't set scheduling policy.");
	goto error_recovery;
      }
    }
    err = pthread_create(&dev->thread, &attr, iiwu_alsa_midi_run, (void*) dev);
    if (err) {
      IIWU_LOG(WARN, "Couldn't set high priority scheduling for the MIDI input");
      if (sched == SCHED_FIFO) {
	sched = SCHED_OTHER;
	continue;
      } else {
	IIWU_LOG(PANIC, "Couldn't create the midi thread.");
	goto error_recovery;
      }
    }
    break;
  }  
  return (iiwu_midi_driver_t*) dev;
  
 error_recovery:
  delete_iiwu_alsa_midi_driver((iiwu_midi_driver_t*) dev);
  return NULL;
  
}

/*
 * delete_iiwu_alsa_midi_driver
 */
int delete_iiwu_alsa_midi_driver(iiwu_midi_driver_t* p)
{
  iiwu_alsa_midi_driver_t* dev;

  dev = (iiwu_alsa_midi_driver_t*) p;
  if (dev == NULL) {
    return IIWU_OK;
  }

  dev->status = IIWU_MIDI_DONE;

  /* cancel the thread and wait for it before cleaning up */
  if (dev->thread) {
    if (pthread_cancel(dev->thread)) {
      IIWU_LOG(ERR, "Failed to cancel the midi thread");
      return IIWU_FAILED;
    }
    if (pthread_join(dev->thread, NULL)) {
      IIWU_LOG(ERR, "Failed to join the midi thread");
      return IIWU_FAILED;
    }
  }
  if (dev->rawmidi_in) {
    snd_rawmidi_close(dev->rawmidi_in);
  }
  if (dev->parser != NULL) {
    delete_iiwu_midi_parser(dev->parser);
  }
  IIWU_FREE(dev);
  return IIWU_OK;
}

/*
 * iiwu_alsa_midi_run
 */
void* iiwu_alsa_midi_run(void* d)
{
  int n, i;
  iiwu_midi_event_t* evt;
  iiwu_alsa_midi_driver_t* dev = (iiwu_alsa_midi_driver_t*) d;

  /* make sure the other threads can cancel this thread any time */
  if (pthread_setcancelstate(PTHREAD_CANCEL_ENABLE, NULL)) {
    IIWU_LOG(ERR, "Failed to set the cancel state of the midi thread");
    pthread_exit(NULL);
  }
  if (pthread_setcanceltype(PTHREAD_CANCEL_ASYNCHRONOUS, NULL)) {
    IIWU_LOG(ERR, "Failed to set the cancel state of the midi thread");
    pthread_exit(NULL);
  }

  /* go into a loop until someone tells us to stop */
  dev->status = IIWU_MIDI_LISTENING;
  while (dev->status == IIWU_MIDI_LISTENING) {

    /* is there something to read? */
    n = poll(dev->pfd, dev->npfd, 1); /* use a 1 milliseconds timeout */
    if (n < 0) {
      perror("poll");
    } else if (n > 0) {

      /* read new data */
      n = snd_rawmidi_read(dev->rawmidi_in, dev->buffer, BUFFER_LENGTH);
      if ((n < 0) && (n != -EAGAIN)) {
	IIWU_LOG(ERR, "Failed to read the midi input");
	dev->status = IIWU_MIDI_DONE;
      }

      /* let the parser convert the data into events */
      for (i = 0; i < n; i++) {
	evt = iiwu_midi_parser_parse(dev->parser, dev->buffer[i]);
	if (evt != NULL) {
	  /* send the events to the midi handler */
	  iiwu_midi_handler_send_event(dev->handler, evt);
	}
      }
    }
  }
  pthread_exit(NULL);
}

/*
 * iiwu_alsa_midi_driver_join
 */
int iiwu_alsa_midi_driver_join(iiwu_midi_driver_t* d)
{
  iiwu_alsa_midi_driver_t* dev = (iiwu_alsa_midi_driver_t*) d;
  if (pthread_join(dev->thread, NULL)) {
    IIWU_LOG(ERR, "Failed to join the midi thread");
    return IIWU_FAILED;
  }
  return IIWU_OK;
}

int iiwu_alsa_midi_driver_status(iiwu_midi_driver_t* d)
{
  iiwu_alsa_midi_driver_t* dev = (iiwu_alsa_midi_driver_t*) d;
  return dev->status;
}

#endif /* #if ALSA_SUPPORT */
