/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "ireplicatedvolume.h"


#include "ibuffer.h"
#include "ierror.h"
#include "iviewmodule.h"
#include "iviewsubjectreplicator.h"

#include <vtkFrustumCoverageCuller.h>
#include <vtkMatrix4x4.h>
#include <vtkRenderer.h>
#include <vtkTransform.h>
#include <vtkVolumeMapper.h>

//
//  Templates
//
#include "iarraytemplate.h"
#include "ibuffertemplate.h"


//
//  Main class
//
iReplicatedVolume* iReplicatedVolume::New(iViewSubjectReplicator *replicator)
{
	return new iReplicatedVolume(replicator);
}


iReplicatedVolume::iReplicatedVolume(iViewSubjectReplicator *replicator) : iReplicated(replicator,false), iViewModuleComponent(replicator->GetViewModule()), mReplicas(Replica())
{
	//
	//  Must create property first - it is not created by default
	//
	IERROR_ASSERT(this->GetProperty());

	//
	//  Create main replica
	//
	this->CreateReplica(0,0,0);

	mCuller = vtkFrustumCoverageCuller::New(); IERROR_ASSERT(mCuller);
	mCuller->SetSortingStyleToBackToFront();
}


iReplicatedVolume::~iReplicatedVolume()
{
	mCuller->Delete();
	while(mReplicas.Size() > 0) mReplicas.RemoveLast().Volume->Delete();
}


void iReplicatedVolume::CreateReplica(int i, int j, int k)
{
	mReplicas.Add(Replica());

	Replica &tmp = mReplicas.Last();
	tmp.Pos[0] = i;
	tmp.Pos[1] = j;
	tmp.Pos[2] = k;
	tmp.Volume = vtkVolume::New();
	tmp.Volume->ShallowCopy(this);
	tmp.Volume->SetPosition(2.0*tmp.Pos[0],2.0*tmp.Pos[1],2.0*tmp.Pos[2]);
}


int iReplicatedVolume::IsA(const char *type)
{
	if(strcmp(type,"iReplicatedVolume") == 0) return 1; else return vtkVolume::IsA(type);
}


double* iReplicatedVolume::GetBounds()
{
	int i;
	
	vtkVolume::GetBounds();

	if(this->Bounds != 0)
	{
		for(i=0; i<3; i++)
		{
			this->Bounds[2*i+0] -= 2.0*mNumReplicas[2*i+0];
			this->Bounds[2*i+1] += 2.0*mNumReplicas[2*i+1];
		}
	}

	return this->Bounds;
}


int iReplicatedVolume::RenderTranslucentGeometry(vtkViewport *vp)
{
	int initialized = 0;
	int i, n = mReplicas.Size(), nren = 0;

	vtkProp** buf = new vtkProp*[n]; IERROR_ASSERT(buf);
	for(i=0; i<mReplicas.Size(); i++)
	{
		if(mReplicas[i].Volume->GetMTime() < this->GetMTime())
		{
			mReplicas[i].Volume->ShallowCopy(this);
		}
		buf[i] = mReplicas[i].Volume;
	}

	//
	//  Reorder volumes
	//
	mCuller->Cull(iRequiredCast<vtkRenderer>(vp),buf,n,initialized);

	//
	//  n can change inside Cull()
	//
	for(i=0; i<n; i++) nren += buf[i]->RenderTranslucentGeometry(vp);
	delete [] buf;

	if(nren > 0) return 1; else return 0;
}


void iReplicatedVolume::UpdateReplicas()
{
	int i, j, k;
	Replica tmp;

	//
	//  Add needed actors
	//
	for(k=-mNumReplicas[4]; k<=mNumReplicas[5]; k++)
	{
		for(j=-mNumReplicas[2]; j<=mNumReplicas[3]; j++)
		{
			for(i=-mNumReplicas[0]; i<=mNumReplicas[1]; i++)
			{
				if(i<-mOldNumReplicas[0] || i>mOldNumReplicas[1] || j<-mOldNumReplicas[2] || j>mOldNumReplicas[3] || k<-mOldNumReplicas[4] || k>mOldNumReplicas[5])
				{
					this->CreateReplica(i,j,k);
				}
			}
		}
	}

	//
	//  Delete unneeded actors
	//
	for(k=0; k<mReplicas.Size(); k++)
	{
		if(mReplicas[k].Pos[0]<-mNumReplicas[0] || mReplicas[k].Pos[0]>mNumReplicas[1] || mReplicas[k].Pos[1]<-mNumReplicas[2] || mReplicas[k].Pos[1]>mNumReplicas[3] || mReplicas[k].Pos[2]<-mNumReplicas[4] || mReplicas[k].Pos[2]>mNumReplicas[5])
		{
			mReplicas[k].Volume->Delete();
			mReplicas.Remove(k);
		}
	}
}

