/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/

/*=========================================================================

This class is based on:

  Program:   Visualization Toolkit
  Module:    $RCSfile: iexpressionparser.h,v $
  Language:  C++
  Date:      $Date: 2006/08/18 02:45:46 $
  Version:   $Revision: 1.1 $

  Copyright (c) 1993-2002 Ken Martin, Will Schroeder, Bill Lorensen 
  All rights reserved.
  See Copyright.txt or http://www.kitware.com/Copyright.htm for details.

     This software is distributed WITHOUT ANY WARRANTY; without even 
     the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
     PURPOSE.  See the above copyright notice for more information.

=========================================================================*/

#ifndef IEXPRESSIONPARSER_H
#define IEXPRESSIONPARSER_H


enum ParserTypes
{
	VTK_PARSER_IMMEDIATE = 1,
	VTK_PARSER_UNARY_MINUS,
	VTK_PARSER_NOT,
	// supported math functions
	VTK_PARSER_OR,
	VTK_PARSER_AND,
	VTK_PARSER_LT,
	VTK_PARSER_GT,
	VTK_PARSER_LE,
	VTK_PARSER_GE,
	VTK_PARSER_EQ,
	VTK_PARSER_NE,
	VTK_PARSER_ADD,
	VTK_PARSER_SUBTRACT,
	VTK_PARSER_DOT_PRODUCT,
	VTK_PARSER_MULTIPLY,
	VTK_PARSER_DIVIDE,
	VTK_PARSER_POWER,
	VTK_PARSER_ABSOLUTE_VALUE,
	VTK_PARSER_EXPONENT,
	VTK_PARSER_CEILING,
	VTK_PARSER_FLOOR,
	VTK_PARSER_LOGARITHM,
	VTK_PARSER_SQUARE_ROOT,
	VTK_PARSER_SINE,
	VTK_PARSER_COSINE,
	VTK_PARSER_TANGENT,
	VTK_PARSER_ARCSINE,
	VTK_PARSER_ARCCOSINE,
	VTK_PARSER_ARCTANGENT,
	VTK_PARSER_HYPERBOLIC_SINE,
	VTK_PARSER_HYPERBOLIC_COSINE,
	VTK_PARSER_HYPERBOLIC_TANGENT,
	// functions involving vectors
	VTK_PARSER_VECTOR_UNARY_MINUS,
	VTK_PARSER_VECTOR_ADD,
	VTK_PARSER_VECTOR_SUBTRACT,
	VTK_PARSER_SCALAR_MULTIPLE,
	VTK_PARSER_MAGNITUDE,
	VTK_PARSER_NORMALIZE,
	// codes for scalar variables come before those for vectors
	VTK_PARSER_BEGIN_VARIABLES
};


// the value that is retuned as a result if there is an error
#define VTK_PARSER_ERROR_RESULT 1.0e300

#include "iarray.h"
#include "istring.h"


class iExpressionParser
{
	
public:
	
	iExpressionParser();
	virtual ~iExpressionParser();
	
	// Decription:
	// Set/Get input string to evaluate. 
	void SetFunction(const iString &function);
	
	// Description:
	// Get a scalar result from evaluating the input function.
	bool GetScalarResult(double &result);
	
	// Description:
	// Get a vector result from evaluating the input function.
	bool GetVectorResult(double result[3]);
	
	// Description:
	// Set the value of a scalar variable.  If a variable with this name
	// exists, then its value will be Set to the new value.  If there is not
	// already a variable with this name, variableName will be added to the
	// list of variables, and its value will be Set to the new value.
	void SetScalarVariableValue(const iString &variableName, double value);
	
	// Description:
	// Set the value of a vector variable.  If a variable with this name
	// exists, then its value will be Set to the new value.  If there is not
	// already a variable with this name, variableName will be added to the
	// list of variables, and its value will be Set to the new value.
	void SetVectorVariableValue(const iString &variableName, double xValue, double yValue, double zValue);
	void SetVectorVariableValue(const iString &variableName, const double values[3]) { this->SetVectorVariableValue(variableName,values[0],values[1],values[2]); }

	// Description:
	// Get the number of scalar variables.
	int GetNumberOfScalarVariables() const { return mScalarVariables.Size(); }
	
	// Description:
	// Get the number of vector variables.
	int GetNumberOfVectorVariables() const { return mVectorVariables.Size(); }
	
	// Description:
	// Remove all the current variables.
	void RemoveAllVariables();

	//  Get the error string & position
	const iString& GetErrorMessage() const { return mErrorMessage; }
	int GetErrorPosition() const;

	//  block/unblock division by zero errors
	bool GetIgnoreDivisionByZero() const { return mIgnoreDivZero; }
	void SetIgnoreDivisionByZero(bool s){ mIgnoreDivZero = s; mModified = true; }

	const char GetElementaryOperatorByteEncoding(ParserTypes op) const;

	bool CheckSyntax(const iString &s);

protected:
	
	int Parse();
	void Evaluate();
	
	int CheckSyntax();
	void RemoveSpaces();
	
	int BuildInternalFunctionStructure();
	void BuildInternalSubstringStructure(int beginIndex, int endIndex);
	void AddInternalByte(unsigned char newByte, int pos);
	
	int IsSubstringCompletelyEnclosed(int beginIndex, int endIndex) const;
	int FindEndOfMathFunction(int beginIndex) const;
	
	int IsVariableName(int currentIndex) const;
	int IsElementaryOperator(int op) const;
	
	int GetMathFunctionNumber(int currentIndex) const;
	int GetMathFunctionStringLength(int mathFunctionNumber) const;
	int GetElementaryOperatorNumber(char op) const;
	int GetOperandNumber(int currentIndex);
	int GetVariableNameLength(int variableNumber) const;
	
	int DisambiguateOperators();
	
	void Error(const iString &text, int bytecode = -1, int position = -1);

	struct VariableBase
	{
		iString Name;
	    bool operator==(const VariableBase &b) const
		{
			return (Name == b.Name);
		}
		bool operator<(const VariableBase &b) const
		{
			return (Name < b.Name);
		}
	};

	struct Scalar : public VariableBase
	{
		double Value;
		Scalar(){ Value = 0.0; }
	};

	struct Vector : public VariableBase
	{
		double Value[3];
		Vector(){ Value[0] = Value[1] = Value[2] = 0.0; }
	};

	iString mFunction;
	int mStackSize, mStackPointer;

	iOrderedArray<Scalar> mScalarVariables;
	iOrderedArray<Vector> mVectorVariables;

	iArray<double> mStack;
	iArray<unsigned char> mByteCode;
	iArray<int> mBytePosition;
	iArray<double> mImmediates;

	int mErrorByteCode, mErrorPosition;
	iString mErrorMessage;
	bool mModified, mValuesChanged, mIgnoreDivZero;
};

#endif  // IEXPRESSIONPARSER_H
