#if defined HAVE_CONFIG_H
#include <config.h>
#endif
                                                                                
#include <iostream.h>
#include <string.h>
#include <id3/tag.h>
#include <getopt.h>
#include <stdlib.h>
#include <id3/misc_support.h>

#define VERSION_NUMBER "$Revision: 0.1 $"
#define MAXNOFRAMES 1000

void PrintUsage(char *sName)
{
  cout << "Usage: " << sName << " [OPTION]... [FILE]..." << endl;
  cout << "Adds/Modifies/Removes/Views id3v2 tags, converts/lists id3v1 tags" << endl;
  cout << endl;
  cout << "  -h,  --help               Display this help and exit" << endl;
  cout << "  -f,  --list-frames        Display all possible frames for id3v2" << endl;
  cout << "  -L,  --list-genres        Lists all id3v1 genres" << endl;
  cout << "  -v,  --version            Display version information and exit" << endl;
  cout << "  -l,  --list               Lists the tag(s) on the file(s)" << endl;
  cout << "  -R,  --list-rfc822        Lists using an rfc822-style format for output" << endl;
  cout << "  -d,  --delete-v2          Deletes id3v2 tags" << endl;
  cout << "  -s,  --delete-v1          Deletes id3v1 tags" << endl;
  cout << "  -D,  --delete-all         Deletes both id3v1 and id3v2 tags" << endl;
  cout << "  -C,  --convert            Converts id3v1 tag to id3v2" << endl;
  cout << "  -a,  --artist  \"ARTIST\"   Set the artist information" << endl;
  cout << "  -A,  --album   \"ALBUM\"    Set the album title information" << endl;
  cout << "  -t,  --song    \"SONG\"     Set the song title information" << endl;
  cout << "  -c,  --comment \"DESCRIPTION\":\"COMMENT\"  "<< endl
       << "                            Set the comment information" << endl;
  cout << "  -g,  --genre   num        Set the genre number" << endl;
  cout << "  -y,  --year    num        Set the year" << endl;
  cout << "  -T,  --track   num/num    Set the track number/(optional) total tracks" << endl;
  cout << endl;
}


void PrintVersion(char *sName)
{
  size_t nIndex;
  cout << sName << " ";
  for (nIndex = 0; nIndex < strlen(VERSION_NUMBER); nIndex++)
    if (VERSION_NUMBER[nIndex] == ' ') break;
  nIndex++;
  for (; nIndex < strlen (VERSION_NUMBER); nIndex++)
  {
    if (VERSION_NUMBER[nIndex] == ' ') break;
    cout << VERSION_NUMBER[nIndex];
  }
  cout << endl;
  cout << "Uses " << ID3LIB_FULL_NAME << endl << endl;

  cout << "This program adds/modifies/removes/views id3v2 tags, " << endl 
       << "and can convert from id3v1 tags" << endl;
}


extern void ListTag(int argc, char *argv[], int optind, int rfc822);
extern void PrintFrameHelp(char *sName);
extern void PrintGenreList();

extern void DeleteTag(int argc, char *argv[], int optind, int whichTags); 
extern void ConvertTag(int argc, char *argv[], int optind);

int main( int argc, char *argv[])
{
  int iOpt;
  int argCounter = 0;
  int ii;

  struct frameInfo {
    enum ID3_FrameID id;
    char *data;
  } frameList[MAXNOFRAMES];
  
  int frameCounter = 0;
  
  while (true)
  {
    int option_index = 0;
    int iLongOpt = 0;
    int optFrameID = ID3FID_NOFRAME;
    static struct option long_options[] = 
    { 
    // help and info
      { "help",    no_argument,       &iLongOpt, 'h' },
      { "list-frames",
                   no_argument,       &iLongOpt, 'f' },
      { "list-genres",   
                  no_argument,        &iLongOpt, 'L' },
      { "version", no_argument,       &iLongOpt, 'v' },

    // list / remove / convert
      { "list",   no_argument,        &iLongOpt, 'l' },
      { "list-rfc822",   
                   no_argument,       &iLongOpt, 'R' },
      { "delete-v2",  no_argument,    &iLongOpt, 'd' },
      { "delete-v1",  
                   no_argument,       &iLongOpt, 's' },
      { "delete-all",  
                   no_argument,       &iLongOpt, 'D' },
      { "convert", no_argument,       &iLongOpt, 'C' },

    // infomation to tag
      { "artist",  required_argument, &iLongOpt, 'a' },
      { "album",   required_argument, &iLongOpt, 'A' },
      { "song",    required_argument, &iLongOpt, 't' },
      { "comment", required_argument, &iLongOpt, 'c' },
      { "genre",   required_argument, &iLongOpt, 'g' },
      { "year",    required_argument, &iLongOpt, 'y' },
      { "track",   required_argument, &iLongOpt, 'T' },
      { "AENC",    required_argument, &optFrameID, ID3FID_AUDIOCRYPTO },
      { "APIC",    required_argument, &optFrameID, ID3FID_PICTURE },
      { "COMM",    required_argument, &optFrameID, ID3FID_COMMENT },
    /* COMR too complex */
      { "ENCR",    required_argument, &optFrameID, ID3FID_CRYPTOREG },
      { "EQUA",    required_argument, &optFrameID, ID3FID_EQUALIZATION },
      { "ETCO",    required_argument, &optFrameID, ID3FID_EVENTTIMING },
      { "GEOB",    required_argument, &optFrameID, ID3FID_GENERALOBJECT },
      { "GRID",    required_argument, &optFrameID, ID3FID_GROUPINGREG },
      { "IPLS",    required_argument, &optFrameID, ID3FID_INVOLVEDPEOPLE },
      { "LINK",    required_argument, &optFrameID, ID3FID_LINKEDINFO },
      { "MCDI",    required_argument, &optFrameID, ID3FID_CDID },
      { "MLLT",    required_argument, &optFrameID, ID3FID_MPEGLOOKUP },
      { "OWNE",    required_argument, &optFrameID, ID3FID_OWNERSHIP },
      { "PRIV",    required_argument, &optFrameID, ID3FID_PRIVATE },
      { "PCNT",    required_argument, &optFrameID, ID3FID_PLAYCOUNTER },
      { "POPM",    required_argument, &optFrameID, ID3FID_POPULARIMETER },
      { "POSS",    required_argument, &optFrameID, ID3FID_POSITIONSYNC },
      { "RBUF",    required_argument, &optFrameID, ID3FID_BUFFERSIZE },
      { "RVAD",    required_argument, &optFrameID, ID3FID_VOLUMEADJ },
      { "RVRB",    required_argument, &optFrameID, ID3FID_REVERB },
      { "SYLT",    required_argument, &optFrameID, ID3FID_SYNCEDLYRICS },
      { "SYTC",    required_argument, &optFrameID, ID3FID_SYNCEDTEMPO },
      { "TALB",    required_argument, &optFrameID, ID3FID_ALBUM },
      { "TBPM",    required_argument, &optFrameID, ID3FID_BPM },
      { "TCOM",    required_argument, &optFrameID, ID3FID_COMPOSER },
      { "TCON",    required_argument, &optFrameID, ID3FID_CONTENTTYPE },
      { "TCOP",    required_argument, &optFrameID, ID3FID_COPYRIGHT },
      { "TDAT",    required_argument, &optFrameID, ID3FID_DATE },
      { "TDLY",    required_argument, &optFrameID, ID3FID_PLAYLISTDELAY },
      { "TENC",    required_argument, &optFrameID, ID3FID_ENCODEDBY },
      { "TEXT",    required_argument, &optFrameID, ID3FID_LYRICIST },
      { "TFLT",    required_argument, &optFrameID, ID3FID_FILETYPE },
      { "TIME",    required_argument, &optFrameID, ID3FID_TIME },
      { "TIT1",    required_argument, &optFrameID, ID3FID_CONTENTGROUP },
      { "TIT2",    required_argument, &optFrameID, ID3FID_TITLE },
      { "TIT3",    required_argument, &optFrameID, ID3FID_SUBTITLE },
      { "TKEY",    required_argument, &optFrameID, ID3FID_INITIALKEY },
      { "TLAN",    required_argument, &optFrameID, ID3FID_LANGUAGE },
      { "TLEN",    required_argument, &optFrameID, ID3FID_SONGLEN },
      { "TMED",    required_argument, &optFrameID, ID3FID_MEDIATYPE },
      { "TOAL",    required_argument, &optFrameID, ID3FID_ORIGALBUM },
      { "TOFN",    required_argument, &optFrameID, ID3FID_ORIGFILENAME },
      { "TOLY",    required_argument, &optFrameID, ID3FID_ORIGLYRICIST },
      { "TOPE",    required_argument, &optFrameID, ID3FID_ORIGARTIST },
      { "TORY",    required_argument, &optFrameID, ID3FID_ORIGYEAR },
      { "TOWN",    required_argument, &optFrameID, ID3FID_FILEOWNER },
      { "TPE1",    required_argument, &optFrameID, ID3FID_LEADARTIST },
      { "TPE2",    required_argument, &optFrameID, ID3FID_BAND },
      { "TPE3",    required_argument, &optFrameID, ID3FID_CONDUCTOR },
      { "TPE4",    required_argument, &optFrameID, ID3FID_MIXARTIST },
      { "TPOS",    required_argument, &optFrameID, ID3FID_PARTINSET },
      { "TPUB",    required_argument, &optFrameID, ID3FID_PUBLISHER },
      { "TRCK",    required_argument, &optFrameID, ID3FID_TRACKNUM },
      { "TRDA",    required_argument, &optFrameID, ID3FID_RECORDINGDATES },
      { "TRSN",    required_argument, &optFrameID, ID3FID_NETRADIOSTATION },
      { "TRSO",    required_argument, &optFrameID, ID3FID_NETRADIOOWNER },
      { "TSRC",    required_argument, &optFrameID, ID3FID_ISRC },
      { "TSSE",    required_argument, &optFrameID, ID3FID_ENCODERSETTINGS },
      { "TXXX",    required_argument, &optFrameID, ID3FID_USERTEXT },
      { "TYER",    required_argument, &optFrameID, ID3FID_YEAR },
      { "UFID",    required_argument, &optFrameID, ID3FID_UNIQUEFILEID },
      { "USER",    required_argument, &optFrameID, ID3FID_TERMSOFUSE },
      { "USLT",    required_argument, &optFrameID, ID3FID_UNSYNCEDLYRICS },
      { "WCOM",    required_argument, &optFrameID, ID3FID_WWWCOMMERCIALINFO },
      { "WCOP",    required_argument, &optFrameID, ID3FID_WWWCOPYRIGHT },
      { "WOAF",    required_argument, &optFrameID, ID3FID_WWWAUDIOFILE },
      { "WOAR",    required_argument, &optFrameID, ID3FID_WWWARTIST },
      { "WOAS",    required_argument, &optFrameID, ID3FID_WWWAUDIOSOURCE },
      { "WORS",    required_argument, &optFrameID, ID3FID_WWWRADIOPAGE },
      { "WPAY",    required_argument, &optFrameID, ID3FID_WWWPAYMENT },
      { "WPUB",    required_argument, &optFrameID, ID3FID_WWWPUBLISHER },
      { "WXXX",    required_argument, &optFrameID, ID3FID_WWWUSER },
      { 0, 0, 0, 0 }
    };
    iOpt = getopt_long (argc, argv, "hfLvlRdsDCa:A:t:c:g:y:T:",
                        long_options, &option_index);

    if (iOpt == -1  && argCounter == 0)
    {
      PrintUsage(argv[0]);
      exit(0);
    } 
    else if (iOpt == -1)
      break;
    argCounter++;

    if (iOpt == 0) iOpt = iLongOpt;
//    if (iOpt == 0) iOpt = optFrameID;
    

    switch (iOpt)
    {
      case 0:
                frameList[frameCounter].id   = (enum ID3_FrameID)optFrameID;
                frameList[frameCounter].data = optarg;
                frameCounter++;
                break;
      case '?': 
      case 'h': PrintUsage(argv[0]);    exit (0);
      case 'f': PrintFrameHelp(argv[0]);exit (0);
      case 'L': PrintGenreList();       exit (0);
      case 'v': PrintVersion(argv[0]);  exit (0);

    // listing / remove / convert -- see other.cpp
      case 'l': ListTag(argc, argv, optind, 0);    
                                        exit (0);
      case 'R': ListTag(argc, argv, optind, 1);    
                                        exit (0);
      case 'd': DeleteTag(argc, argv, optind, 2);    
                                        exit (0);
      case 's': DeleteTag(argc, argv, optind, 1);    
                                        exit (0);
      case 'D': DeleteTag(argc, argv, optind, 0);    
                                        exit (0);
      case 'C': ConvertTag(argc, argv, optind);    
                                        exit (0);

    // Tagging stuff 
      case 'a': 
                frameList[frameCounter].id   = ID3FID_LEADARTIST;
                frameList[frameCounter].data = optarg;
                frameCounter++;
                break;
      case 'A': 
                frameList[frameCounter].id   = ID3FID_ALBUM;
                frameList[frameCounter].data = optarg;
                frameCounter++;
                break;
      case 't': 
                frameList[frameCounter].id   = ID3FID_TITLE;
                frameList[frameCounter].data = optarg;
                frameCounter++;
                break;
      case 'c': 
                frameList[frameCounter].id   = ID3FID_COMMENT;
                frameList[frameCounter].data = optarg;
                frameCounter++;
                break;
      case 'g': 
                frameList[frameCounter].id   = ID3FID_CONTENTTYPE;
                frameList[frameCounter].data = optarg;
                frameCounter++;
                break;
      case 'y': 
                frameList[frameCounter].id   = ID3FID_YEAR;
                frameList[frameCounter].data = optarg;
                frameCounter++;
                break;
      case 'T': 
                frameList[frameCounter].id   = ID3FID_TRACKNUM;
                frameList[frameCounter].data = optarg;
                frameCounter++;
                break;
    // other tags
    
      default:
                cerr << "This isn't supposed to happen" << endl;
                exit(1);
    }
  }
  
  // loop thru the files
  if (optind == argc) 
  {
    cerr << "No file to work on." << endl;
    exit(1);
  }
  
  for (size_t nIndex = optind; nIndex < argc; nIndex++)
  {
      try
      {
        ID3_Tag myTag;

        // cout << "Tagging " << argv[nIndex] << ": ";

        // fix me - not checking to see if we can link to it
        myTag.Link(argv[nIndex], (luint) ID3TT_ID3V2);

        // loop thru the frames we need to add/modify
        for(ii = 0; ii < frameCounter; ii++) 
        {
          ID3_Frame *myFrame;
          myFrame = new ID3_Frame;
          if (NULL == myFrame)
          {
             cout << "\nOut of memory\n" << endl;
             exit(1);
          }
                              
          switch (frameList[ii].id)
          {
          //  strings
            case ID3FID_ALBUM:
            case ID3FID_BPM:
            case ID3FID_COMPOSER:
            case ID3FID_CONTENTTYPE:
            case ID3FID_COPYRIGHT:
            case ID3FID_DATE:
            case ID3FID_PLAYLISTDELAY:
            case ID3FID_ENCODEDBY:
            case ID3FID_LYRICIST:
            case ID3FID_FILETYPE:
            case ID3FID_TIME:
            case ID3FID_CONTENTGROUP:
            case ID3FID_TITLE:
            case ID3FID_SUBTITLE:
            case ID3FID_INITIALKEY:
            case ID3FID_LANGUAGE:
            case ID3FID_SONGLEN:
            case ID3FID_MEDIATYPE:
            case ID3FID_ORIGALBUM:
            case ID3FID_ORIGFILENAME:
            case ID3FID_ORIGLYRICIST:
            case ID3FID_ORIGARTIST:
            case ID3FID_ORIGYEAR:
            case ID3FID_FILEOWNER:
            case ID3FID_LEADARTIST:
            case ID3FID_BAND:
            case ID3FID_CONDUCTOR:
            case ID3FID_MIXARTIST:
            case ID3FID_PARTINSET:
            case ID3FID_PUBLISHER:
            case ID3FID_RECORDINGDATES:
            case ID3FID_NETRADIOSTATION:
            case ID3FID_NETRADIOOWNER:
            case ID3FID_SIZE:
            case ID3FID_ISRC:
            case ID3FID_ENCODERSETTINGS:
            case ID3FID_YEAR:
            {
              ID3_Frame *pFrame = NULL;
              pFrame = myTag.Find(frameList[ii].id);

              if(pFrame != NULL) 
              {
                myTag.RemoveFrame(pFrame);
              }
              myFrame->SetID(frameList[ii].id);
              myFrame->Field(ID3FN_TEXT) = frameList[ii].data;
              myTag.AttachFrame(myFrame);
              break;
            }
            case ID3FID_TRACKNUM:
            {
              // check if there is a total track number and if we only have 
              // the track number for this file.  In this case combine them.
              ID3_Frame *pFrame = NULL;
              char *currentTrackNum, *newTrackNum;
              pFrame = myTag.Find(ID3FID_TRACKNUM);

              if(pFrame != NULL) 
              {
                currentTrackNum = ID3_GetString(pFrame, ID3FN_TEXT);
                if(*currentTrackNum == '/') 
                {
                  newTrackNum = (char *)malloc(strlen(currentTrackNum) 
                                       + strlen(frameList[ii].data)); 
                  strcpy(newTrackNum, frameList[ii].data);
                  strcat(newTrackNum, currentTrackNum);
                }
                else
                {
                  myTag.RemoveFrame(pFrame);
                }
              }
              
              myFrame->SetID(ID3FID_TRACKNUM);
              myFrame->Field(ID3FN_TEXT) = frameList[ii].data;
              myTag.AttachFrame(myFrame);
                                                      
              free(newTrackNum);
              break;
            }
            case ID3FID_USERTEXT:
            {
              ID3_Frame *pFrame = NULL;
              pFrame = myTag.Find(frameList[ii].id);

              if(pFrame != NULL) 
              {
                myTag.RemoveFrame(pFrame);
              }
              
              
              // split the string at the ':' remember if no : then leave descip empty
              char *text;
              text = strchr(frameList[ii].data, ':');
              text++;
              
              break;
            }
            case ID3FID_COMMENT:
            case ID3FID_UNSYNCEDLYRICS:
            {
              char 
                *sText = ID3_GetString(myFrame, ID3FN_TEXT), 
                *sDesc = ID3_GetString(myFrame, ID3FN_DESCRIPTION), 
                *sLang = ID3_GetString(myFrame, ID3FN_LANGUAGE);
              cout << "(" << sDesc << ")[" << sLang << "]: "
                   << sText << endl;
              delete [] sText;
              delete [] sDesc;
              delete [] sLang;
              break;
            }
            case ID3FID_WWWAUDIOFILE:
            case ID3FID_WWWARTIST:
            case ID3FID_WWWAUDIOSOURCE:
            case ID3FID_WWWCOMMERCIALINFO:
            case ID3FID_WWWCOPYRIGHT:
            case ID3FID_WWWPUBLISHER:
            case ID3FID_WWWPAYMENT:
            case ID3FID_WWWRADIOPAGE:
            {
              char *sURL = ID3_GetString(myFrame, ID3FN_URL);
              cout << sURL << endl;
              delete [] sURL;
              break;
            }
            case ID3FID_WWWUSER:
            {
              char 
                *sURL = ID3_GetString(myFrame, ID3FN_URL),
                *sDesc = ID3_GetString(myFrame, ID3FN_DESCRIPTION);
              cout << "(" << sDesc << "): " << sURL << endl;
              delete [] sURL;
              delete [] sDesc;
              break;
            }
            case ID3FID_INVOLVEDPEOPLE:
            {
              // This isn't the right way to do it---will only get first person
              size_t nItems = myFrame->Field(ID3FN_TEXT).GetNumTextItems();
              for (size_t nIndex = 1; nIndex <= nItems; nIndex++)
              {
                char *sPeople = ID3_GetString(myFrame, ID3FN_TEXT, nIndex);
                cout << sPeople;
                delete [] sPeople;
                if (nIndex < nItems)
                { 
                  cout << ", ";
                }
              }
              cout << endl;
              break;
            }
            case ID3FID_PICTURE:
            {
              char
                *sMimeType = ID3_GetString(myFrame, ID3FN_MIMETYPE),
                *sDesc     = ID3_GetString(myFrame, ID3FN_DESCRIPTION),
                *sFormat   = ID3_GetString(myFrame, ID3FN_IMAGEFORMAT);
              size_t
                nPicType   = myFrame->Field(ID3FN_PICTURETYPE).Get(),
                nDataSize  = myFrame->Field(ID3FN_DATA).Size();
              cout << "(" << sDesc << ")[" << sFormat << ", "
                   << nPicType << "]: " << sMimeType << ", " << nDataSize
                   << " bytes" << endl;
              delete [] sMimeType;
              delete [] sDesc;
              delete [] sFormat;
              break;
            }
            case ID3FID_GENERALOBJECT:
            {
              char 
                *sMimeType = ID3_GetString(myFrame, ID3FN_TEXT), 
                *sDesc = ID3_GetString(myFrame, ID3FN_DESCRIPTION), 
                *sFileName = ID3_GetString(myFrame, ID3FN_FILENAME);
              size_t 
              nDataSize = myFrame->Field(ID3FN_DATA).Size();
              cout << "(" << sDesc << ")[" 
                  << sFileName << "]: " << sMimeType << ", " << nDataSize
                  << " bytes" << endl;
              delete [] sMimeType;
              delete [] sDesc;
              delete [] sFileName;
              break;
            }
            case ID3FID_UNIQUEFILEID:
            {
              char *sOwner = ID3_GetString(myFrame, ID3FN_TEXT);
              size_t nDataSize = myFrame->Field(ID3FN_DATA).Size();
              cout << sOwner << ", " << nDataSize
                   << " bytes" << endl;
              delete [] sOwner;
              break;
            }
            case ID3FID_PLAYCOUNTER:
            {
              size_t nCounter = myFrame->Field(ID3FN_COUNTER).Get();
              cout << nCounter << endl;
              break;
            }
            case ID3FID_POPULARIMETER:
            {
              char *sEmail = ID3_GetString(myFrame, ID3FN_EMAIL);
              size_t
                nCounter = myFrame->Field(ID3FN_COUNTER).Get(),
                nRating = myFrame->Field(ID3FN_RATING).Get();
              cout << sEmail << ", counter=" 
                   << nCounter << " rating=" << nRating;
              delete [] sEmail;
              break;
            }
            case ID3FID_CRYPTOREG:
            case ID3FID_GROUPINGREG:
            {
              char *sOwner = ID3_GetString(myFrame, ID3FN_OWNER);
              size_t 
                nSymbol = myFrame->Field(ID3FN_ID).Get(),
                nDataSize = myFrame->Field(ID3FN_DATA).Size();
              cout << "(" << nSymbol << "): " << sOwner
                   << ", " << nDataSize << " bytes";
              break;
            }
            case ID3FID_AUDIOCRYPTO:
            case ID3FID_EQUALIZATION:
            case ID3FID_EVENTTIMING:
            case ID3FID_CDID:
            case ID3FID_MPEGLOOKUP:
            case ID3FID_OWNERSHIP:
            case ID3FID_PRIVATE:
            case ID3FID_POSITIONSYNC:
            case ID3FID_BUFFERSIZE:
            case ID3FID_VOLUMEADJ:
            case ID3FID_REVERB:
            case ID3FID_SYNCEDLYRICS:
            case ID3FID_SYNCEDTEMPO:
            case ID3FID_METACRYPTO:
            {
              cout << " (unimplemented)" << endl;
              break;
            }
            default:
            {
              cout << " frame" << endl;
              break;
            }
          }
        }  // steping thru frames
       
        luint nTags = myTag.Update(ID3TT_ID3V2);

      }
      catch(ID3_Error err)
      {
        cout << endl;
        cout << err.GetErrorFile() << " (" << err.GetErrorLine() << "): "
             << err.GetErrorType() << ": " << err.GetErrorDesc() << endl;
      }
  }

  return 0;
}
