#include "util.h"
#include <features.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <stdio.h>
#include <ctype.h>
#include <errno.h>
#include <signal.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <unistd.h>
#include <limits.h>
#include "common.h"

/* get_basename_no_ext:
 *   Modify `filename' to remove leading directories and first file
 *   extension. Return pointer in `filename' to start character.
 */
char *
get_basename_no_ext(char *filename)
{
	char *str;

	filename = basename(filename);
	str = strrchr(filename, '.');

	if (str != NULL)
		str[0] = '\0';

	return filename;
}

/* is_directory:
 *   Return true if file exists, is readable and is a directory.
 */
boolean
is_directory(char *filename)
{
	struct stat statbuf;

	if (stat(filename, &statbuf) == -1)
		return FALSE;

	if (S_ISDIR(statbuf.st_mode))
		return TRUE;

	return FALSE;
}

/* get_file_size:
 *   Get the size of a file `filename' and put it in `size'.
 *   Returns FALSE if stat system call failed.
 *  
 */
boolean
get_file_size (char *filename, int *size)
{
	struct stat file_stat;

	if (stat(filename, &file_stat))
		return FALSE;

	*size = file_stat.st_size;
	return TRUE;
}

/* str2int: Convert a string to integer, returning boolean indicating whether
 * succeeded.
 */
boolean
str2int (char *str, int *value)
{
	char *endptr;
	long int longvalue;

	endptr = NULL;
	longvalue = strtol(str, &endptr, 0);

	if (endptr[0] != '\0' || (endptr[0] == '\0' && str[0] == '\0')
	  || longvalue == LONG_MIN || longvalue == LONG_MAX
	  || (int) longvalue != longvalue)
		return FALSE;

	*value = (int) longvalue;
	return TRUE;
}

/* int2str: Convert integer value to string. len specifies the length of the
 * string array, including room for null byte at end.
 */
void
int2str (int value, char *str, int len)
{
	int max, c;

	len--;
	c = 0;

	/* negative value */
	if (value < 0) {
		str[0] = '-';
		value = -value;
		c++;
	}

	/* find out how many decimals we need */
	max = 1;
	while (value >= max)
		max *= 10;
	if (max != 1)
		max /= 10;

	for ( ; max >= 1 && c < len; c++) {
		str[c] = (value / max) + '0';
		value = value % max;
		max /= 10;
	}

	str[c] = '\0';
}

/* vec: An implementation of the perl vec() function.
 *  Treats the string in vector as a vector of unsigned integers, and returns the value
 *  of the bit field specified by offset. size specifies the number of bits that are
 *  reserved for each entry in the bit vector.
 *  Note: Due to the win32 image format, we start with bit 7 and work our way down to 0.
 *  In the perl implementation, I believe bit 0 is really the first bit we encounter.
 */
unsigned int
vec(vector, offset, size)
  char *vector;
  int offset;
  int size;
{
	int start_byte, start_bit;
	int end_byte, end_bit;
	unsigned int mod, value;

	start_byte = (offset * size) / 8;
	start_bit = 7 - (offset * size) % 8;

	end_byte = ((offset + 1) * size) / 8;
	end_bit = 7 - ((offset + 1) * size) % 8;

	value = 0;
	mod = 1<<(size-1);

	while (start_byte != end_byte || start_bit != end_bit) {
		if (vector[start_byte] & (1<<start_bit))
			value |= mod;
		mod >>= 1;

		start_bit--;
		if (start_bit < 0) {
			start_bit = 7;
			start_byte++;
		}
	}
	
	return value;
}
