/** 
* @file planartree.c
* @brief data representation part
* @author serge guelton
* @date 2008-02-13
*/
/*
 * This file is part of hyantes.
 *
 * hyantes is free software; you can redistribute it and/or modify
 * it under the terms of the CeCILL-C License
 *
 * You should have received a copy of the CeCILL-C License
 * along with this program.  If not, see <http://www.cecill.info/licences>.
 */

#include "hs_config.h"
#include "planartree.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>

static planartree *add_tree(const QuadT iter_in,    /*@returned@ */
   planartree * iter_out) /*@modifies iter_out@ */ ;

/** 
* @brief allocates a new planar tree from a quadtree
*   the quad tree representation is easy to build, but it does not behave well according to memory (lots of cache miss)
*   so we move it to a easier to allocate and to navigate structure here
* 
* @param qtree origian quadtree
* 
* @return newly allocated planartree, or NULL if an error occured
*/
planartree *create_planartree(const QuadT qtree)
{
    planartree *ptree =
       (planartree *) malloc(sizeof(*ptree) * (1 + qtree->size));
    planartree *ptree_end = NULL;
    if(ptree == NULL)
    {
        perror("[create_planartree::malloc] ");
        return NULL;
    }

    memset(ptree, 0, sizeof(*ptree) * (1 + qtree->size));

    ptree_end = add_tree(qtree, ptree);
    if(ptree_end != ptree + qtree->size)
    {
        fprintf(stderr, "[create_planartree::add_tree] some data miss\n");
        free(ptree);
        return NULL;
    }

    ptree[qtree->size].size = 0;

    return ptree;
}



/**
 *  recursievly adds quadtree to an array
 *  \param iter_in pointer to the quadtree to add
 *  \param iter_out pointeur to a free place in the out array
 *  \return nouvel iterator to a new free place in out array
 */
static planartree *add_tree(const QuadT iter_in, planartree * iter_out)
{
    iter_out->value = iter_in->value;

    switch (iter_in->size)
    {
    case 0:
        fprintf(stderr,
           "[create_planartree::add_tree] quadtree has impossible null size\n");
        exit(EXIT_FAILURE);
    case 1:
        iter_out->size = 1;
        iter_out->coo.mLat = iter_out->coo.MLat = iter_in->coords.mLat;
        iter_out->coo.mLon = iter_out->coo.MLon = iter_in->coords.mLon;
        break;
    default:
        iter_out->size = iter_in->size;
        iter_out->coo.mLat = iter_in->coords.mLat;
        iter_out->coo.mLon = iter_in->coords.mLon;
        iter_out->coo.MLat = iter_in->coords.MLat;
        iter_out->coo.MLon = iter_in->coords.MLon;
        break;
    };

    ++iter_out;
    if(iter_in->fbg)
        iter_out = add_tree(iter_in->fbg, iter_out);
    if(iter_in->fbd)
        iter_out = add_tree(iter_in->fbd, iter_out);
    if(iter_in->fhg)
        iter_out = add_tree(iter_in->fhg, iter_out);
    if(iter_in->fhd)
        iter_out = add_tree(iter_in->fhd, iter_out);

    return iter_out;
}
