// -*- C++ -*-

/**
 *
 * @class  ThreadLocalAllocationBuffer
 * @author Emery Berger <http://www.cs.umass.edu/~emery>
 * @brief  An allocator, meant to be used for thread-local allocation.
 */

#include "dllist.h"
#include "array.h"

using namespace HL;

template <int NumBins,
	  int (*getSizeClass) (size_t),
	  size_t (*getClassSize) (const int),
	  int LocalHeapThreshold,
	  class SuperblockType,
	  int SuperblockSize,
	  class ParentHeap>

class ThreadLocalAllocationBuffer {

public:

  ThreadLocalAllocationBuffer (ParentHeap * parent)
    : _localHeapBytes (0),
      _parentHeap (parent)
  {
  }

  inline void * malloc (size_t sz) {
    // Get memory from the local heap,
    // and deduct that amount from the local heap bytes counter.
    int c = getSizeClass (sz);
    void * ptr = localHeap(c).get();
    if (ptr) {
      _localHeapBytes -= getClassSize(c);
      return ptr;
    } else {
      // No more local memory - get more memory from our parent.
      ptr = _parentHeap->malloc (sz);
      return ptr;
    }
  }

  inline void free (void * ptr) {

    // Get all the size and ownership info about this object.

    if (ptr == NULL) {
      return;
    }

    const SuperblockType * s = getSuperblock (ptr);
    assert (s != NULL);
    assert (s->isValidSuperblock());

    const void * owner = (const void *) (s->getOwner());
    assert (owner != NULL);

    const size_t sz = s->getObjectSize();
    const int c = getSizeClass (sz);

    // Check to see if this pointer is 'remote' or local.

    if (owner == _parentHeap) {
      // It's local. What happens local stays local :).
      localHeap(c).insert ((DLList::Entry *) ptr);
      _localHeapBytes += sz;
    } else {
      // It's remote. Free it directly to the parent heap.
      _parentHeap->free (ptr);
    }
    
    // If we have too much memory stored locally,
    // give half back.
    
    if (_localHeapBytes > LocalHeapThreshold) {
      clearHalf();
    }
  }

  void clear (void) {
    // Free every object to the 'parent' heap.
    for (int i = 0; i < NumBins; i++) {
      while (!localHeap(i).isEmpty()) {
	DLList::Entry * e = localHeap(i).get();
	_parentHeap->free (e);
      }
    }
    _localHeapBytes = 0;
  }

private:

  void clearHalf (void) {
    // Free objects to the 'parent' heap
    // until we've given up half of the threshold amount of memory.
    // We start with the bigger objects first, and work our way down.
    for (int i = NumBins - 1; i >= 0; i--) {
      size_t sz = getClassSize (i);
      while (!localHeap(i).isEmpty()) {
	DLList::Entry * e = localHeap(i).get();
	_parentHeap->free (e);
	_localHeapBytes -= sz;
      }
      if (_localHeapBytes < (LocalHeapThreshold / 2)) {
	return;
      }
    }
  }

  // Disable assignment and copying.

  ThreadLocalAllocationBuffer (const ThreadLocalAllocationBuffer&);
  ThreadLocalAllocationBuffer& operator=(const ThreadLocalAllocationBuffer&);

  /// Use bit masking to find the start of the containing superblock.
  static inline SuperblockType * getSuperblock (void * ptr) {
    SuperblockType * s = reinterpret_cast<SuperblockType *>(reinterpret_cast<size_t>(ptr) & ~(SuperblockSize-1));
    assert (s->isValidSuperblock());
    return s;
  }

  /// This heap's 'parent' (where to go for more memory).
  ParentHeap * _parentHeap;

  /// The number of bytes we currently have on this thread.
  int _localHeapBytes;

  /// The local heap itself.
  Array<NumBins, DLList> localHeap;

};
