/*  -*-objc-*-
 *  SmallIconsPanel.m: Implementation of the SmallIconsPanel Class 
 *  of the GNUstep GWorkspace application
 *
 *  Copyright (c) 2001 Enrico Sersale <enrico@imago.ro>
 *  
 *  Author: Enrico Sersale
 *  Date: August 2001
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#import <Foundation/Foundation.h>
#import <AppKit/AppKit.h>
#include "SmallIconsPanel.h"
#include "SmallIcon.h"
#include "Functions.h"
#include "GNUstep.h"

#ifndef max
#define max(a,b) ((a) >= (b) ? (a):(b))
#endif

#ifndef min
#define min(a,b) ((a) <= (b) ? (a):(b))
#endif

#define ICNWIDTH 32
#define ICNHEIGHT 26
#define LABHEIGHT 14
#define ROWSHEIGHT 32
#define LEFTMARGIN 16

extern NSString *GWorkspaceRecycleOutOperation;

#ifndef GNUSTEP
	NSString *GWIconsCellsWidthChangedNotification = @"GWIconsCellsWidthChangedNotification";
#else
	extern NSString *GWIconsCellsWidthChangedNotification;
#endif

@implementation SmallIconsPanel

- (void)dealloc
{
	[[NSNotificationCenter defaultCenter] removeObserver: self];
  RELEASE (icons);
  RELEASE (currentPath);
  TEST_RELEASE(charBuffer);
  [super dealloc];
}

- (id)initAtPath:(NSString *)path
        delegate:(id)adelegate
{
  self = [super initWithFrame: NSZeroRect];
  if (self) {
    ASSIGN (currentPath, path);
		[self setDelegate: adelegate];
		
		cellsWidth = [delegate getIconViewCellsWidth];
    [self setAutoresizingMask: (NSViewWidthSizable | NSViewHeightSizable)];            
    icons = [[NSMutableArray alloc] initWithCapacity: 1]; 		
		currSelectionSel = @selector(currentSelection);
		currSelection = [self methodForSelector: currSelectionSel];	
		isDragTarget = NO;
    isShiftClick = NO;
		selectInProgress = NO;
		
		lastKeyPressed = 0.;
  	charBuffer = nil;
		
  	[self registerForDraggedTypes: [NSArray arrayWithObjects: NSFilenamesPboardType, nil]];

    [[NSNotificationCenter defaultCenter] addObserver: self 
                				  selector: @selector(cellsWidthChanged:) 
                					    name: GWIconsCellsWidthChangedNotification
                					  object: nil];

		fm = [NSFileManager defaultManager];
    [self makeFileIcons];
  }
	
  return self;
}

- (void)setPath:(NSString *)path
{
  ASSIGN (currentPath, path);
  [self makeFileIcons];
  [self resizeWithOldSuperviewSize: [self frame].size];  
}

- (NSString *)currentPath
{
  return currentPath;
}

- (void)makeFileIcons
{
  NSArray *files, *hiddenFiles;
  NSMutableArray *paths;
	NSUserDefaults *defaults;
	BOOL gsFileBrowserHideDotFiles;
	NSString *h;	
  int i, count;
  int stype;

  for (i = 0; i < [icons count]; i++) {
    SmallIcon *icon = [icons objectAtIndex: i];
		NSTextField *label = [icon label];
		[label setDelegate: nil];
		[label setEditable: NO];
    [label removeFromSuperview]; 							
    [icon removeFromSuperview];
  }
  
  [icons removeAllObjects];
  
  files = [fm directoryContentsAtPath: currentPath];
	
	h = [currentPath stringByAppendingPathComponent: @".hidden"];
	h = [NSString stringWithContentsOfFile: h];
  hiddenFiles = [h componentsSeparatedByString: @"\n"];
	
	defaults = [NSUserDefaults standardUserDefaults];
	gsFileBrowserHideDotFiles = [defaults boolForKey: @"GSFileBrowserHideDotFiles"];
	
	if (hiddenFiles != nil  ||  gsFileBrowserHideDotFiles) {	
		NSMutableArray *mutableFiles = AUTORELEASE ([files mutableCopy]);
	
		if (hiddenFiles != nil) {
	    [mutableFiles removeObjectsInArray: hiddenFiles];
	  }
	
		if (gsFileBrowserHideDotFiles) {
	    int j = [mutableFiles count] - 1;
	    
	    while (j >= 0) {
				NSString *file = (NSString *)[mutableFiles objectAtIndex: j];

				if ([file hasPrefix: @"."]) {
		    	[mutableFiles removeObjectAtIndex: j];
		  	}
				j--;
			}
	  }
		
		files = mutableFiles;
	}
 	
  count = [files count];
  if (count == 0) {
  	[self resizeWithOldSuperviewSize: [self frame].size]; 
		return;
	}

  paths = [NSMutableArray arrayWithCapacity: 1];

  for (i = 0; i < count; ++i) {
    NSString *s = [currentPath stringByAppendingPathComponent: [files objectAtIndex: i]];
    [paths addObject: s];
  }

	stype = [delegate sortTypeForDirectoryAtPath: currentPath];
  paths = AUTORELEASE ([[paths sortedArrayUsingFunction: (int (*)(id, id, void*))sortPaths
                                      context: (void *)stype] mutableCopy]);

  for (i = 0; i < count; ++i) {
    NSString *s = [paths objectAtIndex: i];
    [self addIconsWithPaths: [NSArray arrayWithObject: s]];
  }  

  [self resizeWithOldSuperviewSize: [self frame].size];  
}

- (void)addIconsWithPaths:(NSArray *)iconpaths
{
  int i;
  
  for (i = 0; i < [iconpaths count]; i++) {
		NSString *s = [iconpaths objectAtIndex: i];
    SmallIcon *icon = [[SmallIcon alloc] initForPath: s delegate: self];
    [icons addObject: icon];  
	  [self addSubview: icon];
	  [self addSubview: [icon label]];
    RELEASE (icon);
  }
}

- (void)removeIcon:(id)anIcon
{
  SmallIcon *icon = (SmallIcon *)anIcon;
  [[icon label] removeFromSuperview];
  [icon removeFromSuperview];
  [icons removeObject: icon];
	[self resizeWithOldSuperviewSize: [self frame].size];  
}

- (void)removeIconsWithNames:(NSArray *)names
{
  int i, count = [icons count];
  
  for (i = 0; i < count; i++) {
    SmallIcon *icon = [icons objectAtIndex: i];
    NSString *name = [icon name];

    if ([names containsObject: name] == YES) {
      [[icon label] removeFromSuperview];
      [icon removeFromSuperview];
      [icons removeObject: icon];
      count--;
      i--;
    }
  }

  [self resizeWithOldSuperviewSize: [self frame].size];  
}

- (void)dimIconsWithNames:(NSArray *)names
{
	SmallIcon *icon;
  int i;
  
  for (i = 0; i < [icons count]; i++) {
		icon = [icons objectAtIndex: i];
    if ([names containsObject: [icon name]]) {
			[icon setDimmed: YES];
    }
  }
}

- (void)unselectOtherIcons:(id)anIcon
{
  int i;
  
  if(isShiftClick == YES) {
    return;
  }
  
  for (i = 0; i < [icons count]; i++) {
    SmallIcon *icon = [icons objectAtIndex: i];
    if (icon != anIcon) {  
      [icon unselect];
    }
  }  
}

- (void)selectIconsWithPaths:(NSArray *)paths
{
  int i;
  
  isShiftClick = YES;

  for (i = 0; i < [icons count]; i++) {
    SmallIcon *icon = [icons objectAtIndex: i];    
    NSString *ipath = [icon path];

    if ([paths containsObject: ipath] == YES) {
      [icon select];
    }
  }

  isShiftClick = NO;
}

- (void)selectAllIcons
{
	int i;

	isShiftClick = YES;  
	selectInProgress = YES;
	for(i = 0; i < [icons count] - 1; i++) {
		[[icons objectAtIndex: i] select];
	}
	selectInProgress = NO;
	[[icons objectAtIndex: [icons count] - 1] select];
	isShiftClick = NO;
}

- (void)selectNextIcon
{
	int i, count = [icons count];
    
	for(i = 0; i < [icons count]; i++) {
		SmallIcon *icon = [icons objectAtIndex: i];
    
		if([icon isSelect]) {
			if (i < (count - 1)) {
				[self unselectOtherIcons: icon];
				icon = [icons objectAtIndex: i + 1];
				[icon select];
				[self scrollRectToVisible: [icon frame]];		
				return;
			} else {
				return;
			}
		}
	} 
}

- (void)selectPrevIcon
{
	int i;
    
	for(i = 0; i < [icons count]; i++) {
		SmallIcon *icon = [icons objectAtIndex: i];
    
		if([icon isSelect]) {
			if (i > 0) {
				[self unselectOtherIcons: icon];
				icon = [icons objectAtIndex: i - 1];
				[icon select];
				[self scrollRectToVisible: [icon frame]];		
				return;
			} else {
				return;
			}
		}
	}
}

- (void)selectIconInNextLine
{
	NSPoint sp;
	int i, startpos = -1;
	
	for (i = 0; i < [icons count]; i++) {
		SmallIcon *icon = [icons objectAtIndex: i];
		
		if ([icon isSelect]) {
			startpos = i;
			sp = [icon position];
			break;
		}
	}

	if (startpos != -1) {
		for (i = startpos; i < [icons count]; i++) {
			SmallIcon *icon = [icons objectAtIndex: i];
			NSPoint p = [icon position];

			if ((p.x == sp.x) && (p.y < sp.y)) {
				[icon select];
				[self scrollRectToVisible: [icon frame]];	
				break;
			}
		}
	}	
}

- (void)selectIconInPrevLine
{
	NSPoint sp;
	int i, startpos = -1;
	
	for (i = [icons count] -1; i >= 0; i--) {
		SmallIcon *icon = [icons objectAtIndex: i];
		
		if ([icon isSelect]) {
			startpos = i;
			sp = [icon position];
			break;
		}
	}

	if (startpos != -1) {
		for (i = startpos; i >= 0; i--) {
			SmallIcon *icon = [icons objectAtIndex: i];
			NSPoint p = [icon position];

			if ((p.x == sp.x) && (p.y > sp.y)) {
				[icon select];
				[self scrollRectToVisible: [icon frame]];	
				break;
			}
		}
	}	
}

- (SmallIcon *)iconWithNamePrefix:(NSString *)prefix 
													inRange:(NSRange)range
{
	int i;

	if (range.length == 0) {
		return nil;
	}
	
	for (i = range.location; i < range.location + range.length; i++) {
    SmallIcon *icon = [icons objectAtIndex: i];
    if ([[icon name] hasPrefix: charBuffer]) {
			return icon;	
    }
	}
	
	return nil;
}

- (NSPoint)locationOfIconWithName:(NSString *)name
{
	int i;

  for (i = 0; i < [icons count]; i++) {
    SmallIcon *icon = [icons objectAtIndex: i]; 
		
    if ([[icon name] isEqualToString: name]) {	
			NSPoint p = [icon frame].origin;
			NSSize s = [icon iconShift];
			return NSMakePoint(p.x + s.width, p.y + s.height);
    }
  } 

	return NSMakePoint(0, 0);
}

- (void)setLabelRectOfIcon:(id)anIcon
{
  SmallIcon *icon = (SmallIcon *)anIcon;
	NSTextField *label = [icon label];

	[label setFrame: NSMakeRect([icon frame].origin.x + ICNWIDTH, 
							[icon frame].origin.y + ((ICNHEIGHT - LABHEIGHT) / 2), 
																						[icon labelWidth], LABHEIGHT)];
					
	[label setNeedsDisplay: YES];
}

- (void)setCurrentSelection
{
	if (selectInProgress) {
		return;
	}
	[delegate setTheSelectedPaths: (*currSelection)(self, currSelectionSel)];
}

- (NSArray *)currentSelection
{
  NSMutableArray *allpaths = [NSMutableArray arrayWithCapacity: 1]; 
  int i;

  for (i = 0; i < [icons count]; i++) {
    SmallIcon *icon = [icons objectAtIndex: i];    
    if ([icon isSelect] == YES) {
      [allpaths addObject: [icon path]];
    }
  } 
  
  if ([allpaths count] == 0) {
    return nil;
  }
    
  return allpaths;
}

- (void)openCurrentSelection:(NSArray *)paths newViewer:(BOOL)newv
{
	[delegate openTheCurrentSelection: (*currSelection)(self, currSelectionSel) 
													newViewer: newv];
}

- (void)setShiftClick:(BOOL)value
{
  isShiftClick = value;
}

- (int)cellsWidth
{
  return cellsWidth;
}

- (void)cellsWidthChanged:(NSNotification *)notification
{
	cellsWidth = [delegate getIconViewCellsWidth];	  
  [self resizeWithOldSuperviewSize: [self frame].size];
}

- (id)delegate
{
  return delegate;
}

- (void)setDelegate:(id)anObject
{	
  delegate = anObject;
}

- (void)resizeWithOldSuperviewSize:(NSSize)oldFrameSize
{
  float fw = [[self superview] frame].size.width;
  float fh = [[self superview] frame].size.height;
	float oy = [self frame].origin.y;
	float posx = LEFTMARGIN;
	float posy = ROWSHEIGHT;
	float shiftx = 0;
	int count = [icons count];
	NSRect *irects = NSZoneMalloc (NSDefaultMallocZone(), sizeof(NSRect) * count);
	int i;

	[self setFrame: NSMakeRect(0, oy, fw, fh)];
	
	for (i = 0; i < count; i++) {
		SmallIcon *icon = [icons objectAtIndex: i];
		float labwidth = [icon labelWidth];
		float iconwidth = [icon frame].size.width + labwidth;
		
		posx += shiftx;

    if (posx >= (fw - iconwidth)) {
      posx = LEFTMARGIN;  
			shiftx = 0;    
      posy += ROWSHEIGHT;  
			
			if (posy > fh) {
				fh = posy;
				[self setFrame: NSMakeRect(0, oy, fw, fh)];
			}
    }
		
		irects[i] = NSMakeRect(posx, posy, ICNWIDTH, ICNHEIGHT);
		
		while (iconwidth > shiftx) {
			shiftx += cellsWidth;
		}
	}

	posy += (ROWSHEIGHT / 2);  
	if (posy > fh) {
		fh = posy;
		[self setFrame: NSMakeRect(0, oy, fw, fh)];
	}

	for (i = 0; i < count; i++) {   
		SmallIcon *icon = [icons objectAtIndex: i];
		irects[i].origin.y = fh - irects[i].origin.y;
		[icon setFrame: irects[i]];
		[icon setPosition: irects[i].origin gridIndex: i];
    [icon setNeedsDisplay: YES];
		[self setLabelRectOfIcon: icon];
	}
	
	NSZoneFree (NSDefaultMallocZone(), irects);

  if ([icons count] > 0) {
    [self scrollRectToVisible: NSMakeRect(0, posy, 10, 10)];
  }
	    
	[self setNeedsDisplay: YES];
}

- (void)mouseDown:(NSEvent *)theEvent
{
  [[self window] makeFirstResponder: self];
  
	if([theEvent modifierFlags] != 2) {
		isShiftClick = NO;
		[self unselectOtherIcons: nil];
		[delegate setTheSelectedPaths: [NSArray arrayWithObject: currentPath]];
	}
}

- (void)mouseDragged:(NSEvent *)theEvent
{
  static NSRect	oldRect; 
  NSPoint basep;
  
  basep = [self convertPoint: [theEvent locationInWindow] fromView: nil];  
  oldRect = NSZeroRect;  
  [self setShiftClick: YES];
  selectInProgress = YES;
	
  while (1) {
		NSEvent	*nextEvent;
		NSPoint p;
  	float x, y, w, h;
  	NSRect r;
  	int i;

	  nextEvent = [[self window] nextEventMatchingMask:
    							            NSLeftMouseUpMask | NSLeftMouseDraggedMask];

    if ([nextEvent type] != NSLeftMouseDragged) {
      break;
    }

    p = [self convertPoint: [nextEvent locationInWindow] fromView: nil];

    x = (p.x >= basep.x) ? basep.x : p.x;
    y = (p.y >= basep.y) ? basep.y : p.y;
    w = max(p.x, basep.x) - min(p.x, basep.x);
    w = (w == 0) ? 1 : w;
    h = max(p.y, basep.y) - min(p.y, basep.y);
    h = (h == 0) ? 1 : h;

    r = NSMakeRect(x, y, w, h);
       
    if (NSEqualRects(r, oldRect) == NO) {
	    oldRect = r;
      [self autoscroll: nextEvent];
	  }

    for (i = 0; i < [icons count]; i++) {
      SmallIcon *icon = [icons objectAtIndex: i];
      NSPoint center = [icon center];
			
      if (NSPointInRect(center, r)) {
        [icon select];
      } else {
        [icon unselect];
      }
    }    
  }
  selectInProgress = NO;
	[self setCurrentSelection];
  [self setShiftClick: NO];
}

- (void)keyDown:(NSEvent *)theEvent 
{
	NSString *characters;
	unichar character;
	NSRect vRect, hiddRect;
	NSPoint p;
	float x, y, w, h;
		
	characters = [theEvent characters];
	character = 0;
		
  if ([characters length] > 0) {
		character = [characters characterAtIndex: 0];
	}
		
	switch (character) {
    case NSPageUpFunctionKey:
		  vRect = [self visibleRect];
		  p = vRect.origin;    
		  x = p.x;
		  y = p.y + vRect.size.height;
		  w = vRect.size.width;
		  h = vRect.size.height;
		  hiddRect = NSMakeRect(x, y, w, h);
		  [self scrollRectToVisible: hiddRect];
	    return;

    case NSPageDownFunctionKey:
		  vRect = [self visibleRect];
		  p = vRect.origin;
		  x = p.x;
		  y = p.y - vRect.size.height;
		  w = vRect.size.width;
		  h = vRect.size.height;
		  hiddRect = NSMakeRect(x, y, w, h);
		  [self scrollRectToVisible: hiddRect];
	    return;

    case NSUpArrowFunctionKey:
			isShiftClick = NO;
	    [self selectIconInPrevLine];
      return;

    case NSDownArrowFunctionKey:
			isShiftClick = NO;
	    [self selectIconInNextLine];
      return;

    case NSLeftArrowFunctionKey:
			{
				if ([theEvent modifierFlags] & NSControlKeyMask) {
	      	[super keyDown: theEvent];
	    	} else {
					isShiftClick = NO;
	    		[self selectPrevIcon];
				}
			}
      return;

    case NSRightArrowFunctionKey:			
			{
				if ([theEvent modifierFlags] & NSControlKeyMask) {
	      	[super keyDown: theEvent];
	    	} else {
					isShiftClick = NO;
	    		[self selectNextIcon];
				}
			}
      return;

    case 13:
      [self openCurrentSelection: (*currSelection)(self, currSelectionSel)
											 newViewer: NO];      
      return;

		case NSTabCharacter:
	  	{
	    	if ([theEvent modifierFlags] & NSShiftKeyMask) {
	      	[[self window] selectKeyViewPrecedingView: self];
	    	} else {
	      	[[self window] selectKeyViewFollowingView: self];
				}
	  	}
			return;

      break;
	} 

  if ((character < 0xF700) && ([characters length] > 0)) {
		SEL iwnpSel = @selector(iconWithNamePrefix:inRange:);
		IMP iwnp = [self methodForSelector: iwnpSel];
		SmallIcon *icon;
		int i, s, count;
		
		s = -1;
		count = [icons count];
	  
		if (charBuffer == nil) {
	    charBuffer = [characters substringToIndex: 1];
	    RETAIN (charBuffer);
	  } else {
	    if ([theEvent timestamp] - lastKeyPressed < 2000.0) {
		  	ASSIGN (charBuffer, ([charBuffer stringByAppendingString:
				    															[characters substringToIndex: 1]]));
			} else {
		  	ASSIGN (charBuffer, ([characters substringToIndex: 1]));
			}														
		}

		lastKeyPressed = [theEvent timestamp];

		[self setShiftClick: NO];
		
		for (i = 0; i < count; i++) {
      icon = [icons objectAtIndex: i];
			
			if ([icon isSelect]) {												
      	if ([[icon name] hasPrefix: charBuffer]) {
					[self scrollRectToVisible: [icon frame]];	
					return;
      	} else {
					s = i;
					break;
				}
			}
		}
		
  	icon = (*iwnp)(self, iwnpSel, charBuffer, NSMakeRange(s + 1, count -s -1));
		if (icon) {
  		[icon select];
			[self scrollRectToVisible: [icon frame]];
			return;	
		}
		
		s = (s == -1) ? count - 1 : s;
		
  	icon = (*iwnp)(self, iwnpSel, charBuffer, NSMakeRange(0, s)); 
		if (icon) {
  		[icon select];
			[self scrollRectToVisible: [icon frame]];
			return;	
		}
				
		lastKeyPressed = 0.;			
	}
	
	[super keyDown: theEvent];
}

- (BOOL)acceptsFirstResponder
{
  return YES;
}

- (BOOL)acceptsFirstMouse:(NSEvent *)theEvent
{
	return YES;
}

@end

@implementation SmallIconsPanel (DraggingDestination)

- (unsigned int)draggingEntered:(id <NSDraggingInfo>)sender
{
	NSPasteboard *pb;
  NSDragOperation sourceDragMask;
	NSData *pbData;
	NSArray *sourcePaths;
	NSString *fromPath;
  NSString *buff;
	int i, count;

	isDragTarget = NO;	
    
 	pb = [sender draggingPasteboard];
  if ([[pb types] indexOfObject: NSFilenamesPboardType] != NSNotFound) {
  
    pbData = [pb dataForType: NSFilenamesPboardType];
    sourcePaths = [NSUnarchiver unarchiveObjectWithData: pbData];  
  
	  count = [sourcePaths count];
	  fromPath = [[sourcePaths objectAtIndex: 0] stringByDeletingLastPathComponent];
  
	  if (count == 0) {
		  return NSDragOperationNone;
    } 
  
	  if ([fm isWritableFileAtPath: currentPath] == NO) {
		  return NSDragOperationNone;
	  }
  
	  if ([currentPath isEqualToString: fromPath]) {
		  return NSDragOperationNone;
    }  
  
	  for (i = 0; i < count; i++) {
		  if ([currentPath isEqualToString: [sourcePaths objectAtIndex: i]]) {
		    return NSDragOperationNone;
		  }
	  }
  
	  buff = [NSString stringWithString: currentPath];
	  while (1) {
		  for (i = 0; i < count; i++) {
			  if ([buff isEqualToString: [sourcePaths objectAtIndex: i]]) {
 		      return NSDragOperationNone;
			  }
		  }
      if ([buff isEqualToString: @"/"] == YES) {
        break;
      }            
		  buff = [buff stringByDeletingLastPathComponent];
	  }
  
    isDragTarget = YES;	

		sourceDragMask = [sender draggingSourceOperationMask];
	
		if (sourceDragMask == NSDragOperationCopy) {
			return NSDragOperationCopy;
		} else if (sourceDragMask == NSDragOperationLink) {
			return NSDragOperationLink;
		} else {
			return NSDragOperationAll;
		}		
    return NSDragOperationAll;
  }
  
	isDragTarget = NO;	
  return NSDragOperationNone;
}

- (unsigned int)draggingUpdated:(id <NSDraggingInfo>)sender
{
  NSDragOperation sourceDragMask;
	
	if (isDragTarget == NO) {
		return NSDragOperationNone;
	}

	sourceDragMask = [sender draggingSourceOperationMask];
	
	if (sourceDragMask == NSDragOperationCopy) {
		return NSDragOperationCopy;
	} else if (sourceDragMask == NSDragOperationLink) {
		return NSDragOperationLink;
	} else {
		return NSDragOperationAll;
	}

	return NSDragOperationNone;
}

- (void)draggingExited:(id <NSDraggingInfo>)sender
{
	isDragTarget = NO;
}

- (BOOL)prepareForDragOperation:(id <NSDraggingInfo>)sender
{
	return isDragTarget;
}

- (BOOL)performDragOperation:(id <NSDraggingInfo>)sender
{
	return YES;
}

- (void)concludeDragOperation:(id <NSDraggingInfo>)sender
{
	NSPasteboard *pb;
  NSDragOperation sourceDragMask;
	NSData *pbData;
	NSArray *sourcePaths;
  NSString *operation, *source;
  NSMutableArray *files;
	NSMutableDictionary *opDict;
	NSString *trashPath;
  int i;

  isDragTarget = NO;
	
	sourceDragMask = [sender draggingSourceOperationMask];  	
  pb = [sender draggingPasteboard];
  pbData = [pb dataForType: NSFilenamesPboardType];
  sourcePaths = [NSUnarchiver unarchiveObjectWithData: pbData];
  
  source = [[sourcePaths objectAtIndex: 0] stringByDeletingLastPathComponent];

	trashPath = [delegate getTrashPath];
	if ([source isEqualToString: trashPath]) {
		operation = GWorkspaceRecycleOutOperation;
	} else {
		if (sourceDragMask == NSDragOperationCopy) {
			operation = NSWorkspaceCopyOperation;
		} else if (sourceDragMask == NSDragOperationLink) {
			operation = NSWorkspaceLinkOperation;
		} else {
			operation = NSWorkspaceMoveOperation;
		}
  }
  
  files = [NSMutableArray arrayWithCapacity: 1];    
  for(i = 0; i < [sourcePaths count]; i++) {    
    [files addObject: [[sourcePaths objectAtIndex: i] lastPathComponent]];
  }  

	opDict = [NSMutableDictionary dictionaryWithCapacity: 4];
	[opDict setObject: operation forKey: @"operation"];
	[opDict setObject: source forKey: @"source"];
	[opDict setObject: currentPath forKey: @"destination"];
	[opDict setObject: files forKey: @"files"];
	
	[delegate performFileOperationWithDictionary: opDict];	
}

@end

//
// SmallIcon Delegate Methods
//
@implementation SmallIconsPanel (SmallIconDelegateMethods)

- (NSImage *)smallIconForFile:(NSString*)aPath
{
	return [delegate smallIconForFile: aPath];
}

- (NSImage *)smallIconForFiles:(NSArray*)pathArray
{
	return [delegate smallIconForFiles: pathArray];
}

- (void)unselectIconsDifferentFrom:(id)aicon
{
	[self unselectOtherIcons: aicon];
}

- (void)setShiftClickValue:(BOOL)value
{
	[self setShiftClick: value];
}

- (void)setTheCurrentSelection
{
	[self setCurrentSelection];
}

- (NSArray *)getTheCurrentSelection
{
	return (*currSelection)(self, currSelectionSel);
}

- (void)openTheCurrentSelection:(id)paths newViewer:(BOOL)newv
{
	[self openCurrentSelection: paths newViewer: newv];
}

- (void)performFileOperationWithDictionary:(id)opdict
{
	[delegate performFileOperationWithDictionary: opdict];
}

- (NSString *)getTrashPath
{
	return [delegate getTrashPath];
}

@end
