 /*  -*-objc-*-
 *  PathIcon.m: Implementation of the PathIcon Class 
 *  of the GNUstep GWorkspace application
 *
 *  Copyright (c) 2001 Enrico Sersale <enrico@imago.ro>
 *  
 *  Author: Enrico Sersale
 *  Date: August 2001
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#import <Foundation/Foundation.h>
#import <AppKit/AppKit.h>
#include "PathIcon.h"
#include "PathIconLabel.h"
#include "IconsPath.h"
#include "Functions.h"
#include "GNUstep.h"

#define ICONPOSITION(s1, s2) (NSMakePoint(((int)(s1).width - (int)(s2).width) >> 1, \
((int)(s1).height - (int)(s2).height) >> 1))

#define ONICON(p, s1, s2) ([self mouse: (p) \
inRect: NSMakeRect(((int)(s1).width - (int)(s2).width) >> 1,\
((int)(s1).height - (int)(s2).height) >> 1, 48, 48)])

/* Notifications */
extern NSString *GWFileSystemWillChangeNotification;
extern NSString *GWFileSystemDidChangeNotification;
extern NSString *GWDidSetFileAttributesNotification;
extern NSString *GWSortTypeDidChangeNotification;

/* File Operations */
extern NSString *GWorkspaceRenameOperation;

extern NSString *GWorkspaceRecycleOutOperation;

@implementation PathIcon

- (void)dealloc
{
	[[NSNotificationCenter defaultCenter] removeObserver: self];
  TEST_RELEASE (paths);
  TEST_RELEASE (fullpath);
  TEST_RELEASE (name);
	TEST_RELEASE (hostname);
  TEST_RELEASE (type);
	RELEASE (namelabel);
  TEST_RELEASE (icon);
  RELEASE (highlight);
  RELEASE (arrow); 
  [super dealloc];
}

- (id)initWithDelegate:(id)aDelegate
{
  self = [super init];
  
  if (self) {
    fm = [NSFileManager defaultManager];
		ws = [NSWorkspace sharedWorkspace];
		
    [self setDelegate: aDelegate];    
		
    ASSIGN (highlight, [NSImage imageNamed: @"CellHighlight.tiff"]);
    ASSIGN (arrow, [NSImage imageNamed: @"common_3DArrowRight.tiff"]);
    icon = nil;

    namelabel = [[PathIconLabel alloc] initForPathIcon: self];
    [namelabel setDelegate: self];  
		[namelabel setFont: [NSFont systemFontOfSize: 12]];
		[namelabel setBezeled: NO];
		[namelabel setEditable: NO];
		[namelabel setSelectable: NO];
		[namelabel setAlignment: NSCenterTextAlignment];
	  [namelabel setBackgroundColor: [NSColor windowBackgroundColor]];

		hostname = nil;
    isbranch = NO;
    dimmed = NO;
    singlepath = YES;
    isSelect = NO;
    dragdelay = 0;
    isDragTarget = NO;
		isRootIcon = NO;

    [self registerForDraggedTypes: [NSArray arrayWithObjects: NSFilenamesPboardType, nil]];    
  }
  
  return self;
}

- (void)setPaths:(NSArray *)p
{
	NSString *defapp = nil, *t = nil;
  float width, labwidth;
  int count;

  if ([p isEqualToArray: paths]) {
    return;
  }

  if (p == nil) {
    TEST_RELEASE (paths);
    paths = nil;
    TEST_RELEASE (fullpath);
    TEST_RELEASE (name);
    TEST_RELEASE (type);
    type = nil;
    TEST_RELEASE (icon);
    icon = nil;
    return;
  }

  ASSIGN (paths, p);
  count = [paths count];                    

  if (count == 1) {
    singlepath = YES;   
    ASSIGN (fullpath, [paths objectAtIndex: 0]);    
		if ([fullpath isEqualToString: @"/"]) {
			ASSIGN (name, fullpath);
			isRootIcon = YES;
		} else {
    	ASSIGN (name, [fullpath lastPathComponent]);
			isRootIcon = NO;
		}
    [ws getInfoForFile: fullpath application: &defapp type: &t];      
    ASSIGN (type, t);
  } else {
    singlepath = NO;
    ASSIGN (name, ([NSString stringWithFormat: @"%i items", count]));
    type = nil;
		isRootIcon = NO;
  }

  if (singlepath == YES) {
    ASSIGN (icon, [ws iconForFile: fullpath]);    
  } else {
    ASSIGN (icon, [NSImage imageNamed: @"MultipleSelection.tiff"]);
  }
  
  width = [self frame].size.width;
  labwidth = [[namelabel font] widthOfString: name] + 8;
  if (labwidth > width) {
    labwidth = width;
  }
	[namelabel setFrame: NSMakeRect(0, 0, labwidth, 14)];  

	if (isRootIcon == NO) {
  	[namelabel setStringValue: cutLabel(name, namelabel, labwidth)];
  } else {
		NSHost *host = [NSHost currentHost];
		NSString *hname = [host name];
		NSRange range = [hname rangeOfString: @"."];
			
		if (range.length != 0) {	
			hname = [hname substringToIndex: range.location];
		} 			
		ASSIGN (hostname, hname);
		[namelabel setStringValue: hostname];
	}

	[delegate setLabelFrameOfIcon: self];	
	[self setNeedsDisplay: YES];
}

- (void)setFrame:(NSRect)frameRect
{
  float width, labwidth;

  [super setFrame: frameRect];
  
  width = [self frame].size.width;

	if (isRootIcon == NO) {
  	labwidth = [[namelabel font] widthOfString: name] + 8;
  } else {
  	labwidth = [[namelabel font] widthOfString: hostname] + 8;
	}
  
  [namelabel setFrame: NSMakeRect(0, 0, labwidth, 14)];    
  [delegate setLabelFrameOfIcon: self];
  
	[self setNeedsDisplay: YES];
}

- (void)select
{
  if (isSelect) {
    return;
  }

	if (isRootIcon == NO) {
		[namelabel setFrame: NSMakeRect(0, 0, [[namelabel font] widthOfString: name] + 8, 14)];
		[namelabel setStringValue: name];
  } else {
		[namelabel setFrame: NSMakeRect(0, 0, [[namelabel font] widthOfString: hostname] + 8, 14)];
		[namelabel setStringValue: hostname];
	}
  
	[namelabel setBackgroundColor: [NSColor whiteColor]]; 
  if (singlepath == YES) {
    [namelabel setSelectable: YES];
    [namelabel setEditable: YES];
  }
	[delegate setLabelFrameOfIcon: self];
	[delegate unselectIconsDifferentFrom: self];
	[self setNeedsDisplay: YES];
	isSelect = YES;
}

- (void)selectToEdit
{
  if (isSelect == NO) {
    [self select];
  }
  
  [namelabel selectText: self];
}

- (void)unselect
{
  float width, labwidth;
  
  if (isSelect == NO) {
    return;
  }
  
  if (dimmed == NO) {
    width = [self frame].size.width;  

		if (isRootIcon == NO) {
			labwidth = [[namelabel font] widthOfString: name] + 8;
		} else {
			labwidth = [[namelabel font] widthOfString: hostname] + 8;
		}
    if (labwidth > width) {
      labwidth = width;
    }
	  [namelabel setFrame: NSMakeRect(0, 0, labwidth, 14)];  

		if (isRootIcon == NO) {
			[namelabel setStringValue: cutLabel(name, namelabel, labwidth)];
  	} else {
			[namelabel setStringValue: hostname];
		}     

	  [namelabel setBackgroundColor: [NSColor windowBackgroundColor]];
    [namelabel setEditable: NO];
    [namelabel setSelectable: NO];  
  } else {
    [namelabel setFrame: NSMakeRect(0, 0, 10, 14)];  
  }
	[delegate setLabelFrameOfIcon: self];
  
	[self setNeedsDisplay: YES];  
	isSelect = NO;
}

- (BOOL)isSelect
{
  return isSelect;
}

- (NSTextField *)label
{
  return namelabel;
}

- (void)setBranch:(BOOL)value
{
  isbranch = value;  
  [self setDimmed: NO];
}

- (BOOL)isBranch
{
  return isbranch;
}

- (void)setDimmed:(BOOL)value
{
  dimmed = value;
}

- (BOOL)isDimmed
{
  return dimmed;
}

- (BOOL)isRootIcon
{ 
	return isRootIcon;
}

- (NSArray *)paths
{
  return paths;
}

- (NSString *)name
{
  return name;
}

- (NSString *)type
{
  return type;
}

- (NSImage *)icon
{
  return icon;
}

- (void)clickOnLabel
{
  [self select];
	[delegate clickedIcon: self];
}

- (void)mouseDown:(NSEvent *)theEvent
{
	NSEvent *nextEvent;
  BOOL startdnd = NO;
  NSPoint p;
  
  p = [theEvent locationInWindow];
  p = [self convertPoint: p fromView: nil];
  
  if (ONICON(p, [self frame].size, [icon size]) == NO) {    
    return;  
  }

	if ([theEvent clickCount] > 1) { 
		unsigned int modifier = [theEvent modifierFlags];
		
		[delegate doubleClickedIcon: self newViewer: (modifier == NSControlKeyMask)];
    return;
	}  
    
  if (isSelect == NO) {  
    [self select];
  }
   
  while (1) {
	  nextEvent = [[self window] nextEventMatchingMask:
    							            NSLeftMouseUpMask | NSLeftMouseDraggedMask];

    if ([nextEvent type] == NSLeftMouseUp) {
			[delegate clickedIcon: self];
      break;

    } else if ([nextEvent type] == NSLeftMouseDragged) {
	    if(dragdelay < 5) {
        dragdelay++;
      } else {        
        startdnd = YES;        
        break;
      }
    }
  }
  
  if (startdnd == YES) {  
    [self startExternalDragOnEvent: nextEvent];    
  }               
}

- (BOOL)isSinglePath
{
  return singlepath;
}

- (id)delegate
{
  return delegate;
}

- (void)setDelegate:(id)aDelegate
{
	delegate = aDelegate;
}

- (BOOL)acceptsFirstMouse:(NSEvent *)theEvent 
{
  return YES;
}

- (void)controlTextDidChange:(NSNotification *)aNotification
{
  NSDictionary *info = [aNotification userInfo];
  NSText *text = [info objectForKey: @"NSFieldEditor"];  
  NSString *current = [text string];
  [namelabel setFrame: NSMakeRect(0, 0, [[namelabel font] widthOfString: current] + 8, 14)];
	[delegate setLabelFrameOfIcon: self];
}

- (void)controlTextDidEndEditing:(NSNotification *)aNotification
{
  NSDictionary *info;
  NSString *basePath, *newpath, *newname;
	NSString *title, *msg1, *msg2;
  NSMutableDictionary *notifObj;  
  NSArray *dirContents;
  NSCharacterSet *notAllowSet;
  NSRange range;
  BOOL samename;
  NSEvent *e;
  int i;
  
#define CLEAREDITING \
	if (isRootIcon == NO) [namelabel setStringValue: name]; \
	else [namelabel setStringValue: hostname]; \
  if (isRootIcon == NO) [namelabel setFrame: NSMakeRect(0, 0, [[namelabel font] widthOfString: name] + 8, 14)]; \
  else [namelabel setFrame: NSMakeRect(0, 0, [[namelabel font] widthOfString: hostname] + 8, 14)]; \
  [delegate setLabelFrameOfIcon: self]; \
  return

	if (isRootIcon == YES) {
		CLEAREDITING;
	}
	
  info = [aNotification userInfo];
  newname = [[info objectForKey: @"NSFieldEditor"] string];

  basePath = [fullpath stringByDeletingLastPathComponent];
	
	if ([fm fileExistsAtPath: fullpath] == NO) {
		notifObj = [NSMutableDictionary dictionaryWithCapacity: 1];		
		[notifObj setObject: NSWorkspaceDestroyOperation forKey: @"operation"];	
  	[notifObj setObject: basePath forKey: @"source"];	
  	[notifObj setObject: basePath forKey: @"destination"];	
  	[notifObj setObject: [NSArray arrayWithObjects: fullpath, nil] forKey: @"files"];	

		[[NSNotificationCenter defaultCenter]
 				  postNotificationName: GWFileSystemDidChangeNotification
	 									object: notifObj];
		return;
	}
	
	title = NSLocalizedString(@"Error", @"");
	msg1 = NSLocalizedString(@"You have not write permission\nfor ", @"");
	msg2 = NSLocalizedString(@"Continue", @"");		
  if ([fm isWritableFileAtPath: fullpath] == NO) {
    NSRunAlertPanel(title, [NSString stringWithFormat: @"%@\"%@\"!\n", msg1, fullpath], msg2, nil, nil);   
		CLEAREDITING;
  } else if ([fm isWritableFileAtPath: basePath] == NO) {	
    NSRunAlertPanel(title, [NSString stringWithFormat: @"%@\"%@\"!\n", msg1, basePath], msg2, nil, nil);   
		CLEAREDITING;
  }  
      
  notAllowSet = [NSCharacterSet characterSetWithCharactersInString: @"/\\*$|~\'\"`^!?"];
  range = [newname rangeOfCharacterFromSet: notAllowSet];
  
  if (range.length > 0) {
		msg1 = NSLocalizedString(@"Invalid char in name", @"");
    NSRunAlertPanel(title, msg1, msg2, nil, nil);
    CLEAREDITING;
  }	

  dirContents = [fm directoryContentsAtPath: basePath];

  samename = NO;			
  for (i = 0; i < [dirContents count]; i++) {
    if ([newname isEqualToString: [dirContents objectAtIndex:i]]) {    
      if ([newname isEqualToString: name]) {
        CLEAREDITING;
      } else {
        samename = YES;
        break;
      }
    }
  }	
  if (samename == YES) {
		NSString *msg3 = NSLocalizedString(@"The name ", @"");
		msg1 = NSLocalizedString(@" is already in use!", @"");	
    NSRunAlertPanel(title, [NSString stringWithFormat: @"%@'%@'%@!", msg3, newname, msg1], msg2, nil, nil);   
    CLEAREDITING;
  }

  e = [[self window] nextEventMatchingMask: NSLeftMouseUpMask | NSKeyUpMask];
  [[self window] flushWindow];   

  newpath = [basePath stringByAppendingPathComponent: newname];

	notifObj = [NSMutableDictionary dictionaryWithCapacity: 1];		
	[notifObj setObject: GWorkspaceRenameOperation forKey: @"operation"];	
  [notifObj setObject: fullpath forKey: @"source"];	
  [notifObj setObject: newpath forKey: @"destination"];	
  [notifObj setObject: [NSArray arrayWithObjects: @"", nil] forKey: @"files"];	

	[[NSNotificationCenter defaultCenter]
 				 postNotificationName: GWFileSystemWillChangeNotification
	 								object: notifObj];
    
  [fm movePath: fullpath toPath: newpath handler: self];

  [self setPaths: [NSArray arrayWithObject: newpath]];

	[[NSNotificationCenter defaultCenter]
 				 postNotificationName: GWFileSystemDidChangeNotification
	 								object: notifObj];
}

- (BOOL)fileManager:(NSFileManager *)manager 
              shouldProceedAfterError:(NSDictionary *)errorDict
{
	NSString *title = NSLocalizedString(@"Error", @"");
	NSString *msg1 = NSLocalizedString(@"Cannot rename ", @"");
	NSString *msg2 = NSLocalizedString(@"Continue", @"");

  NSRunAlertPanel(title, [NSString stringWithFormat: @"%@'%@'!", msg1, name], msg2, nil, nil);   
	return NO;
}

- (void)fileManager:(NSFileManager *)manager willProcessPath:(NSString *)path
{
}

- (void)drawRect:(NSRect)rect
{
  if (dimmed == YES) {
    return;
  }
  
	if(isSelect) {
		[highlight compositeToPoint: ICONPOSITION(rect.size, [highlight size]) 
                      operation: NSCompositeSourceOver];
	}
	
  if (icon != nil) {
		[icon compositeToPoint: ICONPOSITION(rect.size, [icon size]) 
                 operation: NSCompositeSourceOver];
  }
  
  if (isbranch == YES) {
		[arrow compositeToPoint: NSMakePoint(rect.size.width - 15, 26)
                  operation: NSCompositeSourceOver];
  }
}

@end


@implementation PathIcon (DraggingSource)

- (void)startExternalDragOnEvent:(NSEvent *)event
{
	NSEvent *nextEvent;
  NSPoint dragPoint;
  NSPasteboard *pb;

	nextEvent = [[self window] nextEventMatchingMask:
    							NSLeftMouseUpMask | NSLeftMouseDraggedMask];

  if([nextEvent type] != NSLeftMouseDragged) {
    [self unselect];
   	return;
  }
  
  dragPoint = [nextEvent locationInWindow];
  dragPoint = [self convertPoint: dragPoint fromView: nil];

	pb = [NSPasteboard pasteboardWithName: NSDragPboard];	
  [self declareAndSetShapeOnPasteboard: pb];

  dragdelay = 0;
      
  [self dragImage: icon
               at: dragPoint 
           offset: NSZeroSize
            event: event
       pasteboard: pb
           source: self
        slideBack: NO];
}

- (void)declareAndSetShapeOnPasteboard:(NSPasteboard *)pb
{
  NSArray *dndtypes;
  NSData *pbData;

  dndtypes = [NSArray arrayWithObject: NSFilenamesPboardType];
  [pb declareTypes: dndtypes owner: nil];
  pbData = [NSArchiver archivedDataWithRootObject: paths];
  [pb setData: pbData forType: NSFilenamesPboardType];
}

- (unsigned int)draggingSourceOperationMaskForLocal:(BOOL)flag
{
  return NSDragOperationAll;
}

@end


@implementation PathIcon (DraggingDestination)

- (unsigned int)draggingEntered:(id <NSDraggingInfo>)sender
{
	NSPasteboard *pb;
  NSDragOperation sourceDragMask;
	NSData *pbData;
	NSArray *sourcePaths;
	NSString *fromPath;
  NSString *buff;
	int i, count;

  if (([type isEqualToString: NSDirectoryFileType] == NO)
          && ([type isEqualToString: NSFilesystemFileType] == NO)) {
    return NSDragOperationNone;
  }

	pb = [sender draggingPasteboard];
  if ([[pb types] indexOfObject: NSFilenamesPboardType] != NSNotFound) {
  
    pbData = [pb dataForType: NSFilenamesPboardType];
    sourcePaths = [NSUnarchiver unarchiveObjectWithData: pbData];  
	  count = [sourcePaths count];
	  fromPath = [[sourcePaths objectAtIndex: 0] stringByDeletingLastPathComponent];
    
	  if (count == 0) {
		  return NSDragOperationNone;
    } 
  
	  if ([fm isWritableFileAtPath: fullpath] == NO) {
		  return NSDragOperationNone;
	  }
  
	  if ([fullpath isEqualToString: fromPath]) {
		  return NSDragOperationNone;
    }  
  
	  for (i = 0; i < count; i++) {
		  if ([fullpath isEqualToString: [sourcePaths objectAtIndex: i]]) {
		    return NSDragOperationNone;
		  }
	  }
       
	  buff = [NSString stringWithString: fullpath];
	  while (1) {
		  for (i = 0; i < count; i++) {
			  if ([buff isEqualToString: [sourcePaths objectAtIndex: i]]) {
 		      return NSDragOperationNone;
			  }
		  }
      if ([buff isEqualToString: @"/"] == YES) {
        break;
      }            
		  buff = [buff stringByDeletingLastPathComponent];
	  }
  
    isDragTarget = YES;
    
    ASSIGN (icon, [NSImage imageNamed: @"FileIcon_Directory_Open.tiff"]);
    [self setNeedsDisplay: YES];   
    
		sourceDragMask = [sender draggingSourceOperationMask];
	
		if (sourceDragMask == NSDragOperationCopy) {
			return NSDragOperationCopy;
		} else if (sourceDragMask == NSDragOperationLink) {
			return NSDragOperationLink;
		} else {
			return NSDragOperationAll;
		}
  }
    
  return NSDragOperationNone;
}

- (unsigned int)draggingUpdated:(id <NSDraggingInfo>)sender
{
  NSDragOperation sourceDragMask;
	
	if (isDragTarget == NO) {
		return NSDragOperationNone;
	}

	sourceDragMask = [sender draggingSourceOperationMask];
	
	if (sourceDragMask == NSDragOperationCopy) {
		return NSDragOperationCopy;
	} else if (sourceDragMask == NSDragOperationLink) {
		return NSDragOperationLink;
	} else {
		return NSDragOperationAll;
	}

	return NSDragOperationNone;
}

- (void)draggingExited:(id <NSDraggingInfo>)sender
{
  if (isDragTarget == YES) {
    isDragTarget = NO;  
    ASSIGN (icon, [ws iconForFile: fullpath]);
    [self setNeedsDisplay: YES];   
  }
}

- (BOOL)prepareForDragOperation:(id <NSDraggingInfo>)sender
{
	return isDragTarget;
}

- (BOOL)performDragOperation:(id <NSDraggingInfo>)sender
{
	return YES;
}

- (void)concludeDragOperation:(id <NSDraggingInfo>)sender
{
	NSPasteboard *pb;
  NSDragOperation sourceDragMask;
	NSData *pbData;
	NSArray *sourcePaths;
  NSString *operation, *source;
  NSMutableArray *files;
	NSMutableDictionary *opDict;
	NSString *trashPath;
  int i;

  ASSIGN (icon, [ws iconForFile: fullpath]);
  [self setNeedsDisplay: YES];
	isDragTarget = NO;  

	sourceDragMask = [sender draggingSourceOperationMask];
  pb = [sender draggingPasteboard];
  pbData = [pb dataForType: NSFilenamesPboardType];
  sourcePaths = [NSUnarchiver unarchiveObjectWithData: pbData];
  
  source = [[sourcePaths objectAtIndex: 0] stringByDeletingLastPathComponent];

	trashPath = [delegate getTrashPath];

	if ([source isEqualToString: trashPath]) {
		operation = GWorkspaceRecycleOutOperation;
	} else {
		if (sourceDragMask == NSDragOperationCopy) {
			operation = NSWorkspaceCopyOperation;
		} else if (sourceDragMask == NSDragOperationLink) {
			operation = NSWorkspaceLinkOperation;
		} else {
			operation = NSWorkspaceMoveOperation;
		}
  }
  
  files = [NSMutableArray arrayWithCapacity: 1];    
  for(i = 0; i < [sourcePaths count]; i++) {    
    [files addObject: [[sourcePaths objectAtIndex: i] lastPathComponent]];
  }  

	opDict = [NSMutableDictionary dictionaryWithCapacity: 4];
	[opDict setObject: operation forKey: @"operation"];
	[opDict setObject: source forKey: @"source"];
	[opDict setObject: fullpath forKey: @"destination"];
	[opDict setObject: files forKey: @"files"];
	
	[delegate performFileOperationWithDictionary: opDict];	
}

@end

