/*
Gwenview - A simple image viewer for KDE
Copyright (C) 2000-2002 Aurlien Gteau

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

*/
// QT includes
#include <qpixmap.h>
#include <qpainter.h>

// KDE includes
#include <kapp.h>
#include <kconfig.h>
#include <kdebug.h>

// Our includes
#include "gvpixmap.h"

#include "fullscreenview.moc"


const char* CONFIG_SHOW_PATH="show path";


FullScreenView::FullScreenView(QWidget* parent, GVPixmap* pixmap,bool enabled)
: QWidget(parent), mGVPixmap(pixmap)
{
	setCursor(BlankCursor);
	setBackgroundColor(black);
	setFocusPolicy(StrongFocus);

	enableView(enabled);
}


void FullScreenView::enableView(bool enabled) {
	disconnect(mGVPixmap,0,this,0);

	if (enabled) {
		connect(mGVPixmap,SIGNAL(loaded()),
			this,SLOT(updateView()) );
		setFocus();
	}
}


void FullScreenView::updateView() {
	repaint();
}


void FullScreenView::paintEvent(QPaintEvent*) {
	QPainter painter(this);

	kapp->setOverrideCursor(WaitCursor);

	painter.eraseRect(0,0,width(),height());
	if (!mGVPixmap->isNull()) {
		paintPixmap(&painter);
	}

	if (mShowPath) {
		paintPath(&painter);
	}

	kapp->restoreOverrideCursor();
}


void FullScreenView::paintPixmap(QPainter* painter) {
	int srcWidth,srcHeight,dstWidth,dstHeight;
	int posX,posY;
	srcWidth=mGVPixmap->width();
	srcHeight=mGVPixmap->height();
	dstWidth=width();
	dstHeight=height();

// Need to scale ?
	if (srcWidth<=dstWidth && srcHeight<=dstHeight) {
	// No scale needed
		posX=(dstWidth-srcWidth)/2;
		posY=(dstHeight-srcHeight)/2;

		painter->drawPixmap( posX,posY,mGVPixmap->pixmap());
	} else {
	// Yes, compute scale
		double widthScale=double(dstWidth) / double(srcWidth);
		double heightScale=double(dstHeight) / double(srcHeight);
		double scale;

		if ( heightScale < widthScale ) {
			scale=heightScale;
			posX=int(dstWidth-srcWidth*scale)/2;
			posY=0;
		} else {
			scale=widthScale;
			posX=0;
			posY=int(dstHeight-srcHeight*scale)/2;
		}

	// Scale image
		painter->scale(scale,scale);
		painter->drawPixmap( int(posX/scale),int(posY/scale),mGVPixmap->pixmap());
		painter->resetXForm();
	}
}


void FullScreenView::paintPath(QPainter* painter) {
	int left=1;
	int top=fontMetrics().height()+1;
	QString path=mGVPixmap->path();

	painter->drawText(left-1,top-1,path);
	painter->drawText(left,top-1,path);
	painter->drawText(left+1,top-1,path);

	painter->drawText(left-1,top,path);
	painter->drawText(left+1,top,path);

	painter->drawText(left-1,top+1,path);
	painter->drawText(left,top+1,path);
	painter->drawText(left+1,top+1,path);

	painter->setPen(white);
	painter->drawText(left,top,path);
}


void FullScreenView::mouseReleaseEvent(QMouseEvent *event) {
	switch(event->button()) {
	case Qt::LeftButton:
		emit selectPrevious();
		return;
	case Qt::RightButton:
		emit selectNext();
		return;
	default:
		break;
	}
}


void FullScreenView::wheelEvent(QWheelEvent *event) {
	if (event->delta()<0) {
		emit selectNext();
	} else {
		emit selectPrevious();
	}
}


void FullScreenView::keyReleaseEvent(QKeyEvent* event) {
	if (event->key()==Key_Escape) {
		emit escapePressed();
		return;
	}
	event->ignore();
}


void FullScreenView::setShowPath(bool value) {
	mShowPath=value;
}


//-Configuration--------------------------
void FullScreenView::readConfig(KConfig* config,const QString& group) {
	config->setGroup(group);
	mShowPath=config->readBoolEntry(CONFIG_SHOW_PATH,true);
}


void FullScreenView::writeConfig(KConfig* config,const QString& group) const {
	config->setGroup(group);
	config->writeEntry(CONFIG_SHOW_PATH,mShowPath);
}
