/*
 * guessnet.c
 * 
 *	(C) Copyright 2001 Enrico Zini <zinie@cs.unibo.it>
 *	Based on laptop-netconf.c by Matt Kern <matt@debian.org>
 *	Based on divine.c by Felix von Leitner <felix@fefe.de>
 * 
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version.
 * 
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 * 
 *	You should have received a copy of the GNU General Public License
 *	along with this program; see the file COPYING.	If not, write to
 *	the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 *	Boston, MA 02111-1307, USA.
 */

#define _GNU_SOURCE	/* for strndup, asprintf */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <pthread.h>
#include <stdio.h>
#include <ctype.h>
#include <net/ethernet.h>	/* needed to compile on S390
							 * (thanks to Gerhard Tonn <GerhardTonn@swol.de>)
							 */
#include <libnet.h>
#include <pcap.h>
#include <unistd.h> /* fork */
#include <sys/types.h>	/* regcomp, regexec, wait, fork */
#include <sys/wait.h>	/* wait */
#include <string.h>	/* strndup, memcmp */
#include <errno.h>	/* errno */
#include <popt.h>	/* commandline parsing */
#include <regex.h>	/* regcomp, regexec */

/* Default time we wait after initializing an interface */
#define DEFAULT_INIT_TIME 3
/* Default timeout after which we decide we haven't found anything */
#define DEFAULT_TIMEOUT 5
/* Default tag to print when nothing is found */
#define DEFAULT_PROFILE "none"
/* Default iterface to use */
#define DEFAULT_DEVICE "eth0"

typedef enum { false = 0, true = 1 } bool;

/* IPV4 address, stored in network byte order */
typedef unsigned char ipv4addr[4];
/* MAC address, stored in network byte order */
typedef unsigned char macaddr[6];

/* #ifdef WORD_BIGENDIAN */
/* TODO: check if byte ordering is ok here or if we need to do the ifdef and
 * byte swap a little bit */
#define IPv4_FROM_LIBNET(target, source) memcpy(target, source, 4)
#define IPv4_MATCHES(addr1, addr2) (memcmp((addr1), (addr2), 4) == 0)
#define MAC_FROM_LIBNET(target, source) memcpy(target, source, 6)
#define MAC_MATCHES(addr1, addr2) (memcmp((addr1), (addr2), 6) == 0)

struct nethost
{
	ipv4addr local_ip_address;
	ipv4addr probe_ip_address;
	macaddr hardware_address;
	const char* profile;
	struct nethost* next;
};

/* Candidate hosts to check */
struct nethost *nethosts;
int nethost_count;

/* Interface to use */
const char* ethernet_device = DEFAULT_DEVICE;

/* Default profile name */
char* defprof = 0;

/* True when operations should be verbose */
bool verbose = false;

/* True when operations should be very verbose */
bool debug = false;

/* True when sending should stop */
bool send_done = false;

/* Timeout after which we decide we haven't found anything */
int timeout = DEFAULT_TIMEOUT;

/* Time we wait after initializing an interface */
int init_time = DEFAULT_INIT_TIME;

/* Sender thread id */
pthread_t sender_thread_id;

/* Exit for a fatal error */
void fatal_error (const char *ptr)
{
	fprintf (stderr, "guessnet: %s\n", ptr);
	exit (1);
}

/* TODO: check if these four work well on big endian machines */

/* Format an IPv4 address in a static char buffer */
const char* format_ipv4(ipv4addr* addr)
{
	static char buf[15];
	char* s;
	int i;
	for (s = buf, i = 0; i < 4; i++)
	{
		if (i > 0)
			*s++ = '.';
		s += snprintf(s, 4, "%d", (int)(((unsigned char*)(*addr))[i]));
	}
	*s = 0;
	return buf;
}

/* Format a MAC address in a static char buffer */
const char* format_mac(macaddr* addr)
{
	static char buf[20];
	char* s;
	int i;
	for (s = buf, i = 0; i < 6; i++)
	{
		if (i > 0)
			*s++ = ':';
		s += snprintf(s, 3, "%02X", (int)(((unsigned char*)(*addr))[i]));
	}
	*s = 0;
	return buf;
}

/* Parse an IPV4 address from a dotted-quad string */
int parse_ipv4(ipv4addr* target, const char* str)
{
	unsigned int a, b, c, d;
	if (sscanf(str, "%u.%u.%u.%u", &a, &b, &c, &d) == 4)
	{
		(*target)[0] = a;
		(*target)[1] = b;
		(*target)[2] = c;
		(*target)[3] = d;
		return 1;
	} else
		return 0;
}

/* Parse a MAC address from its canonical string representation */
int parse_mac(macaddr* target, const char* str)
{
	unsigned int a, b, c, d, e, f;
	if (sscanf(str, "%x:%x:%x:%x:%x:%x", &a, &b, &c, &d, &e, &f) == 6)
	{
		(*target)[0] = a;
		(*target)[1] = b;
		(*target)[2] = c;
		(*target)[3] = d;
		(*target)[4] = e;
		(*target)[5] = f;
		return 1;
	} else
		return 0;
}


/* ARP sender thread
 * Very simple: just send probe packets for all candidates until send_done becomes true
 */
void* send_arp_requests_thread (void *arg)
{
	char buffer[1024];
	struct libnet_link_int* link_interface;
	struct ether_addr* local_hardware_addr;
	unsigned char ether_broadcast_addr[6] = {0xff, 0xff, 0xff, 0xff, 0xff, 0xff};
	unsigned char ether_no_addr[6] = {0x00, 0x00, 0x00, 0x00, 0x00, 0x00};
	struct nethost* cycle;
	int send_count = 0;

	if (verbose)
		fprintf(stderr, "Start of sender thread (%d candidates to probe)\n",
				nethost_count);

	if (!(link_interface = libnet_open_link_interface (ethernet_device, buffer)))
		fatal_error ("Unable to open link interface.");

	if (!(local_hardware_addr = libnet_get_hwaddr (link_interface, ethernet_device, buffer)))
		fatal_error ("Unable to determine local hardware address.");

	while (! send_done)
	{
		for (cycle = nethosts; cycle && ! send_done; cycle = cycle->next)
		{
			/* fprintf(stderr, "Testing %s...\n", cycle->profile); */
			libnet_build_ethernet (
				ether_broadcast_addr,
				local_hardware_addr->ether_addr_octet,
				ETHERTYPE_ARP,
				NULL,
				0,
				buffer);

			libnet_build_arp (
				ARPHRD_ETHER,
				ETHERTYPE_IP,
				ETHER_ADDR_LEN,
				4,
				ARPOP_REQUEST,
				local_hardware_addr->ether_addr_octet,
				(u_char *)&cycle->local_ip_address,
				ether_no_addr,
				(u_char *)&cycle->probe_ip_address,
				NULL,
				0,
				buffer + ETH_H);

			if (debug)
			{
				fprintf(stderr, "Sent ARP probe from %s",
						format_ipv4(&cycle->local_ip_address));
				fprintf(stderr, " to %s (%s)\n",
						format_ipv4(&cycle->probe_ip_address),
						cycle->profile);
			}

			/* printf ("Sending packet %08x.\n", *((int *)(buffer + LIBNET_ETH_H + 14))); */

			if (!(libnet_write_link_layer (link_interface, ethernet_device, buffer, LIBNET_ARP_H + LIBNET_ETH_H)))
				fatal_error ("Unable to send ARP packet.");

			send_count++;
		}

		if (!send_done)
			sleep (1);
	}

	libnet_close_link_interface (link_interface);

	if (verbose)
		fprintf(stderr, "End of sender thread (%d packets sent)\n",
				send_count);

	return 0;
}

/* Start the ARP sender thread */
void send_arp_requests ()
{
	if (pthread_create (&sender_thread_id, 0, send_arp_requests_thread, 0))
		fatal_error ("Unable to create ARP request thread.");
}

/* Stop the ARP sender thread */
void stop_arp_requests ()
{
	send_done = true;
	pthread_join(sender_thread_id, NULL);
}

/* Listen to the network for arp replies */
void receive_arp_replies ()
{
	char buffer[1024];
	unsigned char *packet;
	struct pcap_pkthdr pcap_header;
	pcap_t *pcap_interface;
	time_t start = time(NULL);
	const char* got = NULL;
	int received = 0, recv_arp = 0, recv_replies = 0, recv_match = 0;

	/* fprintf(stderr, "In receive_arp_replies\n"); */

	if (!(pcap_interface = pcap_open_live (
			ethernet_device, LIBNET_ARP_H + LIBNET_ETH_H, 0, 500, buffer)))
		fatal_error ("Unable to open PCAP interface.");

	for ( ; got == NULL && (start + timeout > time(NULL)) &&
			(packet = ((u_char *)pcap_next (pcap_interface, &pcap_header))); )
	{
		struct libnet_ethernet_hdr *packet_header;
		struct libnet_arp_hdr *arp_header;

		/* fprintf(stderr, "Got packet\n"); */
		received++;

		packet_header = (struct libnet_ethernet_hdr *)packet;

		if (ntohs (packet_header->ether_type) == ETHERTYPE_ARP)
		{
			arp_header = (struct libnet_arp_hdr *)(packet + ETH_H);
			/* fprintf(stderr, "ARP packet %08X -> %08X\n", *((int *)&arp_header->ar_spa), *((int *)arp_header->ar_tpa)); */
			recv_arp++;

			if (ntohs (arp_header->ar_op) == ARPOP_REPLY)
			{
				ipv4addr ipv4_me;
				ipv4addr ipv4_him;
				macaddr mac_him;
				struct nethost* cycle;

				IPv4_FROM_LIBNET(ipv4_me, arp_header->ar_tpa); 
				IPv4_FROM_LIBNET(ipv4_him, arp_header->ar_spa); 
				MAC_FROM_LIBNET(mac_him, arp_header->ar_sha);

				/* fprintf(stderr, "ARP REPLY packet\n"); */
				recv_replies++;

				for (cycle = nethosts; cycle; cycle = cycle->next)
				{
					/* Check if IP and MAC addreses match */
					if (	IPv4_MATCHES(cycle->local_ip_address, ipv4_me)
						&&  IPv4_MATCHES(cycle->probe_ip_address, ipv4_him)
						&&  MAC_MATCHES(cycle->hardware_address, mac_him))
					{
						got = cycle->profile;
						if (verbose)
						{
							fprintf(stderr,
									"Received ARP reply packet from %s",
									format_ipv4(&ipv4_him));
							fprintf(stderr, " to %s, mac: %s, matches: %s\n",
									format_ipv4(&ipv4_me), format_mac(&mac_him),
									got);
						}
						recv_match++;
					}

					/*
					if (debug)
						printf ("  test %08X to %08X\n", cycle->probe_ip_address, cycle->local_ip_address);
					if (debug)
						  printf ("%02x:%02x:%02x:%02x:%02x:%02x\n%02x:%02x:%02x:%02x:%02x:%02x\n", cycle->hardware_address[0], cycle->hardware_address[1], cycle->hardware_address[2], cycle->hardware_address[3], cycle->hardware_address[4], cycle->hardware_address[5], arp_header->ar_sha[0], arp_header->ar_sha[1], arp_header->ar_sha[2], arp_header->ar_sha[3], arp_header->ar_sha[4], arp_header->ar_sha[5]);
					if (debug)
						printf ("Found entry:\n\tlocal ip address %08x\n"
						        "\thwaddress %02x:%02x:%02x:%02x:%02x:%02x\n"
								"\tprofile %s\n",
								cycle->local_ip_address,
								cycle->hardware_address[0], cycle->hardware_address[1], cycle->hardware_address[2],
								cycle->hardware_address[3], cycle->hardware_address[4], cycle->hardware_address[5],
								cycle->profile);
					*/
				}
				if (got == NULL)
					if (verbose)
					{
						fprintf(stderr,
								"Received ARP reply packet from %s (%s)",
								format_ipv4(&ipv4_him), format_mac(&mac_him));
						fprintf(stderr, " to %s, matches none\n",
								format_ipv4(&ipv4_me));
					}
			}
		}
	}

	stop_arp_requests();

	if (verbose)
		fprintf(stderr, "Of %d packets received, %d were ARP with "
						"%d replies and %d matching\n",
				received, recv_arp, recv_replies, recv_match);


	if (got == NULL)
	{
		if (verbose)
			fprintf(stderr, "No matches found\n");
		got = defprof;
	}

	puts(got);
}

/* Parse the input from `input', building the nethosts list
 * To make it simple, use regexps on input lines instead of implementing a real
 * parser.
 */
void parse_input(FILE* input)
{
	regex_t nulline;
	regex_t inputline;
	char linebuf[1024];
	int linenum = 1;
	int err;
	if ((err = regcomp(&nulline, "^[[:blank:]]*(#.*)?\n$", REG_EXTENDED | REG_NOSUB)) != 0)
	{
		fprintf(stderr, "Error compiling regexp 1\n");
	}
	if ((err = regcomp(&inputline,
		"^[[:blank:]]*([0-9]+\\.[0-9]+\\.[0-9]+\\.[0-9]+)[[:blank:]]+"
		"([0-9A-Za-z]{2}:[0-9A-Za-z]{2}:[0-9A-Za-z]{2}:[0-9A-Za-z]{2}:[0-9A-Za-z]{2}:[0-9A-Za-z]{2})"
		"[[:blank:]]+([0-9]+\\.[0-9]+\\.[0-9]+\\.[0-9]+)[[:blank:]]+([[:alnum:]_+-]+)", REG_EXTENDED)) != 0)
	{
		char error[1024];
		regerror(err, &inputline, error, 1024);
		fprintf(stderr, "Error compiling regexp 2: %s\n", error);
		
	}
	while (fgets(linebuf, 1024, input))
	{
		if (regexec(&nulline, linebuf, 0, NULL, 0) != 0)
		{
			regmatch_t parts[5];
			if (regexec(&inputline, linebuf, 5, parts, 0) == 0)
			{	
				struct nethost* nh;
				char* source = strndup(linebuf+parts[1].rm_so, parts[1].rm_eo - parts[1].rm_so);
				char* mac = strndup(linebuf+parts[2].rm_so, parts[2].rm_eo - parts[2].rm_so);
				char* target = strndup(linebuf+parts[3].rm_so, parts[3].rm_eo - parts[3].rm_so);
				char* name = strndup(linebuf+parts[4].rm_so, parts[4].rm_eo - parts[4].rm_so);
				if (debug)
					fprintf(stderr, "Found network %s: here %s, there %s (%s)\n", name, source, target, mac);

				nh = (struct nethost*) malloc(sizeof(struct nethost));
				parse_ipv4(&nh->local_ip_address, source);
				parse_ipv4(&nh->probe_ip_address, target);
				parse_mac(&nh->hardware_address, mac);
				nh->profile = strdup(name);
				nh->next = nethosts;
				nethosts = nh;
				nethost_count++;
			} else {
				fprintf(stderr, "Parse error at line %d: line ignored\n", linenum);
			}
		} else {
			/* printf("Line %d: empty\n", linenum); */
		}
		linenum++;
	}
}

/* Initialize the interface (with "ifconfig <iface> up" if it is found down
 * Return true if the interface was down
 */
bool iface_init(const char* iface)
{
	pid_t pid;
	int status;

	if ((pid = fork()) == 0)
	{
		char* cmdbuf;
		if (verbose)
			fprintf(stderr, "Checking if %s is up (%s -c '%s | %s -q %s'): ",
					iface, SH, IFCONFIG, GREP, iface);
		asprintf(&cmdbuf, IFCONFIG " | " GREP " -q %s", iface);
		execlp(SH, SH, "-c", cmdbuf, 0);
		perror("interface check execlp failed");
		_exit(1);
	}
	wait(&status);

	if (WIFEXITED(status))
	{
		if (WEXITSTATUS(status) == 0)
		{
			if (verbose) fprintf(stderr, "up\n");
			return false;
		}
		if (verbose) fprintf(stderr, "down\n");

		if ((pid = fork()) == 0)
		{
			if (verbose)
				fprintf(stderr, "Bringing up %s (%s %s up): ",
						iface, IFCONFIG, iface);
			execlp(IFCONFIG, IFCONFIG, iface, "up", 0);
			perror("interface setup execlp failed");
			_exit(1);
		}
		wait(&status);
		if (WIFEXITED(status))
		{
			if (WEXITSTATUS(status) == 0)
			{
				if (verbose) fprintf(stderr, "succeeded\n");
				/* Wait a little for the interface to initialize */
				sleep(init_time);
				return true;
			}
			fatal_error("child process for interface setup was not successful");
		} else {
			fatal_error("child process for interface setup has been interrupted");
		}
	} else {
		fatal_error("child process for interface checking has been interrupted");
	}
	/* Make gcc stop complaining */
	return false;
}

/* Bring down `iface' */
void iface_shutdown(const char* iface)
{
	pid_t pid;
	int status;

	if ((pid = fork()) == 0)
	{
		if (verbose)
			fprintf(stderr, "Shutting down %s (%s %s down): ",
					iface, IFCONFIG, iface);
		execlp(IFCONFIG, IFCONFIG, iface, "down", 0);
		perror("interface shutdown execlp failed");
		_exit(1);
	}
	wait(&status);
	if (WIFEXITED(status))
	{
		if (WEXITSTATUS(status) != 0)
			fatal_error("child process for interface shutdown was not successful");
		else
			if (verbose) fprintf(stderr, "succeeded\n");
	} else {
		fatal_error("child process for interface shutdown has been interrupted");
	}
}

int main (int argc, const char *argv[])
{
	/* Initialize commandline passing data */
	bool op_ver = false;
	char* help = "Guess the current network location using fake ARP packets.\n";
	struct poptOption emptyTable[] = { POPT_TABLEEND };
	struct poptOption optionsTable[] = {
		/*
		{ "identity", 0, POPT_ARG_STRING, &identity, 0,
			"reader identification tag", "ident" },
		{ "interface", 'i', POPT_ARG_STRING, &ethernet_device, 0,
		  "ethernet device to use for scanning (defaults to \""
			DEFAULT_DEVICE "\")", 0 },
		*/
		{ "default", 'd', POPT_ARG_STRING, &defprof, 0,
		  "profile name to report if no known networks are found"
		  " (defaults to <interface>-" DEFAULT_PROFILE ")", 0 },
		{ "timeout", 't', POPT_ARG_INT, &timeout, 0,
		  "timeout (in seconds) used to wait for response packets"
		  " (defaults to 5 seconds)", 0 },
		{ "init-time", 0, POPT_ARG_INT, &init_time, 0,
		  "time (in seconds) to wait for the interface to initialize"
		  " when not found already up (defaults to 3 seconds)", 0 },
		{ "verbose", 'v', POPT_ARG_NONE, &verbose, 0,
		  "enable verbose operations", 0 },
		{ "very-verbose", 'V', POPT_ARG_NONE, &debug, 0,
		  "enable debugging output", 0 },
		{ "version", 0, POPT_ARG_NONE, &op_ver, 0,
		  "print version and exit", 0 },
		POPT_AUTOHELP
		{ NULL, 0, POPT_ARG_INCLUDE_TABLE, emptyTable, 0, help, 0 },
		POPT_TABLEEND
	};
	int res;
	const char* file;
	FILE* input;
	bool iface_was_down;

	poptContext optCon = poptGetContext(NULL, argc, argv, optionsTable, 0);
	poptSetOtherOptionHelp(optCon, "[options] [ethernet device] [config_file]");

	/*
	if (argc < 2) {
		poptPrintHelp(optCon, stderr, 0);
		return 1;
	}
	*/

	/* Process commandline */
	res = poptGetNextOpt(optCon);
	if (res != -1)
	{
		fprintf(stderr, "%s: %s\n\n",
			poptBadOption(optCon, POPT_BADOPTION_NOALIAS),
			poptStrerror(res));
		poptPrintUsage(optCon, stderr, 0);
		return 1;
	}

	if (op_ver)
	{
		printf("%s ver." VERSION "\n", argv[0]);
		return 0;
	}

	/* Check user id */
	if (geteuid() != 0)
		fatal_error ("You must run this command as root.");

	if (argc > 1)
		ethernet_device = poptGetArg(optCon);
	if (! defprof)
		asprintf(&defprof, "%s-%s", ethernet_device, DEFAULT_PROFILE);
	else
		if (verbose)
			fprintf(stderr, "Default profile set to `%s'\n", defprof);
	file = poptGetArg(optCon);
	input = 0;

	/* very verbose implies verbose */
	if (debug)
		verbose = true;

	/* Open the specified config file or stdin if not specified */
	if (file)
	{
		input = fopen(file, "rt");
		if (! input)
		{
			fprintf(stderr, "Cannot open %s: %s.  Falling back to stdin.\n", file, strerror(errno));
		}
	}
	if (! input)
		input = stdin;
		
	parse_input(input);

	if (input != stdin)
		fclose(input);

	if (nethost_count == 0)
	{
		printf("%s\n", defprof);
		return 0;
	}

	/* Check if we have to bring up the interface; if yes, do it */
	iface_was_down = iface_init(ethernet_device);

	/* Send the arp requests */
	send_arp_requests ();

	/* Record replies */
	receive_arp_replies ();

	/* Bring down the interface if we need it */
	if (iface_was_down)
		iface_shutdown(ethernet_device);

	return 0;
}

// vim:set ts=4 sw=4:
