/* contains some litte helper functions used by parts of Gnometoaster */

#include "int.h"
#include "preferences.h"

#include <gtk/gtk.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <fcntl.h>
#include <unistd.h>

/* uncomment for debugging */
#define DEBUG
//
/* store string results here */
char buffer[MAXSTRINGSIZE];

/* build sum of all a numbers digits */
int helpings_dsum(int val)
{
   int d=0;

   for (;val>0;val/=10)
     {
	d+=val%10;
     }
   ;
   return d;
}
;

/* this returns a GList containing all items in the dnd string passed to
 * this function */
GList *helpings_convertdndlist(const char *dndlist)
{
   char *lc;
   GList *result=NULL;
   char *next;
   char *current;
   char *n13;
   char *n10;
   int  olength;

   if (dndlist!=NULL)
     {
	/* make a local copy of the dnd string */
	lc=strdup(dndlist);
	/* we're storing this value to detect when we'll be done */
	olength=strlen(lc);
	next=lc;
	do
	  {
	     current=next;
	     next=NULL;
	     
	     n13=strchr(current,13);
	     n10=strchr(current,10);
	     next=n13;
	     if ((n10!=NULL)&&((n10<n13)||(!next)))
	       next=n10;
	     if (next==NULL)
	       next=&lc[olength];
	     else
	       *next=0;

	     if (strlen(current))
	       {
		  result=g_list_append(result,
				       (gpointer)strdup(current));
	       };
	     do
	       next++;
	     while (((next-lc)<olength)&&
		    ((*next==13)||(*next==10)));
	  }
	while ((next-lc)<olength);
	free(lc);
     };
   return result;
};

/* returns true if one of the GList entries starts with string */
int helpings_dndlistitemstartswith(GList *dndlist,
				   char *string)
{
   GList *current;
   int result=0;
   for (current=dndlist;
	((current!=NULL)&&(!result));
	current=current->next)
     {
	if ((current->data!=NULL)&&
	    (!strncasecmp((char*)current->data,
			  string,
			  strlen(string))))
	  result=1;
     };
   return result;
};

/* build Glist from a list of strings terminated with NULL
 * this is useful mainly to create a Glist out of a list of constants
 * e.g. to create a combo box with it */
GList *helpings_buildglist(char *stringlist[])
{
   int x;
   GList *list=NULL;

   x=0;
   for (x=0;stringlist[x]!=NULL;x++)
     {
	list=g_list_append(list,stringlist[x]);
     }
   ;

   return list;
}
;

/* return a pointer to the n-th line of a string containing a complex
 * output with CRs and LFs - return NULL if line is not present */
char *helpings_getlinestart(char *s,int n)
{
   char *linestart;
   char *x;
   int cl;

   cl=1;linestart=s;x=linestart;
   while ((cl<n)&&((x-s)<strlen(s)))
     {
	x=linestart;
	while ((*x!=13)&&(*x!=10)) x++;
	if ((*x==13)||(*x==10)) x++; /* allow one more CR */
	linestart=x;
	cl++;
     }
   ;
   if (cl!=n) linestart=NULL;

   return linestart;
}
;

/* get line no. n - returns "" if not found */
char *helpings_getlineno(char *s,int n)
{
   char *beginline;
   char *endline;
   char *newstring;

   newstring=(char*)malloc(strlen(s)+1);
   beginline=helpings_getlinestart(s,n);
   if (beginline!=NULL)
     {
	strcpy(newstring,beginline);

	endline=newstring;
	while ((*endline!=13)&&(*endline!=10))
	  endline++;
	*endline=0;
     }
   else
     {
	strcpy(newstring,"");
     }
   ;
   return newstring;
}
;

/* get the n-th value occurring in a string */
int helpings_getvalueinstring(char *s,int n)
{
   int cv;
   char *x;
   char *nmbstart;
   int num;

   cv=0;x=s;nmbstart=x;
   while ((cv<n)&&((x-s)<strlen(s)))
     {
	while ((*x<'0')||(*x>'9')) x++;
	nmbstart=x;
	while ((*x>='0')&&(*x<='9')) x++;
	cv++;
     }
   ;
#ifdef DEBUG
   printf("getting number from %s\n",nmbstart);
#endif
   if (cv==n) /* if number was found */
     sscanf(nmbstart,"%d",&num);
   else
     num=-1;

   return num;
}
;

void helpings_formatnumber(char *buf,int bufsize,long long unsigned int number)
/* Fills the Buffer with an ascii representation of the number.
 * every 3 digits will be seperated by a '.'  */
{
   int n=1,d=0;
   buf[bufsize-1]=0;/* 0-terminated */
   do
     {
	++d;
	if ((d%3==1)&&(d!=1))
	  buf[bufsize-(++n)]='.';
	buf[bufsize-(++n)]='0'+number%10;
	number/=10;
     }
   while ((number>0)&&(n<bufsize));

   for(d=0;d<n;d++)
     {
	buf[d]=buf[bufsize-n+d];
     };
};

/* convert a given number of seconds into an ascii representation
 * of a "clock counter" */
char *helpings_secs2hms(int secs,int verbose)
{
   char *result;
   result=buffer;
   if (verbose)
     sprintf(result,_("%02i hour(s) %02i min. %02i sec."),
	     secs/3600,
	     (secs%3600)/60,
	     secs%60
	     );
   else
     sprintf(result,"%02i:%02i:%02i",
	     secs/3600,
	     (secs%3600)/60,
	     secs%60
	     );
   return result;
};

/* compare a word to a list of words separated by "sep" and return true if
 * it was found */
int helpings_listcomp(char *list,char *word,char *sep)
{
   char *wc,*current;
   char *key;
   int match=0;

   /* create a local copy of list */
   wc=(char*)malloc(strlen(list)+strlen(sep)+1);
   strcpy(wc,list);
   strcat(wc,sep);

   key=(char*)malloc(strlen(word)+strlen(sep)+1);
   strcpy(key,word);
   strcat(key,sep);

   for (current=wc;
	*current!=0;
	current++)
     if (!strncasecmp(current,key,strlen(key)))
       match=1;

   free(key);free(wc);
   return match;
};

/* get valid location of a file out of a list of possible locations */
char *helpings_locate(char *paths[])
{
   int x;
   int fd;
   char *result=NULL;

   for (x=0;paths[x]!=NULL;x++)
     {
	fd=open(paths[x],O_RDONLY);
	if (fd!=-1)
	  {
	     result=paths[x];
	     close(fd);
	  };
     };

   return result;
};

/* allocate a new list of strings and translate the list specified as
 * parameter using gettext(). The list must be NULL-terminated */
char **helpings_translatestringlist(const char **list)
{
   int x;
   int elements=0;
   char **newlist;

   while (list[elements]!=NULL)
     elements++;

   /* allocate the new string list plus a NULL terminator at the end */
   newlist=(char**)malloc(sizeof(char*)*(elements+1));
   for (x=0;x<elements;x++)
     {
	newlist[x]=(char*)malloc(MAXSTRINGSIZE+1);
	newlist[x][MAXSTRINGSIZE]=0;
	/* avoid possible buffer overflow here */
	strncpy(newlist[x],_(list[x]),1024);
     };
   /* terminate the string list */
   newlist[elements]=NULL;

   return newlist;
};

/* can through name and replace all '/'es by ','.
 * Also ignore all quotation marks */
void helpings_makelegalfilename(char *name)
{
   int i;

   for (i=0;i<strlen(name);i++)
     {
	switch (name[i])
	  {
	   case '/':
	     name[i]=',';
	     break;
	  };
     };
};

/* create full path of a file in path.
 * memory allocated for result is exactly the stringlegth+1, so don't
 * extend the string.
 * String has to be freed by the caller when no longer needed */
char *helpings_fileinpath(const char *path,const char *file)
{
   int  adddelim=((path[strlen(path)-1]!='/')&&(file[0]!='/'));
   char *result=(char*)malloc(strlen(path)+
			      strlen(file)+
			      1+adddelim);
   strcpy(result,path);
   if ((path[strlen(path)-1]=='/')&&(file[0]=='/'))
     result[strlen(result)-1]=0;
   if (adddelim)
     strcat(result,"/");
   strcat(result,file);
   return result;
};

/* get rightmost path component (file or directory) */
char *helpings_rpathcomponent(char *path)
{
   char *result=NULL;
   if (path)
     {
	char *newpath=strdup(path);
	if (newpath[strlen(newpath)]=='/')
	  newpath[strlen(newpath)]=0;
	if (strrchr(newpath,'/'))
	  result=strdup(strrchr(newpath,'/')+1);
	else
	  result=strdup(newpath);
	free(newpath);
     }
   return result;
};

/* get path only. cuts the righmost path component (file or directory) */
char *helpings_pathonly(const char *path)
{
   char *result=strdup(path);
   /* if we have a trailing slash, ignore it */
   if (result[strlen(result)]=='/')
     result[strlen(result)]=0;
   if (strrchr(result,'/'))
     *strrchr(result,'/')=0;
   return result;
};

int helpings_isinpath(const char *file,
		      const char *path)
{
   return (!strncmp(file,path,strlen(path)));
};

int helpings_pathcompare(const char *p1,
			 const char *p2)
{
   char *path1=strdup(p1);
   char *path2=strdup(p2);
   int result;
   if (path1[strlen(path1)-1]=='/')
     path1[strlen(path1)]=0;
   if (path2[strlen(path2)-1]=='/')
     path1[strlen(path2)]=0;
   result=strcmp(path1,path2);
   free(path1);free(path2);
   return result;
};

/* finds occurrence of a substring in a string while not matching the
 * parts of string located between quotation marks.
 * This is necessary for parsing terms containing string constants,
 * e.g. functions expecting a filename as their parameter or
 * the regexp parsing function */
char *helpings_strstr_quote(char *string,char *substring)
{
   char *c;
   int quote_mode=0;
   int esc=0;
   char *found=NULL;

   for (c=string;
	((*c!=0)&&(found==NULL));
	c++)
     {
	/* if '\' enter esc mode */
	if (*c=='\\')
	  esc=1;
	/* only take quotation mark for real if ESC flag isn't set */
	if ((*c=='"')&&(!esc))
	  /* invert quote-mode flag */
	  quote_mode=1-quote_mode;
	/* if we're not within quotation marks */
	if ((!quote_mode) &&
	    (!strncmp(c,substring,strlen(substring))))
	  found=c;
	/* reset ESC flag */
	esc=0;
     };

   return found;
};

void helpings_printstringlist(GList *s)
{
   GList *c;

   for (c=s;c!=NULL;c=c->next)
     {
	printf("%s\n",(char*)(c->data));
     };
};

void helpings_freestringlist(GList *s)
{
   GList *ls=s;

   while (ls!=NULL)
     {
	free(ls->data);
	ls=g_list_remove(ls,ls->data);
     };
};

/* convert special characters represented by %xx within a string
 * into normal characters */
char *helpings_convertspecialchars(char *source)
{
   /* destination never gets bigger than source */
   char *dest=(char*)malloc(strlen(source)+1);
   int x=0;
   while (*source!='\0')
     {
	if (*source!='%')
	  dest[x]=*source;
	else
	  {
	     int temp=0; /* in case of an invalid %xx sequence, stop string here */
	     source++;
	     sscanf(source,"%2x",&temp);
	     dest[x]=temp;
	     source++;
	  };
	source++;
	x++;
     };
   /* terminate destination string */
   dest[x]=0;
   return dest;
};

char *helpings_resolveescapedchars(const char *source)
{
   char *result=NULL;
   if (source)
     {	          
	const char *current=source;
	char *cdest=NULL;
	result=strdup(source);
	cdest=result;
	while (*current)
	  {
	     switch (*current)
	       {
		case '\\':
		  ++current;
		  switch (*current)
		    {
		     case 't':
		       *cdest=9;
		       break;
		     case 'n':
		       *cdest=10;
		       break;
		     case 'r':
		       *cdest=13;
		       break;
		     case '\\':
		       *cdest='\\';
		       break;
		     default:
		       /* unrecognized escape sequence. Keep it */
		       *cdest='\\';++cdest;		       
		       *cdest=*current;
		       break;
		    };
		  break;
		default:
		  *cdest=*current;
		  break;		  
	       };
	     ++current;
	     ++cdest;
	  };
	*cdest=0;
     };
   return result;
};

/* maximum size in pixels */
char *helpings_cutstring(int nmbr,const char *source,GdkFont *cfont)
{
   char *result=NULL;
   if (nmbr<gdk_string_width(cfont,source))
     {
	const char *placeholder="[...]";
	char *left=strdup(source);
	const char *right=&source[strlen(source)>>1];
	int action=0;
	int impossible=0;
	left[strlen(source)>>1]=0;
	do
	  {
	     if (result)
	       free(result);

	     if (action&&strlen(left))
	       left[strlen(left)-1]=0;
	     else
	       {
		  if (strlen(right))
		    ++right;
		  else
		    impossible=1;
	       };
	     action=1-action;
	     result=malloc(strlen(left)+strlen(placeholder)+strlen(right)+1);
	     strcpy(result,left);
	     strcat(result,placeholder);
	     strcat(result,right);
	  }
	while ((gdk_string_width(cfont,result)>nmbr)&&(!impossible));
	free(left);
	if (impossible)
	  {
#ifdef DEBUG
	     printf("helpings_cutstring: can't add placeholder as desired string length is too short\n");
#endif
	     free(result);
	     result=malloc(nmbr+1);
	     strncpy(result,source,(nmbr>>1));
	     strcat(result,&source[strlen(source)-(nmbr-(nmbr>>1))]);
	  };
#ifdef DEBUG
	if (!result)
	  printf("helpings_cutstring: result is %s\n",result);
#endif
     }
   else
     result=strdup(source);
   return result;
};

#define INIT_SIZE 256
char *helpings_getterminatedstring(FILE *input,char terminator,
				   int ignorelistlength,
				   const char *ignorelist)
{
   char *result=(char*)malloc(INIT_SIZE);
   int clength=INIT_SIZE;
   int currentchar=fgetc(input);
   *result=0;
   while ((currentchar!=EOF)&&(currentchar!=(int)terminator))
     {
	/* allow certain characters to be ignored */
	int ignore=0;
	int i;
	for (i=0;i<ignorelistlength;++i)
	  if (currentchar==(int)ignorelist[i])
	    ignore=1;

	if (!ignore)
	  {
	     char cat[2];
	     cat[0]=(char)currentchar;
	     cat[1]=0;
	     /* result will grow dynamically */
	     if (strlen(result)+1>=clength)
	       {
		  char *newstring=NULL;
		  clength<<=1;
		  newstring=(char*)malloc(clength);
		  strcpy(newstring,result);
		  free(result);
		  result=newstring;
	       };
	     strcat(result,cat);
	  };
     };
   return result;
};

char *helpings_escape(const char *input,
		      const char *escapelist,
		      char escape_char)
{
   /* allocate the maximum destination string size */
   char *result=(char*)malloc((strlen(input)<<1)+1);
   const char *current=input;
   char *cdest=result;
   *result=0;
   while (*current)
     {
	if ((strchr(escapelist,*current))||
	    (escape_char==*current))
	  {
	     *cdest=escape_char;
	     ++cdest;
	  };
	*cdest=*current;
	++cdest;++current;
     };
   *cdest=0;
   return result;
};

char *helpings_removecharacters(const char *input, char candidate)
{
   char *result = NULL;
   if (input)
     {
	const char *sourcecount=input;
	char *destcount = NULL;
	result = (char*)malloc(strlen(input)+1);
	destcount = result;
	// Copy the string, including the terminator
	// Take all characters except 'candidate' over into the new string
	do
	  {
	     if ((*sourcecount)!=candidate)
	       {
		  *destcount=*sourcecount;
		  ++destcount;
	       };	     
	     ++sourcecount;
	  }
	while (*(sourcecount-1)); // See if the last character processed was the end of the string
     };
   return result;
};
	
