/***************************************************************************
                          application.cpp  -  description
                             -------------------
    begin                : Mon Jan 7 2002
    copyright            : (C) 1999-2002 by Brian Ashe & Jacques Fortier
    email                : gtkpool@seul.org
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <vector>
#include <algorithm>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <gtk/gtk.h>
#include "application.h"
#include "sound.h"
#include "connectdialog.h"
#include "check_table_collision.h"
#include "move_balls.h"
#include "apply_friction.h"
#include "draw_ball.h"
#include "pointer_selects.h"
#include "check_pocket.h"
#include "vec2d.h"
#include "point2d.h"
#include "ball.h"
#include "indentify_ball.h"

#define PI 3.14159265

static GList *load_directories = NULL;

Application::Application(){
		strcpy(s1, "0");
		strcpy(s2, "0");
		gap = 29;
		width = 722 + gap * 2; height = 361 + gap * 2;
		table.width = width - (gap * 2);
		table.height = height - (gap * 2);
		table.x = 0 + gap;
		table.y = 0 + gap;

		FRICTION = .06;
		BUMPER_DRAG = .8;
		COLLIDE_DRAG = .7;

		selected = NULL;

		// Initialize colors
		black  = new GdkColor;
		red = new GdkColor;
		yellow = new GdkColor;
		table_color = new GdkColor;
		message_colours[1] = new GdkColor;
		message_colours[2] = new GdkColor;
		message_colours[3] = new GdkColor;
		message_colours[4] = new GdkColor;
		message_colours[14] = new GdkColor;

		buf_pixmap = NULL;
		table_pixmaps = new (GdkPixmap *)[4];
		balls_pixmaps = new (GdkPixmap *)[NUM_BALLS];
		balls_pixmap_masks = new (GdkBitmap *)[NUM_BALLS];
		balls_big_pixmaps = new (GdkPixmap *)[NUM_BALLS];
		balls_big_pixmap_masks = new (GdkBitmap *)[NUM_BALLS];
		running = true; placing_cue = false;
		sunk_tf = collide_tf = bounce_tf = false;
		connected = false;
		connect_dialog = NULL;
		game_name = eight_ball;
		table_choice = TABLE_MA;
}

Application::~Application(){
   	delete black;
   	delete red;
   	delete yellow;
   	delete table_color;
   	delete message_colours[1];
   	delete message_colours[2];
   	delete message_colours[3];
   	delete message_colours[4];
   	delete message_colours[14];
   	delete[] table_pixmaps;
   	delete[] balls_pixmaps;
   	delete[] balls_pixmap_masks;
   	delete[] balls_big_pixmaps;
   	delete[] balls_big_pixmap_masks;
   	delete thegame;
}

void Application::init()
{
	visual = gdk_visual_get_system();
	colormap = gdk_colormap_get_system();

	load_ball_xpms();
	if(sound)
		init_sound();

	gdk_color_parse("ForestGreen", table_color);
	gdk_color_parse("Red", message_colours[1]);
	gdk_color_parse("Green", message_colours[2]);
	gdk_color_parse("Blue", message_colours[3]);
	gdk_color_parse("Magenta", message_colours[4]);
	gdk_color_parse("Gray", message_colours[14]);
	gdk_colormap_alloc_color(colormap, table_color, FALSE, TRUE);
	gdk_colormap_alloc_color(colormap, message_colours[1], FALSE, TRUE);
	gdk_colormap_alloc_color(colormap, message_colours[2], FALSE, TRUE);
	gdk_colormap_alloc_color(colormap, message_colours[3], FALSE, TRUE);
	gdk_colormap_alloc_color(colormap, message_colours[4], FALSE, TRUE);
	gdk_colormap_alloc_color(colormap, message_colours[14], FALSE, TRUE);
	
	// Set colors
	gdk_color_black(colormap, black);
	gdk_color_parse("Red", red);
	gdk_colormap_alloc_color(colormap, red, FALSE, TRUE);
	gdk_color_parse("Yellow", yellow);
	gdk_colormap_alloc_color(colormap, yellow, FALSE, TRUE);
	message_colours[0] = black;

	// setup adjustments -- these are used in options dialog
	// to control physics
	friction_adjustment = gtk_adjustment_new(0.06, 0, 1, 0.01, 0.1, 0);
	collide_drag_adjustment = gtk_adjustment_new(0.7, 0, 1, 0.01, 0.05, 0);
	bumper_drag_adjustment = gtk_adjustment_new(0.8, 0.0, 1.5, 0.01, 0.1, 0);

	resetTable();
}

void Application::destroy()
{
	if(buf_pixmap)
		gdk_pixmap_unref(buf_pixmap);
	for(int i = 0; i < NUM_BALLS; i++)
	{
		gdk_pixmap_unref(balls_pixmaps[i]);
		gdk_bitmap_unref(balls_pixmap_masks[i]);
	}

	gdk_colormap_free_colors(colormap, table_color, 1);

	gdk_colormap_free_colors(colormap, message_colours[1], 1);
	gdk_colormap_free_colors(colormap, message_colours[2], 1);
	gdk_colormap_free_colors(colormap, message_colours[3], 1);
	gdk_colormap_free_colors(colormap, message_colours[4], 1);
	gdk_colormap_free_colors(colormap, message_colours[14], 1);
	
	close_sound_device();
}

void Application::updateBalls () {
	/* Every time this function is called, it moves time forward by one
animation frame.  This means that every ball will be moved by its
vel.dx and vel.dy values by the time the function returns.  However, to
process collisions the frame is broken up.  First the balls are
analyzed to see if any collisions will happen in this tick.  The
first collision is selected, and its time noted.  All the balls are
advanced to that time, and the collision is processed.  This is
reiterated until there are no more collisions left in the frame
*/
	double	md, ei, time;
	int cnt = balls.size();
	sunk_tf = collide_tf = bounce_tf = false;
	md = ei = time = 1;
	do {
		Ball *cb1 = NULL, *cb2 = NULL, *bnc = NULL;
		check_table_collision c(ei, time, bnc, table);
		// Check for collisions between balls and the edge of the table
		check_table_collision c2 = for_each(balls.begin(), balls.end(), c);
		bnc = c2.bnc;
		// Check for ball to ball collisions
		for (int ii = 1; ii < cnt; ii++) {
			for (int jj = 0; jj < ii; jj++) {
				double coll_time = balls[ii].pathIntercept(balls[jj]);
				if (coll_time < md  && coll_time >= 0  &&  coll_time < time) {
					md = coll_time;
					cb1 = &balls[ii];
					cb2 = &balls[jj];
				}
			}
		}
		// Determine closest event
		double dt = md < time ? md : time;
		double dt2 = dt < ei ? dt : ei;
		// Move all ball's forward to this point in time
		for_each(balls.begin(), balls.end(), move_balls(dt2));
		// Process any collisions
		if (ei <= dt &&  bnc != NULL) // if this event is a bumper collision
		{
			check_pocket cp(table, ei);
			if(cp(*bnc))           // if the ball is going into a pocket
			{
				int r = BALL_SIZE / 2 + 1;
				update_dirty(int(bnc->x - r), int(bnc->x + r),
				             int(bnc->y - r), int(bnc->y + r));

				sunk_balls |= (1 << cp.ball_num);

				// Put balls in the rack when sunk
				if (cp.ball_num != CUE)
				{
					switch (game_name)
					{
						case eight_ball:
						thegame->eight_ball_rules(cp, bnc);
						break;
						case nine_ball:
						thegame->nine_ball_rules(cp, bnc);
						break;
						case rotation:
						thegame->rotation_rules(cp, bnc);
						break;
						case no_rules:
						thegame->player_1.sunkballs.push_back(bnc->ball_num);
						thegame->first_ball_sunk = true;
						strcpy(thegame->comment, "Nice Shot!\n");
						break;
					}
					
					last_sunk = cp.ball_num;
				} else
				{
					thegame->scratched = true;
				}

				update_rack();
				balls.erase(std::vector<Ball>::iterator(bnc)); // get rid of it
				sunk_tf = true;
			}
			else             // otherwise bounce it off the bumper
			{
				bnc->bounce(ei, bumper_drag());
				bounce_tf = true;
			}
		}
		else if (md <= time  &&  cb1 != NULL)
		{
			Ball *hb;
			if (cb1->ball_num > 0)
				hb = cb1;
			else
				hb = cb2;
				
			if (!thegame->first_ball_hit && thegame->cat_is_set)
			{
				switch (game_name)
				{
					case eight_ball:
						if (thegame->turn == tplayer1)
						{
							if (hb->category == thegame->player_1.category ||
							 (thegame->player_1.sunkballs.size() == 7 && hb->is_eight))
							{
								thegame->first_ball_hit = true;
								thegame->hit_wrong_ball = false;
							} else
							{
								thegame->first_ball_hit = true;
								thegame->hit_wrong_ball = true;
							}
						} else if (thegame->turn == tplayer2)
						{
							if (hb->category == thegame->player_2.category ||
							 (thegame->player_2.sunkballs.size() == 7 && hb->is_eight))
							{
								thegame->first_ball_hit = true;
								thegame->hit_wrong_ball = false;
							} else
							{
								thegame->first_ball_hit = true;
								thegame->hit_wrong_ball = true;
							}
						}						
					break;
					
					case nine_ball:
						if (hb->ball_num == low_ball())
						{
							thegame->first_ball_hit = true;
							thegame->hit_wrong_ball = false;
						} else
						{
							thegame->first_ball_hit = true;
							thegame->hit_wrong_ball = true;
						}
					break;
					
					case rotation:
						if (hb->ball_num == low_ball())
						{
							thegame->first_ball_hit = true;
							thegame->hit_wrong_ball = false;
						} else
						{
							thegame->first_ball_hit = true;
							thegame->hit_wrong_ball = true;
						}
					break;
				}
			}
			cb1->collide(cb2, collide_drag());
			collide_tf = true;
		}
		time -= dt2; // wind the clock forward
		ei = md = time;
	} while (time > 0);
	// Apply friction deceleration to all balls
	for_each(balls.begin(), balls.end(), apply_friction(friction()));
	
	// redraw aimer if it selects a moving ball
	if((hit_moving && selected != NULL) &&
	   (selected->vel.dx != 0 || selected->vel.dy != 0))
	{
		update_dirty(int(shoot.x), int(selected->x), int(shoot.y), int(selected->y));
	}
	
/*	// All balls have been sunk (restart)
	if((balls.size() == 1 && balls.back().is_cue) || balls.size() == 0)
	{
		resetTable();
	}*/
}

void Application::replaceBall() {
	double ox = (table.width * 0.75) + gap;
	double oy = height * 0.5;
   	
   	// If 8-Ball take the last one of the right rack
   	if (game_name == eight_ball)
   	{
	   	if (last_sunk > 0 && last_sunk < 9)
   			thegame->player_1.sunkballs.pop_back();
	   	if (last_sunk > 7 && last_sunk < 16)
	   		thegame->player_1.sunkballs.pop_back();
	}
   	// Put it on the table at the foul spot
   	if (last_sunk > 0 && last_sunk < 16)	
		new_ball(ox, oy, last_sunk, other);
	
   	// Reset so you can't put too many balls on the table
   	last_sunk = 0;
	udl = table.x; udr = table.x + table.width;
	udt = table.y; udb = table.y + table.height;
	// Reset the racks
	update_rack();

}

void Application::ball_in_hand()
{				
	if (placing_cue)
		return;
	std::vector<Ball>::iterator cb;
	cb = find_if(balls.begin(), balls.end(), indentify_ball());
	if (cb->is_cue)
	{
		balls.erase(cb); // remove current cue ball
		update_dirty(table.x, table.x + table.width, table.y, table.y + table.height);
		new_cue_ball();
	}
}

int Application::low_ball() {
	// Get lowest ball number still on table
	int low = 16;
	int cur;
	std::vector<Ball>::iterator bs;
	for (bs=balls.begin(); bs != balls.end(); bs++)
	{
		cur = bs->ball_num;
		if (cur > 0 && cur < 16 && cur < low)
			low = cur;
	}
	return low;
}

void Application::resetTable () {
	double ox = (table.width * 0.75) + gap;
	double oy = height * 0.5;
	double ofx = sqrt((BALL_SIZE * BALL_SIZE) - ((BALL_SIZE / 2) * (BALL_SIZE / 2))) + 1;
	double ofy = BALL_SIZE / 2 + 1;
	
	balls.erase(balls.begin(), balls.end());
	// Setup game
	if (thegame)
		delete thegame;
		
	thegame = new Game;
	
	if(game_name == eight_ball || game_name == no_rules)
	{
		// Standard Rack up
		new_ball((table.width / 4) + gap - 1, oy, CUE, other, true);
		new_ball(ox, oy, PURPLE, solids, false, false, false, true);
		new_ball(ox + ofx, oy - ofy, RED_S, stripes, false, false, false, false, true);
		new_ball(ox + ofx, oy + ofy - .5, BLUE, solids, false, false, false, true);
		new_ball(ox + (ofx * 2) + .4, oy - (ofy * 2) - .3, GREEN_S, stripes, false, false, false, false, true);
		new_ball(ox + (ofx * 2), oy, EIGHT, other, false, true);
		new_ball(ox + (ofx * 2), oy + (ofy * 2), BROWN_S, stripes, false, false, false, false, true);
		new_ball(ox + (ofx * 3) + .3, oy - (ofy * 3) - .6, ORANGE, solids, false, false, false, true);
		new_ball(ox + (ofx * 3), oy - ofy, YELLOW_S, stripes, false, false, false, false, true);
		new_ball(ox + (ofx * 3), oy + ofy, RED, solids, false, false, false, true);
		new_ball(ox + (ofx * 3) + .3, oy + (ofy * 3), BLUE_S, stripes, false, false, false, false, true);
		new_ball(ox + (ofx * 4), oy - (ofy * 4) - .7, GREEN, solids, false, false, false, true);
		new_ball(ox + (ofx * 4), oy - (ofy * 2), PURPLE_S, stripes, false, false, false, false, true);
		new_ball(ox + (ofx * 4), oy, BROWN, solids, false, false, false, true);
		new_ball(ox + (ofx * 4) + .3, oy + (ofy * 2), ORANGE_S, stripes, false, false, false, false, true);
		new_ball(ox + (ofx * 4), oy + (ofy * 4), YELLOW, solids, false, false, false, true);
	}
	else if(game_name == nine_ball)
	{
		thegame->cat_is_set = true;		// Otherwise it will get confused
		
		// Rack up for Game of 9-ball
		new_ball((table.width / 4) + gap, oy, CUE, other, true);
		new_ball(ox, oy, YELLOW);
		new_ball(ox + ofx, oy - ofy, BLUE);
		new_ball(ox + ofx, oy + ofy + .4, GREEN);
		new_ball(ox + (ofx * 2) - .6, oy - (ofy * 2) - .3, BROWN);
		new_ball(ox + (ofx * 2), oy, YELLOW_S, other, false, false, true);
		new_ball(ox + (ofx * 2) - .3, oy + (ofy * 2), RED);
		new_ball(ox + (ofx * 3), oy - ofy, ORANGE);
		new_ball(ox + (ofx * 3) - .4, oy + ofy - .7, EIGHT);
		new_ball(ox + (ofx * 4), oy - .5, PURPLE);
	} else if (game_name == rotation)
	{
		thegame->cat_is_set = true;		// Otherwise it will get confused
		
		// Rack up balls for Rotation
		new_ball((table.width / 4) + gap - 1, oy, CUE, other, true);
		new_ball(ox, oy, YELLOW);
		new_ball(ox + ofx, oy - ofy, RED_S);
		new_ball(ox + ofx, oy + ofy, ORANGE_S);
		new_ball(ox + ofx * 2, oy - ofy * 2, PURPLE);
		new_ball(ox + ofx * 2, oy, BROWN_S);
		new_ball(ox + ofx * 2, oy + ofy * 2, GREEN);
		new_ball(ox + ofx * 3, oy - ofy * 3, YELLOW_S);
		new_ball(ox + ofx * 3, oy - ofy, EIGHT);
		new_ball(ox + ofx * 3, oy + ofy, BLUE_S);
		new_ball(ox + ofx * 3, oy + ofy * 3, BROWN);
		new_ball(ox + ofx * 4, oy - ofy * 4, BLUE);
		new_ball(ox + ofx * 4, oy - ofy * 2, PURPLE_S);
		new_ball(ox + ofx * 4, oy, ORANGE);
		new_ball(ox + ofx * 4, oy + ofy * 2, GREEN_S);
		new_ball(ox + ofx * 4, oy + ofy * 4, RED);
	}
	
	sunk_balls = 0;
	last_sunk = 0;
	udl = table.x; udr = table.x + table.width;
	udt = table.y; udb = table.y + table.height;
	update_rack();
	
	if(placing_cue)
	{
		// this is in case the cue ball was sunk but not placed before the
		// reset, so that there can't be more than one cue ball
		gtk_signal_handler_block(GTK_OBJECT(drawing_area), qbp_hndlr);
		gtk_signal_handler_unblock(GTK_OBJECT(drawing_area),
		                           button_press_handler_id);
		gdk_window_set_cursor(drawing_area->window, (GdkCursor *)NULL);
		placing_cue = false;
	}
}

void Application::paint (GdkDrawable *pixmap, GdkGC *gc) {

	Point2D stick_start;
	Point2D stick_end;
	Vec2D stick;
	
	// Draw Table
	gdk_draw_pixmap(pixmap, gc, table_pixmaps[table_choice], 0, 0, 0, 0, width, height);

	// Place balls on table
	for_each(balls.begin(), balls.end(), draw_ball(pixmap, gc));

	// Draw Shooting Line
	if (selected != NULL) {
		stick_start.setPos(selected->x, selected->y);
		double xdist = stick_start.x - shoot.x;
		double ydist = stick_start.y - shoot.y;
		
		stick.setVec(xdist, ydist);
		double l = stick.mag() / 200;
		stick.dx /= l; stick.dy /= l;
		stick_end.x = stick_start.x - stick.dx;
		stick_end.y = stick_start.y - stick.dy;
				
		// Draw Cue Stick Line
		gdk_gc_set_foreground(gc, black);
		gdk_gc_set_line_attributes(gc, 2, GDK_LINE_SOLID, GDK_CAP_BUTT, GDK_JOIN_MITER);
		gdk_draw_line(pixmap, gc, int(stick_start.x), int(stick_start.y),
		              int(stick_end.x), int(stick_end.y));
		// Draw Power Line
		gdk_gc_set_foreground(gc, red);
		gdk_gc_set_line_attributes(gc, 2, GDK_LINE_SOLID, GDK_CAP_BUTT, GDK_JOIN_MITER);
		gdk_draw_line(pixmap, gc, int(stick_start.x), int(stick_start.y),
		              int(shoot.x), int(shoot.y));
		// Draw Aiming Line
		if (!aimlineoff)
		{
			stick.dx = -stick.dx;
			stick.dy = -stick.dy;
			double a = stick.mag() / 800;
			stick.dx /= a; stick.dy /= a;
			double sx = stick_start.x - stick.dx;
			double sy = stick_start.y - stick.dy;
	
			gdk_gc_set_foreground(gc, black);
			gdk_gc_set_line_attributes(gc, 1, GDK_LINE_ON_OFF_DASH, GDK_CAP_BUTT, GDK_JOIN_MITER);
			gdk_draw_line(pixmap, gc, int(stick_start.x), int(stick_start.y),
			              int(sx), int(sy));
		}
	}
}

void Application::mouse_down (double x, double y) {
	std::vector<Ball>::iterator bb;
 	bb = find_if(balls.begin(), balls.end(), pointer_selects( x, y, hit_moving));
 	if(bb != balls.end())
 	{
#if _CPP_CSTDLIB == 1
		// FIXME: horrible, non-portable, converting a vector iterator
		// to a pointer using g++ 3.0 private interface :-(
		// -- Philip Martin <philip_martin@ntlworld.com>
		selected = bb.base();
#else
 		selected = bb;
#endif
		shoot.x = x; shoot.y = y;
		thegame->new_shot();
	}
}

void Application::mouse_up (double x, double y) {
	if (selected != NULL) {
		selected->shoot(shoot);
		update_dirty(0, width, 0, height);
		thegame->shot = true;
	}
	selected = NULL;
}

void Application::mouse_drag (double x, double y) {
	if (selected != NULL) {
		Vec2D vel;
		double sx = double(selected->x), sy = double(selected->y);
		double xdist = (sx - x), ydist = (sy - y);
		// decrease the following parameter for more accuracy
		double dalpha = 0.00002;

		double x1 = (xdist*cos(dalpha*(sx - x))), x2 = (xdist*sin(dalpha*(sx - x)));
		double y1 = (ydist*cos(dalpha*(sy - y))), y2 = (ydist*sin(dalpha*(sy - y)));
		xdist = (x1 * x2);
		ydist = (y1 * y2);

		vel.setVec(xdist, ydist);
		
		if(!(superhuman || (vel.mag() <= 200)))
		{
			// calculate a vector with the same direction but
			// a magnitude of 200, since we don't want people
			// to be able to make superhuman shots
			double k = (vel.mag() / 200);
			vel.dx /= k; vel.dy /= k;
		}
		shoot.x = selected->x - vel.dx;
		shoot.y = selected->y - vel.dy;
		
		update_dirty(0, width, 0, height);
	}
}

void Application::update_rack()
{
	if(rack_area == (GtkWidget *)NULL)
		return;
	GdkRectangle area;
	area.x = area.y = 0;
	area.width = width; area.height = 70;
	gtk_widget_draw(GTK_WIDGET(rack_area), &area);
}

void Application::update_table()
{
	if(rack_area == (GtkWidget *)NULL)
		return;
	GdkRectangle area;
	area.x = area.y = 0;
	area.width = width; area.height = height;
	gtk_widget_draw(GTK_WIDGET(drawing_area), &area);
}

void Application::new_cue_ball()
{
	GdkCursor *cursor;
	cursor = gdk_cursor_new(GDK_CIRCLE);
	gdk_window_set_cursor(drawing_area->window, cursor);
	gdk_cursor_destroy(cursor);
	gtk_signal_handler_block(GTK_OBJECT(drawing_area), button_press_handler_id);
	selected = NULL; // just in case
	qbp_hndlr = gtk_signal_connect(GTK_OBJECT(drawing_area), "button_press_event",
	                               GTK_SIGNAL_FUNC(cue_ball_placed), this);
	placing_cue = true;
}

gint Application::cue_ball_placed(GtkWidget *widget,
                                           GdkEventButton *event,
								   Application *app)
{
	int x = int(event->x) ; int y = int(event->y);
	std::vector<Ball>::iterator bb;
	bb = find_if(app->balls.begin(), app->balls.end(), pointer_selects(x, y, FALSE));
	if (app->game_name == eight_ball || app->game_name == rotation)
	{
		if(x - BALL_SIZE / 2 > app->table.x &&
//		   x + BALL_SIZE / 2 < app->table.x + int(app->table.width / 4) + 8 &&
		   x + BALL_SIZE / 2 < app->table.x + app->table.width &&
		   y - BALL_SIZE / 2 > app->table.y &&
		   y + BALL_SIZE / 2 < app->table.y + app->table.height)
		{
			app->new_ball(x, y, CUE, other, true);
			update_dirty(x - BALL_SIZE / 2, x + BALL_SIZE / 2, y - BALL_SIZE / 2,
			             y + BALL_SIZE / 2);
			gtk_signal_disconnect(GTK_OBJECT(app->drawing_area), app->qbp_hndlr);
			gtk_signal_handler_unblock(GTK_OBJECT(app->drawing_area),
			                           app->button_press_handler_id);
			gdk_window_set_cursor(app->drawing_area->window, (GdkCursor *)NULL);
		}
	} else
	{
		if(x - BALL_SIZE / 2 > app->table.x &&
		   x + BALL_SIZE / 2 < app->table.x + app->table.width &&
		   y - BALL_SIZE / 2 > app->table.y &&
		   y + BALL_SIZE / 2 < app->table.y + app->table.height)
		{
			app->new_ball(x, y, CUE, other, true);
			update_dirty(x - BALL_SIZE / 2, x + BALL_SIZE / 2, y - BALL_SIZE / 2,
			             y + BALL_SIZE / 2);
			gtk_signal_disconnect(GTK_OBJECT(app->drawing_area), app->qbp_hndlr);
			gtk_signal_handler_unblock(GTK_OBJECT(app->drawing_area),
			                           app->button_press_handler_id);
			gdk_window_set_cursor(app->drawing_area->window, (GdkCursor *)NULL);
		}
	}
	app->placing_cue = false;

	return app->placing_cue;
}

void Application::add_pixmap_directory (const gchar *directory)
{
  load_directories = g_list_prepend (load_directories, g_strdup (directory));
}

gchar *Application::check_file_exists (const gchar *directory, const gchar *filename)
{
  gchar *full_filename;
  struct stat s;
  gint status;

  full_filename = (gchar*) g_malloc (strlen (directory) + 1
                                     + strlen (filename) + 1);
  strcpy (full_filename, directory);
  strcat (full_filename, G_DIR_SEPARATOR_S);
  strcat (full_filename, filename);

  status = stat (full_filename, &s);
  if (status == 0 && S_ISREG (s.st_mode))
    return full_filename;
  g_free (full_filename);
  return NULL;
}

GdkPixmap *Application::load_pixmap(GdkPixmap **mask, char *filename)
{
	GList *elem;
	gchar *found_filename = NULL;

	/* We first try any pixmaps directories set by the application. */
	elem = load_directories;
	while (elem)
	{
		found_filename = check_file_exists ((gchar*)elem->data, filename);
		if (found_filename)
		break;
		elem = elem->next;
	}

	GdkPixmap *pixmap = gdk_pixmap_colormap_create_from_xpm(NULL, colormap,
                            mask, NULL, found_filename);
	if(pixmap == (GdkPixmap *)NULL)
		g_error("Could not load pixmap: %s", found_filename);
	return pixmap;

}

void Application::load_ball_xpms()
{
	add_pixmap_directory("/usr/local/share/gtkpool");
	add_pixmap_directory("/usr/share/gtkpool");
	
	// Small
	balls_pixmaps[CUE] = load_pixmap(&balls_pixmap_masks[CUE], "ball_cue_sm.xpm");
	balls_pixmaps[YELLOW] = load_pixmap(&balls_pixmap_masks[YELLOW], "ball_1_sm.xpm");
	balls_pixmaps[BLUE] = load_pixmap(&balls_pixmap_masks[BLUE], "ball_2_sm.xpm");
	balls_pixmaps[RED] = load_pixmap(&balls_pixmap_masks[RED], "ball_3_sm.xpm");
	balls_pixmaps[PURPLE] = load_pixmap(&balls_pixmap_masks[PURPLE], "ball_4_sm.xpm");
	balls_pixmaps[ORANGE] = load_pixmap(&balls_pixmap_masks[ORANGE], "ball_5_sm.xpm");
	balls_pixmaps[GREEN] = load_pixmap(&balls_pixmap_masks[GREEN], "ball_6_sm.xpm");
	balls_pixmaps[BROWN] = load_pixmap(&balls_pixmap_masks[BROWN], "ball_7_sm.xpm");
	balls_pixmaps[EIGHT] = load_pixmap(&balls_pixmap_masks[EIGHT], "ball_8_sm.xpm");
	balls_pixmaps[YELLOW_S] = load_pixmap(&balls_pixmap_masks[YELLOW_S], "ball_9_sm.xpm");
	balls_pixmaps[BLUE_S] = load_pixmap(&balls_pixmap_masks[BLUE_S], "ball_10_sm.xpm");
	balls_pixmaps[RED_S] = load_pixmap(&balls_pixmap_masks[RED_S], "ball_11_sm.xpm");
	balls_pixmaps[PURPLE_S] = load_pixmap(&balls_pixmap_masks[PURPLE_S], "ball_12_sm.xpm");
	balls_pixmaps[ORANGE_S] = load_pixmap(&balls_pixmap_masks[ORANGE_S], "ball_13_sm.xpm");
	balls_pixmaps[GREEN_S] = load_pixmap(&balls_pixmap_masks[GREEN_S], "ball_14_sm.xpm");
	balls_pixmaps[BROWN_S] = load_pixmap(&balls_pixmap_masks[BROWN_S], "ball_15_sm.xpm");

	// Large
	balls_big_pixmaps[CUE] = load_pixmap(&balls_big_pixmap_masks[CUE], "ball_cue_lg.xpm");
	balls_big_pixmaps[YELLOW] = load_pixmap(&balls_big_pixmap_masks[YELLOW], "ball_1_lg.xpm");
	balls_big_pixmaps[BLUE] = load_pixmap(&balls_big_pixmap_masks[BLUE], "ball_2_lg.xpm");
	balls_big_pixmaps[RED] = load_pixmap(&balls_big_pixmap_masks[RED], "ball_3_lg.xpm");
	balls_big_pixmaps[PURPLE] = load_pixmap(&balls_big_pixmap_masks[PURPLE], "ball_4_lg.xpm");
	balls_big_pixmaps[ORANGE] = load_pixmap(&balls_big_pixmap_masks[ORANGE], "ball_5_lg.xpm");
	balls_big_pixmaps[GREEN] = load_pixmap(&balls_big_pixmap_masks[GREEN], "ball_6_lg.xpm");
	balls_big_pixmaps[BROWN] = load_pixmap(&balls_big_pixmap_masks[BROWN], "ball_7_lg.xpm");
	balls_big_pixmaps[EIGHT] = load_pixmap(&balls_big_pixmap_masks[EIGHT], "ball_8_lg.xpm");
	balls_big_pixmaps[YELLOW_S] = load_pixmap(&balls_big_pixmap_masks[YELLOW_S], "ball_9_lg.xpm");
	balls_big_pixmaps[BLUE_S] = load_pixmap(&balls_big_pixmap_masks[BLUE_S], "ball_10_lg.xpm");
	balls_big_pixmaps[RED_S] = load_pixmap(&balls_big_pixmap_masks[RED_S], "ball_11_lg.xpm");
	balls_big_pixmaps[PURPLE_S] = load_pixmap(&balls_big_pixmap_masks[PURPLE_S], "ball_12_lg.xpm");
	balls_big_pixmaps[ORANGE_S] = load_pixmap(&balls_big_pixmap_masks[ORANGE_S], "ball_13_lg.xpm");
	balls_big_pixmaps[GREEN_S] = load_pixmap(&balls_big_pixmap_masks[GREEN_S], "ball_14_lg.xpm");
	balls_big_pixmaps[BROWN_S] = load_pixmap(&balls_big_pixmap_masks[BROWN_S], "ball_15_lg.xpm");

	// Pool Table
	table_pixmaps[TABLE_MB] = load_pixmap(&table_pixmaps[TABLE_MB], "pool_table_maple-burl.xpm");
	table_pixmaps[TABLE_BL] = load_pixmap(&table_pixmaps[TABLE_BL], "pool_table_black-laquer.xpm");
	table_pixmaps[TABLE_BP] = load_pixmap(&table_pixmaps[TABLE_BP], "pool_table_black-pearl.xpm");
	table_pixmaps[TABLE_MA] = load_pixmap(&table_pixmaps[TABLE_MA], "pool_table_mahogany.xpm");
}

void Application::load_sounds()
{
	gchar *found_filename = NULL;
	gchar *found_filename2 = NULL;
	GList *elem;
	char filename[20] = "ball_hit.raw";
	coll_sndc = 12000; // max size to read
	elem = load_directories;
	while (elem)
	{
		found_filename = check_file_exists ((gchar*)elem->data, filename);
		if (found_filename)
		break;
		elem = elem->next;
	}
	load_sound(found_filename, coll_snd, &coll_sndc);	
	
	/* really ugly way of loading the second sound
	   make sure to fix this
	*/
	char filename2[20] = "ball_drop.raw";
	pock_sndc = 194000; // max size to read
	elem = load_directories;
	while (elem)
	{
		found_filename2 = check_file_exists ((gchar*)elem->data, filename2);
		if (found_filename2)
		break;
		elem = elem->next;
	}
	load_sound(found_filename2, pock_snd, &pock_sndc);	

}

void Application::init_sound()
{
	try
	{
		open_sound_device();
		configure_sound_device();
		sound = true;
	}
	catch (SoundError &se)
	{
		sound = false;
		switch (se.code)
		{
			case SE_OPENFAIL:
				g_message("An error occured while opening the sound device\n\t%s", se.description);
				break;
			case SE_FRAGFAIL:
				g_message("An error occured while setting the sound card's fragment size");
				g_message("\t%s\nThe requested value was: %x", se.description,
				          se.requested);
				break;
		};
	}
	if(sound)
		load_sounds();
}

void Application::print_message(const char *message, int colour = 0)
{
	if (colour > 4)
		colour = 14;
	if (colour < 0)
		colour = 14;
	gtk_text_insert(GTK_TEXT(chat_text), NULL, message_colours[colour], NULL, "\n", -1);
	gtk_text_insert(GTK_TEXT(chat_text), NULL, message_colours[colour], NULL, message, -1);
}
