/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  GThumb
 *
 *  Copyright (C) 2001 The Free Software Foundation, Inc.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Street #330, Boston, MA 02111-1307, USA.
 */

#include <config.h>
#include <string.h>
#include <stdio.h>
#include <gtk/gtkadjustment.h>
#include <gtk/gtksignal.h>
#include <gtk/gtkmain.h>
#include <libgnomeui/gnome-canvas-rect-ellipse.h>
#include <libgnomeui/gnome-icon-item.h>

#include "image-list.h"

#define DEFAULT_MEDIUM_FONT \
          "-adobe-helvetica-bold-r-normal-*-*-100-*-*-p-*-iso8859-1,"  \
          "-*-*-bold-r-normal-*-*-100-*-*-*-*-*-*,*"

#define DEFAULT_SMALL_FONT \
          "-adobe-helvetica-medium-r-normal-*-*-100-*-*-p-*-iso8859-1,"  \
          "-*-*-medium-r-normal-*-*-100-*-*-*-*-*-*,*"

/* default spacings */
#define DEFAULT_ROW_SPACING   4
#define DEFAULT_COL_SPACING   2
#define DEFAULT_TEXT_SPACING  2
#define DEFAULT_IMAGE_BORDER  2
#define TEXT_COMMENT_SPACE    5  /* space between text and comment. */

/* Autoscroll timeout in milliseconds */
#define SCROLL_TIMEOUT 30

/**/
#define gray50_width 2
#define gray50_height 2
static const gchar gray50_bits[] = { 0x02, 0x01, };


/* Signals */
enum {
	SELECT_IMAGE,
	UNSELECT_IMAGE,
	TEXT_CHANGED,
	LAST_SIGNAL
};

typedef enum {
	SYNC_INSERT,
	SYNC_REMOVE
} SyncType;

enum {
	ARG_0,
	ARG_HADJUSTMENT,
	ARG_VADJUSTMENT
};

static guint gil_signals[LAST_SIGNAL] = { 0 };


static GtkContainerClass *parent_class;


/* A row of images */
typedef struct {
	int y;
	int image_height, text_height, comment_height;
	GList *line_images;
} ImageLine;

/* Private data of the ImageList structure */
typedef struct {
	/* List of images and list of rows of images */
	GList *image_list;
	GList *lines;
	
	/* Max of the height of all the image rows and window height */
	int total_height;
	
	/* Selection mode */
	GtkSelectionMode selection_mode;

	/* Veiw mode */
	guint8 view_mode;

	/* Freeze count */
	int frozen;

	/* Width allocated for images */
	int image_width;

	/* Spacing values */
	int row_spacing;
	int col_spacing;
	int text_spacing;
	int image_border;

	/* Separators used to wrap the text below images */
	char *separators;

	/* Index and pointer to last selected image */
	int last_selected_idx;
	Image *last_selected_image;

	/* Timeout ID for autoscrolling */
	guint timer_tag;

	/* Change the adjustment value by this amount when autoscrolling */
	int value_diff;
	
	/* Mouse position for autoscrolling */
	int event_last_x;
	int event_last_y;

	/* Selection start position */
	int sel_start_x;
	int sel_start_y;

	/* Modifier state when the selection began */
	guint sel_state;

	/* Rubberband rectangle */
	GnomeCanvasItem *sel_rect;

	/* Saved event for a pending selection */
	GdkEvent select_pending_event;

	/* Whether the image texts are editable */
	guint is_editable : 1;

	/* Whether the image texts need to be copied */
	guint static_text : 1;

	/* Whether the images need to be laid out */
	guint dirty : 1;

	/* Whether the user is performing a rubberband selection */
	guint selecting : 1;

	/* Whether editing an image is pending after a button press */
	guint edit_pending : 1;

	/* Whether selection is pending after a button press */
	guint select_pending : 1;

	/* Whether the image that is pending selection was selected to 
	 * begin with */
	guint select_pending_was_selected : 1;

	/* Ascending or descending order. */
	GtkSortType sort_type;

	/* Comapre function. */
	GCompareFunc compare;

	/* Whether the image list is editing. */
	gboolean has_focus;

	/* The text before starting to edit. */
	gchar * old_text;
} ImageListPrivate;


static int
image_line_height (ImageList *gil, ImageLine *il)
{
	ImageListPrivate *priv;

	priv = gil->priv;

	return (il->image_height 
		+ ((il->comment_height > 0 || il->text_height > 0) ? priv->text_spacing : 0)
		+ il->comment_height
		+ ((il->comment_height > 0 && il->text_height > 0) ? TEXT_COMMENT_SPACE : 0)
		+ il->text_height
		+ priv->row_spacing);
}


static void
image_get_height (Image *image, 
		  int *image_height, 
		  int *text_height,
		  int *comment_height)
{
	gboolean has_text, has_comment;

	has_text = image->text->text != NULL;
	has_comment = image->comment->text != NULL;

	*image_height = image->height;
	*text_height = has_text ? image->text->ti->height : 0;
	*comment_height = has_comment ? image->comment->ti->height : 0;
}


static int
gil_get_images_per_line (ImageList *gil)
{
	ImageListPrivate *priv;
	gint images_per_line;

	priv = gil->priv;

	images_per_line = GTK_WIDGET (gil)->allocation.width / (priv->image_width + priv->col_spacing);
	if (images_per_line == 0)
		images_per_line = 1;

	return images_per_line;
}


int
image_list_get_images_per_line (ImageList *gil)
{
	g_return_val_if_fail (IS_IMAGE_LIST (gil), 1);

	return gil_get_images_per_line (gil);
}


static void
gil_place_image (ImageList *gil, 
		 Image *image, 
		 gint x, 
		 gint y, 
		 gint image_height,
		 gboolean view_text,
		 gboolean view_comment)
{
	ImageListPrivate *priv;
	gint x_offset, y_offset;

	priv = gil->priv;

	x_offset = (priv->image_width - image->width) / 2;

	if (image_height > image->height)
		y_offset = (image_height - image->height) / 2;
	else
		y_offset = 0;

	gnome_canvas_item_set (GNOME_CANVAS_ITEM (image->image),
			       "x", (gdouble) x + x_offset,
			       "y", (gdouble) y + y_offset,
			       NULL);

	y = y + image_height + priv->text_spacing;

	if (view_comment) {

		gnome_canvas_item_show (GNOME_CANVAS_ITEM (image->comment));
		gnome_icon_text_item_setxy (image->comment,
					    x + 1,
					    y);
		y += image->comment->ti->height + TEXT_COMMENT_SPACE;
	} else
		gnome_canvas_item_hide (GNOME_CANVAS_ITEM (image->comment));

	if (view_text) {
		gnome_canvas_item_show (GNOME_CANVAS_ITEM (image->text));
		gnome_icon_text_item_setxy (image->text,
					    x + 1,
					    y);
	} else
		gnome_canvas_item_hide (GNOME_CANVAS_ITEM (image->text));
}


static void
image_get_view_mode (ImageList *gil,
		     Image *image,
		     gboolean *view_text,
		     gboolean *view_comment)
{
	ImageListPrivate *priv = gil->priv;
	
	*view_text = TRUE;
	*view_comment = TRUE;

	if (priv->view_mode == IMAGE_LIST_VIEW_TEXT)
		*view_comment = FALSE;
	if (priv->view_mode == IMAGE_LIST_VIEW_COMMENTS)
		*view_text = FALSE;
	if ((priv->view_mode == IMAGE_LIST_VIEW_COMMENTS_OR_TEXT)
	    && (*(image->comment->text) != 0))
		*view_text = FALSE;

	if (*(image->comment->text) == 0)
		*view_comment = FALSE;
	if (*(image->text->text) == 0)
		*view_text = FALSE;
}


static void
gil_layout_line (ImageList *gil, 
		 ImageLine *il)
{
	ImageListPrivate *priv;
	GList *l;
	gint x;
	gboolean view_text, view_comment;

	priv = gil->priv;

	x = 0;
	for (l = il->line_images; l; l = l->next) {
		Image *image = l->data;

		image_get_view_mode (gil, image, &view_text, &view_comment);
		
		x += priv->col_spacing;
		gil_place_image (gil, image, x, il->y, il->image_height,
				 view_text, view_comment);
		x += priv->image_width;
	}
}


static void
gil_add_and_layout_line (ImageList *gil, 
			 GList *line_images, 
			 gint y,
			 gint image_height, 
			 gint text_height,
			 gint comment_height)
{
	ImageListPrivate *priv;
	ImageLine *il;

	priv = gil->priv;
	
	il = g_new (ImageLine, 1);
	il->line_images = line_images;
	il->y = y;
	il->image_height = image_height;
	il->text_height = text_height;
	il->comment_height = comment_height;

	gil_layout_line (gil, il);
	priv->lines = g_list_append (priv->lines, il);
}


static void
gil_relayout_images_at (ImageList *gil, 
			gint pos, 
			gint y)
{
	ImageListPrivate *priv;
	gint col, row, text_height, image_height, comment_height;
	gint images_per_line, n;
	GList *line_images, *l;
	gint max_height, tmp_max_height;

	priv = gil->priv;
	images_per_line = gil_get_images_per_line (gil);

	col = row = text_height = image_height = comment_height = 0;
	max_height = 0;
	line_images = NULL;

	l = g_list_nth (priv->image_list, pos);

	for (n = pos; l; l = l->next, n++) {
		Image *image = l->data;
		gint ih, th, ch;
		gboolean view_text, view_comment;

		if (!(n % images_per_line)) {
			if (line_images) {
				gil_add_and_layout_line (gil, line_images, y,
							 image_height, 
							 text_height,
							 comment_height);
				line_images = NULL;

				y += max_height + priv->row_spacing;
			}

			max_height = 0;
			image_height = 0;
			text_height = 0;
			comment_height = 0;
		}

		image_get_height (image, &ih, &th, &ch);
		image_get_view_mode (gil, image, &view_text, &view_comment);

		if (! view_text) th = 0;
		if (! view_comment) ch = 0;

		tmp_max_height = ih + th + ch;
		if ((ch > 0) || (th > 0))
			tmp_max_height += priv->text_spacing;
		if ((ch > 0) && (th > 0))
			tmp_max_height += TEXT_COMMENT_SPACE;
		max_height = MAX (tmp_max_height, max_height);

		image_height = MAX (ih, image_height);
		text_height = MAX (th, text_height);
		comment_height = MAX (ch, comment_height);

		line_images = g_list_append (line_images, image);
	}

	if (line_images)
		gil_add_and_layout_line (gil, line_images, y, 
					 image_height, 
					 text_height,
					 comment_height);
}


static void
gil_free_line_info (ImageList *gil)
{
	ImageListPrivate *priv;
	GList *l;

	priv = gil->priv;

	for (l = priv->lines; l; l = l->next) {
		ImageLine *il = l->data;

		g_list_free (il->line_images);
		g_free (il);
	}

	g_list_free (priv->lines);
	priv->lines = NULL;
	priv->total_height = 0;
}


static void
gil_free_line_info_from (ImageList *gil, 
			 gint first_line)
{
	ImageListPrivate *priv;
	GList *l, *ll;
	
	priv = gil->priv;
	ll = g_list_nth (priv->lines, first_line);

	for (l = ll; l; l = l->next) {
		ImageLine *il = l->data;

		g_list_free (il->line_images);
		g_free (il);
	}

	if (priv->lines) {
		if (ll->prev)
			ll->prev->next = NULL;
		else
			priv->lines = NULL;
	}

	g_list_free (ll);
}


static void
gil_layout_from_line (ImageList *gil, 
		      gint line)
{
	ImageListPrivate *priv;
	GList *l;
	gint height;

	priv = gil->priv;
	
	gil_free_line_info_from (gil, line);

	height = priv->row_spacing;
	for (l = priv->lines; l; l = l->next) {
		ImageLine *il = l->data;

		height += image_line_height (gil, il);
	}

	gil_relayout_images_at (gil, line * gil_get_images_per_line (gil), height);
}


static void
gil_layout_all_images (ImageList *gil)
{
	ImageListPrivate *priv;

	priv = gil->priv;

	if (!GTK_WIDGET_REALIZED (gil))
		return;

	gil_free_line_info (gil);
	gil_relayout_images_at (gil, 0, priv->row_spacing);
	priv->dirty = FALSE;
}


static void
gil_scrollbar_adjust (ImageList *gil)
{
	ImageListPrivate *priv;
	GList *l;
	gdouble wx, wy;
	gint height, step_increment;

	priv = gil->priv;

	if (!GTK_WIDGET_REALIZED (gil))
		return;

	height = priv->row_spacing;
	step_increment = 0;
	for (l = priv->lines; l; l = l->next) {
		ImageLine *il = l->data;

		height += image_line_height (gil, il);

		if (l == priv->lines)
			step_increment = height;
	}

	if (!step_increment)
		step_increment = 10;

	priv->total_height = MAX (height, GTK_WIDGET (gil)->allocation.height);

	wx = wy = 0;
	gnome_canvas_window_to_world (GNOME_CANVAS (gil), 0, 0, &wx, &wy);

	gil->adj->upper = height;
	gil->adj->step_increment = step_increment;
	gil->adj->page_increment = GTK_WIDGET (gil)->allocation.height;
	gil->adj->page_size = GTK_WIDGET (gil)->allocation.height;

	if (wy > gil->adj->upper - gil->adj->page_size)
		wy = gil->adj->upper - gil->adj->page_size;

	gil->adj->value = wy;

	gtk_adjustment_changed (gil->adj);
}


/* Emits the select_image or unselect_image signals as appropriate */
static void
emit_select (ImageList *gil, int sel, int i, GdkEvent *event)
{
	gtk_signal_emit (GTK_OBJECT (gil),
			 gil_signals[sel ? SELECT_IMAGE : UNSELECT_IMAGE],
			 i,
			 event);
}


/**
 * image_list_select_all:
 * @gil: An image list.
 * 
 * Selects all the images in the image list. 
 *
 **/
void
image_list_select_all (ImageList *gil) 
{
	ImageListPrivate *priv;
	GList *l;
	Image *image;
	gint i;

	g_return_if_fail (IS_IMAGE_LIST (gil));
	
	priv = gil->priv;
	
	for (l = priv->image_list, i = 0; l; l = l->next, i++) {
		image = l->data;

		if (!image->selected)
			emit_select (gil, TRUE, i, NULL);
	}
}


/**
 * image_list_unselect_all:
 * @gil:   An image list.
 * @event: Unused, must be NULL.
 * @keep:  For internal use only; must be NULL.
 *
 * Unselects all the images in the image list.  The @event and @keep parameters
 * must be NULL, since they are used only internally.
 *
 * Returns: the number of images in the image list
 */
int
image_list_unselect_all (ImageList *gil, 
			 GdkEvent *event, 
			 gpointer keep)
{
	ImageListPrivate *priv;
	GList *l;
	Image *image;
	gint i, idx = 0;

	g_return_val_if_fail (IS_IMAGE_LIST (gil), 0);

	priv = gil->priv;

	for (l = priv->image_list, i = 0; l; l = l->next, i++) {
		image = l->data;

		if (image == keep)
			idx = i;
		else if (image->selected)
			emit_select (gil, FALSE, i, event);
	}

	return idx;
}


static void
sync_selection (ImageList *gil, 
		gint pos, 
		SyncType type)
{
	GList *list;

	for (list = gil->selection; list; list = list->next) {
		if (GPOINTER_TO_INT (list->data) >= pos) {
			gint i = GPOINTER_TO_INT (list->data);

			switch (type) {
			case SYNC_INSERT:
				list->data = GINT_TO_POINTER (i + 1);
				break;

			case SYNC_REMOVE:
				list->data = GINT_TO_POINTER (i - 1);
				break;

			default:
				g_assert_not_reached ();
			}
		}
	}
}


static gint
gil_image_to_index (ImageList *gil, Image *image)
{
	ImageListPrivate *priv;
	GList *l;
	gint n;

	priv = gil->priv;

	n = 0;
	for (l = priv->image_list; l; n++, l = l->next)
		if (l->data == image)
			return n;

	/*g_assert_not_reached ();*/
	return -1; /* Shut up the compiler */
}


/* Event handler for images when we are in SINGLE or BROWSE mode */
static gint
selection_one_image_event (ImageList *gil, 
			   Image *image, 
			   gint idx, 
			   gint on_text, 
			   GdkEvent *event)
{
	ImageListPrivate *priv;
	GnomeIconTextItem *text;
	gint retval;

	priv = gil->priv;
	retval = FALSE;

	/* We use a separate variable and ref the object because it may be
	 * destroyed by one of the signal handlers.
	 */
	text = image->text;
	gtk_object_ref (GTK_OBJECT (text));

	switch (event->type) {
	case GDK_BUTTON_PRESS:
		priv->edit_pending = FALSE;
		priv->select_pending = FALSE;

		/* Ignore wheel mouse clicks for now */
		if (event->button.button > 3)
			break;

		if (!image->selected) {
			image_list_unselect_all (gil, NULL, NULL);
			emit_select (gil, TRUE, idx, event);
		} else {
			if (priv->selection_mode == GTK_SELECTION_SINGLE
			    && (event->button.state & GDK_CONTROL_MASK))
				emit_select (gil, FALSE, idx, event);
			else if (on_text && priv->is_editable && event->button.button == 1)
				priv->edit_pending = TRUE;
			else
				emit_select (gil, TRUE, idx, event);
		}

		retval = TRUE;
		break;

	case GDK_2BUTTON_PRESS:
	case GDK_3BUTTON_PRESS:
		/* Ignore wheel mouse clicks for now */
		if (event->button.button > 3)
			break;

		emit_select (gil, TRUE, idx, event);
		retval = TRUE;
		break;

	case GDK_BUTTON_RELEASE:
		if (priv->edit_pending) {
			gnome_icon_text_item_start_editing (text);
			priv->edit_pending = FALSE;
		}

		retval = TRUE;
		break;

	default:
		break;
	}

	/* If the click was on the text and we actually did something, stop the
	 * image text item's own handler from executing.
	 */
	if (on_text && retval)
		gtk_signal_emit_stop_by_name (GTK_OBJECT (text), "event");

	gtk_object_unref (GTK_OBJECT (text));

	return retval;
}


/* Handles range selections when clicking on an image */
static void
select_range (ImageList *gil, 
	      Image *image, 
	      gint idx, 
	      GdkEvent *event)
{
	ImageListPrivate *priv;
	gint a, b;
	GList *l;
	Image *i;

	priv = gil->priv;

	if (priv->last_selected_idx == -1) {
		priv->last_selected_idx = idx;
		priv->last_selected_image = image;
	}

	if (idx < priv->last_selected_idx) {
		a = idx;
		b = priv->last_selected_idx;
	} else {
		a = priv->last_selected_idx;
		b = idx;
	}

	l = g_list_nth (priv->image_list, a);

	for (; a <= b; a++, l = l->next) {
		i = l->data;

		if (!i->selected)
			emit_select (gil, TRUE, a, NULL);
	}

	/* Actually notify the client of the event */
	emit_select (gil, TRUE, idx, event);
}


/* Handles image selection for MULTIPLE or EXTENDED selection modes */
static void
do_select_many (ImageList *gil, 
		Image *image,
		gint idx, 
		GdkEvent *event, 
		gint use_event)
{
	ImageListPrivate *priv;
	gint range, additive;

	priv = gil->priv;

	range = (event->button.state & GDK_SHIFT_MASK) != 0;
	additive = (event->button.state & GDK_CONTROL_MASK) != 0;

	if (!additive) {
		if (image->selected)
			image_list_unselect_all (gil, NULL, image);
		else
			image_list_unselect_all (gil, NULL, NULL);
	}

	if (!range) {
		if (additive)
			emit_select (gil, !image->selected, idx, use_event ? event : NULL);
		else 
			emit_select (gil, TRUE, idx, use_event ? event : NULL);

		priv->last_selected_idx = idx;
		priv->last_selected_image = image;
	} else
		select_range (gil, image, idx, use_event ? event : NULL);
}


/* Event handler for images when we are in MULTIPLE or EXTENDED mode */
static gint
selection_many_image_event (ImageList *gil, 
			    Image *image, 
			    gint idx, 
			    gint on_text, 
			    GdkEvent *event)
{
	ImageListPrivate *priv;
	GnomeIconTextItem *text;
	gint retval;
	gint additive, range;
	gint do_select;

	priv = gil->priv;
	retval = FALSE;

	/* We use a separate variable and ref the object because it may be
	 * destroyed by one of the signal handlers.
	 */
	text = image->text;
	gtk_object_ref (GTK_OBJECT (text));
	
	range = (event->button.state & GDK_SHIFT_MASK) != 0;
	additive = (event->button.state & GDK_CONTROL_MASK) != 0;
	
	switch (event->type) {
	case GDK_BUTTON_PRESS:
		priv->edit_pending = FALSE;
		priv->select_pending = FALSE;

		/* Ignore wheel mouse clicks for now */
		if (event->button.button > 3)
			break;

		do_select = TRUE;

		if (additive || range) {
			if (additive && !range) {
				priv->select_pending = TRUE;
				priv->select_pending_event = *event;
				priv->select_pending_was_selected = image->selected;

				/* We have to emit this so that the client will
				 * know about the click.
				 */
				/*emit_select (gil, TRUE, idx, event);*/
				do_select = FALSE;
			}
		} else if (image->selected) {
			priv->select_pending = TRUE;
			priv->select_pending_event = *event;
			priv->select_pending_was_selected = image->selected;
			
			if (on_text && priv->is_editable && event->button.button == 1)
				priv->edit_pending = TRUE;
			
			emit_select (gil, TRUE, idx, event);
			do_select = FALSE;
		}
		if (do_select)
			do_select_many (gil, image, idx, event, TRUE);
		
		retval = TRUE;
		break;
		
	case GDK_2BUTTON_PRESS:
	case GDK_3BUTTON_PRESS:
		/* Ignore wheel mouse clicks for now */
		if (event->button.button > 3)
			break;
		
		emit_select (gil, TRUE, idx, event);
		retval = TRUE;
		break;
		
	case GDK_BUTTON_RELEASE:
		if (priv->select_pending) {
			image->selected = priv->select_pending_was_selected;
			do_select_many (gil, image, idx, &priv->select_pending_event, FALSE);
			priv->select_pending = FALSE;
			retval = TRUE;
		}

		if (priv->edit_pending) {
			gnome_icon_text_item_start_editing (text);
			priv->edit_pending = FALSE;
			retval = TRUE;
		}
		break;
		
	default:
		break;
	}
	
	/* If the click was on the text and we actually did something, stop the
	 * image text item's own handler from executing.
	 */
	if (on_text && retval)
		gtk_signal_emit_stop_by_name (GTK_OBJECT (text), "event");
	
	gtk_object_unref (GTK_OBJECT (text));

	return retval;
}


/* Event handler for images in the image list */
static gint
image_event (GnomeCanvasItem *item,
	     GdkEvent *event, 
	     gpointer data)
{
	Image *image;
	ImageList *gil;
	ImageListPrivate *priv;
	gint idx;
	gint on_text;

	image = data;
	gil = IMAGE_LIST (item->canvas);
	priv = gil->priv;
	idx = gil_image_to_index (gil, image);
	g_assert (idx != -1);
	
	on_text = item == GNOME_CANVAS_ITEM (image->text); 

	switch (priv->selection_mode) {
	case GTK_SELECTION_SINGLE:
	case GTK_SELECTION_BROWSE:
		return selection_one_image_event (gil, image, idx, on_text, event);

	case GTK_SELECTION_MULTIPLE:
	case GTK_SELECTION_EXTENDED:
		return selection_many_image_event (gil, image, idx, on_text, event);

	default:
		g_assert_not_reached ();
		return FALSE; /* Shut up the compiler */
	}
}


/* Handler for the editing_started signal of an image text item.  We block the
 * event handler so that it will not be called while the text is being edited.
 */
static void
editing_started (GnomeIconTextItem *iti, 
		 gpointer data)
{
	Image *image;
	ImageListPrivate *priv;

	image = data;
	gtk_signal_handler_block (GTK_OBJECT (iti), image->text_event_id);
	image_list_unselect_all (IMAGE_LIST (GNOME_CANVAS_ITEM (iti)->canvas), NULL, image);
	
	priv = IMAGE_LIST (GNOME_CANVAS_ITEM (iti)->canvas)->priv;
	priv->has_focus = TRUE;
	
	if (priv->old_text)
		g_free (priv->old_text);
	priv->old_text = g_strdup (gnome_icon_text_item_get_text(image->text));
}


/* Handler for the editing_stopped signal of an image text item. 
 * We unblock the event handler so that we can get events from it again.
 */
static void
editing_stopped (GnomeIconTextItem *iti, 
		 gpointer data)
{
	Image *image;
	ImageListPrivate *priv;

	image = data;
	gtk_signal_handler_unblock (GTK_OBJECT (iti), image->text_event_id);

	priv = IMAGE_LIST (GNOME_CANVAS_ITEM (iti)->canvas)->priv;
	priv->has_focus = FALSE;
}


static gboolean
text_changed (GnomeCanvasItem *item, 
	      Image *image)
{
	ImageList *gil;
	gboolean accept;
	gint idx;

	gil = IMAGE_LIST (item->canvas);
	accept = TRUE;

	idx = gil_image_to_index (gil, image);
	gtk_signal_emit (GTK_OBJECT (gil),
			 gil_signals[TEXT_CHANGED],
			 idx, 
			 gnome_icon_text_item_get_text (image->text),
			 &accept);

	return accept;
}


static void
height_changed (GnomeCanvasItem *item, 
		Image *image)
{
	ImageList *gil;
	ImageListPrivate *priv;
	gint n;

	gil = IMAGE_LIST (item->canvas);
	priv = gil->priv;

	if (!GTK_WIDGET_REALIZED (gil))
		return;

	if (priv->frozen) {
		priv->dirty = TRUE;
		return;
	}

	n = gil_image_to_index (gil, image);
	gil_layout_from_line (gil, n / gil_get_images_per_line (gil));
	gil_scrollbar_adjust (gil);
}


static Image *
image_new_from_pixbuf (ImageList *gil, 
		       GdkPixbuf *pixbuf, 
		       const char *text,
		       const char *comment)
{
	ImageListPrivate *priv;
	GnomeCanvas *canvas;
	GnomeCanvasGroup *group;
	Image *image;
	gdouble width, height;

	priv = gil->priv;
	canvas = GNOME_CANVAS (gil);
	group = GNOME_CANVAS_GROUP (canvas->root);

	image = g_new0 (Image, 1);

	/* Image item. */
	if (pixbuf) {
		image->image = GNOME_CANVAS_THUMB (gnome_canvas_item_new (
			group,
			gnome_canvas_thumb_get_type (),
			"x", 0.0,
			"y", 0.0,
			"width", (double) priv->image_width,
			"height", (double) priv->image_width,
			"pixbuf", pixbuf,
			NULL));

		/*
		image->width = gdk_pixbuf_get_width (pixbuf);
		image->height = gdk_pixbuf_get_height (pixbuf);
		*/
		gtk_object_get (GTK_OBJECT (image->image),
				"width", &width,
				"height", &height,
				NULL);
		image->width = (gint) width;
		image->height = (gint) height;
	} else {
		image->image = GNOME_CANVAS_THUMB (gnome_canvas_item_new (
			group,
			gnome_canvas_thumb_get_type (),
			"x", 0.0,
			"y", 0.0,
			NULL));

		image->width = 0;
		image->height = 0;
	}

	/* Comment item. */
	image->comment = GNOME_ICON_TEXT_ITEM (gnome_canvas_item_new (
		group,
		gnome_icon_text_item_get_type (),
		NULL));
	gnome_canvas_item_set (GNOME_CANVAS_ITEM (image->comment),
			       "use_broken_event_handling", FALSE,
			       NULL);
	gnome_icon_text_item_configure (image->comment,
					0, 
					0, 
					priv->image_width, 
					DEFAULT_MEDIUM_FONT,
					comment, 
					FALSE,
					priv->static_text);

	/* Text item. */
	image->text = GNOME_ICON_TEXT_ITEM (gnome_canvas_item_new (
		group,
		gnome_icon_text_item_get_type (),
		NULL));
	gnome_canvas_item_set (GNOME_CANVAS_ITEM (image->text),
			       "use_broken_event_handling", FALSE,
			       NULL);
	gnome_icon_text_item_configure (image->text,
					0, 
					0, 
					priv->image_width, 
					DEFAULT_SMALL_FONT,
					text, 
					priv->is_editable, 
					priv->static_text);
			   
	/* Signals. */
	gtk_signal_connect (GTK_OBJECT (image->image), "event",
			    GTK_SIGNAL_FUNC (image_event),
			    image);
	image->text_event_id = 
		gtk_signal_connect (GTK_OBJECT (image->text), "event",
				    GTK_SIGNAL_FUNC (image_event),
				    image);

/* FIXME
	gtk_signal_connect (GTK_OBJECT (image->comment), "event",
			    GTK_SIGNAL_FUNC (image_event),
			    image);
*/

	gtk_signal_connect (GTK_OBJECT (image->text), "editing_started",
			    GTK_SIGNAL_FUNC (editing_started),
			    image);
	gtk_signal_connect (GTK_OBJECT (image->text), "editing_stopped",
			    GTK_SIGNAL_FUNC (editing_stopped),
			    image);

	gtk_signal_connect (GTK_OBJECT (image->text), "text_changed",
			    GTK_SIGNAL_FUNC (text_changed),
			    image);
	gtk_signal_connect (GTK_OBJECT (image->text), "height_changed",
			    GTK_SIGNAL_FUNC (height_changed),
			    image);
	return image;
}


static int
image_list_append_image (ImageList *gil, 
			 Image *image)
{
	ImageListPrivate *priv;
	int pos;

	priv = gil->priv;

	pos = gil->images++;
	priv->image_list = g_list_append (priv->image_list, image);

	switch (priv->selection_mode) {
	case GTK_SELECTION_BROWSE:
		image_list_select_image (gil, 0);
		break;

	default:
		break;
	}

	if (! priv->frozen) {
		/* FIXME: this should only layout the last line */
		/* gil_layout_all_images (gil); */
		gil_layout_from_line (gil, pos / gil_get_images_per_line (gil));

		gil_scrollbar_adjust (gil);
	} else
		priv->dirty = TRUE;

	return gil->images - 1;
}


static void
image_list_insert_image (ImageList *gil, 
			 gint pos, 
			 Image *image)
{
	ImageListPrivate *priv;

	priv = gil->priv;

	if (pos == gil->images) {
		image_list_append_image (gil, image);
		return;
	}

	priv->image_list = g_list_insert (priv->image_list, image, pos);
	gil->images++;

	switch (priv->selection_mode) {
	case GTK_SELECTION_BROWSE:
		image_list_select_image (gil, 0);
		break;

	default:
		break;
	}

	if (! priv->frozen) {
		/* FIXME: this should only layout the lines from the one
		 * containing the Image to the end.
		 */

		gil_layout_from_line (gil, pos / gil_get_images_per_line (gil));

		/* gil_layout_all_images (gil); */
		gil_scrollbar_adjust (gil);
	} else
		priv->dirty = TRUE;

	sync_selection (gil, pos, SYNC_INSERT);
}

	     
/**
 * gnome_image_list_insert_imlib:
 * @gil:  An image list.
 * @pos:  Position at which the new image should be inserted.
 * @im:   Imlib image with the image image.
 * @text: Text to be used for the image's caption.
 *
 * Inserts an image in the specified image list.  The image is created from the
 * specified Imlib image, and it is inserted at the @pos index.
 */
void
image_list_insert (ImageList *gil, 
		   gint pos, 
		   GdkPixbuf *pixbuf, 
		   const char *text,
		   const char *comment)
{
	Image *image;

	g_return_if_fail (IS_IMAGE_LIST (gil));
	g_return_if_fail (pixbuf != NULL);

	image = image_new_from_pixbuf (gil, pixbuf, text, comment);
	image_list_insert_image (gil, pos, image);
	return;
}


/**
 * gnome_image_list_append_imlib:
 * @gil:  An image list.
 * @im:   Imlib image with the image image.
 * @text: Text to be used for the image's caption.
 *
 * Appends an image to the specified image list.  The image is created from
 * the specified Imlib image.
 */
int
image_list_append (ImageList *gil, 
		   GdkPixbuf *pixbuf, 
		   const char *text,
		   const char *comment)
{
	Image *image;

	g_return_val_if_fail (IS_IMAGE_LIST (gil), -1);
	g_return_val_if_fail (pixbuf != NULL, -1);

	image = image_new_from_pixbuf (gil, pixbuf, text, comment);
	return image_list_append_image (gil, image);
}


static void
image_destroy (Image *image)
{
	if (image->destroy)
		(* image->destroy) (image->data);

	gtk_object_destroy (GTK_OBJECT (image->image));
	gtk_object_destroy (GTK_OBJECT (image->text));
	gtk_object_destroy (GTK_OBJECT (image->comment));
	g_free (image);
}


/**
 * gnome_image_list_remove:
 * @gil: An image list.
 * @pos: Index of the image that should be removed.
 *
 * Removes the image at index position @pos.  If a destroy handler was specified
 * for that image, it will be called with the appropriate data.
 */
void
image_list_remove (ImageList *gil, gint pos)
{
	ImageListPrivate *priv;
	gint was_selected;
	GList *list;
	Image *image;

	g_return_if_fail (IS_IMAGE_LIST (gil));
	g_return_if_fail (pos >= 0 && pos < gil->images);

	priv = gil->priv;

	was_selected = FALSE;

	list = g_list_nth (priv->image_list, pos);
	image = list->data;

	if (image->selected) {
		was_selected = TRUE;

		switch (priv->selection_mode) {
		case GTK_SELECTION_SINGLE:
		case GTK_SELECTION_BROWSE:
		case GTK_SELECTION_MULTIPLE:
		case GTK_SELECTION_EXTENDED:
			image_list_unselect_image (gil, pos);
			break;

		default:
			break;
		}
	}

	priv->image_list = g_list_remove_link (priv->image_list, list);
	g_list_free_1(list);
	gil->images--;

	sync_selection (gil, pos, SYNC_REMOVE);

	if (was_selected) {
		switch (priv->selection_mode) {
		case GTK_SELECTION_BROWSE:
			if (pos == gil->images)
				image_list_select_image (gil, pos - 1);
			else
				image_list_select_image (gil, pos);

			break;

		default:
			break;
		}
	}

	if (gil->images >= priv->last_selected_idx)
		priv->last_selected_idx = -1;

	if (priv->last_selected_image == image)
		priv->last_selected_image = NULL;

	image_destroy (image);

	if (! priv->frozen) {
		/* FIXME: Optimize, only re-layout from pos to end */
		
		gil_layout_from_line (gil, pos / gil_get_images_per_line (gil));

		gil_layout_all_images (gil);
		gil_scrollbar_adjust (gil);
	} else
		priv->dirty = TRUE;
}


/**
 * gnome_image_list_clear:
 * @gil: An image list.
 *
 * Clears the contents for the image list by removing all the images.  If destroy
 * handlers were specified for any of the images, they will be called with the
 * appropriate data.
 */
void
image_list_clear (ImageList *gil)
{
	ImageListPrivate *priv;
	GList *l;

	g_return_if_fail (IS_IMAGE_LIST (gil));

	priv = gil->priv;

	for (l = priv->image_list; l; l = l->next)
		image_destroy (l->data);

	gil_free_line_info (gil);

	g_list_free (gil->selection);
	gil->selection = NULL;
	priv->image_list = NULL;
	gil->images = 0;
	priv->last_selected_idx = -1;
	priv->last_selected_image = NULL;
	priv->has_focus = FALSE;
}


static void
gil_destroy (GtkObject *object)
{
	ImageList *gil;
	ImageListPrivate *priv;

	gil = IMAGE_LIST (object);
	priv = gil->priv;

	g_free (priv->separators);

	priv->frozen = 1;
	priv->dirty  = TRUE;
	image_list_clear (gil);

	if (priv->timer_tag != 0) {
		gtk_timeout_remove (priv->timer_tag);
		priv->timer_tag = 0;
	}

	if (gil->adj)
		gtk_object_unref (GTK_OBJECT (gil->adj));

	if (gil->hadj)
		gtk_object_unref (GTK_OBJECT (gil->hadj));

	if (priv->old_text)
		g_free (priv->old_text);

	g_free (priv);

	if (GTK_OBJECT_CLASS (parent_class)->destroy)
		(*GTK_OBJECT_CLASS (parent_class)->destroy) (object);
}


static void
select_image (ImageList *gil, 
	      gint pos, 
	      GdkEvent *event)
{
	ImageListPrivate *priv;
	gint i;
	GList *list;
	Image *image;

	priv = gil->priv;

	switch (priv->selection_mode) {
	case GTK_SELECTION_SINGLE:
	case GTK_SELECTION_BROWSE:
		i = 0;

		for (list = priv->image_list; list; list = list->next) {
			image = list->data;

			if (i != pos && image->selected)
				emit_select (gil, FALSE, i, event);

			i++;
		}

		emit_select (gil, TRUE, pos, event);
		break;

	case GTK_SELECTION_MULTIPLE:
	case GTK_SELECTION_EXTENDED:
		emit_select (gil, TRUE, pos, event);
		break;

	default:
		break;
	}
}


/**
 * gnome_image_list_select_image:
 * @gil: An image list.
 * @pos: Index of the image to be selected.
 *
 * Selects the image at the index specified by @pos.
 */
void
image_list_select_image (ImageList *gil, 
			 gint pos)
{
	g_return_if_fail (IS_IMAGE_LIST (gil));
	g_return_if_fail (pos >= 0 && pos < gil->images);

	select_image (gil, pos, NULL);
}


static void
unselect_image (ImageList *gil, 
		gint pos, 
		GdkEvent *event)
{
	ImageListPrivate *priv;

	priv = gil->priv;

	switch (priv->selection_mode) {
	case GTK_SELECTION_SINGLE:
	case GTK_SELECTION_BROWSE:
	case GTK_SELECTION_MULTIPLE:
	case GTK_SELECTION_EXTENDED:
		emit_select (gil, FALSE, pos, event);
		break;

	default:
		break;
	}
}


/**
 * gnome_image_list_unselect_image:
 * @gil: An image list.
 * @pos: Index of the image to be unselected.
 *
 * Unselects the image at the index specified by @pos.
 */
void
image_list_unselect_image (ImageList *gil, 
			   gint pos)
{
	g_return_if_fail (IS_IMAGE_LIST (gil));
	g_return_if_fail (pos >= 0 && pos < gil->images);

	unselect_image (gil, pos, NULL);
}


static void
gil_size_request (GtkWidget *widget, 
		  GtkRequisition *requisition)
{
	requisition->width = 1;
	requisition->height = 1;
}


static void
gil_size_allocate (GtkWidget *widget, 
		   GtkAllocation *allocation)
{
	ImageList *gil;
	ImageListPrivate *priv;

	gil = IMAGE_LIST (widget);
	priv = gil->priv;

	if (GTK_WIDGET_CLASS (parent_class)->size_allocate)
		(* GTK_WIDGET_CLASS (parent_class)->size_allocate) (widget, allocation);

	if (priv->frozen)
		return;

	gil_layout_all_images (gil);
	gil_scrollbar_adjust (gil);
}


static void
gil_realize (GtkWidget *widget)
{
	ImageList *gil;
	ImageListPrivate *priv;
	GtkStyle *style;

	gil = IMAGE_LIST (widget);
	priv = gil->priv;

	priv->frozen++;

	if (GTK_WIDGET_CLASS (parent_class)->realize)
		(* GTK_WIDGET_CLASS (parent_class)->realize) (widget);

	priv->frozen--;

	/* Change the style to use the base color as the background */

	style = gtk_style_copy (gtk_widget_get_style (widget));
	style->bg[GTK_STATE_NORMAL] = style->base[GTK_STATE_NORMAL];
	gtk_widget_set_style (widget, style);

	gdk_window_set_background (GTK_LAYOUT (gil)->bin_window,
				   &widget->style->bg[GTK_STATE_NORMAL]);

	if (priv->frozen)
		return;

	if (priv->dirty) {
		gil_layout_all_images (gil);
		gil_scrollbar_adjust (gil);
	}
}


static void
real_select_image (ImageList *gil, 
		   gint num, 
		   GdkEvent *event)
{
	ImageListPrivate *priv;
	Image *image;

	g_return_if_fail (IS_IMAGE_LIST (gil));
	g_return_if_fail (num >= 0 && num < gil->images);

	priv = gil->priv;

	image = g_list_nth (priv->image_list, num)->data;

	if (image->selected)
		return;

	image->selected = TRUE;

	gnome_icon_text_item_select (image->text, TRUE);
	gnome_icon_text_item_select (image->comment, TRUE);
	gnome_canvas_thumb_select (image->image, TRUE);

	gil->selection = g_list_prepend (gil->selection, GINT_TO_POINTER (num));
}


static void
real_unselect_image (ImageList *gil, 
		     gint num, 
		     GdkEvent *event)
{
	ImageListPrivate *priv;
	Image *image;

	g_return_if_fail (IS_IMAGE_LIST (gil));
	g_return_if_fail (num >= 0 && num < gil->images);

	priv = gil->priv;

	image = g_list_nth (priv->image_list, num)->data;

	if (!image->selected)
		return;

	image->selected = FALSE;

	gnome_icon_text_item_select (image->text, FALSE);
	gnome_icon_text_item_select (image->comment, FALSE); 
	gnome_canvas_thumb_select (image->image, FALSE);

	gil->selection = g_list_remove (gil->selection, GINT_TO_POINTER (num));
}


/* Saves the selection of the image list to temporary storage */
static void
store_temp_selection (ImageList *gil)
{
	ImageListPrivate *priv;
	GList *l;
	Image *image;

	priv = gil->priv;

	for (l = priv->image_list; l; l = l->next) {
		image = l->data;

		image->tmp_selected = image->selected;
	}
}


/* Button press handler for the image list */
static gint
gil_button_press (GtkWidget *widget, 
		  GdkEventButton *event)
{
	ImageList *gil;
	ImageListPrivate *priv;
	gint only_one;
	GdkBitmap *stipple;
	gdouble tx, ty;
	gint pos;

	gil = IMAGE_LIST (widget);
	priv = gil->priv;

	gtk_widget_grab_focus (widget);


	/* ---- FIXME ---- */

	/* Invoke the canvas event handler and see if an item picks up the 
	 * event */
/*
		if ((* GTK_WIDGET_CLASS (parent_class)->button_press_event) (widget, event))
			return TRUE;
*/

	pos = image_list_get_image_at (gil, event->x, event->y);
	if (pos != -1) {
		ImageListPrivate *priv = gil->priv;
		Image *image = g_list_nth (priv->image_list, pos)->data;
		GdkEvent synthetic_event;
		gdouble x1, y1, x2, y2;
		gboolean is_text;
		gdouble x, y, dx, dy;
		
		dx = event->x;
		dy = event->y;
		gnome_canvas_window_to_world (GNOME_CANVAS (gil), 
					      dx, dy, 
					      &x, &y);
		gnome_canvas_item_get_bounds (GNOME_CANVAS_ITEM (image->text), 
					      &x1, &y1, &x2, &y2);
		is_text = (x >= x1 && x <= x2 && y >= y1 && y <= y2);

		if (is_text) {
			(* GTK_WIDGET_CLASS (parent_class)->button_press_event) (widget, event);
			return TRUE;
		}

		synthetic_event.type = GDK_BUTTON_PRESS;
		synthetic_event.any = *((GdkEventAny*)(event));
		synthetic_event.button = *event;
		image_event (GNOME_CANVAS_ITEM (image->image),
			     &synthetic_event, 
			     image);
		return TRUE;
	}

	if (!(event->type == GDK_BUTTON_PRESS
	      && event->button == 1
	      && priv->selection_mode != GTK_SELECTION_BROWSE))
		return FALSE;

	only_one = priv->selection_mode == GTK_SELECTION_SINGLE;

	if (only_one || (event->state & (GDK_SHIFT_MASK | GDK_CONTROL_MASK)) == 0)
		image_list_unselect_all (gil, NULL, NULL);

	if (only_one)
		return TRUE;

	if (priv->selecting)
		return FALSE;

	gnome_canvas_window_to_world (GNOME_CANVAS (gil), event->x, event->y, &tx, &ty);
	priv->sel_start_x = tx;
	priv->sel_start_y = ty;
	priv->sel_state = event->state;
	priv->selecting = TRUE;

	store_temp_selection (gil);

	stipple = gdk_bitmap_create_from_data (NULL, gray50_bits, gray50_width, gray50_height);
	priv->sel_rect = gnome_canvas_item_new (gnome_canvas_root (GNOME_CANVAS (gil)),
						gnome_canvas_rect_get_type (),
						"x1", tx,
						"y1", ty,
						"x2", tx,
						"y2", ty,
						"outline_color", "black",
						"width_pixels", 1,
						"outline_stipple", stipple,
						NULL);
	gdk_bitmap_unref (stipple);

	gnome_canvas_item_grab (priv->sel_rect, 
				GDK_POINTER_MOTION_MASK | GDK_BUTTON_RELEASE_MASK,
				NULL, event->time);

	return TRUE;
}


/* Returns whether the specified image is at least partially inside the 
 * specified rectangle.
 */
static gboolean
image_is_in_area (ImageList *gil,
		  Image *image, 
		  gint x1, 
		  gint y1, 
		  gint x2, 
		  gint y2)
{
	gdouble ix1, iy1, ix2, iy2;   /* image bounds. */
	gdouble tx1, ty1, tx2, ty2;   /* text bounds. */
	gdouble cx1, cy1, cx2, cy2;   /* comment bounds. */
	gdouble sx1, sy1, sx2, sy2;   /* sensible rectangle. */
	gdouble x_ofs, y_ofs;
	gboolean view_text, view_comment;

	if (x1 == x2 && y1 == y2)
		return FALSE;

	image_get_view_mode (gil, image, &view_text, &view_comment);

	gnome_canvas_item_get_bounds (GNOME_CANVAS_ITEM (image->image), 
				      &ix1, &iy1, &ix2, &iy2);
	gnome_canvas_item_get_bounds (GNOME_CANVAS_ITEM (image->text), 
				      &tx1, &ty1, &tx2, &ty2);
	gnome_canvas_item_get_bounds (GNOME_CANVAS_ITEM (image->comment), 
				      &cx1, &cy1, &cx2, &cy2);

	sx1 = ix1;
	sy1 = iy1;
	sx2 = ix2;
	sy2 = iy2;

	if (view_text) {
		sx1 = MIN (sx1, tx1);
		sy1 = MIN (sy1, ty1);
		sx2 = MAX (sx2, tx2);
		sy2 = MAX (sy2, ty2);
	}

	if (view_comment) {
		sx1 = MIN (sx1, cx1);
		sy1 = MIN (sy1, cy1);
		sx2 = MAX (sx2, cx2);
		sy2 = MAX (sy2, cy2);
	}

	x_ofs = (sx2 - sx1) / 6;
	y_ofs = (sy2 - sy1) / 6;

	sx1 += x_ofs;
	sx2 -= x_ofs;
	sy1 += y_ofs;
	sy2 -= y_ofs;
	
	if (sx1 <= x2 && sy1 <= y2 && sx2 >= x1 && sy2 >= y1)
		return TRUE;

	return FALSE;
}


/* Updates the rubberband selection to the specified point */
static void
update_drag_selection (ImageList *gil, 
		       gint x, 
		       gint y)
{
	ImageListPrivate *priv;
	gint x1, x2, y1, y2;
	GList *l, *begin, *end;
	gint i, begin_idx;
	Image *image;
	gint additive, invert;

	priv = gil->priv;

	/* Update rubberband */

	if (priv->sel_start_x < x) {
		x1 = priv->sel_start_x;
		x2 = x;
	} else {
		x1 = x;
		x2 = priv->sel_start_x;
	}

	if (priv->sel_start_y < y) {
		y1 = priv->sel_start_y;
		y2 = y;
	} else {
		y1 = y;
		y2 = priv->sel_start_y;
	}

	if (x1 < 0)
		x1 = 0;

	if (y1 < 0)
		y1 = 0;

	if (x2 >= GTK_WIDGET (gil)->allocation.width)
		x2 = GTK_WIDGET (gil)->allocation.width - 1;

	if (y2 >= priv->total_height)
		y2 = priv->total_height - 1;

	gnome_canvas_item_set (priv->sel_rect,
			       "x1", (double) x1,
			       "y1", (double) y1,
			       "x2", (double) x2,
			       "y2", (double) y2,
			       NULL);

	/* Select or unselect images as appropriate */

	additive = priv->sel_state & GDK_SHIFT_MASK;
	invert = priv->sel_state & GDK_CONTROL_MASK;

	/* Consider only visible images. */

	begin_idx = image_list_get_first_visible (gil);
	begin = g_list_nth (priv->image_list, begin_idx);
	end = g_list_nth (priv->image_list, image_list_get_last_visible (gil));
	if (end != NULL)
		end = end->next;

	for (l = begin, i = begin_idx; l != end; l = l->next, i++) {
		image = l->data;

		if (image_is_in_area (gil, image, x1, y1, x2, y2)) {
			if (invert) {
				if (image->selected == image->tmp_selected)
					emit_select (gil, !image->selected, i, NULL);
			} else if (additive) {
				if (!image->selected)
					emit_select (gil, TRUE, i, NULL);
			} else {
				if (!image->selected)
					emit_select (gil, TRUE, i, NULL);
			}
		} else if (image->selected != image->tmp_selected)
			emit_select (gil, image->tmp_selected, i, NULL);
	}
}


/* Button release handler for the image list */
static gint
gil_button_release (GtkWidget *widget, GdkEventButton *event)
{
	ImageList *gil;
	ImageListPrivate *priv;
	gdouble x, y;

	gil = IMAGE_LIST (widget);
	priv = gil->priv;

	if (!priv->selecting)
		return (* GTK_WIDGET_CLASS (parent_class)->button_release_event) (widget, event);

	if (event->button != 1)
		return FALSE;

	gnome_canvas_window_to_world (GNOME_CANVAS (gil), event->x, event->y, &x, &y);
	update_drag_selection (gil, x, y);
	gnome_canvas_item_ungrab (priv->sel_rect, event->time);

	gtk_object_destroy (GTK_OBJECT (priv->sel_rect));
	priv->sel_rect = NULL;
	priv->selecting = FALSE;

	if (priv->timer_tag != 0) {
		gtk_timeout_remove (priv->timer_tag);
		priv->timer_tag = 0;
	}

	return TRUE;
}


/* Autoscroll timeout handler for the image list */
static gint
scroll_timeout (gpointer data)
{
	ImageList *gil;
	ImageListPrivate *priv;
	gdouble x, y;
	gint value;

	gil = data;
	priv = gil->priv;

	GDK_THREADS_ENTER ();

	value = gil->adj->value + priv->value_diff;
	if (value > gil->adj->upper - gil->adj->page_size)
		value = gil->adj->upper - gil->adj->page_size;

	gtk_adjustment_set_value (gil->adj, value);

	gnome_canvas_window_to_world (GNOME_CANVAS (gil),
				      priv->event_last_x, priv->event_last_y,
				      &x, &y);
	update_drag_selection (gil, x, y);

	GDK_THREADS_LEAVE();

	return TRUE;
}


/* Motion event handler for the image list */
static gint
gil_motion_notify (GtkWidget *widget, GdkEventMotion *event)
{
	ImageList *gil;
	ImageListPrivate *priv;
	gdouble x, y;

	gil = IMAGE_LIST (widget);
	priv = gil->priv;

	if (!priv->selecting)
		return (* GTK_WIDGET_CLASS (parent_class)->motion_notify_event) (widget, event);

	gnome_canvas_window_to_world (GNOME_CANVAS (gil), event->x, event->y, &x, &y);
	update_drag_selection (gil, x, y);

	/* If we are out of bounds, schedule a timeout that will do the 
	 * scrolling */

	if (event->y < 0 || event->y > widget->allocation.height) {
		if (priv->timer_tag == 0)
			priv->timer_tag = gtk_timeout_add (SCROLL_TIMEOUT, scroll_timeout, gil);

		if (event->y < 0)
			priv->value_diff = event->y;
		else
			priv->value_diff = event->y - widget->allocation.height;

		priv->event_last_x = event->x;
		priv->event_last_y = event->y;

		/* Make the steppings be relative to the mouse distance from 
		 * the canvas.  
		 * Also notice the timeout above is small to give a
		 * more smooth movement.
		 */
		priv->value_diff /= 3;
	} else if (priv->timer_tag != 0) {
		gtk_timeout_remove (priv->timer_tag);
		priv->timer_tag = 0;
	}

	return TRUE;
}


static void
gil_set_scroll_adjustments (GtkLayout *layout,
			    GtkAdjustment *hadjustment,
			    GtkAdjustment *vadjustment)
{
	ImageList *gil = IMAGE_LIST (layout);

	image_list_set_hadjustment (gil, hadjustment);
	image_list_set_vadjustment (gil, vadjustment);
}


typedef gboolean (*xGtkSignal_BOOL__INT_POINTER) (GtkObject * object,
						  gint     arg1,
						  gpointer arg2,
						  gpointer user_data);
static void
xgtk_marshal_BOOL__INT_POINTER (GtkObject *object, 
				GtkSignalFunc func, 
				gpointer func_data,
				GtkArg *args)
{
	xGtkSignal_BOOL__INT_POINTER rfunc;
	gboolean *return_val;
	
	return_val = GTK_RETLOC_BOOL (args[2]);
	rfunc = (xGtkSignal_BOOL__INT_POINTER) func;
	*return_val = (*rfunc) (object,
				GTK_VALUE_INT (args[0]),
				GTK_VALUE_POINTER (args[1]),
				func_data);
}


static void
gil_set_arg (GtkObject *object, 
	     GtkArg *arg, 
	     guint arg_id)
{
	ImageList *gil;
	GtkAdjustment *adjustment;

	gil = IMAGE_LIST (object);

	switch (arg_id) {
	case ARG_HADJUSTMENT:
		adjustment = GTK_VALUE_POINTER (*arg);
		image_list_set_hadjustment (gil, adjustment);
		break;

	case ARG_VADJUSTMENT:
		adjustment = GTK_VALUE_POINTER (*arg);
		image_list_set_vadjustment (gil, adjustment);
		break;

	default:
		break;
	}
}


static void
gil_get_arg (GtkObject *object, 
	     GtkArg *arg, 
	     guint arg_id)
{
	ImageList *gil;
	ImageListPrivate *priv;

	gil = IMAGE_LIST (object);
	priv = gil->priv;

	switch (arg_id) {
	case ARG_HADJUSTMENT:
		GTK_VALUE_POINTER (*arg) = gil->hadj;
		break;

	case ARG_VADJUSTMENT:
		GTK_VALUE_POINTER (*arg) = gil->adj;
		break;

	default:
		arg->type = GTK_TYPE_INVALID;
		break;
	}
}


static void
gil_class_init (ImageListClass *gil_class)
{
	GtkObjectClass *object_class;
	GtkWidgetClass *widget_class;
	GtkLayoutClass *layout_class;
	GnomeCanvasClass *canvas_class;

	object_class = (GtkObjectClass *)   gil_class;
	widget_class = (GtkWidgetClass *)   gil_class;
	layout_class = (GtkLayoutClass *)   gil_class;
	canvas_class = (GnomeCanvasClass *) gil_class;

	parent_class = gtk_type_class (gnome_canvas_get_type ());

	gtk_object_add_arg_type ("ImageList::hadjustment",
				 GTK_TYPE_ADJUSTMENT,
				 GTK_ARG_READWRITE,
				 ARG_HADJUSTMENT);
	gtk_object_add_arg_type ("ImageList::vadjustment",
				 GTK_TYPE_ADJUSTMENT,
				 GTK_ARG_READWRITE,
				 ARG_VADJUSTMENT);

	gil_signals[SELECT_IMAGE] =
		gtk_signal_new (
			"select_image",
			GTK_RUN_FIRST,
			object_class->type,
			GTK_SIGNAL_OFFSET (ImageListClass, select_image),
			gtk_marshal_NONE__INT_POINTER,
			GTK_TYPE_NONE, 2,
			GTK_TYPE_INT,
			GTK_TYPE_GDK_EVENT);

	gil_signals[UNSELECT_IMAGE] =
		gtk_signal_new (
			"unselect_image",
			GTK_RUN_FIRST,
			object_class->type,
			GTK_SIGNAL_OFFSET (ImageListClass, unselect_image),
			gtk_marshal_NONE__INT_POINTER,
			GTK_TYPE_NONE, 2,
			GTK_TYPE_INT,
			GTK_TYPE_GDK_EVENT);

	gil_signals[TEXT_CHANGED] =
		gtk_signal_new (
			"text_changed",
			GTK_RUN_LAST,
			object_class->type,
			GTK_SIGNAL_OFFSET (ImageListClass, text_changed),
			xgtk_marshal_BOOL__INT_POINTER,
			GTK_TYPE_BOOL, 2,
			GTK_TYPE_INT,
			GTK_TYPE_POINTER);

	gtk_object_class_add_signals (object_class, gil_signals, LAST_SIGNAL);

	object_class->destroy = gil_destroy;
	object_class->set_arg = gil_set_arg;
	object_class->get_arg = gil_get_arg;

	widget_class->size_request = gil_size_request;
	widget_class->size_allocate = gil_size_allocate;
	widget_class->realize = gil_realize;
	widget_class->button_press_event = gil_button_press;
	widget_class->button_release_event = gil_button_release;
	widget_class->motion_notify_event = gil_motion_notify;

	/* we override GtkLayout's set_scroll_adjustments signal instead
	 * of creating a new signal so as to keep binary compatibility.
	 * Anyway, a widget class only needs one of these signals, and
	 * this gives the correct implementation for ImageList */
	layout_class->set_scroll_adjustments = gil_set_scroll_adjustments;

	gil_class->select_image = real_select_image;
	gil_class->unselect_image = real_unselect_image;
}


static gint
default_compare (gconstpointer  ptr1,
                 gconstpointer  ptr2)
{
	return 0;
}


static void
gil_init (ImageList *gil)
{
	ImageListPrivate *priv;

	GTK_WIDGET_SET_FLAGS (gil, GTK_CAN_FOCUS);

	priv = g_new0 (ImageListPrivate, 1);
	gil->priv = priv;

	priv->row_spacing = DEFAULT_ROW_SPACING;
	priv->col_spacing = DEFAULT_COL_SPACING;
	priv->text_spacing = DEFAULT_TEXT_SPACING;
	priv->image_border = DEFAULT_IMAGE_BORDER;
	priv->separators = g_strdup (" ");

	priv->selection_mode = GTK_SELECTION_SINGLE;
	priv->view_mode = IMAGE_LIST_VIEW_ALL /*COMMENTS_OR_TEXT*/ /*ALL*/;
	priv->dirty = TRUE;

	priv->compare = default_compare;
	priv->sort_type = GTK_SORT_ASCENDING;

	priv->has_focus = FALSE;
	priv->old_text = NULL;

	/* FIXME:  Make the image list use the normal canvas/layout 
	 * adjustments */
	gnome_canvas_set_scroll_region (GNOME_CANVAS (gil), 0.0, 0.0, 1000000.0, 1000000.0);
	gnome_canvas_scroll_to (GNOME_CANVAS (gil), 0, 0);
}


/**
 * gnome_image_list_get_type:
 *
 * Registers the &ImageList class if necessary, and returns the type ID
 * associated to it.
 *
 * Returns: The type ID of the &ImageList class.
 */
guint
image_list_get_type (void)
{
	static guint gil_type = 0;

	if (!gil_type) {
		GtkTypeInfo gil_info = {
			"ImageList",
			sizeof (ImageList),
			sizeof (ImageListClass),
			(GtkClassInitFunc) gil_class_init,
			(GtkObjectInitFunc) gil_init,
			(GtkArgSetFunc) NULL,
			(GtkArgGetFunc) NULL
		};

		gil_type = gtk_type_unique (gnome_canvas_get_type (),
					    &gil_info);
	}

	return gil_type;
}


/**
 * gnome_image_list_set_image_width:
 * @gil: An image list.
 * @w:   New width for the image columns.
 *
 * Sets the amount of horizontal space allocated to the images, i.e. the column
 * width of the image list.
 */
void
image_list_set_image_width (ImageList *gil, int w)
{
	ImageListPrivate *priv;

	g_return_if_fail (IS_IMAGE_LIST (gil));

	priv = gil->priv;

	priv->image_width = w;

	if (priv->frozen) {
		priv->dirty = TRUE;
		return;
	}

	gil_layout_all_images (gil);
	gil_scrollbar_adjust (gil);
}


static void
gil_adj_value_changed (GtkAdjustment *adj, 
		       ImageList *gil)
{
	gnome_canvas_scroll_to (GNOME_CANVAS (gil), 0, adj->value);
}


/**
 * gnome_image_list_set_hadjustment:
 * @gil: An image list.
 * @hadj: Adjustment to be used for horizontal scrolling.
 *
 * Sets the adjustment to be used for horizontal scrolling.  This is normally
 * not required, as the image list can be simply inserted in a &GtkScrolledWindow
 * and scrolling will be handled automatically.
 **/
void
image_list_set_hadjustment (ImageList *gil, 
			    GtkAdjustment *hadj)
{
	ImageListPrivate *priv;
	GtkAdjustment *old_adjustment;

	/* hadj isn't used but is here for compatibility with 
	 * GtkScrolledWindow */

	g_return_if_fail (IS_IMAGE_LIST (gil));

	if (hadj)
		g_return_if_fail (GTK_IS_ADJUSTMENT (hadj));

	priv = gil->priv;

	if (gil->hadj == hadj)
		return;

	old_adjustment = gil->hadj;

	if (gil->hadj)
		gtk_object_unref (GTK_OBJECT (gil->hadj));

	gil->hadj = hadj;

	if (gil->hadj) {
		gtk_object_ref (GTK_OBJECT (gil->hadj));
		/* The horizontal adjustment is not used, so set some default
		 * values to indicate that everything is visible horizontally.
		 */
		gil->hadj->lower = 0.0;
		gil->hadj->upper = 1.0;
		gil->hadj->value = 0.0;
		gil->hadj->step_increment = 1.0;
		gil->hadj->page_increment = 1.0;
		gil->hadj->page_size = 1.0;
		gtk_adjustment_changed (gil->hadj);
	}

	if (!gil->hadj || !old_adjustment)
		gtk_widget_queue_resize (GTK_WIDGET (gil));
}


/**
 * gnome_image_list_set_vadjustment:
 * @gil: An image list.
 * @hadj: Adjustment to be used for horizontal scrolling.
 *
 * Sets the adjustment to be used for vertical scrolling.  This is normally not
 * required, as the image list can be simply inserted in a &GtkScrolledWindow and
 * scrolling will be handled automatically.
 **/
void
image_list_set_vadjustment (ImageList *gil, 
			    GtkAdjustment *vadj)
{
	ImageListPrivate *priv;
	GtkAdjustment *old_adjustment;

	g_return_if_fail (IS_IMAGE_LIST (gil));

	if (vadj)
		g_return_if_fail (GTK_IS_ADJUSTMENT (vadj));

	priv = gil->priv;

	if (gil->adj == vadj)
		return;

	old_adjustment = gil->adj;

	if (gil->adj) {
		gtk_signal_disconnect_by_data (GTK_OBJECT (gil->adj), gil);
		gtk_object_unref (GTK_OBJECT (gil->adj));
	}

	gil->adj = vadj;

	if (gil->adj) {
		gtk_object_ref (GTK_OBJECT (gil->adj));
		gtk_object_sink (GTK_OBJECT (gil->adj));
		gtk_signal_connect (GTK_OBJECT (gil->adj), "value_changed",
				    GTK_SIGNAL_FUNC (gil_adj_value_changed),
				    gil);
		gtk_signal_connect (GTK_OBJECT (gil->adj), "changed",
				    GTK_SIGNAL_FUNC (gil_adj_value_changed), 
				    gil);
	}

	if (!gil->adj || !old_adjustment)
		gtk_widget_queue_resize (GTK_WIDGET (gil));
}


/**
 * gnome_image_list_construct:
 * @gil: An image list.
 * @image_width: Width for the image columns.
 * @adj: Adjustment to be used for vertical scrolling.
 * @flags: A combination of %GNOME_IMAGE_LIST_IS_EDITABLE and %GNOME_IMAGE_LIST_STATIC_TEXT.
 *
 * Constructor for the image list, to be used by derived classes.
 **/
void
image_list_construct (ImageList *gil, 
		      guint image_width, 
		      GtkAdjustment *adj, 
		      gint flags)
{
	ImageListPrivate *priv;

	g_return_if_fail (IS_IMAGE_LIST (gil));

	priv = gil->priv;

	image_list_set_image_width (gil, image_width);
	priv->is_editable = (flags & IMAGE_LIST_IS_EDITABLE) != 0;
	priv->static_text = (flags & IMAGE_LIST_STATIC_TEXT) != 0;

	if (!adj)
		adj = GTK_ADJUSTMENT (gtk_adjustment_new (0, 0, 1, 0.1, 0.1, 0.1));

	image_list_set_vadjustment (gil, adj);
}


/**
 * gnome_image_list_new_flags: [constructor]
 * @image_width: Width for the image columns.
 * @adj:        Adjustment to be used for vertical scrolling.
 * @flags:      A combination of %GNOME_IMAGE_LIST_IS_EDITABLE and 
 *              %GNOME_IMAGE_LIST_STATIC_TEXT.
 *
 * Creates a new image list widget.  The image columns are allocated a width of
 * @image_width pixels.  Image captions will be word-wrapped to this width as
 * well.
 *
 * The adjustment is used to pass an existing adjustment to be used to control
 * the image list's vertical scrolling.  Normally NULL can be passed here; if
 *  the
 * image list is inserted into a &GtkScrolledWindow, it will handle scrolling
 * automatically.
 *
 * If @flags has the %GNOME_IMAGE_LIST_IS_EDITABLE flag set, then the user 
 * will be
 * able to edit the text in the image captions, and the "text_changed" signal
 * will be emitted when an image's text is changed.
 *
 * If @flags has the %GNOME_IMAGE_LIST_STATIC_TEXT flags set, then the text
 * for the image captions will not be copied inside the image list; it will 
 * only
 * store the pointers to the original text strings specified by the 
 * application.
 * This is intended to save memory.  If this flag is not set, then the text
 * strings will be copied and managed internally.
 *
 * Returns: a newly-created image list widget
 */
GtkWidget *
image_list_new_flags (guint image_width, 
		      GtkAdjustment *adj, 
		      gint flags)
{
	ImageList *gil;

	gtk_widget_push_visual (gdk_rgb_get_visual ());
	gtk_widget_push_colormap (gdk_rgb_get_cmap ());
	gil = IMAGE_LIST (gtk_type_new (image_list_get_type ()));
	gtk_widget_pop_visual ();
	gtk_widget_pop_colormap ();

	image_list_construct (gil, image_width, adj, flags);

	return GTK_WIDGET (gil);
}


/**
 * image_list_new:
 * @image_width: Width for the image columns.
 * @adj: Adjustment to be used for vertical scrolling.
 * @flags: A combination of %GNOME_IMAGE_LIST_IS_EDITABLE and 
 *         %GNOME_IMAGE_LIST_STATIC_TEXT.
 *
 * This function is kept for binary compatibility with old applications.  It is
 * similar in purpose to gnome_image_list_new_flags(), but it will always turn
 * on the %GNOME_IMAGE_LIST_IS_EDITABLE flag.
 *
 * Return value: a newly-created image list widget
 **/
GtkWidget *
image_list_new (guint image_width, 
		GtkAdjustment *adj, 
		gint flags)
{
	return image_list_new_flags (image_width, adj, flags & IMAGE_LIST_IS_EDITABLE);
}


/**
 * gnome_image_list_freeze:
 * @gil:  An image list.
 *
 * Freezes an image list so that any changes made to it will not be
 * reflected on the screen until it is thawed with gnome_image_list_thaw().
 * It is recommended to freeze the image list before inserting or deleting
 * many images, for example, so that the layout process will only be executed
 * once, when the image list is finally thawed.
 *
 * You can call this function multiple times, but it must be balanced with the
 * same number of calls to gnome_image_list_thaw() before the changes will take
 * effect.
 */
void
image_list_freeze (ImageList *gil)
{
	ImageListPrivate *priv;

	g_return_if_fail (IS_IMAGE_LIST (gil));

	priv = gil->priv;

	priv->frozen++;

	/* We hide the root so that the user will not see any changes while the
	 * image list is doing stuff.
	 */

	if (priv->frozen == 1)
		gnome_canvas_item_hide (GNOME_CANVAS (gil)->root);
}


/**
 * gnome_image_list_thaw:
 * @gil:  An image list.
 *
 * Thaws the image list and performs any pending layout operations.  This
 * is to be used in conjunction with gnome_image_list_freeze().
 */
void
image_list_thaw (ImageList *gil)
{
        ImageListPrivate *priv;

	g_return_if_fail (IS_IMAGE_LIST (gil));

	priv = gil->priv;

	g_return_if_fail (priv->frozen > 0);

	priv->frozen--;

	if (priv->dirty) {
                gil_layout_all_images (gil);
                gil_scrollbar_adjust (gil);
        }

	if (priv->frozen == 0)
		gnome_canvas_item_show (GNOME_CANVAS (gil)->root);
}


/**
 * gnome_image_list_set_selection_mode
 * @gil:  An image list.
 * @mode: New selection mode.
 *
 * Sets the selection mode for an image list.  The %GTK_SELECTION_MULTIPLE and
 * %GTK_SELECTION_EXTENDED modes are considered equivalent.
 */
void
image_list_set_selection_mode (ImageList *gil, GtkSelectionMode mode)
{
	ImageListPrivate *priv;

	g_return_if_fail (IS_IMAGE_LIST (gil));

	priv = gil->priv;

	priv->selection_mode = mode;
	priv->last_selected_idx = -1;
	priv->last_selected_image = NULL;
}


/**
 * gnome_image_list_set_image_data_full:
 * @gil:     An image list.
 * @pos:     Index of an image.
 * @data:    User data to set on the image.
 * @destroy: Destroy notification handler for the image.
 *
 * Associates the @data pointer to the image at the index specified by @pos.
 * The
 * @destroy argument points to a function that will be called when the image is
 * destroyed, or NULL if no function is to be called when this happens.
 */
void
image_list_set_image_data_full (ImageList *gil,
				gint pos, 
				gpointer data,
				GtkDestroyNotify destroy)
{
	ImageListPrivate *priv;
	Image *image;

	g_return_if_fail (IS_IMAGE_LIST (gil));
	g_return_if_fail (pos >= 0 && pos < gil->images);

	priv = gil->priv;

	image = g_list_nth (priv->image_list, pos)->data;
	image->data = data;
	image->destroy = destroy;
}


/**
 * gnome_image_list_get_image_data:
 * @gil:  An image list.
 * @pos:  Index of an image.
 * @data: User data to set on the image.
 *
 * Associates the @data pointer to the image at the index specified by @pos.
 */
void
image_list_set_image_data (ImageList *gil, 
			   gint pos, 
			   gpointer data)
{
	image_list_set_image_data_full (gil, pos, data, NULL);
}


/**
 * gnome_image_list_get_image_data:
 * @gil: An image list.
 * @pos: Index of an image.
 *
 * Returns the user data pointer associated to the image at the index specified
 * by @pos.
 */
gpointer
image_list_get_image_data (ImageList *gil, 
			   gint pos)
{
	ImageListPrivate *priv;
	Image *image;

	g_return_val_if_fail (IS_IMAGE_LIST (gil), NULL);
	g_return_val_if_fail (pos >= 0 && pos < gil->images, NULL);

	priv = gil->priv;

	image = g_list_nth (priv->image_list, pos)->data;
	return image->data;
}


/**
 * gnome_image_list_find_image_from_data:
 * @gil:    An image list.
 * @data:   Data pointer associated to an image.
 *
 * Returns the index of the image whose user data has been set to @data,
 * or -1 if no image has this data associated to it.
 */
gint
image_list_find_image_from_data (ImageList *gil, 
				 gpointer data)
{
	ImageListPrivate *priv;
	GList *list;
	gint n;
	Image *image;

	g_return_val_if_fail (IS_IMAGE_LIST (gil), -1);

	priv = gil->priv;

	for (n = 0, list = priv->image_list; list; n++, list = list->next) {
		image = list->data;
		if (image->data == data)
			return n;
	}

	return -1;
}


/* Sets an integer value in the private structure of the image list, and updates it */
static void
set_value (ImageList *gil, 
	   ImageListPrivate *priv, 
	   gint *dest, 
	   gint val)
{
	if (val == *dest)
		return;

	*dest = val;

	if (!priv->frozen) {
		gil_layout_all_images (gil);
		gil_scrollbar_adjust (gil);
	} else
		priv->dirty = TRUE;
}


/**
 * gnome_image_list_set_row_spacing:
 * @gil:    An image list.
 * @pixels: Number of pixels for inter-row spacing.
 *
 * Sets the spacing to be used between rows of images.
 */
void
image_list_set_row_spacing (ImageList *gil, 
			    gint pixels)
{
	ImageListPrivate *priv;

	g_return_if_fail (IS_IMAGE_LIST (gil));

	priv = gil->priv;
	set_value (gil, priv, &priv->row_spacing, pixels);
}


/**
 * gnome_image_list_set_col_spacing:
 * @gil:    An image list.
 * @pixels: Number of pixels for inter-column spacing.
 *
 * Sets the spacing to be used between columns of images.
 */
void
image_list_set_col_spacing (ImageList *gil, 
			    gint pixels)
{
	ImageListPrivate *priv;

	g_return_if_fail (IS_IMAGE_LIST (gil));

	priv = gil->priv;
	set_value (gil, priv, &priv->col_spacing, pixels);
}


/**
 * gnome_image_list_set_text_spacing:
 * @gil:    An image list.
 * @pixels: Number of pixels between an image's image and its caption.
 *
 * Sets the spacing to be used between an image's image and its text caption.
 */
void
image_list_set_text_spacing (ImageList *gil, 
			     gint pixels)
{
	ImageListPrivate *priv;

	g_return_if_fail (IS_IMAGE_LIST (gil));

	priv = gil->priv;
	set_value (gil, priv, &priv->text_spacing, pixels);
}


/**
 * gnome_image_list_set_image_border:
 * @gil:    An image list.
 * @pixels: Number of border pixels to be used around an image's image.
 *
 * Sets the width of the border to be displayed around an image's image.  This is
 * currently not implemented.
 */
void
image_list_set_image_border (ImageList *gil, 
			     gint pixels)
{
	ImageListPrivate *priv;

	g_return_if_fail (IS_IMAGE_LIST (gil));

	priv = gil->priv;
	set_value (gil, priv, &priv->image_border, pixels);
}


/**
 * gnome_image_list_set_separators:
 * @gil: An image list.
 * @sep: String with characters to be used as word separators.
 *
 * Sets the characters that can be used as word separators when doing
 * word-wrapping in the image text captions.
 */
void
image_list_set_separators (ImageList *gil, 
			   const gchar *sep)
{
	ImageListPrivate *priv;

	g_return_if_fail (IS_IMAGE_LIST (gil));
	g_return_if_fail (sep != NULL);

	priv = gil->priv;

	if (priv->separators)
		g_free (priv->separators);

	priv->separators = g_strdup (sep);

	if (priv->frozen) {
		priv->dirty = TRUE;
		return;
	}

	gil_layout_all_images (gil);
	gil_scrollbar_adjust (gil);
}


/**
 * gnome_image_list_moveto:
 * @gil:    An image list.
 * @pos:    Index of an image.
 * @yalign: Vertical alignment of the image.
 *
 * Makes the image whose index is @pos be visible on the screen.  The image list
 * gets scrolled so that the image is visible.  An alignment of 0.0 represents
 * the top of the visible part of the image list, and 1.0 represents the bottom.
 * An image can be centered on the image list.
 */
void
image_list_moveto (ImageList *gil, 
		   gint pos, 
		   gdouble yalign)
{
	ImageListPrivate *priv;
	ImageLine *il;
	GList *l;
	gint i, y, uh, line;

	g_return_if_fail (IS_IMAGE_LIST (gil));
	g_return_if_fail (pos >= 0 && pos < gil->images);
	g_return_if_fail (yalign >= 0.0 && yalign <= 1.0);

	priv = gil->priv;

	if (priv->lines == NULL)
		return;

	line = pos / gil_get_images_per_line (gil);

	y = priv->row_spacing;
	for (i = 0, l = priv->lines; l && i < line; l = l->next, i++) {
		il = l->data;
		y += image_line_height (gil, il);
	}

	il = l->data;

	uh = GTK_WIDGET (gil)->allocation.height - image_line_height (gil,il);
	gtk_adjustment_set_value (gil->adj, y - uh * yalign);
}


/**
 * gnome_image_list_is_visible:
 * @gil: An image list.
 * @pos: Index of an image.
 *
 * Returns whether the image at the index specified by @pos is visible.  This
 * will be %GTK_VISIBILITY_NONE if the image is not visible at all,
 * %GTK_VISIBILITY_PARTIAL if the image is at least partially shown, and
 * %GTK_VISIBILITY_FULL if the image is fully visible.
 */
GtkVisibility
image_list_image_is_visible (ImageList *gil, 
			     gint pos)
{
	ImageListPrivate *priv;
	ImageLine *il;
	GList *l;
	gint line, y1, y2, i;

	g_return_val_if_fail (IS_IMAGE_LIST (gil), GTK_VISIBILITY_NONE);
	g_return_val_if_fail (pos >= 0 && pos < gil->images, GTK_VISIBILITY_NONE);

	priv = gil->priv;

	if (priv->lines == NULL)
		return GTK_VISIBILITY_NONE;

	line = pos / gil_get_images_per_line (gil);
	y1 = priv->row_spacing /*0*/;
	for (i = 0, l = priv->lines; l && i < line; l = l->next, i++) {
		il = l->data;
		y1 += image_line_height (gil, il);
	}

	y2 = y1 + image_line_height (gil, (ImageLine *) l->data);

	if (y2 < gil->adj->value)
		return GTK_VISIBILITY_NONE;

	if (y1 > gil->adj->value + GTK_WIDGET (gil)->allocation.height)
		return GTK_VISIBILITY_NONE;

	if (y2 <= gil->adj->value + GTK_WIDGET (gil)->allocation.height &&
	    y1 >= gil->adj->value)
		return GTK_VISIBILITY_FULL;

	return GTK_VISIBILITY_PARTIAL;
}


/**
 * gnome_image_list_get_image_at:
 * @gil: An image list.
 * @x:   X position in the image list window.
 * @y:   Y position in the image list window.
 *
 * Returns the index of the image that is under the specified coordinates, 
 * which
 * are relative to the image list's window.  If there is no image in that
 * position, -1 is returned.
 */
gint
image_list_get_image_at (ImageList *gil, 
			 gint x, 
			 gint y)
{
	ImageListPrivate *priv;
	GList *l;
	gdouble wx, wy;
	gdouble dx, dy;
	gint cx, cy;
	gint n;
	GnomeCanvasItem *item;
	gdouble dist;

	g_return_val_if_fail (IS_IMAGE_LIST (gil), -1);

	priv = gil->priv;

	dx = x;
	dy = y;

	gnome_canvas_window_to_world (GNOME_CANVAS (gil), dx, dy, &wx, &wy);
	gnome_canvas_w2c (GNOME_CANVAS (gil), wx, wy, &cx, &cy);

	for (n = 0, l = priv->image_list; l; l = l->next, n++) {
		Image *image = l->data;
		GnomeCanvasItem *cimage = GNOME_CANVAS_ITEM (image->image);
		GnomeCanvasItem *text = GNOME_CANVAS_ITEM (image->text);
		GnomeCanvasItem *comment = GNOME_CANVAS_ITEM (image->comment);
		gboolean view_text, view_comment;

		image_get_view_mode (gil, image, &view_text, &view_comment);

		/* on image. */
		if ((wx >= cimage->x1) && (wx <= cimage->x2)
		    && (wy >= cimage->y1) && (wy <= cimage->y2)) {
			dist = (* GNOME_CANVAS_ITEM_CLASS (cimage->object.klass)->point) (cimage, wx, wy, cx, cy, &item);

			if ((gint) (dist * GNOME_CANVAS (gil)->pixels_per_unit + 0.5) <= GNOME_CANVAS (gil)->close_enough)
				return n;
		}

		/* on text. */
		if (view_text &&
		    ((wx >= text->x1) && (wx <= text->x2) 
		     && (wy >= text->y1) && (wy <= text->y2))) {
			dist = (* GNOME_CANVAS_ITEM_CLASS (text->object.klass)->point) (text, wx, wy, cx, cy, &item);

			if ((gint) (dist * GNOME_CANVAS (gil)->pixels_per_unit + 0.5) <= GNOME_CANVAS (gil)->close_enough)
				return n;
		}

		/* on comment. */
		if (view_comment &&
		    ((wx >= comment->x1) && (wx <= comment->x2) 
		     && (wy >= comment->y1) && (wy <= comment->y2))) {
			dist = (* GNOME_CANVAS_ITEM_CLASS (comment->object.klass)->point) (comment, wx, wy, cx, cy, &item);

			if ((gint) (dist * GNOME_CANVAS (gil)->pixels_per_unit + 0.5) <= GNOME_CANVAS (gil)->close_enough)
				return n;
		}
	}

	return -1;
}


void
image_list_set_compare_func (ImageList *gil,
			     GCompareFunc cmp_func)
{
	ImageListPrivate *priv;

	g_return_if_fail (IS_IMAGE_LIST (gil));
	
	priv = gil->priv;  
	priv->compare = (cmp_func) ? cmp_func : default_compare;
}


void
image_list_set_sort_type (ImageList *gil,
			  GtkSortType sort_type)
{
	ImageListPrivate *priv;

	g_return_if_fail (IS_IMAGE_LIST (gil));

	priv = gil->priv;  
	priv->sort_type = sort_type;
}


void
image_list_sort (ImageList *gil)
{
	ImageListPrivate *priv;
	GList *scan;
	gint num;

	g_return_if_fail (IS_IMAGE_LIST (gil));

	priv = gil->priv;

	priv->image_list = g_list_sort (priv->image_list, priv->compare);
	if (priv->sort_type == GTK_SORT_DESCENDING)
		priv->image_list = g_list_reverse (priv->image_list);

	/* update selection list. */

	if (gil->selection != NULL) {
		g_list_free (gil->selection);
		gil->selection = NULL;
	}

	scan = priv->image_list;
	num = 0;
	while (scan) {
		Image *image = scan->data;

		if (image->selected)
			gil->selection = g_list_prepend (gil->selection, GINT_TO_POINTER (num));

		num++;
		scan = scan->next;
	}

	if (!priv->frozen) {
		gil_layout_all_images (gil);
		gil_scrollbar_adjust (gil);
	} else
		priv->dirty = TRUE;
}


void
image_list_set_image_pixbuf (ImageList *gil,
			     gint pos,
			     GdkPixbuf *pixbuf)
{
	ImageListPrivate *priv;
	Image *image;

	g_return_if_fail (IS_IMAGE_LIST (gil));
	g_return_if_fail (pos >= 0 && pos < gil->images);
	g_return_if_fail (pixbuf != NULL);

	priv = gil->priv;
	image = g_list_nth (priv->image_list, pos)->data;

	gtk_object_set (GTK_OBJECT (image->image),
			"pixbuf", pixbuf,
			NULL);
}


void
image_list_set_image_text (ImageList *gil,
			   gint pos,
			   const gchar *text)
{
	/* FIXME */
}


void
image_list_set_image_comment (ImageList *gil,
			      gint pos,
			      const gchar *comment)
{
	ImageListPrivate *priv;
	Image *image;

	g_return_if_fail (IS_IMAGE_LIST (gil));
	g_return_if_fail (pos >= 0 && pos < gil->images);

	if (!GTK_WIDGET_REALIZED (gil))
		return;

	priv = gil->priv;
	image = g_list_nth (priv->image_list, pos)->data;
	gnome_icon_text_item_configure (image->comment,
					0, 
					0, 
					priv->image_width, 
					DEFAULT_MEDIUM_FONT,
					comment, 
					FALSE,
					priv->static_text);

	if (priv->frozen) {
		priv->dirty = TRUE;
		return;
	}

	gil_layout_from_line (gil, pos / gil_get_images_per_line (gil));
	gil_scrollbar_adjust (gil);

}


GdkPixbuf*
image_list_get_image_pixbuf (ImageList *gil,
			     gint pos)
{
	ImageListPrivate *priv;
	Image *image;

	g_return_val_if_fail (IS_IMAGE_LIST (gil), NULL);
	g_return_val_if_fail (pos >= 0 && pos < gil->images, NULL);

	priv = gil->priv;
	image = g_list_nth (priv->image_list, pos)->data;

	return image->image->pixbuf;
}


gchar*
image_list_get_image_text (ImageList *gil,
			   gint pos)
{
	ImageListPrivate *priv;
	Image *image;

	g_return_val_if_fail (IS_IMAGE_LIST (gil), NULL);
	g_return_val_if_fail (pos >= 0 && pos < gil->images, NULL);

	priv = gil->priv;
	image = g_list_nth (priv->image_list, pos)->data;

	return gnome_icon_text_item_get_text (image->text);
}


gchar*
image_list_get_image_comment (ImageList *gil,
			      gint pos)
{
	ImageListPrivate *priv;
	Image *image;

	g_return_val_if_fail (IS_IMAGE_LIST (gil), NULL);
	g_return_val_if_fail (pos >= 0 && pos < gil->images, NULL);

	priv = gil->priv;
	image = g_list_nth (priv->image_list, pos)->data;

	return gnome_icon_text_item_get_text (image->comment);
}


gchar*
image_list_get_old_text (ImageList *gil)
{
	ImageListPrivate *priv;

	g_return_val_if_fail (IS_IMAGE_LIST (gil), NULL);

	priv = gil->priv;

	return priv->old_text;
}


GList*
image_list_get_list (ImageList *gil)
{
	ImageListPrivate *priv;

	g_return_val_if_fail (IS_IMAGE_LIST (gil), NULL);

	priv = gil->priv;
	
	return priv->image_list;
}


int
image_list_get_first_visible (ImageList *gil)
{
	gint pos, line;
	ImageListPrivate *priv;
	gdouble ofs;
	GList *l;

	g_return_val_if_fail (IS_IMAGE_LIST (gil), -1);

	if (gil->images == 0)
		return -1;

	priv = gil->priv;

	line = 0;
	ofs = gil->adj->value;
	for (l = priv->lines; l && (ofs > 0.0); l = l->next) {
		ofs -= image_line_height (gil, l->data);
		line++;
	}
	
	pos = image_list_get_images_per_line (gil) * (line - 1);

	if (pos < 0)
		pos = 0;
	if (pos >= gil->images)
		pos = gil->images - 1;

	return pos;
}


int
image_list_get_last_visible (ImageList *gil)
{
	gint pos, line;
	ImageListPrivate *priv;
	gdouble ofs;
	GList *l;

	g_return_val_if_fail (IS_IMAGE_LIST (gil), -1);

	if (gil->images == 0)
		return -1;

	priv = gil->priv;

	line = 0;
	ofs = gil->adj->value + gil->adj->page_size;
	for (l = priv->lines; l && (ofs > 0.0); l = l->next) {
		ofs -= image_line_height (gil, l->data);
		line++;
	}
	
	pos = image_list_get_images_per_line (gil) * line - 1;

	if (pos < 0)
		pos = 0;
	if (pos >= gil->images)
		pos = gil->images - 1;

	return pos;
}


gboolean
image_list_has_focus (ImageList *gil)
{
	g_return_val_if_fail (IS_IMAGE_LIST (gil), FALSE);
	return ((ImageListPrivate*) gil->priv)->has_focus;
}


void
image_list_start_editing (ImageList *gil,
			  gint pos)
{
	Image *image;
	ImageListPrivate *priv;

	g_return_if_fail (IS_IMAGE_LIST (gil));
	g_return_if_fail (pos >= 0 && pos < gil->images);

	priv = gil->priv;
	image = g_list_nth (priv->image_list, pos)->data;

	gnome_icon_text_item_start_editing (image->text);
}


gboolean
image_list_editing (ImageList *gil)
{
	ImageListPrivate *priv;
	Image *image;
	GList *scan;

	g_return_val_if_fail (IS_IMAGE_LIST (gil), FALSE);

	priv = gil->priv;

	for (scan = priv->image_list; scan; scan = scan->next) {
		image = (Image*) scan->data;
		if (image->text->editing)
			return TRUE;
	}

	return FALSE;
}


void
image_list_set_view_mode (ImageList *gil,
			  guint8 mode)
{
	ImageListPrivate *priv;

	g_return_if_fail (IS_IMAGE_LIST (gil));
	priv = gil->priv;

	if (priv->view_mode == mode)
		return;
	priv->view_mode = mode;
}


guint8
image_list_get_view_mode (ImageList *gil)
{
	ImageListPrivate *priv;

	g_return_val_if_fail (IS_IMAGE_LIST (gil), 0);	
	priv = gil->priv;

	return priv->view_mode;
}
