/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  GThumb
 *
 *  Copyright (C) 2001 The Free Software Foundation, Inc.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Street #330, Boston, MA 02111-1307, USA.
 */

#include <config.h>
#include <gtk/gtk.h>
#include <string.h>
#include <libgnome/libgnome.h>
#include <libgnomevfs/gnome-vfs-types.h>
#include <libgnomevfs/gnome-vfs-ops.h>
#include <libgnomevfs/gnome-vfs-file-info.h>
#include <libgnomevfs/gnome-vfs-utils.h>

#include "gthumb-init.h"
#include "file-data.h"
#include "file-list.h"
#include "file-utils.h"
#include "image-list.h"
#include "thumb-loader.h"
#include "typedefs.h"
#include "icons/img_unknown.xpm"

#define THUMB_BORDER 10

static GdkPixbuf *unknown_pixbuf = NULL;


static gint
comp_func_name (gconstpointer  ptr1,
		gconstpointer  ptr2)
{
        const Image *row1 = ptr1, *row2 = ptr2;
	const FileData *fd1, *fd2;

	fd1 = row1->data;
	fd2 = row2->data;

	if ((fd1 == NULL) || (fd2 == NULL))
		return 0;

	return strcasecmp (fd1->name, fd2->name);
}


static gint
comp_func_size (gconstpointer  ptr1,
		gconstpointer  ptr2)
{
        const Image *row1 = ptr1, *row2 = ptr2;
	const FileData *fd1, *fd2;

	fd1 = row1->data;
	fd2 = row2->data;

	if ((fd1 == NULL) || (fd2 == NULL))
		return 0;

	if (fd1->size < fd2->size) return -1;
	if (fd1->size > fd2->size) return 1;

	/* if the size is the same order by name. */
	return comp_func_name (ptr1, ptr2);
}


static gint
comp_func_time (gconstpointer  ptr1,
		gconstpointer  ptr2)
{
        const Image *row1 = ptr1, *row2 = ptr2;
	const FileData *fd1, *fd2;

	fd1 = row1->data;
	fd2 = row2->data;

	if ((fd1 == NULL) || (fd2 == NULL))
		return 0;

	if (fd1->mtime < fd2->mtime) return -1;
	if (fd1->mtime > fd2->mtime) return 1;

	/* if time is the same order by name. */
	return comp_func_name (ptr1, ptr2);
}


static gint
comp_func_path (gconstpointer  ptr1,
		gconstpointer  ptr2)
{
        const Image *row1 = ptr1, *row2 = ptr2;
	const FileData *fd1, *fd2;

	fd1 = row1->data;
	fd2 = row2->data;

	if ((fd1 == NULL) || (fd2 == NULL))
		return 0;

	return strcmp (fd1->path, fd2->path);
}


static gint
comp_func_none (gconstpointer  ptr1,
		gconstpointer  ptr2)
{
	return 0;
}


static void
set_unknown_icon (FileList *file_list, gint pos)
{
	image_list_set_image_pixbuf (IMAGE_LIST (file_list->ilist), 
				     pos, unknown_pixbuf);
}


static void 
update_thumb_in_clist (FileList *file_list)
{
	image_list_set_image_pixbuf (IMAGE_LIST (file_list->ilist), 
				     file_list->thumb_row, 
				     thumb_loader_get_pixbuf (file_list->thumb_loader));
}


static void file_list_update_next_thumb (FileList *file_list);


static void 
load_thumb_error_cb (ThumbLoader *tl,
		     gpointer data)
{
	FileList *file_list = data;
	FileData *fd;

	set_unknown_icon (file_list, file_list->thumb_row);
	fd = image_list_get_image_data (IMAGE_LIST (file_list->ilist),
					file_list->thumb_row);
	fd->error = TRUE;

	if (file_list->doing_thumbs)
		file_list_update_next_thumb (file_list);
}


static void 
load_thumb_done_cb (ThumbLoader *tl,
		    gpointer data)
{
	FileList *file_list = data;
	gfloat percent;

	update_thumb_in_clist (file_list);

	percent = ((float) (file_list->thumbs_num) 
		   / IMAGE_LIST (file_list->ilist)->images);
	if (file_list->progress_func) 
		file_list->progress_func (percent, file_list->progress_data);

	file_list_update_next_thumb (file_list);
}


static GCompareFunc
get_compfunc_from_method (SortMethod sort_method)
{
	GCompareFunc func;

	switch (sort_method) {
	case SORT_BY_NAME:
		func = comp_func_name;
		break;
	case SORT_BY_TIME:
		func = comp_func_time;
		break;
	case SORT_BY_SIZE:
		func = comp_func_size;
		break;
	case SORT_BY_PATH:
		func = comp_func_path;
		break;
	case SORT_NONE:
		func = comp_func_none;
		break;
	default:
		func = comp_func_none;
	}

	return func;
}


FileList*   
file_list_new ()
{
	FileList *file_list;
	GtkWidget *ilist;
	GtkAdjustment *adj;
        GtkWidget *vsb, *hbox;
	GtkWidget *frame;
	ImageListViewMode view_mode;

	file_list = g_new (FileList, 1);
	
	/* set the default values. */
	file_list->list              = NULL;
	file_list->sort_method       = preferences.file_list_sort;
	file_list->sort_type         = preferences.file_list_sort_type;
	file_list->show_dot_files    = preferences.show_dot_files;
	file_list->enable_thumbs     = preferences.enable_thumbnails;
	file_list->thumb_size        = preferences.thumb_size;
	file_list->doing_thumbs      = FALSE;
	file_list->thumb_loader      = THUMB_LOADER (thumb_loader_new (NULL, file_list->thumb_size, file_list->thumb_size));
	file_list->thumbs_done       = NULL;
	file_list->thumbs_num        = 0;
	file_list->thumb_fd          = NULL;
	file_list->thumb_row         = -1;
	file_list->progress_func     = NULL;
	file_list->progress_data     = NULL;
	file_list->interrupt_done_func = NULL;
	file_list->interrupt_done_data = NULL;

	if (unknown_pixbuf == NULL)
		unknown_pixbuf = gdk_pixbuf_new_from_xpm_data ((const char**) img_unknown_xpm);
	else
		gdk_pixbuf_ref (unknown_pixbuf);

	gtk_signal_connect (GTK_OBJECT (file_list->thumb_loader), "done",
			    load_thumb_done_cb, 
			    file_list);
	gtk_signal_connect (GTK_OBJECT (file_list->thumb_loader), "error",
			    load_thumb_error_cb, 
			    file_list);
	
	/* Create the widgets. */

	/* vertical scrollbar. */
	adj = GTK_ADJUSTMENT (gtk_adjustment_new (0, 0, 1, 1, 1, 1));
        vsb = gtk_vscrollbar_new (adj);

	/* image list. */
	ilist = image_list_new (preferences.thumb_size + THUMB_BORDER,
				adj, 
				IMAGE_LIST_IS_EDITABLE);
        image_list_set_selection_mode (IMAGE_LIST (ilist), 
                                       GTK_SELECTION_MULTIPLE);
        image_list_set_row_spacing (IMAGE_LIST (ilist), 15);
        image_list_set_col_spacing (IMAGE_LIST (ilist), 15);
	image_list_set_compare_func (IMAGE_LIST (ilist),
				    get_compfunc_from_method (file_list->sort_method));
	image_list_set_sort_type (IMAGE_LIST (ilist), file_list->sort_type);
	if (preferences.show_comments)
		view_mode = IMAGE_LIST_VIEW_ALL;
	else
		view_mode = IMAGE_LIST_VIEW_TEXT;
	image_list_set_view_mode (IMAGE_LIST (ilist), view_mode);

	/**/
	frame = gtk_frame_new (NULL);
	gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_IN);
	gtk_container_add (GTK_CONTAINER (frame), ilist);

	/**/
	hbox = gtk_hbox_new (FALSE, 3);
        gtk_box_pack_start (GTK_BOX (hbox), frame, TRUE, TRUE, 0);
        gtk_box_pack_start (GTK_BOX (hbox), vsb, FALSE, FALSE, 0);

	file_list->ilist = ilist;
	file_list->root_widget = hbox;

	return file_list;
}


static void
free_file_data_list (GList *list)
{
	GList *scan;
	for (scan = list; scan; scan = scan->next) 
		file_data_free (scan->data);
	g_list_free (list);
}


static void
file_list_free_list (FileList *file_list)
{
	g_return_if_fail (file_list != NULL);

	free_file_data_list (file_list->list);
	file_list->list = NULL;
}


void
file_list_free (FileList *file_list)
{
	file_list_free_list (file_list);
	gtk_object_unref (GTK_OBJECT (file_list->thumb_loader));
	g_free (file_list);
	gdk_pixbuf_unref (unknown_pixbuf);
}


/* used by file_list_update_list. */
static void file_list_update_thumbs (FileList *file_list);


static void
file_list_update_list (FileList *file_list)
{
	GtkWidget *ilist;
	gint pos;
	GList *scan;

	file_list_interrupt_thumbs (file_list);

	ilist = file_list->ilist;
	image_list_freeze (IMAGE_LIST (ilist));
	image_list_clear (IMAGE_LIST (ilist));

	for (scan = file_list->list; scan; scan = scan->next) {
		FileData *fd = scan->data;

		pos = image_list_append (IMAGE_LIST (ilist),
					 unknown_pixbuf,
					 fd->name,
					 fd->comment);

		image_list_set_image_data (IMAGE_LIST (ilist), pos, fd);
	}

	image_list_sort (IMAGE_LIST (ilist));
	image_list_thaw (IMAGE_LIST (ilist));

	if ((file_list->list != NULL) && file_list->enable_thumbs)
		file_list_update_thumbs (file_list);
}


/* ----- file_list_set_list help functions ----- */


typedef struct {
	FileList *file_list;
	GList *filtered;
	GList *list;
	gchar *full_path;
	DoneFunc done_func;
	gpointer done_func_data;
	gboolean call_done_func;
} GetFileInfoData;


static GetFileInfoData *
get_file_info_data_new (FileList *file_list, 
			GList *new_list,
			DoneFunc done_func,
			gpointer done_func_data)
{
	GetFileInfoData * data;

	data = g_new0 (GetFileInfoData, 1);
	data->file_list = file_list;
	data->filtered = NULL;
	data->list = NULL;
	data->full_path = NULL;
	data->done_func = done_func;
	data->done_func_data = done_func_data;
	data->call_done_func = TRUE;

	return data;
}


static void
get_file_info_data_free (GetFileInfoData *data)
{
	if (data == NULL)
		return;

	if (data->list) {
		g_list_foreach (data->list, (GFunc) g_free, NULL);
		g_list_free (data->list);
	}

	if (data->full_path)
		g_free (data->full_path);

	g_free (data);
}


/* called in : get_file_info_done_cb, file_list_set_list. */
static void
file_list_set_list_terminate (GetFileInfoData * gfi_data)
{
	FileList *file_list = gfi_data->file_list;

	file_list->list = gfi_data->filtered;
	file_list_update_list (file_list);
	if (gfi_data->call_done_func) {
		if (gfi_data->done_func != NULL)
			(*gfi_data->done_func) (gfi_data->done_func_data);
	} else { /* interrupted */
		if (file_list->interrupt_done_func != NULL)
			(*file_list->interrupt_done_func) (file_list->interrupt_done_data);
	}
	get_file_info_data_free (gfi_data);
}


static gboolean start_next__get_file_info (GetFileInfoData *gfi_data);


static void 
get_file_info_done_cb (GnomeVFSAsyncHandle *handle,
		       GList *results, /* GnomeVFSGetFileInfoResult* items */
		       gpointer callback_data)
{
	GnomeVFSGetFileInfoResult *info_result = results->data;
	GetFileInfoData * gfi_data = callback_data;
	FileData *fd;
	gboolean at_least_one_added = FALSE;

	if (info_result->result == GNOME_VFS_OK) {
		fd = file_data_new (gfi_data->full_path, 
				    info_result->file_info);
		file_data_update_comment (fd);
		gfi_data->filtered = g_list_prepend (gfi_data->filtered, fd);
	}

	at_least_one_added = start_next__get_file_info (gfi_data);

	/* if the process has terminated free the memory and update the 
	 * view. */
	if (!at_least_one_added) 
		file_list_set_list_terminate (gfi_data);
}


static void
remove_first_element (GList **list)
{
	GList *first = *list;

	*list = g_list_remove_link (*list, first);
	g_free (first->data);
	g_list_free (first);
}


static gboolean
start_next__get_file_info (GetFileInfoData *gfi_data)
{
	GList *scan;
	gchar *full_path;
	const gchar *name_only;
	gchar *escaped;
	GnomeVFSAsyncHandle *handle;
	GList *uri_list = NULL;
	GnomeVFSURI *uri;

	while (gfi_data->list != NULL) {
		scan = gfi_data->list;
		full_path = scan->data;
		name_only = file_name_from_path (full_path);

		if (gfi_data->file_list->interrupt_set_list) {
			free_file_data_list (gfi_data->filtered);
			gfi_data->filtered = NULL;
			gfi_data->call_done_func = FALSE;
			return FALSE;
		}

		if ((! gfi_data->file_list->show_dot_files 
		     && file_is_hidden (name_only))
		    || ! file_is_image (full_path, preferences.fast_file_type)) {
			remove_first_element (& (gfi_data->list));
			continue;
		}

		if (gfi_data->full_path != NULL)
			g_free (gfi_data->full_path);
		gfi_data->full_path = g_strdup (full_path);

		escaped = gnome_vfs_escape_path_string (full_path);
		uri = gnome_vfs_uri_new (escaped);
		g_free (escaped);
		uri_list = g_list_prepend (uri_list, uri);

		remove_first_element (& (gfi_data->list));

		gnome_vfs_async_get_file_info (&handle,
					       uri_list,
					       (GNOME_VFS_FILE_INFO_DEFAULT 
						| GNOME_VFS_FILE_INFO_FOLLOW_LINKS),
					       get_file_info_done_cb,
					       gfi_data);
		return TRUE;
	}

	return FALSE;
}


void 
file_list_set_list (FileList *file_list,
		    GList *new_list,
		    DoneFunc done_func,
		    gpointer done_func_data)
{
	GetFileInfoData *gfi_data;
	gboolean at_least_one_added = FALSE;
	GList *scan;

	g_return_if_fail (file_list != NULL);

	file_list->interrupt_set_list = FALSE;

	if (file_list->doing_thumbs)
		file_list_interrupt_thumbs (file_list);

 	file_list_free_list (file_list);
	file_list->list = NULL;

	gfi_data = get_file_info_data_new (file_list, 
					   new_list, 
					   done_func, 
					   done_func_data);

	for (scan = new_list; scan; scan = scan->next) {
		gchar *path = g_strdup ((char*)(scan->data));
		gfi_data->list = g_list_prepend (gfi_data->list, path);
	}

	at_least_one_added = start_next__get_file_info (gfi_data);

	/* Update now if the list is empty. */
	if (!at_least_one_added) 
		file_list_set_list_terminate (gfi_data);
}


void
file_list_interrupt_set_list (FileList *file_list,
			      DoneFunc done_func,
			      gpointer done_data)
{
	g_return_if_fail (file_list != NULL);

	file_list->interrupt_set_list = TRUE;
	file_list->interrupt_done_func = done_func;
	file_list->interrupt_done_data = done_data;
}


void 
file_list_set_sort_method (FileList *file_list,
			   SortMethod new_method)
{
	gboolean restart_thumbs;

	g_return_if_fail (file_list != NULL);

	if (file_list->sort_method == new_method) return;

	restart_thumbs = FALSE;
	if (file_list->doing_thumbs) {
		restart_thumbs = TRUE;
		file_list_interrupt_thumbs (file_list);
	}

	file_list->sort_method = new_method;

	image_list_set_compare_func (IMAGE_LIST (file_list->ilist), 
				     get_compfunc_from_method (new_method));
	image_list_sort (IMAGE_LIST (file_list->ilist));

	if (restart_thumbs)
		file_list_update_thumbs (file_list);
}


void
file_list_set_sort_type (FileList *file_list,
			 GtkSortType sort_type)
{
	gboolean restart_thumbs;

	g_return_if_fail (file_list != NULL);

	restart_thumbs = FALSE;
	if (file_list->doing_thumbs) {
		restart_thumbs = TRUE;
		file_list_interrupt_thumbs (file_list);
	}

	file_list->sort_type = sort_type;
	image_list_set_sort_type (IMAGE_LIST (file_list->ilist), sort_type);
	image_list_sort (IMAGE_LIST (file_list->ilist));

	if (restart_thumbs)
		file_list_update_thumbs (file_list);
}


static void file_list_thumb_cleanup (FileList *file_list);


void 
file_list_interrupt_thumbs (FileList *file_list)
{
	g_return_if_fail (file_list != NULL);

        if (file_list->doing_thumbs) {
		thumb_loader_stop (file_list->thumb_loader);
		file_list_thumb_cleanup (file_list);
	}
}


gint 
file_list_row_from_path (FileList *file_list, 
			 const gchar *path)
{
	GList *scan;
	gint   i;

	g_return_val_if_fail (file_list != NULL, -1);

	if (path == NULL)
		return -1;

	i = 0;
	scan = image_list_get_list (IMAGE_LIST (file_list->ilist));
	for (; scan; scan = scan->next) {
		FileData *fd = IMAGE (scan->data)->data;

		if (strcmp (fd->path, path) == 0)
			return i;

		i++;
	}

	return -1;
}


GList *
file_list_get_all (FileList *file_list)
{
	GList *list;
	GList *scan;

	g_return_val_if_fail (file_list != NULL, NULL);

	list = NULL;
	for (scan = file_list->list; scan; scan = scan->next) {
		FileData *fd = scan->data;
		list = g_list_prepend (list, g_strdup (fd->path));
	}

	return g_list_reverse (list);
}


gint 
file_list_get_length (FileList *file_list)
{
	g_return_val_if_fail (file_list != NULL, 0);
	return g_list_length (file_list->list);
}


GList *
file_list_get_selection (FileList *file_list)
{
	GList *list = NULL;
	GList *scan;

	g_return_val_if_fail (file_list != NULL, NULL);

	scan = IMAGE_LIST (file_list->ilist)->selection;
	for (; scan; scan = scan->next) {
		FileData *fd = image_list_get_image_data (IMAGE_LIST (file_list->ilist), GPOINTER_TO_INT (scan->data));
		if (fd) list = g_list_prepend (list, g_strdup (fd->path));
	}

	return g_list_reverse (list);
}


GList *
file_list_get_selection_as_fd (FileList *file_list)
{
	GList *list = NULL;
	GList *scan;

	g_return_val_if_fail (file_list != NULL, NULL);

	scan = IMAGE_LIST (file_list->ilist)->selection;
	for (; scan; scan = scan->next) {
		FileData *fd = image_list_get_image_data (IMAGE_LIST (file_list->ilist), GPOINTER_TO_INT (scan->data));
		if (fd) list = g_list_prepend (list, fd);
	}

	return g_list_reverse (list);
}


gint 
file_list_get_selection_length (FileList *file_list)
{
	g_return_val_if_fail (file_list != NULL, 0);
	return g_list_length (IMAGE_LIST (file_list->ilist)->selection);
}


gchar *
file_list_path_from_row (FileList *file_list,
			 gint row)
{
	FileData *fd;

	g_return_val_if_fail (file_list != NULL, NULL);

	fd = image_list_get_image_data (IMAGE_LIST (file_list->ilist), row);

	if (fd)	return g_strdup (fd->path);

	return NULL;
}


gboolean
file_list_is_selected (FileList *file_list,
		       gint row)
{
	GList *scan;

	g_return_val_if_fail (file_list != NULL, FALSE);

	scan = IMAGE_LIST (file_list->ilist)->selection;
	for (; scan; scan = scan->next) 
		if (GPOINTER_TO_INT (scan->data) == row) 
			return TRUE;

	return FALSE;
}


void 
file_list_select_image_by_row (FileList *file_list,
			       gint row)
{
	GtkWidget *ilist;

	g_return_if_fail (file_list != NULL);

	ilist = file_list->ilist;

	image_list_unselect_all (IMAGE_LIST (ilist), NULL, FALSE);
	image_list_select_image (IMAGE_LIST (ilist), row);
	if (image_list_image_is_visible (IMAGE_LIST (ilist), row) != GTK_VISIBILITY_FULL)
		image_list_moveto (IMAGE_LIST (ilist), row, 0.5);
}


void 
file_list_select_all (FileList *file_list)
{
	g_return_if_fail (file_list != NULL);
	image_list_select_all (IMAGE_LIST (file_list->ilist));
}


void 
file_list_unselect_all (FileList *file_list)
{
	g_return_if_fail (file_list != NULL);
	image_list_unselect_all (IMAGE_LIST (file_list->ilist), NULL, FALSE);
}


void
file_list_enable_thumbs (FileList *file_list,
			 gboolean enable)
{
	g_return_if_fail (file_list != NULL);

	file_list->enable_thumbs = enable;
	file_list_update_list (file_list);
}


void
file_list_set_progress_func (FileList *file_list,
			    ProgressFunc func,
			    gpointer data)
{
	g_return_if_fail (file_list != NULL);

	file_list->progress_func = func;
	file_list->progress_data = data;
}


/**
 * file_list_next_image:
 * @file_list: 
 * @row: 
 * @without_error: 
 * 
 * returns the next row that contains an image starting from row 
 * @row, if @without_error is %TRUE ignore the rows that contains 
 * broken images.
 * 
 * Return value: the next row that contains an image or -1 if no row could
 * be found.
 **/
gint
file_list_next_image (FileList *file_list,
		      gint row,
		      gboolean without_error)
{
	gint num_rows;

	g_return_val_if_fail (file_list != NULL, -1);

	num_rows = IMAGE_LIST (file_list->ilist)->images;

	row++;
	if (! without_error) {
		if (row >= num_rows)
			return -1;
		return row;
	}

	while (row < num_rows) {
		FileData *fd;

		fd = image_list_get_image_data (IMAGE_LIST (file_list->ilist),
						row);
		if (! fd->error)
			break;
		row++;
	}

	if (row >= num_rows)
		return -1;

	return row;
}


gint
file_list_prev_image (FileList *file_list,
		      gint row,
		      gboolean without_error)
{
	gint num_rows;

	g_return_val_if_fail (file_list != NULL, -1);

	num_rows = IMAGE_LIST (file_list->ilist)->images;

	row--;
	if (! without_error) {
		if (row < 0)
			return -1;
		return row;
	}

	while (row >= 0) {
		FileData *fd;

		fd = image_list_get_image_data (IMAGE_LIST (file_list->ilist),
						row);
		if (! fd->error)
			break;
		row--;
	}

	if (row < 0)
		return -1;

	return row;
}


void
file_list_delete_row (FileList *file_list,
		      gint row)
{
	FileData *fd;
	ImageList *ilist;
	gboolean restart_thumbs;

	g_return_if_fail (file_list != NULL);
	
	ilist = IMAGE_LIST (file_list->ilist);

	if ((row < 0) || (row >= ilist->images))
		return;

	restart_thumbs = FALSE;
	if (file_list->doing_thumbs) {
		restart_thumbs = TRUE;
		file_list_interrupt_thumbs (file_list);
	}

	fd = image_list_get_image_data (ilist, row);
	file_list->list = g_list_remove (file_list->list, fd);
	file_data_free (fd);

	image_list_remove (ilist, row);

	if (restart_thumbs)
		file_list_update_thumbs (file_list);
}


void
file_list_rename_row (FileList *file_list,
		      gint row, 
		      const gchar *path)
{
	FileData  *fd;
	ImageList *ilist;
	gboolean   restart_thumbs;

	g_return_if_fail (file_list != NULL);
	
	ilist = IMAGE_LIST (file_list->ilist);

	if ((row < 0) || (row >= ilist->images))
		return;

	restart_thumbs = FALSE;
	if (file_list->doing_thumbs) {
		restart_thumbs = TRUE;
		file_list_interrupt_thumbs (file_list);
	}

	/* update the FileData structure. */
	fd = image_list_get_image_data (ilist, row);
	file_data_set_path (fd, path);

	/* Set the new name. */
	image_list_freeze (ilist);
	image_list_set_image_text (ilist, row, fd->name);
	image_list_sort (ilist);
	image_list_thaw (ilist);

	if (restart_thumbs)
		file_list_update_thumbs (file_list);
}


void
file_list_update_comment (FileList *file_list,
			  gint row)
{
	FileData  *fd;
	ImageList *ilist;
	gboolean   restart_thumbs;

	g_return_if_fail (file_list != NULL);
	
	ilist = IMAGE_LIST (file_list->ilist);

	if ((row < 0) || (row >= ilist->images))
		return;

	restart_thumbs = FALSE;
	if (file_list->doing_thumbs) {
		restart_thumbs = TRUE;
		file_list_interrupt_thumbs (file_list);
	}

	/* update the FileData structure. */
	fd = image_list_get_image_data (ilist, row);
	file_data_update_comment (fd);

	/* Set the new name. */
	image_list_freeze (ilist);
	image_list_set_image_comment (ilist, row, fd->comment);
	image_list_thaw (ilist);

	if (restart_thumbs)
		file_list_update_thumbs (file_list);
}


void
file_list_set_thumbs_size (FileList *file_list,
			   gint size)
{
	gboolean restart_thumbs;

	g_return_if_fail (file_list != NULL);
	
	if (file_list->thumb_size == size) return;

	restart_thumbs = FALSE;
	if (file_list->doing_thumbs) {
		restart_thumbs = TRUE;
		file_list_interrupt_thumbs (file_list);
	}

	file_list->thumb_size = size;

	gtk_object_unref (GTK_OBJECT (file_list->thumb_loader));
	file_list->thumb_loader = THUMB_LOADER (thumb_loader_new (NULL, size, size));
	gtk_signal_connect (GTK_OBJECT (file_list->thumb_loader), "done",
			    load_thumb_done_cb, 
			    file_list);
	gtk_signal_connect (GTK_OBJECT (file_list->thumb_loader), "error",
			    load_thumb_error_cb, 
			    file_list);

	image_list_set_image_width (IMAGE_LIST (file_list->ilist), 
				    size + THUMB_BORDER);

	if (restart_thumbs)
		file_list_update_thumbs (file_list);
}



/* -- thumbs update -- */


static void 
file_list_thumb_cleanup (FileList *file_list)
{
	g_free (file_list->thumbs_done);
	file_list->thumbs_done = NULL;
	file_list->thumbs_num = 0;
	file_list->doing_thumbs = FALSE;

	if (file_list->progress_func) 
		file_list->progress_func (0.0, file_list->progress_data);
}


static void
file_list_update_next_thumb (FileList *file_list)
{
	GtkWidget *ilist;
	gint new_row = -1;
	gint row = -1;

	ilist = file_list->ilist;

	/* Find first visible undone. */
	row = image_list_get_first_visible (IMAGE_LIST (ilist));
	if (row != -1) {  
		gint n = IMAGE_LIST (file_list->ilist)->images;

		while ((row < n) && 
		       image_list_image_is_visible (IMAGE_LIST (ilist), row)) {
			if (! file_list->thumbs_done[row]) {
				file_list->thumbs_done[row] = TRUE;
				new_row = row;
				break;
			}
			row++;
		}
	}

	/* If not found then search the first undone. */
	if (new_row == -1) {
		gint n = IMAGE_LIST (file_list->ilist)->images;
		row = 0;
		while ((row < n) && (new_row == -1)) 
			if (! file_list->thumbs_done[row]) {
				file_list->thumbs_done[row] = TRUE;
				new_row = row;
			} else
				row++;
		
		if (row == n) {
			/* Thumbnails creation terminated. */
			file_list_thumb_cleanup (file_list);
			return;
		}
	}

	g_assert (new_row != -1);

	file_list->thumb_row = new_row;
	file_list->thumbs_num++;
	file_list->thumb_fd = image_list_get_image_data (IMAGE_LIST (ilist), 
							 new_row);

	if (path_is_file (file_list->thumb_fd->path)) {
		thumb_loader_set_path (file_list->thumb_loader, 
				       file_list->thumb_fd->path);
		thumb_loader_start (file_list->thumb_loader);
	} else /* Error: the file does not exists. */
		gtk_signal_emit_by_name (GTK_OBJECT (file_list->thumb_loader),
					 "error",
					 file_list);
}


static void 
file_list_update_thumbs (FileList *file_list)
{
	g_assert (file_list->thumbs_done == NULL);

	file_list->thumbs_done = g_new0 (gboolean, 
					 IMAGE_LIST(file_list->ilist)->images);

	file_list->doing_thumbs = TRUE;
	file_list_update_next_thumb (file_list);
}
