/* gt_lesstif.c - widget routines for X/lesstif version of etalk
 *
 * Copyright (C) 1997, 1998, 1999 Free Software Foundation
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, you can either send email to this
 * program's author (see below) or write to:
 * 
 *              The Free Software Foundation, Inc.
 *              675 Mass Ave.
 *              Cambridge, MA 02139, USA. 
 * 
 * Please send bug reports, etc. to zappo@gnu.org.
 * 
 * Description:
 *   These routines manage the lesstif widget set specific interface,
 * where _x handles those routines which would be shared between
 * widget sets.
 *   
 * $Log: gt_lesstif.c,v $
 * Revision 1.15  1999/08/26 11:58:03  zappo
 * Added Ctxt to fix_user_windows.  Fixed all occurances of XRead to pass
 * down Ctxt.
 *
 * Revision 1.14  1998/10/15 14:28:02  zappo
 * Fixed serious memory leak.  Apparently XmTextGetString returns a copy
 * that must be freed.
 *
 * Revision 1.13  1997/12/14 19:16:03  zappo
 * Renamed package to gtalk, renamed symbols and files apropriately
 * Fixed copyright and email address.
 *
 * Revision 1.12  1997/11/01 13:29:02  zappo
 * Tried to make the Motif version work better.  Now regular motif and lesstif
 * should both creation of new talk buffers to work the way they are supposed to.
 *
 * Revision 1.11  1997/10/25 02:41:58  zappo
 * Replaced the read w/ a flush in set_echo_label. Prevents recursion
 * and messing up the order of execution of typed characters.
 *
 * Revision 1.10  1997/10/25 01:49:32  zappo
 * Fixed a text-insertion problem that failed to use the length part.
 *
 * Revision 1.9  1997/10/23 00:54:54  zappo
 * Added net-paste translation to text widgets
 *
 * Revision 1.8  1997/10/22 11:15:41  zappo
 * Added font metrics for sizing things correctly.  Fixed "fix windows" resize
 * bug, and made that section of code more streamlined.  Fixed popup sizing
 * problems.  Fixed popup death for real Motif. Fixed poor startup clear alg.
 *
 * Revision 1.7  1997/10/03 23:21:12  zappo
 * Made "fix windows" work a little smarter by knowing about sash_height.
 *
 * Revision 1.6  1997/07/27 17:37:24  zappo
 * Changed to use the generic list for all windows.
 *
 * Revision 1.5  1997/07/23 01:23:39  zappo
 * Small changes for new rev of lesstif
 *
 * Revision 1.4  1997/06/26 23:29:24  zappo
 * Memory management changes
 *
 * Revision 1.3  1997/03/23 17:04:54  zappo
 * Fixed up some comments, and messed around with the window-fixer.
 * It's still screwed up. ;(
 *
 * Revision 1.2  1997/03/21 12:33:28  zappo
 * Moved hierarchy realization into build_mainwindow
 *
 * Revision 1.1  1997/03/21 04:05:14  zappo
 * Initial revision
 *
 * Tokens: ::Header:: gtalkx.h
 */
#include <X11/Xlib.h>

#include "gtalklib.h"
#include "gtalkc.h"
#include "gtalkx.h"

#include <X11/IntrinsicP.h>

#include <Xm/DialogS.h>

#include <Xm/PanedW.h>
#include <Xm/Form.h>
#include <Xm/RowColumn.h>
#include <Xm/BulletinB.h>
#include <Xm/Label.h>
#include <Xm/PushB.h>
#include <Xm/CascadeB.h>
#include <Xm/MenuShell.h>
#include <Xm/Text.h>
#include <Xm/ScrolledW.h>

/* Now for some static local variables... */
static Widget echo_area = NULL;
static Widget gtalk_pane, etalk_pane, menu_box;
static Widget PopupText;

static void XW_show_message();
static void XW_paste_talk();

/*
 * Variables: font_list, jay
 * Function: font_width, font_height
 *
 * I didn't like dealing with fonts to get window sizes very much, so I
 * made these accessors.  Here I assume that the whole app uses the same
 * font.
 *
 * Returns:     A dimension
 *
 * History:
 * zappo   10/19/97    Created
 */
static XmFontList font_list;
static XmString jay = NULL;
static int font_width()
{
  if(jay == NULL) {
    XtVaGetValues(echo_area, XmNfontList, &font_list, NULL);
    jay = XmStringCreateSimple("j"); /* Why j?  Max size courier? */
  }

  return jay?XmStringWidth(font_list, jay):8;
}
static int font_height()
{
  if(jay == NULL) {
    XtVaGetValues(echo_area, XmNfontList, &font_list, NULL);
    jay = XmStringCreateSimple("j"); /* Why j?  Max size courier? */
  }

  return jay?XmStringHeight(font_list, jay):8;
}

/*
 * Function: XW_build_mainwindow
 *
 *   Creates the top level windows needed to run etalk in X window mode.
 *
 * Returns:     Nothing
 * Parameters:  None
 *
 * History:
 * zappo   9/16/95    Created
 */
void XW_build_mainwindow(XCtxt)
     struct Xcontext *XCtxt;	/* x context to initialize */
{
  /* Now, initialize the shells we need to put our text widgets into */
  gtalk_pane = 
    XtVaCreateWidget("GtalkPane", xmFormWidgetClass, XCtxt->topLevel,
		     /* This makes Lesstif upset */
		     /*XmNallowShellResize, True,*/
		     XmNmargin,           0,
		     NULL);

  /* Create a menu up on the top. */
  menu_box = XmCreateMenuBar(gtalk_pane, "Menubox", NULL, 0);
  XtVaSetValues(menu_box,
		XmNallowResize,      False,
		XmNtopAttachment,    XmATTACH_FORM,
		XmNleftAttachment,   XmATTACH_FORM,
		XmNrightAttachment,  XmATTACH_FORM,
		NULL);  
  XtManageChild(menu_box);

  /* Make the menus */
  X_make_menus(menu_box);

  /* Start with the echo area */
  echo_area = XtVaCreateManagedWidget("EchoArea", xmTextWidgetClass,
				      gtalk_pane,
				      XmNallowResize,       False,
				      XmNeditMode,          XmSINGLE_LINE_EDIT,
				      XmNpaneMinimum,       10,
				      XmNresize,            False,
				      /*XmNshowGrip,          False,*/
				      XmNrows,              1,
				      /* 81 for scrollbar on talk windows */
				      XmNcolumns,           81,
				      XmNskipAdjust,        True,
				      XmNeditable,          False,
				      XmNbottomAttachment,  XmATTACH_FORM,
				      XmNleftAttachment,    XmATTACH_FORM,
				      XmNrightAttachment,   XmATTACH_FORM,
				      NULL);


  /* Create the Pane in which we will stuff the text widgets... */
  etalk_pane = 
    XtVaCreateManagedWidget("EtalkPane", xmPanedWindowWidgetClass, gtalk_pane,
			    XmNallowResize,       True,
			    XmNresize,            True,
			    XmNrefigureMode,      True,
			    XmNseparatorOn,       False,
			    XmNleftAttachment,    XmATTACH_FORM,
			    XmNrightAttachment,   XmATTACH_FORM,
			    XmNtopAttachment,     XmATTACH_WIDGET,
			    XmNtopWidget,         menu_box,
			    XmNbottomAttachment,  XmATTACH_WIDGET,
			    XmNbottomWidget,      echo_area,
			    XmNmarginHeight,      0,
			    XmNspacing,           0,
			    NULL);

  XtManageChild(gtalk_pane);
}


/*
 * Function: XW_fix_windows
 *
 *   This routine realigns windows when users are added or lost from
 * the window list.
 *
 * Returns:     Nothing
 * Parameters:  Ctxt - Current context
 *              list - Pointer toWindow list
 *
 * History:
 * zappo   9/18/95    Created
 */
void XW_fix_user_windows(Ctxt, list)
     struct TalkContext *Ctxt;
     struct WindowList *list;
{
  int sash_height = 0;
  int shadow_size = 0;
  Dimension winsize, panesize;
  struct WindowList *loop;
  int count, done;
  
  /* count up the number of windows. In this loop, LOCAL won't count,
   * so start at 1. */
  for(loop = NextElement(list), count = 1; loop; loop = NextElement(loop))
    {
      if(DISP_window_displayed(loop))
	count++;
    }

  /* Now find out how big a sash is.  Assume they are all the same size */
  XtVaGetValues(etalk_pane,
		XmNsashHeight, &sash_height,
		XmNshadowThickness, &shadow_size,
		NULL);

  /* Now, lets see how much space we have */
  XtVaGetValues(etalk_pane, XmNheight, &winsize, NULL);

  if(winsize/count < (10*font_height() + sash_height + 2*shadow_size))
    {
      if(list->window == NULL)
	/* First window is bigger to hold the greeting.  10 lines of text,
	 * plus the label, the sash, and margins...
	 */
	panesize = font_height() * 14 + sash_height + 2*shadow_size;
      else
	panesize = font_height() * 10 + sash_height + 2*shadow_size;

      winsize = panesize*count;

      XtVaSetValues(etalk_pane, XmNheight, winsize, NULL);
    } else {
      panesize = winsize/count;
    }

  /* Allow our widgets to resize themselves in the below loop. */
  XtVaSetValues(gtalk_pane, XmNresizePolicy, XmRESIZE_ANY, NULL);

  /* rebuild the window list */
  for(loop = list, done = 0; loop; loop = NextElement(loop))
    {
      /* c short circuit protects us from a seg-fault */
      /* a non-user window is local guy, and all others have user pointers */
      /* if a user is cleaned, windows will have been cleaned */
      if(DISP_window_displayed(loop))
	{
	  if(loop->window == NULL)
	    {
	      Widget scroller;
	      char *panename;
	      Pixel bg, fg;

	      if(loop->user)
		{
		  panename = malloc(strlen(loop->user->name) + 5);
		  sprintf(panename, "%sPane", loop->user->name);
		}
	      else
		{
		  panename = "localPane";
		}

	      /* Create a new window... */
	      loop->group =
		XtVaCreateWidget(panename,
				 xmFormWidgetClass,  etalk_pane,
				 XmNpaneMinimum,     50,
				 XmNallowResize,     True,
				 XmNheight,          panesize,
				 XmNallowOverlap,    False,
				 XmNmargin,          0,
				 XmNverticalSpacing, 0,
				 XmNrubberPositioning, True,
				 NULL);

	      loop->statusline =
		XtVaCreateManagedWidget("UserLabel", 
					xmLabelWidgetClass,   loop->group,
					XmNjustify,           XmALIGNMENT_BEGINNING,
					XmNresize,            True,
					XmNbottomAttachment,  XmATTACH_FORM,
					XmNleftAttachment,    XmATTACH_FORM,
					XmNrightAttachment,   XmATTACH_FORM,
					NULL);
	      /* Invert the colors.  This way someone can just specify the
	       * pane, and inversion looks good.
	       */
	      XtVaGetValues(loop->statusline,
			    XmNforeground, &fg,
			    XmNbackground, &bg,
			    NULL);
	      XtVaSetValues(loop->statusline,
			    XmNforeground, bg,
			    XmNbackground, fg,
			    NULL);

	      scroller = 
		XtVaCreateManagedWidget("UserScroller",
					xmScrolledWindowWidgetClass, 
					loop->group,
					XmNscrollVertical,    XmSTATIC,
					XmNscrollHorizontal,  XmAS_NEEDED,
					XmNbottomAttachment,  XmATTACH_WIDGET,
					XmNbottomWidget,      loop->statusline,
					XmNtopAttachment,     XmATTACH_FORM,
					XmNleftAttachment,    XmATTACH_FORM,
					XmNrightAttachment,   XmATTACH_FORM,
					XmNtopOffset,         0,
					XmNbottomOffset,      0,
					XmNleftOffset,        0,
					XmNrightOffset,       0,
					NULL);
	      loop->window = 
		XtVaCreateManagedWidget("UserText",
					xmTextWidgetClass,    scroller,
					XmNeditMode,          XmMULTI_LINE_EDIT,
					XmNeditable,          False,
					NULL);

	      XtOverrideTranslations
		(loop->window,
		 XtParseTranslationTable
		 ("<Btn2Down>:paste-with-net()\n<Btn2Up>:paste-with-net()\n"));

	      XtManageChild(loop->group);
	    }
	  /* Always do this to force some version of Motif to
	   * update some of their positions.
	   */
	  XtVaSetValues(loop->group, XmNheight, panesize, NULL);
	}
      else
	{
	  /* We must free up the windows if they exist! */
	  if(loop->group)
	    {
	      X_delwin(loop->group);
	      loop->group = NULL;
	    }
	}
    }

  /* Resize is only turned off for the widgets.  The pane can still
   * adjust things.  This protects against real-motif which seems to
   * want to shrink newly created widgets.
   */
  XtVaSetValues(gtalk_pane, XmNresizePolicy, XmRESIZE_NONE, NULL);

  /* This routine is oft called from deep within other things which do
   * not call the X read routine, so call it here just to be sure */
  X_xread(Ctxt, NULL);
}


/*
 * Function: XW_popup
 *
 *   Creates a dialog shell, with a text widget in it just large
 * enough to hold the text specified by the parameters.  (Maybe, if I
 * feel like doing all that font-extents stuff.)  Returns as a void
 * pointer the shell widget.
 *
 * Returns:     Nothing
 * Parameters:  width  - Number of width
 *              height - Number of height
 * History:
 * zappo   9/18/95    Created
 */
Widget XW_popup(XCtxt, width, height)
     struct Xcontext *XCtxt;
     int width, height;
{
  Widget shell, dialog, button;

  shell = XtVaCreatePopupShell("PopupInformationShell", 
			       xmDialogShellWidgetClass, XCtxt->topLevel,
			       XmNallowShellResize,   True,
			       XmNtitle,              "GNU Talk Information",
			       NULL);

  dialog = XtVaCreateWidget("PopupInformationDialog",
			    xmFormWidgetClass,   shell, 
			    XmNautoUnmanage,     False,
			    XmNallowOverlap,     False,
			    XmNallowResize,      True,
			    XmNfractionBase,     4,
			    NULL);

  button =
    XtVaCreateManagedWidget("Ok", xmPushButtonWidgetClass, dialog,
			    XmNmargin, 5,
			    XmNbottomAttachment, XmATTACH_FORM,
			    XmNleftAttachment,   XmATTACH_POSITION,
			    XmNleftPosition,     1,
			    XmNrightAttachment,  XmATTACH_POSITION,
			    XmNrightPosition,    3,
			    XmNtopOffset,        5,
			    XmNbottomOffset,     5,
			    XmNleftOffset,       5,
			    XmNrightOffset,      5,
			    NULL);

  XtAddCallback(button, XmNactivateCallback, simulate_key_callback, "\033");

  PopupText
    = XtVaCreateManagedWidget("PopupInformationText",
			      xmTextWidgetClass,   dialog,
			      XmNeditMode,         XmMULTI_LINE_EDIT,
			      XmNcolumns,          width+1,/* Not perfect */
			      XmNrows,             height-1,/* too much*/
			      XmNtopAttachment,    XmATTACH_FORM,
			      XmNleftAttachment,   XmATTACH_FORM,
			      XmNrightAttachment,  XmATTACH_FORM,
			      XmNbottomAttachment, XmATTACH_WIDGET,
			      XmNbottomWidget,     button,
			      XmNtopOffset,        5,
			      XmNbottomOffset,     5,
			      XmNleftOffset,       5,
			      XmNrightOffset,      5,
			      XmNresizePolicy,     XmRESIZE_NONE,
			      NULL);
  XmTextSetEditable(PopupText, False);
  
  /* Turn off the menu so they can't do any new dialogs */
  XtSetSensitive(menu_box, False);

  return dialog;
}
void XW_kill_popup(XCtxt)
     struct Xcontext *XCtxt;
{
  PopupText = NULL;  

  /* Fix the menus */
  XtSetSensitive(menu_box, True);
}

/*
 * Function: XW_popup_text, XW_add_text, and support functions
 *
 *   Adds text to an existing popup window or text window
 *
 * Returns:     Nothing
 * Parameters:  w    - Widget containing the text widget
 *              text - Pointer toCharacter of text
 *              len  - length of text
 * History:
 * zappo   9/19/95    Created
 */
void XW_popup_text(text, len)
     char *text;
     int   len;
{
  if(PopupText)
    {
      XW_add_text(PopupText, text, len);
      XW_add_text(PopupText, "\012", 1);
    }
}
static void add_text_engine(w, text, len)
     Widget w;
     char *text;
     int   len;
{
  XmTextPosition tp;

  /* Give ourselves permission to add stuff */
  XmTextSetEditable(w, True);

  /* Get the last position */
  tp = XmTextGetLastPosition(w);

  if(text[len] != 0)
    {
      static char *insert_buffer = NULL;
      static int buffer_size = 0;
      /* Not all buffers are NULL terminated.  We must copy the text to
       * be inserted if it is to work correctly since motif doesn't support
       * an insert method that takes a number of characters.
       */
      if((len + 1) > buffer_size)
	{
	  if(insert_buffer) free(insert_buffer);
	  
	  insert_buffer = (char *)malloc(len + 10);
	  buffer_size = len+10;
	}
      
      memcpy(insert_buffer, text, len);
      insert_buffer[len] = 0;

      XmTextInsert(w, tp, insert_buffer);
    }
  else
    {
      XmTextInsert(w, tp, text);
    }

  XmTextSetCursorPosition(w, XmTextGetLastPosition(w));

  /* Reset this value */
  XmTextSetEditable(w, False);
}
int XW_winwidth(window)
     void *window;
{
  Dimension    pixwidth;

  /* Get the pixel width and current font in the window */
  XtVaGetValues((Widget)window,
		XmNwidth, &pixwidth, 
		NULL);

  return pixwidth / font_width();
}
static int current_column(window)
     void *window;
{
  XmTextPosition tp, np;
  char           *buff;
  int             linecolumn = 0;

  /* Find the current column of the "insertion point" */
  buff = XmTextGetString((Widget)window);

  tp = XmTextGetLastPosition((Widget)window);

  if(tp)
    {
      /* Scan over everything till we find a newline */
      for(np = tp; (np > 0) && (buff[np] != 10); np--);

      /* move back over the newline we just found */
      if(np != 0) np++;

      linecolumn = tp - np;
    }
  XtFree(buff);

  return linecolumn;
}
void XW_add_text(w, text, len)
     Widget w;
     char *text;
     int   len;
{
  add_text_engine(w, text, len);
  /* Don't auto-fill info boxes.  That would be silly. */
  if((w != PopupText) && (current_column(w)+2) >= XW_winwidth(w))
    add_text_engine(w, "\n", 1);
}

int XW_fillcolumn(window)
     void *window;
{
  /* Logic to determine if we should fill right now. */
  return (current_column(window) + 10) > XW_winwidth(window);
}
void XW_delchar(window)
     void *window;
{
  XmTextPosition tp;
  char           *buff;

  buff = XmTextGetString((Widget)window);

  tp = XmTextGetLastPosition((Widget)window);

  /* Only delete back to the last line feed */
  if((tp > 0) && (buff[tp - 1] != 10))
    {
      /* Give ourselves permission to add stuff */
      XmTextSetEditable((Widget)window, True);

      XmTextReplace((Widget)window, tp -1, tp, "");
      XmTextSetCursorPosition((Widget)window,
			      XmTextGetLastPosition((Widget)window));

      /* Reset this value */
      XmTextSetEditable((Widget)window, False);
    }

  XtFree(buff);
}
void XW_delline(window)
     void *window;
{
  XmTextPosition  tp, np;
  char           *buff;

  buff = XmTextGetString((Widget)window);

  tp = XmTextGetLastPosition((Widget)window);

  if(tp)
    {
      /* Scan over everything till we find a newline */
      for(np = tp; (np > 0) && (buff[np] != 10); np--);
      /* move back over the newline we just found */
      if(np != 0) np++;
      
      /* Give ourselves permission to add stuff */
      XmTextSetEditable((Widget)window, True);
      
      XmTextReplace((Widget)window, np, tp, "");
      XmTextSetCursorPosition((Widget)window, 
			      XmTextGetLastPosition((Widget)window));

      /* Reset this value */
      XmTextSetEditable((Widget)window, False);
    }

  XtFree(buff);
}
void XW_delword(window)
     void *window;
{
  XmTextPosition  tp, np;
  char           *buff;

  buff = XmTextGetString((Widget)window);

  tp = XmTextGetLastPosition((Widget)window);

  if(tp)
    {
      /* Scan over spaces first... */
      for(np = tp-1; (np > 0) && ((buff[np] == ' ') || (buff[np] == '\t')); np--);
      /* Now scan over the word... */
      for(;(np > 0) && (buff[np] != ' ') && (buff[np] != '\t') && (buff[np] != 10);
	  np--);
      /* move back over the space we just found */
      if(np != 0) np++;
      
      /* Give ourselves permission to add stuff */
      XmTextSetEditable((Widget)window, True);
      
      XmTextReplace((Widget)window, np, tp, "");
      XmTextSetCursorPosition((Widget)window,
			      XmTextGetLastPosition((Widget)window));

      /* Reset this value */
      XmTextSetEditable((Widget)window, False);
    }

  XtFree(buff);
}


/*
 * Function: XW_clearwin
 *
 *   This will clear one text window specified in WINDOW
 *
 * Returns:     Nothing
 * Parameters:  window - Widget to window
 *
 * History:
 * zappo   12/9/95    Created
 */
void XW_clearwin(window)
     void *window;
{
  XmTextPosition  tp;
  char           *buff;

  buff = XmTextGetString((Widget)window);

  tp = XmTextGetLastPosition((Widget)window);

  if(tp)
    {
      /* Give ourselves permission to add stuff */
      XmTextSetEditable((Widget)window, True);

      XmTextReplace((Widget)window, 0, tp, "");
      XmTextSetCursorPosition((Widget)window,
			      XmTextGetLastPosition((Widget)window));
      
      /* Reset this value */
      XmTextSetEditable((Widget)window, False);
    }

  XtFree(buff);
}

/*
 * Function: XW_setlabel
 *
 *   Sets a label's value.
 *
 * Returns:     Nothing
 * Parameters:  win   - EmptyPointer to the window to set
 *              label - Pointer toCharacter of label
 * History:
 * zappo   9/18/95    Created
 */
void XW_setlabel(win, label)
     void *win;
     char *label;
{
  XmString xs;

  xs = XmStringCreate(label, XmSTRING_DEFAULT_CHARSET);

  XtVaSetValues((Widget)win, XmNlabelString, xs, NULL);

  XmStringFree(xs);
}

/*
 * Function: XW_build_menu
 *
 *   Use this to build one widget menu with callbacks.  All callbacks
 * are given data 
 *
 * Returns:     Nothing
 * Parameters:  parent   - Widget parent
 *              name     - Name of
 *              items    -  items
 *              numitems - Number of number
 * History:
 * zappo   9/16/95    Created
 */
void XW_build_menu(parent, name, items, numitems)
     Widget parent;
     char *name;
     struct MenuItemConstructor *items;
     int numitems;
{
  Widget mbutton, menu, item;
  XmString xs;
  int i;

  xs = XmStringCreate(name, XmSTRING_DEFAULT_CHARSET);

  menu = XmCreatePulldownMenu(parent, name, NULL, 0);

  mbutton = XtVaCreateManagedWidget(name, 
				    xmCascadeButtonWidgetClass, parent,
				    XmNlabelString,    xs,
				    XmNsubMenuId,      menu,
				    NULL);
  XmStringFree(xs);

  if(strcmp(name, "Help") == 0) 
    {
      XtVaSetValues(parent, XmNmenuHelpWidget, mbutton, NULL);
    }

  for (i=0; i < numitems; i++) {

    xs = XmStringCreate(items[i].button_name, XmSTRING_DEFAULT_CHARSET);
    item = XtVaCreateManagedWidget(items[i].button_name,
				   xmPushButtonWidgetClass,
				   menu, 
				   XmNlabelString, xs,
				   NULL);
    XmStringFree(xs);

    XtAddCallback(item, XmNarmCallback, XW_show_message, items[i].describe);
    XtAddCallback(item, XmNdisarmCallback, XW_show_message, "");
    XtAddCallback(item, XmNactivateCallback, items[i].callback, items[i].data);
  }
}

/*
 * Function: XW_set_echo_label
 *
 *   Set the value of the echo area label.
 *
 * Returns:     Nothing
 * Parameters:  str        - String Pointer  string
 *              showcursor - State of the cursor
 * History:
 * zappo   9/17/95    Created
 * zappo   1/30/97    Added parameters showcursor
 */
static int last_message_was_menu = FALSE;

void XW_set_echo_label(str, showcursor)
     char *str;
     int showcursor;
{
  if(echo_area)
    {
      XmTextSetEditable(echo_area, True);

      XmTextSetString(echo_area, str);

      XmTextSetCursorPosition(echo_area, XmTextGetLastPosition(echo_area));

      XmTextSetEditable(echo_area, False);

      XtVaSetValues(echo_area, XmNautoShowCursorPosition, showcursor,
		    NULL);

      XFlush(XtDisplay(echo_area));
    }
  last_message_was_menu = FALSE;
}

static void XW_show_message(w, ud, cd)
     Widget    w;
     char *ud;
     XtPointer cd;
{
  /* If ud == "" then only reset if we are going to erase a message
   * we put up ourselves.  This is because the disarm happens after
   * the action callback.  Is that correct behavior?
   */
  if((ud[0] != 0) || last_message_was_menu)
    {
      XW_set_echo_label(ud, False);
    }
  last_message_was_menu = TRUE;
}
