/* gt_forkap.c - Forks applications for etalk
 *
 * Copyright (C) 1995, 1996, 1997 Eric M. Ludlam
 * Copyright (C) 1997 Free Software Foundation
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, you can either send email to this
 * program's author (see below) or write to:
 * 
 *              The Free Software Foundation, Inc.
 *              675 Mass Ave.
 *              Cambridge, MA 02139, USA. 
 * 
 * Please send bug reports, etc. to zappo@gnu.org.
 * 
 * Description:
 *   This file manages associations of file suffixes to programs, and
 * the ability to launch these programs.  When a file is sent from one
 * etalk to another, the correct method of display is determined, and
 * forked (if necessary).
 * 
 * $Log: gt_forkap.c,v $
 * Revision 1.7  1997/12/14 19:15:54  zappo
 * Renamed package to gtalk, renamed symbols and files apropriately
 * Fixed copyright and email address.
 *
 * Revision 1.6  1997/01/28 03:20:18  zappo
 * Changed paramter of app_def to non-pointer
 *
 * Revision 1.5  1997/01/26  15:30:04  zappo
 * Fixed parameters to FORK_save_pid so that it made more sense.
 *
 * Revision 1.4  1996/03/02  03:16:15  zappo
 * Fixed grabkey to also be useful for emacs interface, plus some
 * warnings
 *
 * Revision 1.3  1996/02/01  01:16:03  zappo
 * Removed all ASSOC and SHAPP code, and moved them to their own files.
 * Added pipe support.
 *
 * Revision 1.2  1995/12/09  23:59:37  zappo
 * Fixed some status printing problems when listing processes and
 * associations
 *
 * Revision 1.1  1995/11/21  04:01:12  zappo
 * Initial revision
 *
 * Tokens: ::Header:: gtproc.h
 */

#include "gtalklib.h"
#include "gtalkc.h"
#include "gtproc.h"
#include "sitecnfg.h"

#if HAVE_SIGNAL_H == 1
#include <signal.h>
#endif

#if HAVE_SYS_WAIT_H == 1
#include <sys/wait.h>
#else
/* What should I do in this case?? */
#endif

/* The PID struct keeps track of all launched apps 
 */
enum pid_state { Running, Killed, Dead };

struct pid_node {
  pid_t              kid;	/* pid */
  enum pid_state     state;	/* current state */
  int                exit_code;	/* exit code (when killed) */
  enum app_def_type  type;	/* it's type */
  union app_def      app;	/* definition of the running app */
  char              *opt_data;	/* optional data */
  struct UserObject *pipes;	/* pipes used to talk to subprocess */
  struct pid_node *next, *prev;
};

static struct pid_node *Q_first, *Q_last;


/*
 * Function: FORK_pid_number
 *
 *   Returns the number of active jobs
 *
 * Returns:     int  - 
 * Parameters:  None
 *
 * History:
 * zappo   11/19/95   Created
 */
int FORK_pid_number()
{
  int i;
  struct pid_node *loop;

  for(i=0, loop=Q_first; loop; i++, loop=loop->next);

  return i;
}


/*
 * Function: FORK_pid_print
 *
 *   Prints out the list of all active jobs forked off by etalk.
 *
 * Returns:     Nothing
 * Parameters:  Ctxt - Context
 *
 * History:
 * zappo   11/19/95   Created
 */
void FORK_pid_print(Ctxt)
     struct TalkContext *Ctxt;
{
  static char *pidstates[] = { "Run", "Killed", "Dead" };
  char buffer[200];
  char comminf[100];
  struct pid_node *loop;

  DISP_message(Ctxt, "PID\tState\tExit\tCommand Info");
  for(loop=Q_first; loop; loop=loop->next)
    {
      switch(loop->type)
	{
	case APP_ASSOC:
	  sprintf(comminf, "%s %s", loop->app.assoc->app_exec,
		  loop->opt_data);
	  break;
	case APP_SHAPP:
	  sprintf(comminf, "%s", loop->app.shapp->commandline);
	  break;
	default:
	  strcpy(comminf, "Invalid type");
	}
      sprintf(buffer, "%d\t%s\t%d\t%s",
	      loop->kid, pidstates[loop->state], loop->exit_code, comminf);
	      
      DISP_message(Ctxt, buffer);
    }
}



/*
 * Function: FORK_grabkey
 *
 *   Examines CH and returns TRUE if there is a running shared app
 * which should view it, and then dispatches that key to the sub-process
 *
 * Returns:     int  - 
 * Parameters:  Ctxt - Context
 *              ch   - Character of ch
 *              uo   - Pointer to uo
 * History:
 * zappo   1/30/96    Created
 */
int FORK_grabkey(Ctxt, ch, uo)
     struct TalkContext *Ctxt;
     char ch;
     struct UserObject *uo;
{
  struct pid_node *loop;
  int returnme = 0;

  for(loop = Q_first;loop; loop = loop->next)
    {
      if((loop->type == APP_SHAPP) &&
	 loop->app.shapp->trapkeys &&
	 strchr(loop->app.shapp->trapkeys, ch) &&
	 (!uo || (uo == loop->pipes)))
	{
	  char buff[3];
	  sprintf(buff, "%c\n", ch);
	  GT_send(loop->pipes->outpipe, buff, 2);
	  DISP_message(Ctxt, "Sent message to subprocess.");
	  returnme++;
	}
    }
  return returnme;
}

/*
 * Function: FORK_try
 *
 *   Try to fork EXECSTRING with PARAMETER.  If execstring has spaces
 * in it, break it into tokens, and thereby fill in an array of
 * pointers representing the extra parameters. If IO is not specified,
 * then delete IO devices, otherwise delete all but IO.  If pipe, then
 * create and save a pipe to the child.
 *
 * Returns:     int  - the child process id.
 * Parameters:  Ctxt       - Context
 *              execstring - Pointer toCharacter of string
 *              parameter  - Pointer toCharacter of parameter
 *              io         - Pointer to io to keep
 *              intp       - Pipe fd for input of subprocess or 1
 *              outp       - Pipe fd for output of subprocess or 0
 * History:
 * zappo   11/8/95    Created
 */
int FORK_try(Ctxt, execstring, parameter, io, inp, outp)
     struct TalkContext *Ctxt;
     char *execstring;
     char *parameter;
     struct InputDevice *io;
     int   inp, outp;
{
  char *args[10];
  int i;
  pid_t kid;

  if(!execstring)
    return 0;

  i = 0;
  args[i++] = strtok(execstring, " \t");
  while(args[i-1])
    {
      args[i++] = strtok(NULL, " \t");
    }

  args[i-1] = parameter;
  args[i] = NULL;

  if(verbose)
    {
      int j;
      for(j=0;j<i;j++) printf("Argv[%d] = [%s]\n", j, args[j]);
      /* Push these to longs (needed for some systems...) */
      printf("Input = %ld Output = %ld\n", (long)inp, (long)outp);
    }

  if(i == 1)
    {
      return 0;
    }

  kid = fork();
  if(kid == 0)
    {
      /* Dup our pipes to the right places.  Leave stderr alone
       * in case the app needs it's errors displayed
       */
      if(inp && outp)
	{
	  close(0);
	  dup(inp);
	  close(1);
	  dup(outp);
	}

      /* Ah!  Lets nuke all extraneous IO devices */
      GT_close_all(Ctxt, io);

      /* We are child, EXEC! */
      execvp(args[0], args);

      fprintf(stdout, "Error execing child %s\n", args[0]);
      exit(1);			/* we failed! */
    }
  else if(kid == -1)
    {
      return 0;
    }

  /* Close the input/output devices which only child should be paying
   * attention to.
   */
  if(inp && outp)
    {
      close(inp);
      close(outp);
    }

  return kid;
}


/*
 * Function: handle_child
 *
 *   Handle dead children by doing waits and nice things like that.
 *
 * Returns:     RETSIGTYPE  - 
 * Parameters:  val - Number of val
 *
 * History:
 * zappo   11/9/95    Created
 */
RETSIGTYPE handle_child(val)
     int val;
{
  char             msgbuff[100];
  int              kidstat;
  int              kid;
  struct pid_node *loop;

  /* Get the dead process */
  kid = wait(&kidstat);
  
  /* Look up the process */
  for(loop = Q_first; loop && (loop->kid != kid); loop=loop->next);
  
  if(!loop)
    gtalk_shutdown("Child died and I don't know who it is!");

  /* Store exit code for posterity.  It's acutally signal values, and
   * exit() values, but discover that if we print it's value */
  loop->exit_code = kidstat;
  loop->state = Dead;

  /* Refresh the handler, in case we are on one of _those_ systems */
  signal(SIGCHLD, handle_child);

  sprintf(msgbuff, "Process [%d] exit (%d)", loop->kid, kidstat & 255);
  DISP_message(NULL, msgbuff);

#if RETSIGTYPE != void
  /* What are we supposed to return?  This is a guess. */
  return kidstat;
#endif
}

/*
 * Function: FORK_save_pid
 *
 *   Function which stores a pid which has just been launched.  This
 * lets us keep track of children as we recieve thier death signals.
 *
 * Returns:     static void  - 
 * Parameters:  pid   - Process id
 *              type  - Type of subprocess
 *              app   - the application definition
 *              opt   - String Pointer optional data
 *              pipes - Pointer to pipes
 * History:
 * zappo 11/9/95 Created 
 */
void FORK_save_pid(pid, type, app, opt, pipes)
     pid_t                pid;
     enum   app_def_type  type;
     union  app_def       app;
     char                *opt;
     struct UserObject   *pipes;
{
  struct pid_node *new;

  new = (struct pid_node *)malloc(sizeof (struct pid_node));

  new->kid       = pid;
  new->state     = Running;	/* assume we save cause it's running */
  new->type      = type;
  new->app       = app;		/* casn't doesn't really matter */
  new->exit_code = 0;
  if(opt)
    {
      new->opt_data = strdup(opt);
      if(!new->opt_data)
	gtalk_shutdown("strdup failed in save_pid");
    }
  new->pipes = pipes;
  if(Q_last) {
    Q_last->next = new;
  } else {
    Q_first = new;
  }
  new->prev = Q_last;
  new->next = NULL;
  Q_last = new;
  /* Lastly, set our signal handler now that we have a child process. */
  signal(SIGCHLD, handle_child);
}


/*
 * Function: FORK_nuke_child_processes
 *
 *   Locally defined function which will kill all child processes
 * still running.
 *
 * Returns:     static void  - 
 * Parameters:  Ctxt   - Context
 *              answer - Number of answer
 * History:
 * zappo   11/18/95   Created
 */
void FORK_nuke_child_processes(Ctxt)
     struct TalkContext *Ctxt;
{
  struct pid_node *loop;
  loop = Q_first;
  while(loop)
    {
      if(loop->state == Running)
	{
	  if(kill(loop->kid, SIGINT) == -1)
	    {
	      DISP_message(Ctxt, "Error sending signal!");
	      /* Should I do something else here? */
	    }
	  /* Mark that we tried to avoid an infinite loop */
	  loop->state = Killed;
	  /* The signal handler for SIGCHLD is going to be
	   * called a whole bunch now
	   */
	}
      loop = loop->next;
    }
}

/*
 * Function: FORK_kill_children
 *
 *   Kill any still running children.  Will ask if we really want to
 * kill them before actually doing it.
 *
 * Returns:     Nothing
 * Parameters:  Ctxt - Context
 *
 * History:
 * zappo   11/18/95   Created
 */
int FORK_kill_children(Ctxt)
     struct TalkContext *Ctxt;
{
  struct pid_node *loop;

  /* Find at least one running child */
  for(loop = Q_first; loop && (loop->state != Running); loop=loop->next);

  /* And ask the query */
  if(loop)
    {
      if(DISP_yes_or_no(Ctxt, "Some processes are active: Kill them?"))
	{
	  FORK_nuke_child_processes(Ctxt);
	}
    }

  return 0;
}

