/* interface.c --- creation of the UI objects  */

/* Copyright (c) E. Lassauge, 2000-2001.
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted,
 * provided that the above copyright notice appear in all copies and that
 * both that copyright notice and this permission notice appear in
 * supporting documentation.
 *
 * This file is provided AS IS with no warranties of any kind.  The author
 * shall have no liability with respect to the infringement of copyrights,
 * trade secrets or any patents by this file or any part thereof.  In no
 * event will the author be liable for any lost revenue or profits or
 * other special, indirect and consequential damages.
 *
 * mailto:lassauge@mail.dotcom.fr
 * http://lassauge.free.fr/
 *
 * REVISION HISTORY: see Changelog
 *
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <dirent.h>
#include <sys/stat.h>
#include <unistd.h>

#include <gnome.h>

#include "sms.h"
#include "callbacks.h"
#include "interface.h"

#include "support.h"

/* Common widgets */
GtkWidget   *About = (GtkWidget *)NULL;
GtkWidget   *Scripts_fs = (GtkWidget *)NULL;
GtkWidget   *Preference = (GtkWidget *)NULL;
GtkWidget   *Alias = (GtkWidget *)NULL;
GtkWidget   *Scripts = (GtkWidget *)NULL;
GtkWidget   *GnomeSmssend = (GtkWidget *)NULL;
GtkWidget   *Notebook = (GtkWidget *)NULL;
GtkWidget   *Appbar = (GtkWidget *)NULL;
GtkTooltips *Tooltips = (GtkTooltips *)NULL;

/* Local functions */
static void scan_dir(gchar *dirname);
static void update_clist(gpointer key,
                  gpointer value,
                  gpointer user_data);

GtkWidget *create_about()
{
    /* The about box */
    const gchar *authors[] = {
	"Eric Lassauge <lassauge@mail.dotcom.fr>",
	NULL
    };
    GtkWidget *about;

    about = gnome_about_new("Gnome SmsSend",
                            VERSION,
			    _("Copyright 2000-2001"), 
                            authors,
                            _("GNOME interface to SmsSend"), 
#ifndef DEVEL
			    "gsmssend/gsms-about.png"
#else
			    "../pixmaps/gsms-about.png"
#endif
			    );
    gtk_window_set_modal(GTK_WINDOW(about), TRUE);

    return about;
}

GtkWidget *create_scripts_file_selection()
{
    GtkWidget *scripts;
    /* create a script file selection dialog */

    scripts = gtk_file_selection_new(_("Choose script"));
    gtk_signal_connect (GTK_OBJECT(scripts), "destroy",
		(GtkSignalFunc) gtk_widget_destroy, GTK_OBJECT (scripts));
    gtk_signal_connect(GTK_OBJECT (GTK_FILE_SELECTION (scripts)->ok_button),
		"clicked", (GtkSignalFunc) on_script_ok_pressed, scripts);
    gtk_signal_connect_object(GTK_OBJECT (GTK_FILE_SELECTION
		    (scripts)->cancel_button),
		"clicked", (GtkSignalFunc) gtk_widget_destroy,
		GTK_OBJECT (scripts));
    return scripts;
}

GtkWidget *create_preferences()
{
    /* create the preference dialog box */
    GtkWidget *preferences;
    GtkWidget *dialog_vbox;
    GtkWidget *main_vbox;
    GtkWidget *pref_proxy;
    GtkWidget *proxy_label;
    GtkWidget *use_proxy;
    GtkWidget *proxy_table;
    GtkWidget *proxyhost_label;
    GtkWidget *proxyport_label;
    GtkWidget *proxy_host;
    GtkWidget *proxy_port;
    GtkWidget *proxyuser_label;
    GtkWidget *proxypass_label;
    GtkWidget *proxy_user;
    GtkWidget *proxy_password;
    GtkWidget *config_vbox;
    GtkWidget *config_label;
    GtkWidget *config_hbox1;
    GtkWidget *config_hbox2;
    GtkWidget *debug_label;
    GtkObject *debug_level_adj;
    GtkWidget *debug_level;
    GtkWidget *timeout_label;
    GtkObject *timeout_level_adj;
    GtkWidget *timeout_level;
    GtkWidget *auto_check;
    GtkWidget *keep_pass;
    GtkWidget *header_table;
    GtkWidget *header_label;
    GtkWidget *fileentry;
    GtkWidget *header_file;
    GtkWidget *usage_label;
    GtkWidget *usage_hbox1;
    GtkWidget *delay_send;
    GtkWidget *retry_label;
    GtkObject *retries_level_adj;
    GtkWidget *retries_level;
    GtkWidget *dialog_action_area;
    GtkWidget *ok_button;
    GtkWidget *apply_button;
    GtkWidget *cancel_button;

    preferences = gnome_dialog_new(_("Preferences"), NULL);
    gnome_dialog_set_close(GNOME_DIALOG(preferences), TRUE);
    gtk_window_set_policy(GTK_WINDOW(preferences), FALSE, FALSE, FALSE);
    gtk_window_set_wmclass(GTK_WINDOW(preferences), PACKAGE, PACKAGE);

    dialog_vbox = GNOME_DIALOG(preferences)->vbox;
    gtk_widget_show(dialog_vbox);

    main_vbox = gtk_vbox_new(FALSE, 0);
    gtk_widget_show(main_vbox);
    gtk_box_pack_start(GTK_BOX(dialog_vbox), main_vbox, TRUE, TRUE, 0);

    proxy_label = gtk_frame_new(_("Proxy configuration"));
    gtk_widget_show(proxy_label);
    gtk_box_pack_start(GTK_BOX(main_vbox), proxy_label, TRUE, TRUE, 0);
    gtk_container_set_border_width(GTK_CONTAINER(proxy_label), 2);
    gtk_frame_set_label_align(GTK_FRAME(proxy_label), 0.02,0.5);

    pref_proxy = gtk_vbox_new(FALSE, 2);
    gtk_widget_set_name(pref_proxy, "pref_proxy");
    gtk_object_set_data(GTK_OBJECT(preferences), "pref_proxy", pref_proxy);
    gtk_widget_show(pref_proxy);
    gtk_container_add(GTK_CONTAINER(proxy_label), pref_proxy);
    gtk_container_set_border_width(GTK_CONTAINER(pref_proxy), 2);

    use_proxy = gtk_check_button_new_with_label(_("Use proxy"));
    gtk_widget_set_name(use_proxy, "use_proxy");
    gtk_object_set_data(GTK_OBJECT(preferences), "use_proxy", use_proxy);
    gtk_widget_show(use_proxy);
    gtk_box_pack_start(GTK_BOX(pref_proxy), use_proxy, FALSE, FALSE, 0);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(use_proxy), UseProxy);
    gtk_tooltips_set_tip(Tooltips, use_proxy, _("Check if you need to use a proxy server"), NULL);

    proxy_table = gtk_table_new(4, 2, FALSE);
    gtk_widget_show(proxy_table);
    gtk_box_pack_end(GTK_BOX(pref_proxy), proxy_table, FALSE, FALSE, 0);
    gtk_table_set_row_spacings(GTK_TABLE(proxy_table), 1);
    gtk_table_set_col_spacings(GTK_TABLE(proxy_table), 2);

    proxyhost_label = gtk_label_new(_("Proxy Host"));
    gtk_widget_show(proxyhost_label);
    gtk_table_attach(GTK_TABLE(proxy_table), proxyhost_label, 0, 1, 0, 1, (GtkAttachOptions) (0), (GtkAttachOptions) (0), 0, 0);
    gtk_label_set_justify(GTK_LABEL(proxyhost_label), GTK_JUSTIFY_RIGHT);

    proxyport_label = gtk_label_new(_("Proxy Port"));
    gtk_widget_show(proxyport_label);
    gtk_table_attach(GTK_TABLE(proxy_table), proxyport_label, 0, 1, 1, 2, (GtkAttachOptions) (0), (GtkAttachOptions) (0), 0, 0);
    gtk_label_set_justify(GTK_LABEL(proxyport_label), GTK_JUSTIFY_RIGHT);

    proxy_host = gtk_entry_new_with_max_length(P_LENGTH-1);
    gtk_widget_set_name(proxy_host, "proxy_host");
    gtk_object_set_data(GTK_OBJECT(preferences), "proxy_host", proxy_host);
    gtk_widget_show(proxy_host);
    gtk_table_attach(GTK_TABLE(proxy_table), proxy_host, 1, 2, 0, 1,
		     (GtkAttachOptions) (GTK_EXPAND | GTK_FILL), (GtkAttachOptions) (0), 0, 0);
    gtk_tooltips_set_tip(Tooltips, proxy_host, _("Name or IP address of proxy server"), NULL);
    if (UseProxy)
	gtk_entry_set_text(GTK_ENTRY(proxy_host), ProxyHost);

    proxy_port = gtk_entry_new_with_max_length(4);
    gtk_widget_set_name(proxy_port, "proxy_port");
    gtk_object_set_data(GTK_OBJECT(preferences), "proxy_port", proxy_port);
    gtk_widget_show(proxy_port);
    gtk_table_attach(GTK_TABLE(proxy_table), proxy_port, 1, 2, 1, 2,
		     (GtkAttachOptions) (GTK_EXPAND | GTK_FILL), (GtkAttachOptions) (0), 0, 0);
    gtk_tooltips_set_tip(Tooltips, proxy_port, _("Port number for the proxy (usually 80 or 8080)"), NULL);
    if (UseProxy)
    {
	char port[5];
	g_snprintf(port, 5, "%d", ProxyPort);
	gtk_entry_set_text(GTK_ENTRY(proxy_port), port);
    }

    proxyuser_label = gtk_label_new(_("Proxy User"));
    gtk_widget_show(proxyuser_label);
    gtk_table_attach(GTK_TABLE(proxy_table), proxyuser_label, 0, 1, 2, 3, (GtkAttachOptions) (0), (GtkAttachOptions) (0), 0, 0);
    gtk_label_set_justify(GTK_LABEL(proxyuser_label), GTK_JUSTIFY_RIGHT);

    proxypass_label = gtk_label_new(_("Proxy Password"));
    gtk_widget_show(proxypass_label);
    gtk_table_attach(GTK_TABLE(proxy_table), proxypass_label, 0, 1, 3, 4, (GtkAttachOptions) (0), (GtkAttachOptions) (0), 0, 0);
    gtk_label_set_justify(GTK_LABEL(proxypass_label), GTK_JUSTIFY_RIGHT);

    proxy_user = gtk_entry_new_with_max_length(P_LENGTH-1);
    gtk_widget_set_name(proxy_user, "proxy_user");
    gtk_object_set_data(GTK_OBJECT(preferences), "proxy_user", proxy_user);
    gtk_widget_show(proxy_user);
    gtk_table_attach(GTK_TABLE(proxy_table), proxy_user, 1, 2, 2, 3,
		     (GtkAttachOptions) (GTK_EXPAND | GTK_FILL), (GtkAttachOptions) (0), 0, 0);
    gtk_tooltips_set_tip(Tooltips, proxy_user, _("User name for proxy connection if needed"), NULL);
    if (UseProxy)
	gtk_entry_set_text(GTK_ENTRY(proxy_user), ProxyUser);

    proxy_password = gtk_entry_new_with_max_length(P_LENGTH-1);
    gtk_widget_set_name(proxy_password, "proxy_password");
    gtk_object_set_data(GTK_OBJECT(preferences), "proxy_password", proxy_password);
    gtk_widget_show(proxy_password);
    gtk_table_attach(GTK_TABLE(proxy_table), proxy_password, 1, 2, 3, 4,
		     (GtkAttachOptions) (GTK_EXPAND | GTK_FILL), (GtkAttachOptions) (0), 0, 0);
    gtk_entry_set_visibility(GTK_ENTRY(proxy_password), FALSE);
    gtk_tooltips_set_tip(Tooltips, proxy_password, _("Password for proxy connection if needed"), NULL);
    if (UseProxy)
	gtk_entry_set_text(GTK_ENTRY(proxy_password), ProxyPass);

    config_label = gtk_frame_new(_("SmsSend Configuration"));
    gtk_widget_show(config_label);
    gtk_box_pack_start(GTK_BOX(main_vbox), config_label, TRUE, TRUE, 0);
    gtk_container_set_border_width(GTK_CONTAINER(config_label), 2);
    gtk_frame_set_label_align(GTK_FRAME(config_label), 0.02,0.5);

    config_vbox = gtk_vbox_new(FALSE, 3);
    gtk_widget_show(config_vbox);
    gtk_container_add(GTK_CONTAINER(config_label), config_vbox);
    gtk_container_set_border_width(GTK_CONTAINER(config_vbox), 2);

    config_hbox1 = gtk_hbox_new(FALSE, 2);
    gtk_widget_show(config_hbox1);
    gtk_box_pack_start(GTK_BOX(config_vbox), config_hbox1, TRUE, TRUE, 0);
    gtk_container_set_border_width(GTK_CONTAINER(config_hbox1), 2);

    debug_label = gtk_label_new(_("Debug Level"));
    gtk_widget_show(debug_label);
    gtk_box_pack_start(GTK_BOX(config_hbox1), debug_label, FALSE, FALSE, 0);
    gtk_label_set_justify(GTK_LABEL(debug_label), GTK_JUSTIFY_RIGHT);

    debug_level_adj = gtk_adjustment_new(DebugLevel, 0, 15, 1, 5, 5);
    debug_level = gtk_spin_button_new(GTK_ADJUSTMENT(debug_level_adj), 1, 0);
    gtk_widget_set_name(debug_level, "debug_level");
    gtk_object_set_data(GTK_OBJECT(preferences), "debug_level", debug_level);
    gtk_widget_show(debug_level);
    gtk_box_pack_start(GTK_BOX(config_hbox1), debug_level, FALSE, FALSE, 0);
    gtk_spin_button_set_numeric(GTK_SPIN_BUTTON(debug_level), TRUE);
    gtk_tooltips_set_tip(Tooltips, debug_level, _("Set debug level of smssend"), NULL);

    timeout_label = gtk_label_new(_("Timeout"));
    gtk_widget_show(timeout_label);
    gtk_box_pack_start(GTK_BOX(config_hbox1), timeout_label, FALSE, FALSE, 0);
    gtk_label_set_justify(GTK_LABEL(timeout_label), GTK_JUSTIFY_RIGHT);

    timeout_level_adj = gtk_adjustment_new(TimeOut, 0, 1000, 10, 100, 100);
    timeout_level = gtk_spin_button_new(GTK_ADJUSTMENT(timeout_level_adj), 1, 0);
    gtk_widget_set_name(timeout_level, "timeout");
    gtk_object_set_data(GTK_OBJECT(preferences), "timeout", timeout_level);
    gtk_widget_show(timeout_level);
    gtk_box_pack_start(GTK_BOX(config_hbox1), timeout_level, FALSE, FALSE, 0);
    gtk_spin_button_set_numeric(GTK_SPIN_BUTTON(timeout_level), TRUE);
    gtk_tooltips_set_tip(Tooltips, timeout_level, _("Set timeout value for smssend"), NULL);

    config_hbox2 = gtk_hbox_new(FALSE, 2);
    gtk_widget_show(config_hbox2);
    gtk_box_pack_start(GTK_BOX(config_vbox), config_hbox2, FALSE, FALSE, 0);
    gtk_container_set_border_width(GTK_CONTAINER(config_hbox2), 2);

    auto_check = gtk_check_button_new_with_label(_("Auto check new version of scripts"));
    gtk_widget_set_name(auto_check, "auto_check");
    gtk_object_set_data(GTK_OBJECT(preferences), "auto_check", auto_check);
    gtk_widget_show(auto_check);
    gtk_box_pack_start(GTK_BOX(config_hbox2), auto_check, FALSE, FALSE, 0);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(auto_check), AutoCheck);

    keep_pass = gtk_check_button_new_with_label(_("Keep passwords"));
    gtk_widget_set_name(keep_pass, "keep_pass");
    gtk_object_set_data(GTK_OBJECT(preferences), "keep_pass", keep_pass);
    gtk_widget_show(keep_pass);
    gtk_box_pack_start(GTK_BOX(config_hbox2), keep_pass, FALSE, FALSE, 0);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(keep_pass), KeepPass);
    gtk_tooltips_set_tip(Tooltips, keep_pass, _("Check if you want to keep provider's password in the preferences file"), NULL);

    header_table = gtk_table_new(1, 2, FALSE);
    gtk_widget_show(header_table);
    gtk_box_pack_start(GTK_BOX(config_vbox), header_table, FALSE, FALSE, 0);
    gtk_table_set_row_spacings(GTK_TABLE(header_table), 1);
    gtk_table_set_col_spacings(GTK_TABLE(header_table), 2);

    header_label = gtk_label_new(_("Header file: "));
    gtk_widget_show(header_label);
    gtk_table_attach(GTK_TABLE(header_table), header_label, 0, 1, 0, 1, (GtkAttachOptions) (0), (GtkAttachOptions) (0), 0, 0);

    fileentry = gnome_file_entry_new (NULL, _("Header file: "));
    gtk_widget_show (fileentry);
    gtk_container_set_border_width (GTK_CONTAINER (fileentry), 2);

    header_file = gnome_file_entry_gtk_entry (GNOME_FILE_ENTRY (fileentry));
    gtk_widget_show (header_file);
    gtk_table_attach(GTK_TABLE(header_table), fileentry, 1, 2, 0, 1,
		     (GtkAttachOptions) (GTK_EXPAND | GTK_FILL), (GtkAttachOptions) (0), 0, 0);
    gtk_object_set_data(GTK_OBJECT(preferences), "header_file", header_file);
    gtk_tooltips_set_tip(Tooltips, header_file, _("Header file for smssend"), NULL);

    usage_label = gtk_frame_new(_("GNOME SmsSend Configuration"));
    gtk_widget_show(usage_label);
    gtk_box_pack_start(GTK_BOX(main_vbox), usage_label, TRUE, TRUE, 0);
    gtk_container_set_border_width(GTK_CONTAINER(usage_label), 2);
    gtk_frame_set_label_align(GTK_FRAME(usage_label), 0.02,0.5);

    usage_hbox1 = gtk_hbox_new(FALSE, 0);
    gtk_widget_show(usage_hbox1);
    gtk_container_add(GTK_CONTAINER(usage_label), usage_hbox1);
    gtk_container_set_border_width(GTK_CONTAINER(usage_hbox1), 2);

    delay_send = gtk_check_button_new_with_label(_("Delay SmsSend calls"));
    gtk_widget_set_name(delay_send, "delay_send");
    gtk_object_set_data(GTK_OBJECT(preferences), "delay_send", delay_send);
    gtk_widget_show(delay_send);
    gtk_box_pack_start(GTK_BOX(usage_hbox1), delay_send, FALSE, FALSE, 0);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(delay_send), DelaySend);
    gtk_tooltips_set_tip(Tooltips, delay_send, _("Check if you want to postpone all SmsSend calls"), NULL);

    retry_label = gtk_label_new(_("Retries"));
    gtk_widget_show(retry_label);
    gtk_box_pack_start(GTK_BOX(usage_hbox1), retry_label, FALSE, FALSE, 0);
    gtk_label_set_justify(GTK_LABEL(retry_label), GTK_JUSTIFY_RIGHT);

    retries_level_adj = gtk_adjustment_new(Retries, 1, 10, 1, 10, 10);
    retries_level = gtk_spin_button_new(GTK_ADJUSTMENT(retries_level_adj), 1, 0);
    gtk_widget_set_name(retries_level, "retries");
    gtk_object_set_data(GTK_OBJECT(preferences), "retries", retries_level);
    gtk_widget_show(retries_level);
    gtk_box_pack_start(GTK_BOX(usage_hbox1), retries_level, FALSE, FALSE, 0);
    gtk_spin_button_set_numeric(GTK_SPIN_BUTTON(retries_level), TRUE);
    gtk_tooltips_set_tip(Tooltips, retries_level, _("Set number of retries for SmsSend"), NULL);

    dialog_action_area = GNOME_DIALOG(preferences)->action_area;
    gtk_widget_show(dialog_action_area);
    gtk_button_box_set_layout(GTK_BUTTON_BOX(dialog_action_area), GTK_BUTTONBOX_END);
    gtk_button_box_set_spacing(GTK_BUTTON_BOX(dialog_action_area), 8);

    gnome_dialog_append_button(GNOME_DIALOG(preferences), GNOME_STOCK_BUTTON_OK);
    ok_button = g_list_last(GNOME_DIALOG(preferences)->buttons)->data;
    gtk_widget_show(ok_button);
    GTK_WIDGET_SET_FLAGS(ok_button, GTK_CAN_DEFAULT);

    gnome_dialog_append_button(GNOME_DIALOG(preferences), GNOME_STOCK_BUTTON_APPLY);
    apply_button = g_list_last(GNOME_DIALOG(preferences)->buttons)->data;
    gtk_widget_show(apply_button);

    gnome_dialog_append_button(GNOME_DIALOG(preferences), GNOME_STOCK_BUTTON_CANCEL);
    cancel_button = g_list_last(GNOME_DIALOG(preferences)->buttons)->data;
    gtk_widget_show(cancel_button);

    gtk_signal_connect(GTK_OBJECT(ok_button),
		      "pressed",
		      GTK_SIGNAL_FUNC(on_pref_ok_pressed),
		      (gpointer)OK_PRESSED);
    gtk_signal_connect(GTK_OBJECT(apply_button),
		      "pressed",
		      GTK_SIGNAL_FUNC(on_pref_ok_pressed),
		      (gpointer)APPLY_PRESSED);

    return preferences;
}

static void update_clist(gpointer key,
                  gpointer value,
                  gpointer user_data)
{
    /* Local function for updating the clist on each alias */
    GtkWidget *clist = (GtkWidget *)user_data;
    gchar *tmp[ALIAS_CLIST_COLS];

    tmp[KEY_COL] = (gchar *) key;
    tmp[VAL_COL] = (gchar *) value;
    gtk_clist_append (GTK_CLIST(clist),tmp);
}

GtkWidget *create_aliases(void)
{
    /* create the alias dialog box */
    GtkWidget *aliases;
    GtkWidget *dialog_vbox;
    GtkWidget *dialog_hbox;
    GtkWidget *list_vbox;
    GtkWidget *scrolledwindow;
    GtkWidget *clist;
    GtkWidget *key;
    GtkWidget *value;
    GtkWidget *list_buttonbox;
    GtkWidget *up_button;
    GtkWidget *down_button;
    GtkWidget *right_vbox;
    GtkWidget *table;
    GtkWidget *key_label;
    GtkWidget *value_label;
    GtkWidget *key_entry;
    GtkWidget *value_entry;
    GtkWidget *command_buttonbox;
    GtkWidget *add;
    GtkWidget *delete;
    GtkWidget *dialog_action_area1;
    GtkWidget *ok_button;
    GtkWidget *apply_button;
    GtkWidget *cancel_button;
  
    aliases = gnome_dialog_new (_("Alias"), NULL);
    gtk_window_set_position (GTK_WINDOW (aliases), GTK_WIN_POS_CENTER);
    gtk_window_set_policy (GTK_WINDOW (aliases), FALSE, FALSE, FALSE);
    gnome_dialog_set_close (GNOME_DIALOG (aliases), TRUE);
    gnome_dialog_close_hides (GNOME_DIALOG (aliases), TRUE);
    gtk_window_set_wmclass(GTK_WINDOW(aliases), "Smssend", "Smssend");
  
    dialog_vbox = GNOME_DIALOG (aliases)->vbox;
    gtk_widget_show (dialog_vbox);
  
    dialog_hbox = gtk_hbox_new (FALSE, 0);
    gtk_widget_show (dialog_hbox);
    gtk_box_pack_start (GTK_BOX (dialog_vbox), dialog_hbox, FALSE, FALSE, 0);
  
    list_vbox = gtk_vbox_new (FALSE, 0);
    gtk_widget_show (list_vbox);
    gtk_box_pack_start (GTK_BOX (dialog_hbox), list_vbox, TRUE, TRUE, 0);
  
    scrolledwindow = gtk_scrolled_window_new (NULL, NULL);
    gtk_widget_show (scrolledwindow);
    gtk_box_pack_start (GTK_BOX (list_vbox), scrolledwindow, TRUE, TRUE, 2);
    gtk_widget_set_usize (scrolledwindow, -2, 250);
    gtk_container_set_border_width (GTK_CONTAINER (scrolledwindow), 2);
    gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolledwindow), GTK_POLICY_ALWAYS, GTK_POLICY_AUTOMATIC);
  
    clist = gtk_clist_new (ALIAS_CLIST_COLS);
    gtk_object_set_data(GTK_OBJECT(aliases), "alias_clist", clist);
    gtk_widget_show (clist);
    gtk_container_add (GTK_CONTAINER (scrolledwindow), clist);
    gtk_clist_set_column_width (GTK_CLIST (clist), KEY_COL, 80);
    gtk_clist_set_column_width (GTK_CLIST (clist), VAL_COL, 80);
    gtk_clist_column_titles_show (GTK_CLIST (clist));
    gtk_clist_set_shadow_type (GTK_CLIST (clist), GTK_SHADOW_ETCHED_IN);
  
    key = gtk_label_new (_("Key"));
    gtk_widget_show (key);
    gtk_clist_set_column_widget (GTK_CLIST (clist), KEY_COL, key);
    gtk_label_set_justify (GTK_LABEL (key), GTK_JUSTIFY_LEFT);
  
    value = gtk_label_new (_("Value"));
    gtk_widget_show (value);
    gtk_clist_set_column_widget (GTK_CLIST (clist), VAL_COL, value);
    gtk_label_set_justify (GTK_LABEL (value), GTK_JUSTIFY_LEFT);
  
    gtk_clist_column_titles_passive(GTK_CLIST (clist));

    list_buttonbox = gtk_hbutton_box_new ();
    gtk_widget_show (list_buttonbox);
    gtk_box_pack_start (GTK_BOX (list_vbox), list_buttonbox, FALSE, FALSE, 2);
    gtk_container_set_border_width (GTK_CONTAINER (list_buttonbox), 2);
    gtk_button_box_set_spacing (GTK_BUTTON_BOX (list_buttonbox), 8);
    gtk_button_box_set_child_size (GTK_BUTTON_BOX (list_buttonbox), 80, 20);
    gtk_button_box_set_child_ipadding (GTK_BUTTON_BOX (list_buttonbox), 2, 2);
  
    up_button = gnome_stock_button (GNOME_STOCK_BUTTON_UP);
    gtk_widget_show (up_button);
    gtk_container_add (GTK_CONTAINER (list_buttonbox), up_button);
  
    down_button = gnome_stock_button (GNOME_STOCK_BUTTON_DOWN);
    gtk_widget_show (down_button);
    gtk_container_add (GTK_CONTAINER (list_buttonbox), down_button);
  
    right_vbox = gtk_vbox_new (FALSE, 0);
    gtk_widget_show (right_vbox);
    gtk_box_pack_start (GTK_BOX (dialog_hbox), right_vbox, TRUE, TRUE, 0);
  
    table = gtk_table_new (2, 2, FALSE);
    gtk_widget_show (table);
    gtk_box_pack_start (GTK_BOX (right_vbox), table, TRUE, FALSE, 8);
    gtk_container_set_border_width (GTK_CONTAINER (table), 2);
    gtk_table_set_row_spacings (GTK_TABLE (table), 2);
    gtk_table_set_col_spacings (GTK_TABLE (table), 2);
  
    key_label = gtk_label_new (_("Key : "));
    gtk_widget_show (key_label);
    gtk_table_attach (GTK_TABLE (table), key_label, 0, 1, 0, 1,
                      (GtkAttachOptions) (0),
                      (GtkAttachOptions) (0), 0, 0);
    gtk_label_set_justify (GTK_LABEL (key_label), GTK_JUSTIFY_LEFT);
  
    value_label = gtk_label_new (_("Value : "));
    gtk_widget_show (value_label);
    gtk_table_attach (GTK_TABLE (table), value_label, 0, 1, 1, 2,
                      (GtkAttachOptions) (0),
                      (GtkAttachOptions) (0), 0, 0);
    gtk_label_set_justify (GTK_LABEL (value_label), GTK_JUSTIFY_LEFT);
  
    key_entry = gtk_entry_new ();
    gtk_widget_show (key_entry);
    gtk_object_set_data(GTK_OBJECT(aliases), "alias_key", key_entry);
    gtk_table_attach (GTK_TABLE (table), key_entry, 1, 2, 0, 1,
                      (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
                      (GtkAttachOptions) (0), 0, 0);
  
    value_entry = gtk_entry_new ();
    gtk_widget_show (value_entry);
    gtk_object_set_data(GTK_OBJECT(aliases), "alias_value", value_entry);
    gtk_table_attach (GTK_TABLE (table), value_entry, 1, 2, 1, 2,
                      (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
                      (GtkAttachOptions) (0), 0, 0);
  
    command_buttonbox = gtk_hbutton_box_new ();
    gtk_widget_show (command_buttonbox);
    gtk_box_pack_start (GTK_BOX (right_vbox), command_buttonbox, TRUE, FALSE, 100);
    gtk_container_set_border_width (GTK_CONTAINER (command_buttonbox), 2);
    gtk_button_box_set_layout (GTK_BUTTON_BOX (command_buttonbox), GTK_BUTTONBOX_EDGE);
    gtk_button_box_set_spacing (GTK_BUTTON_BOX (command_buttonbox), 8);
    gtk_button_box_set_child_size (GTK_BUTTON_BOX (command_buttonbox), 80, 20);
    gtk_button_box_set_child_ipadding (GTK_BUTTON_BOX (command_buttonbox), 20, 2);
  
    add = gtk_button_new_with_label (_("Add"));
    gtk_widget_show (add);
    gtk_container_add (GTK_CONTAINER (command_buttonbox), add);
    gtk_tooltips_set_tip(Tooltips, add, _("Add current entries to the list"), NULL);
  
    delete = gtk_button_new_with_label (_("Delete"));
    gtk_widget_show (delete);
    gtk_container_add (GTK_CONTAINER (command_buttonbox), delete);
    gtk_tooltips_set_tip(Tooltips, delete, _("Delete selected row in the list"), NULL);
  
    dialog_action_area1 = GNOME_DIALOG (aliases)->action_area;
    gtk_button_box_set_layout (GTK_BUTTON_BOX (dialog_action_area1), GTK_BUTTONBOX_END);
    gtk_button_box_set_spacing (GTK_BUTTON_BOX (dialog_action_area1), 8);
  
    gnome_dialog_append_button (GNOME_DIALOG (aliases), GNOME_STOCK_BUTTON_OK);
    ok_button = g_list_last (GNOME_DIALOG (aliases)->buttons)->data;
    gtk_widget_show (ok_button);
    GTK_WIDGET_SET_FLAGS (ok_button, GTK_CAN_DEFAULT);
  
    gnome_dialog_append_button (GNOME_DIALOG (aliases), GNOME_STOCK_BUTTON_APPLY);
    apply_button = g_list_last (GNOME_DIALOG (aliases)->buttons)->data;
    gtk_widget_show (apply_button);
  
    cancel_button = g_list_last (GNOME_DIALOG (aliases)->buttons)->data;
    gtk_widget_show (cancel_button);

    /* Connect signals */
  
    gtk_signal_connect (GTK_OBJECT (clist), "select_row",
                        GTK_SIGNAL_FUNC (on_alias_clist_select_row),
                        (gpointer)SEL_PRESSED);
    gtk_signal_connect (GTK_OBJECT (clist), "unselect_row",
                        GTK_SIGNAL_FUNC (on_alias_clist_select_row),
                        (gpointer)UNSEL_PRESSED);
    gtk_signal_connect (GTK_OBJECT (up_button), "clicked",
                        GTK_SIGNAL_FUNC (on_updown_clicked),
                        (gpointer)UP_PRESSED);
    gtk_signal_connect (GTK_OBJECT (down_button), "clicked",
                        GTK_SIGNAL_FUNC (on_updown_clicked),
                        (gpointer)DOWN_PRESSED);
    gtk_signal_connect (GTK_OBJECT (add), "clicked",
                        GTK_SIGNAL_FUNC (on_add_clicked),
                        NULL);
    gtk_signal_connect (GTK_OBJECT (delete), "clicked",
                        GTK_SIGNAL_FUNC (on_delete_clicked),
                        NULL);
  
    gtk_signal_connect(GTK_OBJECT(ok_button),
                      "pressed",
                      GTK_SIGNAL_FUNC(on_alias_ok_pressed),
                      (gpointer)OK_PRESSED);
    gtk_signal_connect(GTK_OBJECT(apply_button),
                      "pressed",
                      GTK_SIGNAL_FUNC(on_alias_ok_pressed),
                      (gpointer)APPLY_PRESSED);

    /* update the clist */
    if (AliasHash)
    {
	gtk_clist_freeze(GTK_CLIST(clist));
	g_hash_table_foreach(AliasHash, update_clist,(gpointer)clist);
	gtk_clist_thaw(GTK_CLIST(clist));
    }

    return aliases;
}

GtkWidget *create_scripts(void)
{
    /* create the scripts dialog box */
  GtkWidget *scripts;
  GtkWidget *dialog_vbox;
  GtkWidget *scrolledwindow;
  GtkWidget *script_clist;
  GtkWidget *filename_col;
  GtkWidget *use_col;
  GtkWidget *dialog_action_area;
  GtkWidget *button_val;
  GtkWidget *button_apply;
  GtkWidget *button_cancel;

  scripts = gnome_dialog_new (_("Scripts"), NULL);
  gtk_window_set_policy (GTK_WINDOW (scripts), FALSE, FALSE, FALSE);

  dialog_vbox = GNOME_DIALOG (scripts)->vbox;
  gtk_widget_show (dialog_vbox);

  scrolledwindow = gtk_scrolled_window_new (NULL, NULL);
  gtk_widget_show (scrolledwindow);
  gtk_box_pack_start (GTK_BOX (dialog_vbox), scrolledwindow, TRUE, TRUE, 0);
  gtk_container_set_border_width (GTK_CONTAINER (scrolledwindow), 2);
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolledwindow), GTK_POLICY_ALWAYS, GTK_POLICY_AUTOMATIC);
  gtk_widget_set_usize (scrolledwindow, 0, 256);

  script_clist = gtk_clist_new (SCRIPT_CLIST_COLS);
  gtk_object_set_data (GTK_OBJECT (scripts), "script_clist", script_clist);
  gtk_widget_show (script_clist);
  gtk_container_add (GTK_CONTAINER (scrolledwindow), script_clist);
  gtk_clist_set_column_width (GTK_CLIST (script_clist), SCRIPT_COL, 256);
  gtk_clist_set_column_width (GTK_CLIST (script_clist), SHOW_COL, 64);
  gtk_clist_set_column_width (GTK_CLIST (script_clist), SAVE_COL, 64);
  gtk_clist_set_selection_mode (GTK_CLIST (script_clist), GTK_SELECTION_MULTIPLE);
  gtk_clist_column_titles_show (GTK_CLIST (script_clist));
  gtk_clist_set_column_justification (GTK_CLIST (script_clist),SHOW_COL,GTK_JUSTIFY_CENTER);
  gtk_clist_set_column_justification (GTK_CLIST (script_clist),SAVE_COL,GTK_JUSTIFY_CENTER);

  filename_col = gtk_label_new (_("Script"));
  gtk_widget_show (filename_col);
  gtk_clist_set_column_widget (GTK_CLIST (script_clist), SCRIPT_COL, filename_col);
  gtk_misc_set_padding (GTK_MISC (filename_col), 2, 2);

  use_col = gtk_label_new (_("Show"));
  gtk_widget_show (use_col);
  gtk_clist_set_column_widget (GTK_CLIST (script_clist), SHOW_COL, use_col);
  gtk_misc_set_padding (GTK_MISC (use_col), 2, 2);

  use_col = gtk_label_new (_("Save"));
  gtk_widget_show (use_col);
  gtk_clist_set_column_widget (GTK_CLIST (script_clist), SAVE_COL, use_col);
  gtk_misc_set_padding (GTK_MISC (use_col), 2, 2);

  dialog_action_area = GNOME_DIALOG (scripts)->action_area;
  gtk_widget_show (dialog_action_area);
  gtk_button_box_set_layout (GTK_BUTTON_BOX (dialog_action_area), GTK_BUTTONBOX_END);
  gtk_button_box_set_spacing (GTK_BUTTON_BOX (dialog_action_area), 8);

  gnome_dialog_append_button (GNOME_DIALOG (scripts), GNOME_STOCK_BUTTON_OK);
  button_val = g_list_last (GNOME_DIALOG (scripts)->buttons)->data;
  gtk_widget_show (button_val);
  gtk_tooltips_set_tip(Tooltips, button_val, _("Save scripts in user file"), NULL);

  gnome_dialog_append_button (GNOME_DIALOG (scripts), GNOME_STOCK_BUTTON_APPLY);
  button_apply = g_list_last (GNOME_DIALOG (scripts)->buttons)->data;
  gtk_widget_show (button_apply);
  gtk_tooltips_set_tip(Tooltips, button_apply, _("Change show status"), NULL);
  GTK_WIDGET_SET_FLAGS (button_apply, GTK_CAN_DEFAULT);

  gnome_dialog_append_button (GNOME_DIALOG (scripts), GNOME_STOCK_BUTTON_CANCEL);
  button_cancel = g_list_last (GNOME_DIALOG (scripts)->buttons)->data;
  gtk_widget_show (button_cancel);

  gtk_signal_connect (GTK_OBJECT (button_val), "pressed",
                      GTK_SIGNAL_FUNC (on_script_val_pressed),
                      (gpointer)OK_PRESSED);
  gtk_signal_connect (GTK_OBJECT (button_apply), "pressed",
                      GTK_SIGNAL_FUNC (on_script_val_pressed),
                      (gpointer)APPLY_PRESSED);
  gtk_signal_connect (GTK_OBJECT (button_cancel), "pressed",
                      GTK_SIGNAL_FUNC (on_script_val_pressed),
                      (gpointer)CANCEL_PRESSED);

  gtk_signal_connect (GTK_OBJECT (script_clist), "click_column",
                      GTK_SIGNAL_FUNC (on_script_clist_click_column),
                      NULL);
  gtk_signal_connect (GTK_OBJECT (script_clist), "select_row",
                        GTK_SIGNAL_FUNC (on_script_clist_select_row),
                        (gpointer)SEL_PRESSED);
  gtk_signal_connect (GTK_OBJECT (script_clist), "unselect_row",
                        GTK_SIGNAL_FUNC (on_script_clist_select_row),
                        (gpointer)UNSEL_PRESSED);
  return scripts;
}

/* gnome UI variables */
#include "updates_icon.xpm"
#include "home_icon.xpm"
#include "alias_icon.xpm"
#include "scripts_icon.xpm"
#include "send_icon.xpm"

static GnomeUIInfo file_menu_uiinfo[] = {
    GNOMEUIINFO_MENU_NEW_ITEM(N_("_Add provider File"), NULL, on_new_file_activate, NULL),
    {
     GNOME_APP_UI_ITEM, N_("_Go to SmsSend home page for more scripts"),
     NULL,
     on_goto_smssend_activate, NULL, NULL,
     GNOME_APP_PIXMAP_DATA, home_icon,
     0, 0, NULL},
    {
     GNOME_APP_UI_ITEM, N_("_Execute SmsSend for delayed connections"),
     NULL,
     on_sendnow_activate, NULL, NULL,
     GNOME_APP_PIXMAP_DATA, send_icon,
     0, 0, NULL},
    GNOMEUIINFO_SEPARATOR,
    GNOMEUIINFO_MENU_CLOSE_ITEM(on_close_activate, NULL),
    GNOMEUIINFO_MENU_EXIT_ITEM(on_exit_activate, NULL),
    GNOMEUIINFO_END
};

static GnomeUIInfo settings_menu_uiinfo[] = {
    {
     GNOME_APP_UI_ITEM, N_("Manage _aliases"),
     NULL,
     on_manage_alias_activate, NULL, NULL,
     GNOME_APP_PIXMAP_DATA, alias_icon,
     0, 0, NULL},
    {
     GNOME_APP_UI_ITEM, N_("Manage _scripts"),
     NULL,
     on_manage_scripts_activate, NULL, NULL,
     GNOME_APP_PIXMAP_DATA, scripts_icon,
     0, 0, NULL},
    {
     GNOME_APP_UI_ITEM, N_("Check for _updates"),
     NULL,
     on_check_updates_activate, NULL, NULL,
     GNOME_APP_PIXMAP_DATA, updates_icon,
     0, 0, NULL},
    GNOMEUIINFO_SEPARATOR,
    GNOMEUIINFO_MENU_PREFERENCES_ITEM(on_preferences_activate, NULL),
    GNOMEUIINFO_END
};

static GnomeUIInfo help_menu_uiinfo[] = {
#ifndef DEVEL
    GNOMEUIINFO_HELP(PACKAGE),
    GNOMEUIINFO_SEPARATOR,
#endif
    GNOMEUIINFO_MENU_ABOUT_ITEM(on_about_activate, NULL),
    GNOMEUIINFO_END
};

static GnomeUIInfo menubar_uiinfo[] = {
    GNOMEUIINFO_MENU_FILE_TREE(file_menu_uiinfo),
    GNOMEUIINFO_MENU_SETTINGS_TREE(settings_menu_uiinfo),
    GNOMEUIINFO_MENU_HELP_TREE(help_menu_uiinfo),
    GNOMEUIINFO_END
};

GtkWidget *create_gnome_smssend()
{
    /* create the application main window */
    GtkWidget *smssend;
    GtkWidget *dock;

    Tooltips = gtk_tooltips_new();

    smssend = gnome_app_new(PACKAGE, PACKAGE " - " VERSION);
    gtk_window_set_wmclass(GTK_WINDOW(smssend), "Smssend", "SmsSend");

    dock = GNOME_APP(smssend)->dock;
    gtk_widget_show(dock);

    gnome_app_create_menus(GNOME_APP(smssend), menubar_uiinfo);

    Notebook = gtk_notebook_new();
    gtk_widget_show(Notebook);
    gnome_app_set_contents(GNOME_APP(smssend), Notebook);
    gtk_notebook_set_scrollable (GTK_NOTEBOOK (Notebook), TRUE);
    gtk_container_set_border_width(GTK_CONTAINER(Notebook), 2);
    gtk_notebook_set_tab_hborder (GTK_NOTEBOOK (Notebook), 2);
    gtk_notebook_set_tab_vborder (GTK_NOTEBOOK (Notebook), 2);

    Appbar = gnome_appbar_new(TRUE, TRUE, GNOME_PREFERENCES_NEVER);
    gtk_widget_show(Appbar);
    gnome_app_set_statusbar(GNOME_APP(smssend), Appbar);

    gtk_signal_connect (GTK_OBJECT (smssend), "destroy",
                        GTK_SIGNAL_FUNC (on_exit_activate),
                        NULL);
    return smssend;
}


PProvider_t create_provider(const char filename[])
{
    /* create a provider tab in the notebook */
    PProvider_t Pv;

    /* read provider script */
    Pv = CreateProviderFromFile(filename);

    if (Pv == NULL)
	return (NULL);

    /* update provider's list */
    g_hash_table_insert(ProviderHash,(gpointer) filename, (gpointer) Pv);
    NumProvider++;
    if ( (SavedProviderHash) && (g_hash_table_lookup (SavedProviderHash, (gconstpointer) filename) == NULL) )
	/* if not saved don't display it but keep it for further use */
	return (NULL);
    show_provider((gchar *)filename,Pv);
    return (Pv);
}

void show_provider(gchar *filename,PProvider_t Pv)
{
    /* create a provider tab in the notebook */
    GtkWidget *provider_box;
    GtkWidget *param_frame;
    GtkWidget *table_params;
    GtkWidget *rightbox;
    GtkWidget *remaining;
    GtkWidget *send_button;
    GtkWidget *provider_label;
    GtkWidget *message=(GtkWidget *)NULL;

    gchar *label, *dot;

    int i, msgindex = -1;

    /* filename is to be shown */
    label = g_strdup(g_basename(filename));
    dot   = (gchar *)g_extension_pointer (label);
    *--dot = '\000';
    /* use provider name only as the label */
    provider_label = gtk_label_new(label);

    /* update shown provider's list */
    g_hash_table_insert(ShownProviderHash,(gpointer) filename, (gpointer) Pv);

    /* save filename and label in structure for DoSms() */
    Pv->Label    = label;
    Pv->Filename = g_strdup(filename);

    provider_box = gtk_hbox_new(FALSE, 2);
    gtk_widget_show(provider_box);
    gtk_container_add(GTK_CONTAINER(Notebook), provider_box);
    gtk_container_set_border_width(GTK_CONTAINER(provider_box), 2);
    /* save provider_box in structure */
    Pv->PageWidget = provider_box;

    param_frame = gtk_frame_new(_("Parameters"));
    gtk_widget_show(param_frame);
    gtk_box_pack_start(GTK_BOX(provider_box), param_frame, TRUE, TRUE, 0);
    gtk_container_set_border_width(GTK_CONTAINER(param_frame), 2);
    gtk_frame_set_label_align(GTK_FRAME(param_frame), 0.02, 0.5);

    table_params = gtk_table_new(Pv->NbParams, 2, FALSE);
    gtk_widget_show(table_params);
    gtk_container_add(GTK_CONTAINER(param_frame), table_params);
    gtk_container_set_border_width(GTK_CONTAINER(table_params), 2);
    gtk_table_set_row_spacings(GTK_TABLE(table_params), 2);
    gtk_table_set_col_spacings(GTK_TABLE(table_params), 2);

    /* create entries for the parameters */
    for (i = 0; i < Pv->NbParams; i++)
    {
	GtkWidget *param_label;
	GtkWidget *param_value;

	if (g_strncasecmp(Pv->Params[i].Name, "Message",9) == 0)
	{
	  /* message parameter is treated below */
	  msgindex = i;
	  goto end_for;
	}
	param_label = gtk_label_new(Pv->Params[i].Name);
	gtk_widget_show(param_label);
	gtk_table_attach(GTK_TABLE(table_params), param_label, 0, 1, i, i + 1,
			 (GtkAttachOptions) (0), (GtkAttachOptions) (0), 2, 2);
	gtk_label_set_justify(GTK_LABEL(param_label), GTK_JUSTIFY_LEFT);
	gtk_misc_set_padding(GTK_MISC(param_label), 2, 2);

	if (Pv->Params[i].Size)
	    param_value = gtk_entry_new_with_max_length(Pv->Params[i].Size);
	else
	    param_value = gtk_entry_new();
	gtk_widget_show(param_value);

	if (Pv->Params[i].Hidden)
	    gtk_entry_set_visibility(GTK_ENTRY(param_value), FALSE);
	else
	{
            /* Attach events to check for aliases */
    	    gtk_signal_connect(GTK_OBJECT(param_value), "focus-out-event", 
                      GTK_SIGNAL_FUNC(on_value_check), (gpointer) &Pv->Params[i]);
    	    gtk_signal_connect(GTK_OBJECT(param_value), "activate", 
                      GTK_SIGNAL_FUNC(on_value_check), (gpointer) &Pv->Params[i]);
	}
	gtk_table_attach(GTK_TABLE(table_params), param_value, 1, 2, i, i + 1,
			 (GtkAttachOptions) (GTK_EXPAND | GTK_FILL), (GtkAttachOptions) (0), 2, 2);

	Pv->Params[i].Widget = param_value;

	if (Pv->Params[i].Help)
	    gtk_tooltips_set_tip(Tooltips, param_value, Pv->Params[i].Help, NULL);
	if (KeepPass && g_strncasecmp(Pv->Params[i].Name, "Password",10) == 0)
	{
	  gchar *config_file;
	  gchar *path;
	  gchar *password;
	  /* password is perhaps in preference file */
	  config_file=gnome_util_prepend_user_home(".smssend/smssendrc");
	  path = g_strdup_printf("=%s=/Passwords/%s",config_file,label);
	  password  = gnome_config_get_string_with_default(path,NULL);
	  if (password && strlen(password))
	  {
		crypt_password(password);
		Pv->Params[i].Value = password;
	  }
	  g_free(path);
	  g_free(config_file);	
	}
	if (Pv->Params[i].Value)
	    gtk_entry_set_text(GTK_ENTRY(param_value), Pv->Params[i].Value);
end_for:
    }

    rightbox = gtk_vbox_new(FALSE, 2);
    gtk_widget_show(rightbox);
    gtk_box_pack_start(GTK_BOX(provider_box), rightbox, TRUE, TRUE, 0);
    gtk_container_set_border_width(GTK_CONTAINER(rightbox), 2);

    /* message widget and remaining widget value */
    if (msgindex != -1)
    {
    	GtkWidget *remainbox;
	GtkWidget *remainlabl;
	char remain[4];

	message = gtk_text_new(NULL, NULL);
	gtk_widget_show(message);
	gtk_box_pack_start(GTK_BOX(rightbox), message, TRUE, FALSE, 0);
	gtk_text_set_editable(GTK_TEXT(message), TRUE);

	remainbox = gtk_hbox_new(TRUE, 2);
	gtk_widget_show(remainbox);
	gtk_box_pack_start(GTK_BOX(rightbox), remainbox, FALSE, TRUE, 0);
	gtk_container_set_border_width(GTK_CONTAINER(remainbox), 2);

	remainlabl = gtk_label_new(_("Remaining characters"));
	gtk_widget_show(remainlabl);
	gtk_box_pack_start(GTK_BOX(remainbox), remainlabl, FALSE, FALSE, 0);
	gtk_label_set_justify(GTK_LABEL(remainlabl), GTK_JUSTIFY_RIGHT);

	remaining = gtk_entry_new_with_max_length(3);
 	gtk_widget_show(remaining);
	gtk_box_pack_start(GTK_BOX(remainbox), remaining, FALSE, FALSE, 2);
	gtk_entry_set_editable(GTK_ENTRY(remaining), FALSE);
	gtk_tooltips_set_tip(Tooltips, message, Pv->Params[msgindex].Help, NULL);
	g_snprintf(remain, 5, "%d", Pv->Params[msgindex].Size);
	gtk_entry_set_text(GTK_ENTRY(remaining), remain);
	Pv->Params[msgindex].Widget = message;
        gtk_signal_connect(GTK_OBJECT(message), "changed",
		       GTK_SIGNAL_FUNC(on_message_changed), (gpointer) Pv->Params[msgindex].Size);
	gtk_object_set_data(GTK_OBJECT(message), "remain", remaining);
    }

    /* send button */
    send_button = gnome_stock_button(GNOME_STOCK_BUTTON_APPLY);
    gtk_widget_show(send_button);
    gtk_tooltips_set_tip(Tooltips, send_button, _("Send SMS !"), NULL);
    gtk_box_pack_start(GTK_BOX(rightbox), send_button, FALSE, FALSE, 0);
    gtk_container_set_border_width(GTK_CONTAINER(send_button), 2);
    gtk_widget_set_name(send_button,"send_button");
    gtk_signal_connect(GTK_OBJECT(send_button), "pressed", GTK_SIGNAL_FUNC(on_send_pressed), (gpointer) Pv);

    /* modify notebook's label */
    gtk_widget_show(provider_label);
    gtk_notebook_set_tab_label(GTK_NOTEBOOK(Notebook), provider_box, provider_label);

    /* cross save Pv and widget */
    Pv->MessageWidget = message;
    gtk_object_set_data(GTK_OBJECT(provider_box), "provider", Pv);
}

static void scan_dir(gchar *dirname)
{
    DIR           *dir;
    struct dirent *dirent;
    gchar         *ext;
    PProvider_t   Pv = NULL;

    dir = opendir(dirname);
    if (!dir)
	return;
    while ((dirent = readdir(dir)))
    {
	/* Search for SMSSEND extension */
	ext = (gchar *)g_extension_pointer(dirent->d_name);
	if ((strlen(ext) == strlen(SMSSEND_SUF)) && (g_strncasecmp(ext,SMSSEND_SUF,3) == 0))
	{
    	    Pv = create_provider(g_strconcat(dirname,"/",dirent->d_name,NULL));
	}
    }        
    closedir(dir);
}

void create_notebook()
{
    /* Create all notebook's tabs */
    gchar *dirname;
    gchar *status;

    if (ScriptFilename)
    {
    	gchar         *ext;
    	PProvider_t   Pv = NULL;
	/* Search for SMSSEND extension */
	ext = (gchar *)g_extension_pointer(ScriptFilename);
	if ((strlen(ext) == strlen(SMSSEND_SUF)) && (g_strncasecmp(ext,SMSSEND_SUF,3) == 0))
	{
    	    Pv = create_provider(ScriptFilename);
	}
    }
    else
    {
    	if (ScriptPath) /* shouldn't be set together with 'only-user' option */
    	{
    		scan_dir(ScriptPath);
    	} 

    	/* First try to load local .sms script */
    	scan_dir(g_get_current_dir());
    	/* Then try to load user's scripts */
    	scan_dir(gnome_util_prepend_user_home(".smssend/"));
	if (!MyScripts)
	{
    	    /* Then try to load global shared scripts */
    	    scan_dir(SMSSEND_SHAREPATH);
    	    /* Use an environment variable */
    	    if ( (dirname = getenv("SMSSEND_SCRIPTPATH")) != NULL)    
    	    {
        	    scan_dir(dirname);
    	    }
	}
    }
    
    if (NumProvider == 0)
    {
	g_printerr(_("No provider file found!\n"));
    	gtk_exit(-1);
    }

    status = g_strdup_printf(_("%s : %d provider(s) found"),PACKAGE,NumProvider);
    gnome_appbar_set_status(GNOME_APPBAR(Appbar), status);
    g_free(status);
}
