#
# Copyright 2010 Martin Owens
#
# This program is free software: you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation, either version 3 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program.  If not, see <http://www.gnu.org/licenses/>
#
"""
Sets up some common gtk windows for generic use.
"""

import os
import time
import logging

from GroundControl.gtkviews import ThreadedWindow, GtkApp, Window, ChildWindow

class StatusWindow(ThreadedWindow):
    """Display a status bar to the user"""
    name = 'status_window'

    def load(self, *args, **kwargs):
        """Load our status window"""
        # hopefully filter out variables
        self.gapp.load_vars(args, kwargs)
        self.window.set_title(self.gapp.task_name)
        self.update()
        self._close_thread = False
        super(StatusWindow, self).load(*args, **kwargs)

    def inital_thread(self):
        """Pass on the action"""
        self.gapp.do_work()

    def thread_exited(self):
        """What to do when a thread exits"""
        self.done = True
        self.destroy_window()

    def update(self, value=None, name=None, transport=None):
        """Update gtk after some time"""
        # Update our progress bar and make sure the widgets still exist
        if name and self.widget('label'):
            self.widget('label').set_markup('<b>%s</b>' % name)
        if value != None and self.widget('progress'):
            self.widget('progress').set_fraction(value)
        if self.widget('transport'):
            if transport:
                self.widget('transport').set_text(transport)
            else:
                self.widget('transport').set_text('')

    def pop_error(self, error, title=None):
        """Popup a Gtk Error message box of some kind"""
        PopupApp(title=unicode(title), message=unicode(error))


class PromptWindow(ChildWindow):
    """A simple prompt window"""
    name = 'prompt'

    def load(self, title, label):
        """what to do when we load"""
        self.window.set_title(title)
        self.widget('label').set_text(label)

    def hold(self):
        """Hold the prompt until answered"""
        while not self.dead:
            time.sleep(0.5)
        return self.done


class StatusApp(GtkApp):
    """An application for showing any status"""
    gtkfile = 'common.glade'
    windows = [ StatusWindow, PromptWindow ]
    task_name = _("Unknown Action")

    def do_work(self, *args, **kwargs):
        """Replace this method in your own status updates"""
        raise NotImplimentedError("Replace do_work in your status window")

    def load_vars(self):
        """Strip out any useful variables for the status window"""
        # This is a bit of a juggling act, perhaps someone has a better idea
        # That we can move towards for this.
        pass

    @property
    def window(self):
        """Return the status window"""
        # Get the right window by name, more than one window could be loaded
        return self._loaded['status_window']

    def update(self, *args, **kwargs):
        """Pass an update call back to the window in a thread friendly way"""
        # We're going to make sure that updates don't collide
        kwargs['unique_call'] = True
        # Now call the threaded caller to do an update
        self.window.call('update', *args, **kwargs )

    def pop_error(self, title, error):
        """Call window's creation of popup windows"""
        return self.window.call('pop_error', error, title)

class PopupWindow(Window):
    """A simple popup warning window"""
    name = 'popup'

    def load(self, *args, **kwargs):
        """Load the title, icon"""
        self.window.set_title(kwargs.pop('title', _('Message!')))
        message = kwargs.pop('message')
        logging.error(message)
        self.widget('label').set_text(message)
        super(PopupWindow, self).load(*args, **kwargs)


class PopupApp(GtkApp):
    """An application for showing any status"""
    gtkfile = 'common-popup.glade'
    windows = [ PopupWindow ]

