#
# Copyright 2009 Martin Owens
#
# This program is free software: you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation, either version 3 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program.  If not, see <http://www.gnu.org/licenses/>
#
"""
Small app for search for and selecting launchpad bugs
"""

# Import standard python libs
import os
import gtk
import gtk.gdk
import pango
import logging

# Various required variables and locations
from GroundControl.gtkviews import (
    GtkApp, ThreadedWindow,
    IconManager, TreeView
)
from GroundControl.launchpad import LaunchpadBugs, get_launchpad
from GroundControl.projects import ProjectCreateApp

ICONMAN = IconManager('bugs')

class SelectionWindow(ThreadedWindow):
    """Select a bug to import."""
    name = 'bug_search'

    def load(self, project, *args, **kwargs):
        """Load the bug selection GUI"""
        self.slist      = None
        self.child      = None
        self._project   = project
        self._selected  = None
        self._launchpad = None
        self._ensure    = kwargs.pop('ensure_project', None)
        # Setup the list of bugs from a search
        self.unselected()
        self.slist = BugsView(self.widget('bugslist'),
            selected=self.selected,
            unselected=self.unselected)
        super(SelectionWindow, self).load(*args, **kwargs)

    @property
    def launchpad(self):
        """Return a launchpad object"""
        if not self._launchpad:
            self._launchpad = get_launchpad()
        return self._launchpad

    def inital_thread(self):
        """What to run when we execute the thread."""
        # Set up the launchpad bugs object and connect search
        lpa = self.launchpad.launchpad
        # Make sure we can get a project if required
        if type(self._project) in (str, unicode):
            try:
                self._project = lpa.projects[self._project]
            except KeyError:
                self._project = None
        # Get a bugs object to do searches
        self.lpp = LaunchpadBugs(lpa, project=self._project)
        # Tie up the project search
        self.lpp.connect_signal("search_finish", self.call, 'finish_search')
        self.lpp.connect_signal("search_result", self.call, 'add_search_item')
        self.call('finish_search')

    def add_search_item(self, item):
        """Add an item to the slist"""
        return self.slist.add_item(item)

    def signals(self):
        """bug window signals"""
        return {
            'search'  : self.bug_search,
            'project' : self.select_project,
        }

    def selected(self, item=None):
        """An item has surely been selected."""
        if item:
            self._selected = item
            self.widget('buttonok').set_sensitive(True)
        return self._selected

    def unselected(self):
        """All items unselected"""
        self._selected = None
        self.widget('buttonok').set_sensitive(False)

    def select_project(self):
        """Select the required project"""
        pass
        # TODO: link up with the project selection gui in order
        # To allow the user to do a real bug search.

    def get_args(self):
        """Return out bug number"""
        return {
            'bug'     : self._selected,
            'project' : self._project,
        }

    def bug_search(self, button=None):
        """Search for bugs in launchpad."""
        self.slist.clear()
        self.widget("findButton").hide()
        self.widget("term").hide()
        self.widget("searchLabel").set_text(
            _("Searching Launchpad - please wait..."))
        result = self.widget("term").get_text()
        #Trad: Thread(target=self.lpp.search, args=(result))
        self.start_thread(self.lpp.search, result)
        #Old: self.lpp.search(result)

    def finish_search(self, widget=None):
        """Event of finishing the search process."""
        self.widget("findButton").show()
        self.widget("term").show()
        if self._project:
            self.widget("searchLabel").set_text(_("Name or Bug ID:"))
            self.widget("findButton").set_label(_("Find in %s") % self._project.name)
        else:
            self.widget("searchLabel").set_text(_("Bug ID:"))
            self.widget("findButton").set_label(_("Get Bug"))

    def is_valid(self):
        """Return true is a bug is selected"""
        return self._selected != None

    def post_process(self):
        """Create a project is required"""
        if not self._project and self._ensure:
            project = self._selected.project
            path = os.path.join(self._ensure, project.id)
            if not os.path.exists(path):
                ProjectCreateApp(path=self._ensure,
                    project=project, start_loop=True,
                    parent=self.window)


class BugSelection(GtkApp):
    """Application for loading a bug"""
    gtkfile = 'bug-search.glade'
    windows = [ SelectionWindow ]


class BugsView(TreeView):
    """Controls and operates a table as a bugs view."""
    def setup(self):
        """Setup a treeview for showing services"""
        def text_cell_func(cell_layout, renderer, model, item_iter):
            """Render the text of the services tree view"""
            item = model.get_value(item_iter, 0)
            markup = "<b><i>%s</i></b>" % (item.id)
            renderer.set_property("markup", markup)

        def number_cell_func(cell_layout, renderer, model, item_iter):
            """Render the text of the services tree view"""
            item = model.get_value(item_iter, 0)
            name = str(item.name).replace('&', '&amp;')
            renderer.set_property("markup", name)

        def icon_cell_func(column, cell, model, item_iter):
            """Reender the icons of the services tree view"""
            item = model.get_value(item_iter, 0)
            img = ICONMAN.get_icon(item.task.importance.lower())
            img2 = img.scale_simple(24, 24, gtk.gdk.INTERP_BILINEAR)
            cell.set_property("pixbuf", img2)
            cell.set_property("visible", True)

        svlist = super(BugsView, self).setup()
        column = gtk.TreeViewColumn((_("Bug")))
        column.set_sizing(gtk.TREE_VIEW_COLUMN_AUTOSIZE)
        column.set_expand(False)
        column.set_resizable(True)
        # The icon
        renderer_icon = gtk.CellRendererPixbuf()
        renderer_icon.set_property("ypad", 4)
        renderer_icon.set_property("xpad", 2)
        column.pack_start(renderer_icon, False)
        column.set_cell_data_func(renderer_icon, icon_cell_func)
        # The name
        renderer = gtk.CellRendererText()
        column.pack_start(renderer, True)
        column.set_cell_data_func(renderer, text_cell_func)
        # Add the required coluns to the treeview
        svlist.append_column(column)
        # Bug number
        column2 = gtk.TreeViewColumn(("Title"))
        column2.set_sizing(gtk.TREE_VIEW_COLUMN_AUTOSIZE)
        column2.set_resizable(True)
        renderer = gtk.CellRendererText()
        column2.pack_start(renderer, True)
        column2.set_cell_data_func(renderer, number_cell_func)
        # Add the required coluns to the treeview
        svlist.append_column(column2)


