/*
    GQ -- a GTK-based LDAP client
    Copyright (C) 1998-2003 Bert Vermeulen
    Copyright (C) 2002-2003 Peter Stamfest <peter@stamfest.at>

    This program is released under the Gnu General Public License with
    the additional exemption that compiling, linking, and/or using
    OpenSSL is allowed.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

/* $Id: ref-browse.c,v 1.10 2003/11/03 21:07:57 stamfest Exp $ */

#include <glib.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>

#include <errno.h>
#include <string.h>
#include <stdlib.h>		/* free */

#include <config.h>

#include "common.h"

#include "dn-browse.h"
#include "server-browse.h"
#include "ref-browse.h"

#include "configfile.h"		/* free_ldapserver */
#include "prefs.h"		/* create_edit_server_window */

#include "util.h"
#include "i18n.h"
#include "errorchain.h"
#include "encode.h"
#include "browse-export.h"

/**************************************************************************/



static void free_ref_browse_entry(browse_entry *e) 
{
     ref_browse_entry *entry;
     if (!e) return;
     assert(IS_REF_ENTRY(e));
     entry = REF_BROWSE_ENTRY(e);

     g_free_if(entry->uri);
     if (entry->server) {
	  ldapserver_unref(entry->server);	/* removes from transient list as well */
	  entry->server = NULL;
     }

     g_free(entry);
}

/*
 * a ref browse entry was selected in the tree widget.
 *
 * put up some info.
 */
static void ref_browse_entry_selected(browse_entry *be,
				      int error_context,
				      GtkCTree *ctree,
				      GtkCTreeNode *node,
				      struct tab *tab)
{
     GtkWidget *pane2_scrwin, *pane2_vbox, *label, *e;
     GtkWidget *table;
     int row = 0;
     ref_browse_entry *entry;

     assert(IS_REF_ENTRY(be));
     entry = REF_BROWSE_ENTRY(be);

     record_path(tab, (browse_entry *) entry, ctree, node);

     pane2_scrwin = BROWSETAB(tab)->pane2_scrwin;
     pane2_vbox = BROWSETAB(tab)->pane2_vbox;

     /*  	  gtk_widget_destroy(pane2_vbox); */
     /* remove the viewport of the scrolled window. This should
	_really_ destroy the widgets below it. The pane2_scrwin
	is a GtkBin Object and thus has only one child, use this
	to obtain the viewport */

     gtk_container_remove(GTK_CONTAINER(pane2_scrwin),
			  GTK_BIN(pane2_scrwin)->child);

     pane2_vbox = gtk_vbox_new(FALSE, 2);
     BROWSETAB(tab)->pane2_vbox = pane2_vbox;

     gtk_widget_show(pane2_vbox);
     gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(pane2_scrwin),
					   pane2_vbox);

     table = gtk_table_new(5, 2, FALSE);
     gtk_container_border_width(GTK_CONTAINER(table), 5);
     gtk_widget_show(table);
     gtk_box_pack_start(GTK_BOX(pane2_vbox), table, FALSE, FALSE, 5);

     /* URI */
     label = gtk_label_new(_("Referral URI"));
     gtk_widget_show(label);
     gtk_table_attach(GTK_TABLE(table),
		      label,
		      0, 1, row, row+1,
		      GTK_SHRINK,
		      GTK_EXPAND | GTK_SHRINK | GTK_FILL,
		      0, 0);

     e = gtk_entry_new();
     gtk_entry_set_text(GTK_ENTRY(e), entry->uri);
     gtk_widget_set_sensitive(e, FALSE);
     gtk_widget_show(e);
     gtk_table_attach(GTK_TABLE(table),
		      e,
		      1, 2, row, row+1,
		      GTK_EXPAND | GTK_SHRINK | GTK_FILL,
		      GTK_EXPAND | GTK_SHRINK | GTK_FILL,
		      0, 0);
     row++;
}

static void ref_browse_entry_expand(browse_entry *be,
				    int error_context,
				    GtkCTree *ctree,
				    GtkCTreeNode *node,
				    struct tab *tab)
{
     ref_browse_entry *entry;
     assert(IS_REF_ENTRY(be));
     entry = REF_BROWSE_ENTRY(be);

     if (!entry->expanded) {
	  LDAPURLDesc *desc = NULL;

	  while (GTK_CTREE_ROW(node)->children) {
	       gtk_ctree_remove_node(ctree, GTK_CTREE_ROW(node)->children);
	  }

	  if (ldap_url_parse(entry->uri, &desc) == 0) {
	       struct ldapserver *parent = NULL, *newserver = NULL;
	       const char *labels[] = { desc->lud_dn, NULL };
	       char *dummy[] = { "dummy", NULL };
	       GtkCTreeNode *new_item, *added = NULL;
	       browse_entry *new_entry;

	       
	       /* find parent server */
	       GtkCTreeRow *row = NULL;
	       GtkCTreeNode *n;
	       browse_entry *e;
	       
	       n = GTK_CTREE_ROW(node)->parent;
	       for ( ; n ; n = row->parent ) {
		    row = GTK_CTREE_ROW(n);
		    e = (browse_entry *) 
			 gtk_ctree_node_get_row_data(ctree, n);
		    
		    /* FIXME: This is not OO */
		    if (IS_SERVER_ENTRY(e)) {
			 parent = ((server_browse_entry*) e)->server;
			 break;
		    }
		    if (IS_REF_ENTRY(e)) {
			 parent = ((ref_browse_entry*) e)->server;
			 break;
		    }
	       }

	       if (!parent) {
		    return;
	       }

	       newserver = get_referral_server(error_context, 
					       parent, entry->uri);

		  
#if 0

	       const char *labels[] = { desc->lud_dn, NULL };
	       char *dummy[] = { "dummy", NULL };
	       struct ldapserver *server, *newserver;
	       GtkCTreeNode *new_item, *added = NULL;
	       browse_entry *new_entry;



	       GString *new_uri = 
		    g_string_sized_new(strlen(entry->uri));

	       g_string_sprintf(new_uri, "%s://%s:%d/", 
				desc->lud_scheme,
				desc->lud_host,
				desc->lud_port);

	       newserver = new_ldapserver();

	       /* some sensible settings for the "usual" case:
		  Anonymous bind. Also show referrals */
	       newserver->ask_pw   = 0;
	       newserver->show_ref = 1;

#warning "ADD CONFIG FOR EXTENDED REFERENCE CHASING"
	       /* check: do we have this server around already??? */
	       server = server_by_canon_name(new_uri->str, TRUE);

	       if (!server) {
		    /* find parent server */
		    GtkCTreeRow *row = NULL;
		    GtkCTreeNode *n;
		    browse_entry *e;

		    n = GTK_CTREE_ROW(node)->parent;
		    for ( ; n ; n = row->parent ) {
			 row = GTK_CTREE_ROW(n);
			 e = (browse_entry *) 
			      gtk_ctree_node_get_row_data(ctree, n);

			 /* FIXME: This is not OO */
			 if (IS_SERVER_ENTRY(e)) {
			      server = ((server_browse_entry*) e)->server;
			      break;
			 }
			 if (IS_REF_ENTRY(e)) {
			      server = ((ref_browse_entry*) e)->server;
			      break;
			 }
		    }
	       }

	       if (server) {
		    copy_ldapserver(newserver, server);
		    statusbar_msg(_("Initialized temporary server-definition '%1$s' from existing server '%2$s'"), new_uri->str, server->name);
	       } else {
		    statusbar_msg(_("Created temporary server-definition '%1$s' with no pre-set values."), new_uri->str);
	       }

	       g_free_and_dup(newserver->name,     new_uri->str);
	       g_free_and_dup(newserver->ldaphost, new_uri->str);
	       g_free_and_dup(newserver->basedn,   desc->lud_dn);
#endif
	       newserver->quiet = 1;
	       canonicalize_ldapserver(newserver);

	       transient_add_server(newserver);

	       entry->server = newserver;
	       ldapserver_ref(newserver);

	       entry->expanded = TRUE;

	       gtk_clist_freeze(GTK_CLIST(ctree));

	       new_entry = new_dn_browse_entry(desc->lud_dn);

	       added = gtk_ctree_insert_node(ctree,
					     node, NULL,
					     (char**) labels,  /* bug in the GTK2 API: should be const */
					     0,
					     NULL, NULL, NULL, NULL,
					     FALSE, FALSE);
	       
	       gtk_ctree_node_set_row_data_full(ctree,
						added,
						new_entry,
						(GtkDestroyNotify) destroy_browse_entry);
	       
	       /* add dummy node */
	       new_item = gtk_ctree_insert_node(ctree,
						added, NULL,
						dummy, 
						0,
						NULL, NULL, NULL, NULL,
						TRUE, FALSE);
	       
	       gtk_clist_thaw(GTK_CLIST(ctree));

	       ldap_free_urldesc(desc);
	  }
     }
}

static char* ref_browse_entry_get_name(browse_entry *entry, gboolean long_form)
{
     char *g;
#if GTK_MAJOR < 2
     char *l;
#endif

     assert(IS_REF_ENTRY(entry));

#if GTK_MAJOR >= 2
     g = g_strdup(REF_BROWSE_ENTRY(entry)->uri);
#else
     l = decoded_string(REF_BROWSE_ENTRY(entry)->uri);
     /* impedance match -> malloc to g_malloc */
     g = g_strdup(l);
     free(l);
#endif

     return g;
}

static void ref_browse_entry_refresh(browse_entry *entry,
				     int error_context, 
				     GtkCTree *ctree,
				     GtkCTreeNode *node,
				     struct tab *tab)
{
     assert(IS_REF_ENTRY(entry));

     REF_BROWSE_ENTRY(entry)->expanded = 0;

     gtk_clist_freeze(GTK_CLIST(ctree));

     ref_browse_entry_selected(entry, error_context, ctree, node, tab);

     /* toggle expansion twice to fire the expand callback and to
	return to the current expansion state */
     gtk_ctree_toggle_expansion(ctree, node);
     gtk_ctree_toggle_expansion(ctree, node);

/*       server_browse_entry_expand(entry, ctree, node, tab); */

     gtk_clist_thaw(GTK_CLIST(ctree));

}

static void add_to_permanent_servers(GtkWidget *menu_item, 
				     struct ldapserver *server)
{
     /* no assertion, it could happen... */
     if (is_transient_server(server)) {
	  int ctx = error_new_context(_("Adding server permanently"),
				      menu_item);
	  
	  if (server_by_name(server->name) == NULL) {
	       struct ldapserver *s = new_ldapserver();
	       copy_ldapserver(s, server);
	       config_add_server(config, s);
	       if (save_config_ext(ctx)) {
		    update_serverlist(&mainwin);
	       } else {
		    /* save failed - undo changes */
		    config_remove_server(config, s);
	       }
	  } else {
	       error_push(ctx,
			  _("Another server with the name '%s' already exists."),
			  server->name);
	  }
	  /* popup error, if any */
	  error_flush(ctx);
     }
}


struct edit_server_cb_data {
     struct ldapserver *server;
     struct tab *tab;
};

static void free_edit_server_cb_data(struct edit_server_cb_data *cbd)
{
     ldapserver_unref(cbd->server);
     g_free(cbd);
}

static void edit_server_activated(struct edit_server_cb_data *cbd)
{
     create_edit_server_window(cbd->server, cbd->tab->win->mainwin);
}

static void dump_ref(GtkWidget *widget, struct tab *tab)
{
     GtkCTree *ctree;
     GtkCTreeNode *node;
     browse_entry *e;
     struct ldapserver *server;
     GList *bases = NULL;
     GList *to_export = NULL, *I;
     struct dn_on_server *dos;
     int error_context;

     ctree = BROWSETAB(tab)->ctreeroot;
     node = BROWSETAB(tab)->tree_row_popped_up;
     e = (browse_entry *) gtk_ctree_node_get_row_data(ctree, node);

     assert(IS_REF_ENTRY(e));

     server = server_from_node(ctree, node);

     if (e == NULL || server == NULL)
	  return;

     error_context = error_new_context(_("Exporting referred-to server/DN to LDIF"),
				       tab->win->mainwin);

     bases = get_suffixes(error_context, ((ref_browse_entry *)e)->server);

     /* turn suffixes list into a list of dn_on_server objects
	(impedance match) */
     for (I = g_list_first(bases) ; I ; I = g_list_next(I) ) {
	  dos = new_dn_on_server(I->data, server);
	  dos->flags = LDAP_SCOPE_SUBTREE; /* default is LDAP_SCOPE_BASE */
	  to_export = g_list_append(to_export, dos);
	  g_free(I->data);
	  I->data = NULL;
     }
     g_list_free(bases);

     export_many(error_context, tab->win->mainwin, to_export);

     error_flush(error_context);

}


static void ref_browse_entry_popup(browse_entry *entry,
				   GtkWidget *menu, 
				   GtkWidget *ctreeroot,
				   GtkCTreeNode *ctree_node,
				   struct tab *tab) 
{
     GtkWidget *menu_item;
     struct ldapserver *server;
     struct edit_server_cb_data *cbd;

     assert(IS_REF_ENTRY(entry));
     
     server = server_from_node(GTK_CTREE(ctreeroot), ctree_node);

     /* Edit Server settings */
     menu_item = gtk_menu_item_new_with_label(_("Edit Server"));
     gtk_menu_append(GTK_MENU(menu), menu_item);
     gtk_widget_show(menu_item);
	  
     cbd = (struct edit_server_cb_data *)
	  g_malloc0(sizeof(struct edit_server_cb_data));
     cbd->server = server;
     cbd->tab = tab;

     ldapserver_ref(server); /* not strictly necessary, but ... */

     gtk_signal_connect_object(GTK_OBJECT(menu_item), "activate",
			       GTK_SIGNAL_FUNC(edit_server_activated),
			       (gpointer) cbd);

     /* explicitly attach cbd to assure call to destructor */
     gtk_object_set_data_full(GTK_OBJECT(menu_item), "cbd", 
			      cbd, (GtkDestroyNotify)free_edit_server_cb_data);

     gtk_widget_show(menu_item);

     if (server == NULL) {
	  gtk_widget_set_sensitive(menu_item, FALSE);
     }

     /* Add to permanent list of servers */
     menu_item = gtk_menu_item_new_with_label(_("Add to permanent list of servers"));
     gtk_menu_append(GTK_MENU(menu), menu_item);
     gtk_widget_show(menu_item);
	  
     gtk_signal_connect(GTK_OBJECT(menu_item), "activate",
			GTK_SIGNAL_FUNC(add_to_permanent_servers),
			(gpointer) server);

     gtk_widget_show(menu_item);

     if (server == NULL || !is_transient_server(server)) {
	  gtk_widget_set_sensitive(menu_item, FALSE);
     }
     
     /* Export to LDIF */
     menu_item = gtk_menu_item_new_with_label(_("Export to LDIF"));
     gtk_menu_append(GTK_MENU(menu), menu_item);
     gtk_signal_connect(GTK_OBJECT(menu_item), "activate",
			GTK_SIGNAL_FUNC(dump_ref),
			(gpointer) tab);
     gtk_widget_show(menu_item);

     /* Close connection */
     menu_item = gtk_menu_item_new_with_label(_("Close Connection"));
     gtk_menu_append(GTK_MENU(menu), menu_item);
     gtk_signal_connect(GTK_OBJECT(menu_item), "activate",
			GTK_SIGNAL_FUNC(tree_row_close_connection),
			(gpointer) tab);
     gtk_widget_show(menu_item);

     if (server == NULL) {
	  gtk_widget_set_sensitive(menu_item, FALSE);
     }
}

static struct browse_entry_vtab ref_vtab = {
     free_ref_browse_entry,		/* destroy */
     ref_browse_entry_expand,		/* expand */
     ref_browse_entry_selected,		/* select */
     ref_browse_entry_refresh,		/* refresh */
     ref_browse_entry_get_name,		/* get_name */
     ref_browse_entry_popup,		/* popup */
};


browse_entry *new_ref_browse_entry(const char *uri) 
{
     ref_browse_entry *e;
     e = g_malloc0(sizeof(ref_browse_entry));

     e->type = REF_BROWSE_ENTRY_ID;
     e->base_methods = &ref_vtab;

/*      e->server = server; */

     e->uri = g_strdup(uri);

     return (browse_entry *) e;
}



/* 
   Local Variables:
   c-basic-offset: 5
   End:
*/
