#include "gpsshogi/gui/abstractBoard.h"
#include "osl/pieceStand.h"
#include <QPainter>

gpsshogi::gui::AbstractBoard::AbstractBoard(QWidget *parent)
  : QWidget(parent), selectedPiece(osl::Piece::EMPTY())
{
}

QSize gpsshogi::gui::
AbstractBoard::sizeHint() const
{
  return QSize(520, 520);
}

const QPixmap gpsshogi::gui::
AbstractBoard::pieceToImage(osl::Piece p) const
{
  QString prefix;
  if ((p.owner() == osl::BLACK && !reversed())
      || (p.owner() == osl::WHITE && reversed()))
  {
    prefix = "";
  }
  else
  {
    prefix = "r";
  }
  QString imagename;
  switch (p.ptype())
  {
  case osl::PAWN:
    imagename = "fu.png";
    break;
  case osl::LANCE:
    imagename = "kyo.png";
    break;
  case osl::KNIGHT:
    imagename = "kei.png";
    break;
  case osl::SILVER:
    imagename = "gin.png";
    break;
  case osl::GOLD:
    imagename = "kin.png";
    break;
  case osl::BISHOP:
    imagename = "kaku.png";
    break;
  case osl::ROOK:
    imagename = "hi.png";
    break;
  case osl::KING:
    imagename = "gyoku.png";
    break;
  case osl::PPAWN:
    imagename = "to.png";
    break;
  case osl::PLANCE:
    imagename = "nkyo.png";
    break;
  case osl::PKNIGHT:
    imagename = "nkei.png";
    break;
  case osl::PSILVER:
    imagename = "ngin.png";
    break;
  case osl::PBISHOP:
    imagename = "uma.png";
    break;
  case osl::PROOK:
    imagename = "ryu.png";
    break;
  default:
    abort();
  }
#if QT_VERSION >= 0x040000
  return QPixmap(":/images/" + prefix + imagename);
#else  
  return QPixmap::fromMimeSource(prefix + imagename);
#endif
}

void gpsshogi::gui::
AbstractBoard::paintPiece(QPainter *painter, int x, int y, osl::Piece piece)
{
  painter->drawPixmap(x, y, pieceToImage(piece));
}

void gpsshogi::gui::
AbstractBoard::paintEvent(QPaintEvent *)
{
  QPainter painter(this);

  for (int i = 0; i < 10; i++)
  {
    painter.drawLine(STAND_SIZE, i * BOX_SIZE + MARGIN_HEAD,
		     STAND_SIZE + BOX_SIZE * 9, i * BOX_SIZE + MARGIN_HEAD);
    painter.drawLine(STAND_SIZE + i * BOX_SIZE, 0 + MARGIN_HEAD,
		     STAND_SIZE + i * BOX_SIZE, BOX_SIZE * 9 + MARGIN_HEAD);
  }

  osl::misc::CArray<int, osl::PTYPE_MAX + 1> edit_pieces;
  edit_pieces.fill(0);
  for (int i = 0; i < 40; i++)
  {
    osl::Piece p = getPieceOf(i);
    if (!p.position().isValid())
    {
      edit_pieces[osl::unpromote(p.ptype())]++;
    }
    else if (p.isOnBoard() && p != selectedPiece)
    {
      QPoint point = positionToPoint(p.position());
      int x = point.x() + 10; // magic number to center pieces
      int y = point.y() + 10;

      paintPiece(&painter, x, y, p);
    }
  }
  // Black
  for (unsigned int i = 0; i < osl::PieceStand::order.size(); i++)
  {
    int y = BOX_SIZE * 2 + i * BOX_SIZE;
    osl::Ptype pt = osl::PieceStand::order[i];
    osl::Player player = !reversed() ? osl::BLACK : osl::WHITE;
    int count = countPiecesOnStand(player, pt);
    if (! selectedPiece.isEmpty() && selectedPiece.owner() == player
	&& selectedPiece.position().isValid()
	&& !selectedPiece.isOnBoard() && selectedPiece.ptype() == pt)
      count--;
    if (count > 0)
    {
      osl::Piece p(player, pt, 0, osl::Position::STAND());
      painter.drawPixmap(BLACK_STAND_X + 10, y + 10, pieceToImage(p));
      if (count != 1)
	painter.drawText(BLACK_STAND_X + 20, y + 30, QString(" %1").arg(count));
    }
  }
  // White
  for (unsigned int i = 0; i < osl::PieceStand::order.size(); i++)
  {
    int x = 0;
    int y = BOX_SIZE * 9 - i * BOX_SIZE;
    osl::Ptype pt = osl::PieceStand::order[i];
    osl::Player player = !reversed() ? osl::WHITE : osl::BLACK;
    int count = countPiecesOnStand(player, pt);
    if (! selectedPiece.isEmpty() && selectedPiece.owner() == player
	&& selectedPiece.position().isValid()
	&& !selectedPiece.isOnBoard() && selectedPiece.ptype() == pt)
      count--;
    if (count > 0)
    {
      osl::Piece p(player, pt, 0, osl::Position::STAND());
      painter.drawPixmap(x + 10, y + 10, pieceToImage(p));
      if (count != 1)
	painter.drawText(x + 20, y + 30, QString(" %1").arg(count));
    }
  }
  // Edit pieces
  if (edit_pieces[osl::KING] > 0)
  {
    int count = edit_pieces[osl::KING];
    if (! selectedPiece.isEmpty() && selectedPiece.owner() == osl::BLACK
	&& !selectedPiece.position().isValid()
	&& !selectedPiece.ptype() == osl::KING)
      count--;
    osl::Piece p(osl::BLACK, osl::KING, 0, osl::Position::STAND());
    painter.drawPixmap(STAND_SIZE + 10, 10 * BOX_SIZE + 10,
		       pieceToImage(p));
    if (count != 1)
      painter.drawText(STAND_SIZE + 10, 11 * BOX_SIZE + 10,
		       QString("%1").arg(count));
  }
  for (unsigned int i = 0; i < osl::PieceStand::order.size(); i++)
  {
    int count = edit_pieces[osl::PieceStand::order[i]];
    if (! selectedPiece.isEmpty() && selectedPiece.owner() == osl::BLACK
	&& !selectedPiece.position().isValid()
	&& !selectedPiece.ptype() == osl::KING)
      count--;
    osl::Piece p(osl::BLACK, osl::PieceStand::order[i], 0, osl::Position::STAND());
    if (count > 0)
      painter.drawPixmap(STAND_SIZE + 10 + BOX_SIZE * (i + 1), 10 * BOX_SIZE + 10,
			 pieceToImage(p));
    if (count > 1)
      painter.drawText(STAND_SIZE + 20 + BOX_SIZE * (i + 1), 11 * BOX_SIZE + 10,
		       QString("%1").arg(count));
  }
  painter.end();
}


bool gpsshogi::gui::
AbstractBoard::isOnBoard(int x, int y) const
{
  if ((STAND_SIZE <= x && x <= STAND_SIZE + 9 * BOX_SIZE)
      && (0 <= y && y <= BOX_SIZE * 9))
    return true;
  else
    return false;
}

const osl::Position gpsshogi::gui::
AbstractBoard::getPosition(int x, int y) const
{
  y -= MARGIN_HEAD;
  if (isOnBoard(x, y))
  {
    int board_x = 9 - (x - STAND_SIZE) / BOX_SIZE;
    int board_y = y / BOX_SIZE + 1;
    if (reversed())
    {
      board_x = 10 - board_x;
      board_y = 10 - board_y;
    }
    return osl::Position(board_x, board_y);
  }
  else if (STAND_SIZE <= x &&
	   x <= STAND_SIZE + BOX_SIZE * ((int)osl::PieceStand::order.size()  + 1) &&
	   y >= 10 * BOX_SIZE && y <= 11 * BOX_SIZE)
    return osl::Position(0, 0);
  else
    return osl::Position::STAND();
}

const osl::Piece gpsshogi::gui::
AbstractBoard::getPiece(int x, int y)
{
  y -= MARGIN_HEAD;
  if (isOnBoard(x, y))
  {
    int board_x = 9 - (x - STAND_SIZE) / BOX_SIZE;
    int board_y = y / BOX_SIZE + 1;
    if (reversed())
    {
      board_x = 10 - board_x;
      board_y = 10 - board_y;
    }
    return getPieceAt(osl::Position(board_x, board_y));
  }
  else if (STAND_SIZE <= x &&
	   x <= STAND_SIZE + BOX_SIZE * ((int)osl::PieceStand::order.size()  + 1) &&
	   y >= 10 * BOX_SIZE && y <= 11 * BOX_SIZE)
  {
    if (x <= STAND_SIZE + BOX_SIZE)
    {
      return getReservePiece(osl::KING);
    }
    return getReservePiece(osl::PieceStand::order[(x - STAND_SIZE) / BOX_SIZE - 1]);
  }
  else if (BLACK_STAND_X <= x && x <= BLACK_STAND_X + BOX_SIZE &&
	   y >= BOX_SIZE * 2 && y <= BOX_SIZE * 11)
  {
    return getStandPiece(osl::BLACK,
			 osl::PieceStand::order[(y - BOX_SIZE * 2) / BOX_SIZE]);
  }
  else if (0 <= x && x <= BOX_SIZE &&
	   y >= BOX_SIZE * 2 && y <= BOX_SIZE * 11)
  {
    return getStandPiece(osl::WHITE,
			 osl::PieceStand::order[osl::PieceStand::order.size() -
						(y - BOX_SIZE * 2) / BOX_SIZE]);
  }

  return osl::Piece::EMPTY();
}

const QPoint gpsshogi::gui::
AbstractBoard::positionToPoint(osl::Position position) const
{
  int x = position.x();
  int y = position.y();
  assert (1 <= x &&  x <= 9);
  assert (1 <= y &&  y <= 9);
  if (reversed())
  {
    x = 10 - x;
    y = 10 - y;
  }
  return QPoint(STAND_SIZE + BOX_SIZE * (9 - x),
		BOX_SIZE * (y - 1) + MARGIN_HEAD);
}
