/* gpx-read.c
 * Written by Daniel Foote.
 * Started Feb 2005.
 *
 * This file contains routines to read the XML GPX files,
 * containing GPS data.
 */

/* Copyright 2005 Daniel Foote.
 *
 * This file is part of gpscorrelate.
 *
 * gpscorrelate is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * gpscorrelate is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with gpscorrelate; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <stdio.h>
#include <string.h>
#include <locale.h>
#include <libxml/parser.h>
#include <libxml/tree.h>

#include "gpx-read.h"
#include "unixtime.h"
#include "gpsstructure.h"

struct GPSPoint* FirstPoint;
struct GPSPoint* LastPoint;

void ExtractTrackPoints(xmlNodePtr Start)
{
	/* The pointer passed to us should be the start
	 * of a heap of trkpt's. So walk though them,
	 * extracting what we need. */
	xmlNodePtr Current = NULL;
	xmlNodePtr CCurrent = NULL;
	xmlAttrPtr Properties = NULL;
	char* Lat;
	char* Long;
	char* Elev;
	char* Time;

	for (Current = Start; Current; Current = Current->next)
	{
		if ((Current->type == XML_ELEMENT_NODE) &&
			(strcmp(Current->name, "trkpt") == 0))
		{
			/* This is indeed a trackpoint. Extract! */

			/* Reset the vars... so we don't get
			 * the data from last run. */
			Lat = NULL;
			Long = NULL;
			Elev = NULL;
			Time = 0;
			
			/* To get the Lat and Long, we have to
			 * extract the properties... another 
			 * linked list to walk. */
			for (Properties = Current->properties;
					Properties;
					Properties = Properties->next)
			{
				if (strcmp(Properties->name, "lat") == 0)
				{
					Lat = Properties->children->content;
				}
				if (strcmp(Properties->name, "lon") == 0)
				{
					Long = Properties->children->content;
				}
			}

			/* Now, grab the elevation and time.
			 * These are children of trkpt. */
			/* Oh, and what's the deal with the
			 * Node->children->content thing? */
			for (CCurrent = Current->children;
					CCurrent;
					CCurrent = CCurrent->next)
			{
				if (strcmp(CCurrent->name, "ele") == 0)
				{
					Elev = CCurrent->children->content;
				}
				if (strcmp(CCurrent->name, "time") == 0)
				{
					Time = CCurrent->children->content;
				}
			}

			/* Right, now we theoretically have all the data.
			 * Allocate ourselves some memory and go for it... */
			if (FirstPoint)
			{
				/* Ok, adding to the list... */
				LastPoint->Next = malloc(sizeof(struct GPSPoint));
				LastPoint = LastPoint->Next;
				LastPoint->Next = NULL;
			} else {
				/* This is the first one. */
				FirstPoint = malloc(sizeof(struct GPSPoint));
				LastPoint = FirstPoint;
			}

			/* Clear the structure first... */
			LastPoint->Lat = 0;
			LastPoint->Long = 0;
			LastPoint->Elev = 0;
			LastPoint->Time = 0;
			LastPoint->EndOfSegment = 0;

			/* Write the data into LastPoint, which should be a new point. */
			LastPoint->Lat = atof(Lat);
			LastPoint->Long = atof(Long);
			LastPoint->Elev = atof(Elev);
			LastPoint->Time = ConvertToUnixTime(Time, GPX_DATE_FORMAT, 0, 0);
			
			/* Debug...
			printf("TrackPoint. Lat %s (%f), Long %s (%f). Elev %s (%f), Time %d.\n",
					Lat, atof(Lat), Long, atof(Long), Elev, atof(Elev),
					ConvertToUnixTime(Time, GPX_DATE_FORMAT, 0, 0));
			*/
			
					
		}
	} /* End For. */

	/* Return control to the recursive function... */
}

void FindTrackSeg(xmlNodePtr Start)
{
	/* Go recursive till we find a <trgseg> tag. */
	xmlNodePtr Current = NULL;

	for (Current = Start; Current; Current = Current->next)
	{
		if ((Current->type == XML_ELEMENT_NODE) &&
			(strcmp(Current->name, "trkseg") == 0))
		{
			/* Found it... the children should
			 * all be trkpt's. */
			ExtractTrackPoints(Current->children);
			
			/* Mark the last point as being the end
			 * of a track segment. */
			if (LastPoint) LastPoint->EndOfSegment = 1;
			
		}
		
		/* And again, with children of this node. */
		FindTrackSeg(Current->children);
		
	} /* End For */

}


struct GPSPoint* ReadGPX(char* File)
{
	/* Init the libxml library. Also checks version. */
	LIBXML_TEST_VERSION

	xmlDocPtr GPXData;
	
	/* Read the GPX data from file. */
	GPXData = xmlParseFile(File);
	
	if (GPXData == NULL)
	{
		fprintf(stderr, "Failed to parse GPX data from %s.\n", File);
		return 0;
	}

	/* Now grab the "root" node. */
	xmlNodePtr GPXRoot;
	GPXRoot = xmlDocGetRootElement(GPXData);

	if (GPXRoot == NULL)
	{
		fprintf(stderr, "GPX file has no root. Not healthy.\n");
		return 0;
	}

	/* Check that this is indeed a GPX - the root node
	 * should be "gpx". */
	if (strcmp(GPXRoot->name, "gpx") == 0)
	{
		/* Ok, it is a GPX file. */
	} else {
		/* Not valid. */
		fprintf(stderr, "Invalid GPX file.\n");
		return 0;
	}

	/* Now comes the messy part... walking the tree to find
	 * what we want.
	 * I've chosen to do it with two functions, one of which
	 * is recursive, rather than a clever inside-this-function
	 * walk the tree thing.
	 * 
	 * We start by calling the recursive function to look for
	 * <trkseg> tags, and then that function calls another
	 * when it has found one... this sub function then
	 * hauls out the <trkpt> tags with the actual data.
	 * Messy, convoluted, but it seems to work... */
	/* As to where to store the data? Again, its messy.
	 * We maintain two global vars, FirstPoint and LastPoint.
	 * FirstPoint points to the first GPSPoint done, and
	 * LastPoint is the last point done, used for the next
	 * point... we use this to build a singly-linked list. */
	/* (I think I'll just be grateful for the work that libxml
	 * puts in for me... imagine having to write an XML parser!
	 * Nasty.) */
	/* Before we go into this function, we also setlocale to "C".
	 * The GPX def indicates that the decimal seperator should be
	 * ".", but certain locales specify otherwise. Which has caused issues.
	 * So we set the locale for this function, and then revert it.
	 */
	
	FirstPoint = NULL;
	LastPoint = NULL;
	
	char* OldLocale = setlocale(LC_NUMERIC, NULL);
	setlocale(LC_NUMERIC, "C");
	
	FindTrackSeg(GPXRoot);

	setlocale(LC_NUMERIC, OldLocale);

	/* Clean up stuff for the XML library. */
	xmlFreeDoc(GPXData);
	xmlCleanupParser();

	return FirstPoint;
};


void FreePointList(struct GPSPoint* List)
{
	/* Free the memory associated with the
	 * GPSPoint list... */
	struct GPSPoint* NextFree = NULL;
	struct GPSPoint* CurrentFree = List;
	while (1)
	{
		if (CurrentFree == NULL) break;
		NextFree = CurrentFree->Next;
		free(CurrentFree);
		CurrentFree = NextFree;
	}
};
		
